/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AmazonWebServiceRequest;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a single raw email using Amazon SES. For more information, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-raw.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class SendRawEmailRequest extends AmazonWebServiceRequest implements
        ToCopyableBuilder<SendRawEmailRequest.Builder, SendRawEmailRequest> {
    private final String source;

    private final List<String> destinations;

    private final RawMessage rawMessage;

    private final String fromArn;

    private final String sourceArn;

    private final String returnPathArn;

    private final List<MessageTag> tags;

    private final String configurationSetName;

    private SendRawEmailRequest(BuilderImpl builder) {
        this.source = builder.source;
        this.destinations = builder.destinations;
        this.rawMessage = builder.rawMessage;
        this.fromArn = builder.fromArn;
        this.sourceArn = builder.sourceArn;
        this.returnPathArn = builder.returnPathArn;
        this.tags = builder.tags;
        this.configurationSetName = builder.configurationSetName;
    }

    /**
     * <p>
     * The identity's email address. If you do not provide a value for this parameter, you must specify a "From" address
     * in the raw text of the message. (You can also specify both.)
     * </p>
     * <p>
     * By default, the string must be 7-bit ASCII. If the text must contain any other characters, then you must use MIME
     * encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the following form:
     * <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
     * href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
     * </p>
     * <note>
     * <p>
     * If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
     * complaints will be sent to this email address. This takes precedence over any Return-Path header that you might
     * include in the raw text of the message.
     * </p>
     * </note>
     * 
     * @return The identity's email address. If you do not provide a value for this parameter, you must specify a "From"
     *         address in the raw text of the message. (You can also specify both.)</p>
     *         <p>
     *         By default, the string must be 7-bit ASCII. If the text must contain any other characters, then you must
     *         use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the
     *         following form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
     *         href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
     *         complaints will be sent to this email address. This takes precedence over any Return-Path header that you
     *         might include in the raw text of the message.
     *         </p>
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
     */
    public List<String> destinations() {
        return destinations;
    }

    /**
     * <p>
     * The raw text of the message. The client is responsible for ensuring the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Message must contain a header and a body, separated by a blank line.
     * </p>
     * </li>
     * <li>
     * <p>
     * All required header fields must be present.
     * </p>
     * </li>
     * <li>
     * <p>
     * Each part of a multipart MIME message must be formatted properly.
     * </p>
     * </li>
     * <li>
     * <p>
     * MIME content types must be among those supported by Amazon SES. For more information, go to the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer Guide</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must be base64-encoded.
     * </p>
     * </li>
     * <li>
     * <p>
     * Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length of each line
     * of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The raw text of the message. The client is responsible for ensuring the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Message must contain a header and a body, separated by a blank line.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         All required header fields must be present.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Each part of a multipart MIME message must be formatted properly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MIME content types must be among those supported by Amazon SES. For more information, go to the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer
     *         Guide</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must be base64-encoded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length of
     *         each line of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
     *         </p>
     *         </li>
     */
    public RawMessage rawMessage() {
        return rawMessage;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to specify a particular "From" address in the header of the raw
     * email.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message of the
     * email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon SES uses the
     * value of the <code>FromArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a href=
     * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     * >Amazon SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to specify a particular "From" address in the
     *         header of the raw email.</p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message
     *         of the email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon
     *         SES uses the value of the <code>FromArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a href=
     *         "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public String fromArn() {
        return fromArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to send for the email address specified in the <code>Source</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
     * <code>user@example.com</code>.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw message of the
     * email. If you use both the <code>SourceArn</code> parameter and the corresponding X-header, Amazon SES uses the
     * value of the <code>SourceArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a href=
     * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     * >Amazon SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to send for the email address specified in the
     *         <code>Source</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to send from <code>user@example.com</code>, then you would specify the
     *         <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
     *         the <code>Source</code> to be <code>user@example.com</code>.
     *         </p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw
     *         message of the email. If you use both the <code>SourceArn</code> parameter and the corresponding
     *         X-header, Amazon SES uses the value of the <code>SourceArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a href=
     *         "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the <code>ReturnPath</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
     * <code>feedback@example.com</code>.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw message
     * of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding X-header, Amazon SES
     * uses the value of the <code>ReturnPathArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a href=
     * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     * >Amazon SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>ReturnPath</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to use <code>feedback@example.com</code>, then you would specify the
     *         <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
     *         and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
     *         </p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw
     *         message of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding
     *         X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a href=
     *         "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public String returnPathArn() {
        return returnPathArn;
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send using
     * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
     * publish email sending events.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send using
     *         <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you
     *         can publish email sending events.
     */
    public List<MessageTag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
     * </p>
     * 
     * @return The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
     */
    public String configurationSetName() {
        return configurationSetName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(destinations());
        hashCode = 31 * hashCode + Objects.hashCode(rawMessage());
        hashCode = 31 * hashCode + Objects.hashCode(fromArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(returnPathArn());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendRawEmailRequest)) {
            return false;
        }
        SendRawEmailRequest other = (SendRawEmailRequest) obj;
        return Objects.equals(source(), other.source()) && Objects.equals(destinations(), other.destinations())
                && Objects.equals(rawMessage(), other.rawMessage()) && Objects.equals(fromArn(), other.fromArn())
                && Objects.equals(sourceArn(), other.sourceArn()) && Objects.equals(returnPathArn(), other.returnPathArn())
                && Objects.equals(tags(), other.tags()) && Objects.equals(configurationSetName(), other.configurationSetName());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (source() != null) {
            sb.append("Source: ").append(source()).append(",");
        }
        if (destinations() != null) {
            sb.append("Destinations: ").append(destinations()).append(",");
        }
        if (rawMessage() != null) {
            sb.append("RawMessage: ").append(rawMessage()).append(",");
        }
        if (fromArn() != null) {
            sb.append("FromArn: ").append(fromArn()).append(",");
        }
        if (sourceArn() != null) {
            sb.append("SourceArn: ").append(sourceArn()).append(",");
        }
        if (returnPathArn() != null) {
            sb.append("ReturnPathArn: ").append(returnPathArn()).append(",");
        }
        if (tags() != null) {
            sb.append("Tags: ").append(tags()).append(",");
        }
        if (configurationSetName() != null) {
            sb.append("ConfigurationSetName: ").append(configurationSetName()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Source":
            return Optional.of(clazz.cast(source()));
        case "Destinations":
            return Optional.of(clazz.cast(destinations()));
        case "RawMessage":
            return Optional.of(clazz.cast(rawMessage()));
        case "FromArn":
            return Optional.of(clazz.cast(fromArn()));
        case "SourceArn":
            return Optional.of(clazz.cast(sourceArn()));
        case "ReturnPathArn":
            return Optional.of(clazz.cast(returnPathArn()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        case "ConfigurationSetName":
            return Optional.of(clazz.cast(configurationSetName()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, SendRawEmailRequest> {
        /**
         * <p>
         * The identity's email address. If you do not provide a value for this parameter, you must specify a "From"
         * address in the raw text of the message. (You can also specify both.)
         * </p>
         * <p>
         * By default, the string must be 7-bit ASCII. If the text must contain any other characters, then you must use
         * MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the following
         * form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
         * href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         * </p>
         * <note>
         * <p>
         * If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
         * complaints will be sent to this email address. This takes precedence over any Return-Path header that you
         * might include in the raw text of the message.
         * </p>
         * </note>
         * 
         * @param source
         *        The identity's email address. If you do not provide a value for this parameter, you must specify a
         *        "From" address in the raw text of the message. (You can also specify both.)</p>
         *        <p>
         *        By default, the string must be 7-bit ASCII. If the text must contain any other characters, then you
         *        must use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax
         *        uses the following form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
         *        href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces
         *        and complaints will be sent to this email address. This takes precedence over any Return-Path header
         *        that you might include in the raw text of the message.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * </p>
         * 
         * @param destinations
         *        A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<String> destinations);

        /**
         * <p>
         * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * </p>
         * 
         * @param destinations
         *        A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(String... destinations);

        /**
         * <p>
         * The raw text of the message. The client is responsible for ensuring the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Message must contain a header and a body, separated by a blank line.
         * </p>
         * </li>
         * <li>
         * <p>
         * All required header fields must be present.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each part of a multipart MIME message must be formatted properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * MIME content types must be among those supported by Amazon SES. For more information, go to the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer Guide</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must be base64-encoded.
         * </p>
         * </li>
         * <li>
         * <p>
         * Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length of each
         * line of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param rawMessage
         *        The raw text of the message. The client is responsible for ensuring the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Message must contain a header and a body, separated by a blank line.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        All required header fields must be present.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Each part of a multipart MIME message must be formatted properly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MIME content types must be among those supported by Amazon SES. For more information, go to the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer
         *        Guide</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must be base64-encoded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length
         *        of each line of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rawMessage(RawMessage rawMessage);

        /**
         * <p>
         * The raw text of the message. The client is responsible for ensuring the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Message must contain a header and a body, separated by a blank line.
         * </p>
         * </li>
         * <li>
         * <p>
         * All required header fields must be present.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each part of a multipart MIME message must be formatted properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * MIME content types must be among those supported by Amazon SES. For more information, go to the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer Guide</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must be base64-encoded.
         * </p>
         * </li>
         * <li>
         * <p>
         * Per <a href="https://tools.ietf.org/html/rfc5321#section-4.5.3.1.6">RFC 5321</a>, the maximum length of each
         * line of text, including the &lt;CRLF&gt;, must not exceed 1,000 characters.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link RawMessage.Builder} avoiding the need to create
         * one manually via {@link RawMessage#builder()}.
         *
         * When the {@link Consumer} completes, {@link RawMessage.Builder#build()} is called immediately and its result
         * is passed to {@link #rawMessage(RawMessage)}.
         * 
         * @param rawMessage
         *        a consumer that will call methods on {@link RawMessage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rawMessage(RawMessage)
         */
        default Builder rawMessage(Consumer<RawMessage.Builder> rawMessage) {
            return rawMessage(RawMessage.builder().apply(rawMessage).build());
        }

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to specify a particular "From" address in the header of the
         * raw email.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message of
         * the email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon SES uses
         * the value of the <code>FromArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a href=
         * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param fromArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to specify a particular "From"
         *        address in the header of the raw email.</p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw
         *        message of the email. If you use both the <code>FromArn</code> parameter and the corresponding
         *        X-header, Amazon SES uses the value of the <code>FromArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromArn(String fromArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to send for the email address specified in the
         * <code>Source</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
         * <code>user@example.com</code>.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw message of
         * the email. If you use both the <code>SourceArn</code> parameter and the corresponding X-header, Amazon SES
         * uses the value of the <code>SourceArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a href=
         * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param sourceArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to send for the email address
         *        specified in the <code>Source</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to send from <code>user@example.com</code>, then you would specify the
         *        <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
         *        the <code>Source</code> to be <code>user@example.com</code>.
         *        </p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw
         *        message of the email. If you use both the <code>SourceArn</code> parameter and the corresponding
         *        X-header, Amazon SES uses the value of the <code>SourceArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>ReturnPath</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
         * <code>feedback@example.com</code>.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw
         * message of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding
         * X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a href=
         * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param returnPathArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>ReturnPath</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to use <code>feedback@example.com</code>, then you would specify the
         *        <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
         *        and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
         *        </p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the
         *        raw message of the email. If you use both the <code>ReturnPathArn</code> parameter and the
         *        corresponding X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPathArn(String returnPathArn);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that
         *        you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<MessageTag> tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that
         *        you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(MessageTag... tags);

        /**
         * <p>
         * The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);
    }

    static final class BuilderImpl implements Builder {
        private String source;

        private List<String> destinations;

        private RawMessage rawMessage;

        private String fromArn;

        private String sourceArn;

        private String returnPathArn;

        private List<MessageTag> tags;

        private String configurationSetName;

        private BuilderImpl() {
        }

        private BuilderImpl(SendRawEmailRequest model) {
            source(model.source);
            destinations(model.destinations);
            rawMessage(model.rawMessage);
            fromArn(model.fromArn);
            sourceArn(model.sourceArn);
            returnPathArn(model.returnPathArn);
            tags(model.tags);
            configurationSetName(model.configurationSetName);
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final Collection<String> getDestinations() {
            return destinations;
        }

        @Override
        public final Builder destinations(Collection<String> destinations) {
            this.destinations = AddressListCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(String... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        public final void setDestinations(Collection<String> destinations) {
            this.destinations = AddressListCopier.copy(destinations);
        }

        public final RawMessage.Builder getRawMessage() {
            return rawMessage != null ? rawMessage.toBuilder() : null;
        }

        @Override
        public final Builder rawMessage(RawMessage rawMessage) {
            this.rawMessage = rawMessage;
            return this;
        }

        public final void setRawMessage(RawMessage.BuilderImpl rawMessage) {
            this.rawMessage = rawMessage != null ? rawMessage.build() : null;
        }

        public final String getFromArn() {
            return fromArn;
        }

        @Override
        public final Builder fromArn(String fromArn) {
            this.fromArn = fromArn;
            return this;
        }

        public final void setFromArn(String fromArn) {
            this.fromArn = fromArn;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        public final String getReturnPathArn() {
            return returnPathArn;
        }

        @Override
        public final Builder returnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
            return this;
        }

        public final void setReturnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
        }

        public final Collection<MessageTag.Builder> getTags() {
            return tags != null ? tags.stream().map(MessageTag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<MessageTag> tags) {
            this.tags = MessageTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(MessageTag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        public final void setTags(Collection<MessageTag.BuilderImpl> tags) {
            this.tags = MessageTagListCopier.copyFromBuilder(tags);
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        @Override
        public SendRawEmailRequest build() {
            return new SendRawEmailRequest(this);
        }
    }
}
