/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the event destination that the specified email sending events will be published to.
 * </p>
 * <note>
 * <p>
 * When you create or update an event destination, you must provide one, and only one, destination. The destination can
 * be Amazon CloudWatch, Amazon Kinesis Firehose or Amazon Simple Notification Service (Amazon SNS).
 * </p>
 * </note>
 * <p>
 * Event destinations are associated with configuration sets, which enable you to publish email sending events to Amazon
 * CloudWatch, Amazon Kinesis Firehose, or Amazon Simple Notification Service (Amazon SNS). For information about using
 * configuration sets, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/monitor-sending-activity.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class EventDestination implements ToCopyableBuilder<EventDestination.Builder, EventDestination> {
    private final String name;

    private final Boolean enabled;

    private final List<String> matchingEventTypes;

    private final KinesisFirehoseDestination kinesisFirehoseDestination;

    private final CloudWatchDestination cloudWatchDestination;

    private final SNSDestination snsDestination;

    private EventDestination(BuilderImpl builder) {
        this.name = builder.name;
        this.enabled = builder.enabled;
        this.matchingEventTypes = builder.matchingEventTypes;
        this.kinesisFirehoseDestination = builder.kinesisFirehoseDestination;
        this.cloudWatchDestination = builder.cloudWatchDestination;
        this.snsDestination = builder.snsDestination;
    }

    /**
     * <p>
     * The name of the event destination. The name must:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     * </p>
     * </li>
     * <li>
     * <p>
     * Contain less than 64 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the event destination. The name must:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Contain less than 64 characters.
     *         </p>
     *         </li>
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Sets whether Amazon SES publishes events to this destination when you send an email with the associated
     * configuration set. Set to <code>true</code> to enable publishing to this destination; set to <code>false</code>
     * to prevent publishing to this destination. The default value is <code>false</code>.
     * </p>
     * 
     * @return Sets whether Amazon SES publishes events to this destination when you send an email with the associated
     *         configuration set. Set to <code>true</code> to enable publishing to this destination; set to
     *         <code>false</code> to prevent publishing to this destination. The default value is <code>false</code>.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The type of email sending events to publish to the event destination.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The type of email sending events to publish to the event destination.
     */
    public List<EventType> matchingEventTypes() {
        return TypeConverter.convert(matchingEventTypes, EventType::fromValue);
    }

    /**
     * <p>
     * The type of email sending events to publish to the event destination.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The type of email sending events to publish to the event destination.
     */
    public List<String> matchingEventTypesStrings() {
        return matchingEventTypes;
    }

    /**
     * <p>
     * An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis Firehose
     * event destination.
     * </p>
     * 
     * @return An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
     *         Firehose event destination.
     */
    public KinesisFirehoseDestination kinesisFirehoseDestination() {
        return kinesisFirehoseDestination;
    }

    /**
     * <p>
     * An object that contains the names, default values, and sources of the dimensions associated with an Amazon
     * CloudWatch event destination.
     * </p>
     * 
     * @return An object that contains the names, default values, and sources of the dimensions associated with an
     *         Amazon CloudWatch event destination.
     */
    public CloudWatchDestination cloudWatchDestination() {
        return cloudWatchDestination;
    }

    /**
     * <p>
     * An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon SNS) event
     * destination.
     * </p>
     * 
     * @return An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon SNS)
     *         event destination.
     */
    public SNSDestination snsDestination() {
        return snsDestination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(matchingEventTypesStrings());
        hashCode = 31 * hashCode + Objects.hashCode(kinesisFirehoseDestination());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchDestination());
        hashCode = 31 * hashCode + Objects.hashCode(snsDestination());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventDestination)) {
            return false;
        }
        EventDestination other = (EventDestination) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(enabled(), other.enabled())
                && Objects.equals(matchingEventTypesStrings(), other.matchingEventTypesStrings())
                && Objects.equals(kinesisFirehoseDestination(), other.kinesisFirehoseDestination())
                && Objects.equals(cloudWatchDestination(), other.cloudWatchDestination())
                && Objects.equals(snsDestination(), other.snsDestination());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (enabled() != null) {
            sb.append("Enabled: ").append(enabled()).append(",");
        }
        if (matchingEventTypesStrings() != null) {
            sb.append("MatchingEventTypes: ").append(matchingEventTypesStrings()).append(",");
        }
        if (kinesisFirehoseDestination() != null) {
            sb.append("KinesisFirehoseDestination: ").append(kinesisFirehoseDestination()).append(",");
        }
        if (cloudWatchDestination() != null) {
            sb.append("CloudWatchDestination: ").append(cloudWatchDestination()).append(",");
        }
        if (snsDestination() != null) {
            sb.append("SNSDestination: ").append(snsDestination()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "Enabled":
            return Optional.of(clazz.cast(enabled()));
        case "MatchingEventTypes":
            return Optional.of(clazz.cast(matchingEventTypesStrings()));
        case "KinesisFirehoseDestination":
            return Optional.of(clazz.cast(kinesisFirehoseDestination()));
        case "CloudWatchDestination":
            return Optional.of(clazz.cast(cloudWatchDestination()));
        case "SNSDestination":
            return Optional.of(clazz.cast(snsDestination()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, EventDestination> {
        /**
         * <p>
         * The name of the event destination. The name must:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Contain less than 64 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the event destination. The name must:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Contain less than 64 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Sets whether Amazon SES publishes events to this destination when you send an email with the associated
         * configuration set. Set to <code>true</code> to enable publishing to this destination; set to
         * <code>false</code> to prevent publishing to this destination. The default value is <code>false</code>.
         * </p>
         * 
         * @param enabled
         *        Sets whether Amazon SES publishes events to this destination when you send an email with the
         *        associated configuration set. Set to <code>true</code> to enable publishing to this destination; set
         *        to <code>false</code> to prevent publishing to this destination. The default value is
         *        <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypes(Collection<String> matchingEventTypes);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypes(String... matchingEventTypes);

        /**
         * <p>
         * An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
         * Firehose event destination.
         * </p>
         * 
         * @param kinesisFirehoseDestination
         *        An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
         *        Firehose event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination);

        /**
         * <p>
         * An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
         * Firehose event destination.
         * </p>
         * This is a convenience that creates an instance of the {@link KinesisFirehoseDestination.Builder} avoiding the
         * need to create one manually via {@link KinesisFirehoseDestination#builder()}.
         *
         * When the {@link Consumer} completes, {@link KinesisFirehoseDestination.Builder#build()} is called immediately
         * and its result is passed to {@link #kinesisFirehoseDestination(KinesisFirehoseDestination)}.
         * 
         * @param kinesisFirehoseDestination
         *        a consumer that will call methods on {@link KinesisFirehoseDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kinesisFirehoseDestination(KinesisFirehoseDestination)
         */
        default Builder kinesisFirehoseDestination(Consumer<KinesisFirehoseDestination.Builder> kinesisFirehoseDestination) {
            return kinesisFirehoseDestination(KinesisFirehoseDestination.builder().apply(kinesisFirehoseDestination).build());
        }

        /**
         * <p>
         * An object that contains the names, default values, and sources of the dimensions associated with an Amazon
         * CloudWatch event destination.
         * </p>
         * 
         * @param cloudWatchDestination
         *        An object that contains the names, default values, and sources of the dimensions associated with an
         *        Amazon CloudWatch event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchDestination(CloudWatchDestination cloudWatchDestination);

        /**
         * <p>
         * An object that contains the names, default values, and sources of the dimensions associated with an Amazon
         * CloudWatch event destination.
         * </p>
         * This is a convenience that creates an instance of the {@link CloudWatchDestination.Builder} avoiding the need
         * to create one manually via {@link CloudWatchDestination#builder()}.
         *
         * When the {@link Consumer} completes, {@link CloudWatchDestination.Builder#build()} is called immediately and
         * its result is passed to {@link #cloudWatchDestination(CloudWatchDestination)}.
         * 
         * @param cloudWatchDestination
         *        a consumer that will call methods on {@link CloudWatchDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchDestination(CloudWatchDestination)
         */
        default Builder cloudWatchDestination(Consumer<CloudWatchDestination.Builder> cloudWatchDestination) {
            return cloudWatchDestination(CloudWatchDestination.builder().apply(cloudWatchDestination).build());
        }

        /**
         * <p>
         * An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon SNS)
         * event destination.
         * </p>
         * 
         * @param snsDestination
         *        An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon
         *        SNS) event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsDestination(SNSDestination snsDestination);

        /**
         * <p>
         * An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon SNS)
         * event destination.
         * </p>
         * This is a convenience that creates an instance of the {@link SNSDestination.Builder} avoiding the need to
         * create one manually via {@link SNSDestination#builder()}.
         *
         * When the {@link Consumer} completes, {@link SNSDestination.Builder#build()} is called immediately and its
         * result is passed to {@link #snsDestination(SNSDestination)}.
         * 
         * @param snsDestination
         *        a consumer that will call methods on {@link SNSDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #snsDestination(SNSDestination)
         */
        default Builder snsDestination(Consumer<SNSDestination.Builder> snsDestination) {
            return snsDestination(SNSDestination.builder().apply(snsDestination).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Boolean enabled;

        private List<String> matchingEventTypes;

        private KinesisFirehoseDestination kinesisFirehoseDestination;

        private CloudWatchDestination cloudWatchDestination;

        private SNSDestination snsDestination;

        private BuilderImpl() {
        }

        private BuilderImpl(EventDestination model) {
            name(model.name);
            enabled(model.enabled);
            matchingEventTypes(model.matchingEventTypes);
            kinesisFirehoseDestination(model.kinesisFirehoseDestination);
            cloudWatchDestination(model.cloudWatchDestination);
            snsDestination(model.snsDestination);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final Collection<String> getMatchingEventTypes() {
            return matchingEventTypes;
        }

        @Override
        public final Builder matchingEventTypes(Collection<String> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copy(matchingEventTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchingEventTypes(String... matchingEventTypes) {
            matchingEventTypes(Arrays.asList(matchingEventTypes));
            return this;
        }

        public final void setMatchingEventTypes(Collection<String> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copy(matchingEventTypes);
        }

        public final KinesisFirehoseDestination.Builder getKinesisFirehoseDestination() {
            return kinesisFirehoseDestination != null ? kinesisFirehoseDestination.toBuilder() : null;
        }

        @Override
        public final Builder kinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination) {
            this.kinesisFirehoseDestination = kinesisFirehoseDestination;
            return this;
        }

        public final void setKinesisFirehoseDestination(KinesisFirehoseDestination.BuilderImpl kinesisFirehoseDestination) {
            this.kinesisFirehoseDestination = kinesisFirehoseDestination != null ? kinesisFirehoseDestination.build() : null;
        }

        public final CloudWatchDestination.Builder getCloudWatchDestination() {
            return cloudWatchDestination != null ? cloudWatchDestination.toBuilder() : null;
        }

        @Override
        public final Builder cloudWatchDestination(CloudWatchDestination cloudWatchDestination) {
            this.cloudWatchDestination = cloudWatchDestination;
            return this;
        }

        public final void setCloudWatchDestination(CloudWatchDestination.BuilderImpl cloudWatchDestination) {
            this.cloudWatchDestination = cloudWatchDestination != null ? cloudWatchDestination.build() : null;
        }

        public final SNSDestination.Builder getSNSDestination() {
            return snsDestination != null ? snsDestination.toBuilder() : null;
        }

        @Override
        public final Builder snsDestination(SNSDestination snsDestination) {
            this.snsDestination = snsDestination;
            return this;
        }

        public final void setSNSDestination(SNSDestination.BuilderImpl snsDestination) {
            this.snsDestination = snsDestination != null ? snsDestination.build() : null;
        }

        @Override
        public EventDestination build() {
            return new EventDestination(this);
        }
    }
}
