/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Message-related information to include in the Delivery Status Notification (DSN) when an email that Amazon SES
 * receives on your behalf bounces.
 * </p>
 * <p>
 * For information about receiving email through Amazon SES, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class MessageDsn implements ToCopyableBuilder<MessageDsn.Builder, MessageDsn> {
    private final String reportingMta;

    private final Instant arrivalDate;

    private final List<ExtensionField> extensionFields;

    private MessageDsn(BuilderImpl builder) {
        this.reportingMta = builder.reportingMta;
        this.arrivalDate = builder.arrivalDate;
        this.extensionFields = builder.extensionFields;
    }

    /**
     * <p>
     * The reporting MTA that attempted to deliver the message, formatted as specified in <a
     * href="https://tools.ietf.org/html/rfc3464">RFC 3464</a> (<code>mta-name-type; mta-name</code>). The default value
     * is <code>dns; inbound-smtp.[region].amazonaws.com</code>.
     * </p>
     * 
     * @return The reporting MTA that attempted to deliver the message, formatted as specified in <a
     *         href="https://tools.ietf.org/html/rfc3464">RFC 3464</a> (<code>mta-name-type; mta-name</code>). The
     *         default value is <code>dns; inbound-smtp.[region].amazonaws.com</code>.
     */
    public String reportingMta() {
        return reportingMta;
    }

    /**
     * <p>
     * When the message was received by the reporting mail transfer agent (MTA), in <a
     * href="https://www.ietf.org/rfc/rfc0822.txt">RFC 822</a> date-time format.
     * </p>
     * 
     * @return When the message was received by the reporting mail transfer agent (MTA), in <a
     *         href="https://www.ietf.org/rfc/rfc0822.txt">RFC 822</a> date-time format.
     */
    public Instant arrivalDate() {
        return arrivalDate;
    }

    /**
     * <p>
     * Additional X-headers to include in the DSN.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Additional X-headers to include in the DSN.
     */
    public List<ExtensionField> extensionFields() {
        return extensionFields;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(reportingMta());
        hashCode = 31 * hashCode + Objects.hashCode(arrivalDate());
        hashCode = 31 * hashCode + Objects.hashCode(extensionFields());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageDsn)) {
            return false;
        }
        MessageDsn other = (MessageDsn) obj;
        return Objects.equals(reportingMta(), other.reportingMta()) && Objects.equals(arrivalDate(), other.arrivalDate())
                && Objects.equals(extensionFields(), other.extensionFields());
    }

    @Override
    public String toString() {
        return ToString.builder("MessageDsn").add("ReportingMta", reportingMta()).add("ArrivalDate", arrivalDate())
                .add("ExtensionFields", extensionFields()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReportingMta":
            return Optional.of(clazz.cast(reportingMta()));
        case "ArrivalDate":
            return Optional.of(clazz.cast(arrivalDate()));
        case "ExtensionFields":
            return Optional.of(clazz.cast(extensionFields()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, MessageDsn> {
        /**
         * <p>
         * The reporting MTA that attempted to deliver the message, formatted as specified in <a
         * href="https://tools.ietf.org/html/rfc3464">RFC 3464</a> (<code>mta-name-type; mta-name</code>). The default
         * value is <code>dns; inbound-smtp.[region].amazonaws.com</code>.
         * </p>
         * 
         * @param reportingMta
         *        The reporting MTA that attempted to deliver the message, formatted as specified in <a
         *        href="https://tools.ietf.org/html/rfc3464">RFC 3464</a> (<code>mta-name-type; mta-name</code>). The
         *        default value is <code>dns; inbound-smtp.[region].amazonaws.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportingMta(String reportingMta);

        /**
         * <p>
         * When the message was received by the reporting mail transfer agent (MTA), in <a
         * href="https://www.ietf.org/rfc/rfc0822.txt">RFC 822</a> date-time format.
         * </p>
         * 
         * @param arrivalDate
         *        When the message was received by the reporting mail transfer agent (MTA), in <a
         *        href="https://www.ietf.org/rfc/rfc0822.txt">RFC 822</a> date-time format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arrivalDate(Instant arrivalDate);

        /**
         * <p>
         * Additional X-headers to include in the DSN.
         * </p>
         * 
         * @param extensionFields
         *        Additional X-headers to include in the DSN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extensionFields(Collection<ExtensionField> extensionFields);

        /**
         * <p>
         * Additional X-headers to include in the DSN.
         * </p>
         * 
         * @param extensionFields
         *        Additional X-headers to include in the DSN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extensionFields(ExtensionField... extensionFields);
    }

    static final class BuilderImpl implements Builder {
        private String reportingMta;

        private Instant arrivalDate;

        private List<ExtensionField> extensionFields;

        private BuilderImpl() {
        }

        private BuilderImpl(MessageDsn model) {
            reportingMta(model.reportingMta);
            arrivalDate(model.arrivalDate);
            extensionFields(model.extensionFields);
        }

        public final String getReportingMta() {
            return reportingMta;
        }

        @Override
        public final Builder reportingMta(String reportingMta) {
            this.reportingMta = reportingMta;
            return this;
        }

        public final void setReportingMta(String reportingMta) {
            this.reportingMta = reportingMta;
        }

        public final Instant getArrivalDate() {
            return arrivalDate;
        }

        @Override
        public final Builder arrivalDate(Instant arrivalDate) {
            this.arrivalDate = arrivalDate;
            return this;
        }

        public final void setArrivalDate(Instant arrivalDate) {
            this.arrivalDate = arrivalDate;
        }

        public final Collection<ExtensionField.Builder> getExtensionFields() {
            return extensionFields != null ? extensionFields.stream().map(ExtensionField::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder extensionFields(Collection<ExtensionField> extensionFields) {
            this.extensionFields = ExtensionFieldListCopier.copy(extensionFields);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder extensionFields(ExtensionField... extensionFields) {
            extensionFields(Arrays.asList(extensionFields));
            return this;
        }

        public final void setExtensionFields(Collection<ExtensionField.BuilderImpl> extensionFields) {
            this.extensionFields = ExtensionFieldListCopier.copyFromBuilder(extensionFields);
        }

        @Override
        public MessageDsn build() {
            return new MessageDsn(this);
        }
    }
}
