/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to set whether Amazon SES includes the original email headers in the Amazon SNS notifications of
 * a specified type. For information about notifications, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/notifications-via-sns.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SetIdentityHeadersInNotificationsEnabledRequest extends SesRequest
        implements
        ToCopyableBuilder<SetIdentityHeadersInNotificationsEnabledRequest.Builder, SetIdentityHeadersInNotificationsEnabledRequest> {
    private static final SdkField<String> IDENTITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SetIdentityHeadersInNotificationsEnabledRequest::identity)).setter(setter(Builder::identity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Identity").build()).build();

    private static final SdkField<String> NOTIFICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SetIdentityHeadersInNotificationsEnabledRequest::notificationTypeAsString))
            .setter(setter(Builder::notificationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationType").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(SetIdentityHeadersInNotificationsEnabledRequest::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTITY_FIELD,
            NOTIFICATION_TYPE_FIELD, ENABLED_FIELD));

    private final String identity;

    private final String notificationType;

    private final Boolean enabled;

    private SetIdentityHeadersInNotificationsEnabledRequest(BuilderImpl builder) {
        super(builder);
        this.identity = builder.identity;
        this.notificationType = builder.notificationType;
        this.enabled = builder.enabled;
    }

    /**
     * <p>
     * The identity for which to enable or disable headers in notifications. Examples: <code>user@example.com</code>,
     * <code>example.com</code>.
     * </p>
     * 
     * @return The identity for which to enable or disable headers in notifications. Examples:
     *         <code>user@example.com</code>, <code>example.com</code>.
     */
    public String identity() {
        return identity;
    }

    /**
     * <p>
     * The notification type for which to enable or disable headers in notifications.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return The notification type for which to enable or disable headers in notifications.
     * @see NotificationType
     */
    public NotificationType notificationType() {
        return NotificationType.fromValue(notificationType);
    }

    /**
     * <p>
     * The notification type for which to enable or disable headers in notifications.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return The notification type for which to enable or disable headers in notifications.
     * @see NotificationType
     */
    public String notificationTypeAsString() {
        return notificationType;
    }

    /**
     * <p>
     * Sets whether Amazon SES includes the original email headers in Amazon SNS notifications of the specified
     * notification type. A value of <code>true</code> specifies that Amazon SES will include headers in notifications,
     * and a value of <code>false</code> specifies that Amazon SES will not include headers in notifications.
     * </p>
     * <p>
     * This value can only be set when <code>NotificationType</code> is already set to use a particular Amazon SNS
     * topic.
     * </p>
     * 
     * @return Sets whether Amazon SES includes the original email headers in Amazon SNS notifications of the specified
     *         notification type. A value of <code>true</code> specifies that Amazon SES will include headers in
     *         notifications, and a value of <code>false</code> specifies that Amazon SES will not include headers in
     *         notifications.</p>
     *         <p>
     *         This value can only be set when <code>NotificationType</code> is already set to use a particular Amazon
     *         SNS topic.
     */
    public Boolean enabled() {
        return enabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(identity());
        hashCode = 31 * hashCode + Objects.hashCode(notificationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SetIdentityHeadersInNotificationsEnabledRequest)) {
            return false;
        }
        SetIdentityHeadersInNotificationsEnabledRequest other = (SetIdentityHeadersInNotificationsEnabledRequest) obj;
        return Objects.equals(identity(), other.identity())
                && Objects.equals(notificationTypeAsString(), other.notificationTypeAsString())
                && Objects.equals(enabled(), other.enabled());
    }

    @Override
    public String toString() {
        return ToString.builder("SetIdentityHeadersInNotificationsEnabledRequest").add("Identity", identity())
                .add("NotificationType", notificationTypeAsString()).add("Enabled", enabled()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Identity":
            return Optional.ofNullable(clazz.cast(identity()));
        case "NotificationType":
            return Optional.ofNullable(clazz.cast(notificationTypeAsString()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SetIdentityHeadersInNotificationsEnabledRequest, T> g) {
        return obj -> g.apply((SetIdentityHeadersInNotificationsEnabledRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, SetIdentityHeadersInNotificationsEnabledRequest> {
        /**
         * <p>
         * The identity for which to enable or disable headers in notifications. Examples: <code>user@example.com</code>, <code>example.com</code>.
         * </p>
         * 
         * @param identity
         *        The identity for which to enable or disable headers in notifications. Examples:
         *        <code>user@example.com</code>, <code>example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identity(String identity);

        /**
         * <p>
         * The notification type for which to enable or disable headers in notifications.
         * </p>
         * 
         * @param notificationType
         *        The notification type for which to enable or disable headers in notifications.
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(String notificationType);

        /**
         * <p>
         * The notification type for which to enable or disable headers in notifications.
         * </p>
         * 
         * @param notificationType
         *        The notification type for which to enable or disable headers in notifications.
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(NotificationType notificationType);

        /**
         * <p>
         * Sets whether Amazon SES includes the original email headers in Amazon SNS notifications of the specified
         * notification type. A value of <code>true</code> specifies that Amazon SES will include headers in
         * notifications, and a value of <code>false</code> specifies that Amazon SES will not include headers in
         * notifications.
         * </p>
         * <p>
         * This value can only be set when <code>NotificationType</code> is already set to use a particular Amazon SNS
         * topic.
         * </p>
         * 
         * @param enabled
         *        Sets whether Amazon SES includes the original email headers in Amazon SNS notifications of the
         *        specified notification type. A value of <code>true</code> specifies that Amazon SES will include
         *        headers in notifications, and a value of <code>false</code> specifies that Amazon SES will not include
         *        headers in notifications.</p>
         *        <p>
         *        This value can only be set when <code>NotificationType</code> is already set to use a particular
         *        Amazon SNS topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String identity;

        private String notificationType;

        private Boolean enabled;

        private BuilderImpl() {
        }

        private BuilderImpl(SetIdentityHeadersInNotificationsEnabledRequest model) {
            super(model);
            identity(model.identity);
            notificationType(model.notificationType);
            enabled(model.enabled);
        }

        public final String getIdentity() {
            return identity;
        }

        @Override
        public final Builder identity(String identity) {
            this.identity = identity;
            return this;
        }

        public final void setIdentity(String identity) {
            this.identity = identity;
        }

        public final String getNotificationType() {
            return notificationType;
        }

        @Override
        public final Builder notificationType(String notificationType) {
            this.notificationType = notificationType;
            return this;
        }

        @Override
        public final Builder notificationType(NotificationType notificationType) {
            this.notificationType(notificationType.toString());
            return this;
        }

        public final void setNotificationType(String notificationType) {
            this.notificationType = notificationType;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SetIdentityHeadersInNotificationsEnabledRequest build() {
            return new SetIdentityHeadersInNotificationsEnabledRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
