/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the event destination that the specified email sending events will be published to.
 * </p>
 * <note>
 * <p>
 * When you create or update an event destination, you must provide one, and only one, destination. The destination can
 * be Amazon CloudWatch, Amazon Kinesis Firehose or Amazon Simple Notification Service (Amazon SNS).
 * </p>
 * </note>
 * <p>
 * Event destinations are associated with configuration sets, which enable you to publish email sending events to Amazon
 * CloudWatch, Amazon Kinesis Firehose, or Amazon Simple Notification Service (Amazon SNS). For information about using
 * configuration sets, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/monitor-sending-activity.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventDestination implements SdkPojo, Serializable,
        ToCopyableBuilder<EventDestination.Builder, EventDestination> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EventDestination::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(EventDestination::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<List<String>> MATCHING_EVENT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(EventDestination::matchingEventTypesAsStrings))
            .setter(setter(Builder::matchingEventTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchingEventTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<KinesisFirehoseDestination> KINESIS_FIREHOSE_DESTINATION_FIELD = SdkField
            .<KinesisFirehoseDestination> builder(MarshallingType.SDK_POJO)
            .getter(getter(EventDestination::kinesisFirehoseDestination))
            .setter(setter(Builder::kinesisFirehoseDestination))
            .constructor(KinesisFirehoseDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KinesisFirehoseDestination").build())
            .build();

    private static final SdkField<CloudWatchDestination> CLOUD_WATCH_DESTINATION_FIELD = SdkField
            .<CloudWatchDestination> builder(MarshallingType.SDK_POJO).getter(getter(EventDestination::cloudWatchDestination))
            .setter(setter(Builder::cloudWatchDestination)).constructor(CloudWatchDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchDestination").build())
            .build();

    private static final SdkField<SNSDestination> SNS_DESTINATION_FIELD = SdkField
            .<SNSDestination> builder(MarshallingType.SDK_POJO).getter(getter(EventDestination::snsDestination))
            .setter(setter(Builder::snsDestination)).constructor(SNSDestination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SNSDestination").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(NAME_FIELD, ENABLED_FIELD, MATCHING_EVENT_TYPES_FIELD,
                    KINESIS_FIREHOSE_DESTINATION_FIELD, CLOUD_WATCH_DESTINATION_FIELD, SNS_DESTINATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Boolean enabled;

    private final List<String> matchingEventTypes;

    private final KinesisFirehoseDestination kinesisFirehoseDestination;

    private final CloudWatchDestination cloudWatchDestination;

    private final SNSDestination snsDestination;

    private EventDestination(BuilderImpl builder) {
        this.name = builder.name;
        this.enabled = builder.enabled;
        this.matchingEventTypes = builder.matchingEventTypes;
        this.kinesisFirehoseDestination = builder.kinesisFirehoseDestination;
        this.cloudWatchDestination = builder.cloudWatchDestination;
        this.snsDestination = builder.snsDestination;
    }

    /**
     * <p>
     * The name of the event destination. The name must:
     * </p>
     * <ul>
     * <li>
     * <p>
     * This value can only contain ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     * </p>
     * </li>
     * <li>
     * <p>
     * Contain less than 64 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the event destination. The name must:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         This value can only contain ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Contain less than 64 characters.
     *         </p>
     *         </li>
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Sets whether Amazon SES publishes events to this destination when you send an email with the associated
     * configuration set. Set to <code>true</code> to enable publishing to this destination; set to <code>false</code>
     * to prevent publishing to this destination. The default value is <code>false</code>.
     * </p>
     * 
     * @return Sets whether Amazon SES publishes events to this destination when you send an email with the associated
     *         configuration set. Set to <code>true</code> to enable publishing to this destination; set to
     *         <code>false</code> to prevent publishing to this destination. The default value is <code>false</code>.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The type of email sending events to publish to the event destination.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMatchingEventTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type of email sending events to publish to the event destination.
     */
    public List<EventType> matchingEventTypes() {
        return EventTypesCopier.copyStringToEnum(matchingEventTypes);
    }

    /**
     * Returns true if the MatchingEventTypes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMatchingEventTypes() {
        return matchingEventTypes != null && !(matchingEventTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of email sending events to publish to the event destination.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMatchingEventTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type of email sending events to publish to the event destination.
     */
    public List<String> matchingEventTypesAsStrings() {
        return matchingEventTypes;
    }

    /**
     * <p>
     * An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis Firehose
     * event destination.
     * </p>
     * 
     * @return An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
     *         Firehose event destination.
     */
    public KinesisFirehoseDestination kinesisFirehoseDestination() {
        return kinesisFirehoseDestination;
    }

    /**
     * <p>
     * An object that contains the names, default values, and sources of the dimensions associated with an Amazon
     * CloudWatch event destination.
     * </p>
     * 
     * @return An object that contains the names, default values, and sources of the dimensions associated with an
     *         Amazon CloudWatch event destination.
     */
    public CloudWatchDestination cloudWatchDestination() {
        return cloudWatchDestination;
    }

    /**
     * <p>
     * An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon SNS) event
     * destination.
     * </p>
     * 
     * @return An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon SNS)
     *         event destination.
     */
    public SNSDestination snsDestination() {
        return snsDestination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(matchingEventTypesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(kinesisFirehoseDestination());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchDestination());
        hashCode = 31 * hashCode + Objects.hashCode(snsDestination());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventDestination)) {
            return false;
        }
        EventDestination other = (EventDestination) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(enabled(), other.enabled())
                && Objects.equals(matchingEventTypesAsStrings(), other.matchingEventTypesAsStrings())
                && Objects.equals(kinesisFirehoseDestination(), other.kinesisFirehoseDestination())
                && Objects.equals(cloudWatchDestination(), other.cloudWatchDestination())
                && Objects.equals(snsDestination(), other.snsDestination());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EventDestination").add("Name", name()).add("Enabled", enabled())
                .add("MatchingEventTypes", matchingEventTypesAsStrings())
                .add("KinesisFirehoseDestination", kinesisFirehoseDestination())
                .add("CloudWatchDestination", cloudWatchDestination()).add("SNSDestination", snsDestination()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "MatchingEventTypes":
            return Optional.ofNullable(clazz.cast(matchingEventTypesAsStrings()));
        case "KinesisFirehoseDestination":
            return Optional.ofNullable(clazz.cast(kinesisFirehoseDestination()));
        case "CloudWatchDestination":
            return Optional.ofNullable(clazz.cast(cloudWatchDestination()));
        case "SNSDestination":
            return Optional.ofNullable(clazz.cast(snsDestination()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventDestination, T> g) {
        return obj -> g.apply((EventDestination) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventDestination> {
        /**
         * <p>
         * The name of the event destination. The name must:
         * </p>
         * <ul>
         * <li>
         * <p>
         * This value can only contain ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Contain less than 64 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the event destination. The name must:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        This value can only contain ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Contain less than 64 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Sets whether Amazon SES publishes events to this destination when you send an email with the associated
         * configuration set. Set to <code>true</code> to enable publishing to this destination; set to
         * <code>false</code> to prevent publishing to this destination. The default value is <code>false</code>.
         * </p>
         * 
         * @param enabled
         *        Sets whether Amazon SES publishes events to this destination when you send an email with the
         *        associated configuration set. Set to <code>true</code> to enable publishing to this destination; set
         *        to <code>false</code> to prevent publishing to this destination. The default value is
         *        <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypesWithStrings(Collection<String> matchingEventTypes);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypesWithStrings(String... matchingEventTypes);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypes(Collection<EventType> matchingEventTypes);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchingEventTypes(EventType... matchingEventTypes);

        /**
         * <p>
         * An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
         * Firehose event destination.
         * </p>
         * 
         * @param kinesisFirehoseDestination
         *        An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
         *        Firehose event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination);

        /**
         * <p>
         * An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
         * Firehose event destination.
         * </p>
         * This is a convenience that creates an instance of the {@link KinesisFirehoseDestination.Builder} avoiding the
         * need to create one manually via {@link KinesisFirehoseDestination#builder()}.
         *
         * When the {@link Consumer} completes, {@link KinesisFirehoseDestination.Builder#build()} is called immediately
         * and its result is passed to {@link #kinesisFirehoseDestination(KinesisFirehoseDestination)}.
         * 
         * @param kinesisFirehoseDestination
         *        a consumer that will call methods on {@link KinesisFirehoseDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kinesisFirehoseDestination(KinesisFirehoseDestination)
         */
        default Builder kinesisFirehoseDestination(Consumer<KinesisFirehoseDestination.Builder> kinesisFirehoseDestination) {
            return kinesisFirehoseDestination(KinesisFirehoseDestination.builder().applyMutation(kinesisFirehoseDestination)
                    .build());
        }

        /**
         * <p>
         * An object that contains the names, default values, and sources of the dimensions associated with an Amazon
         * CloudWatch event destination.
         * </p>
         * 
         * @param cloudWatchDestination
         *        An object that contains the names, default values, and sources of the dimensions associated with an
         *        Amazon CloudWatch event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchDestination(CloudWatchDestination cloudWatchDestination);

        /**
         * <p>
         * An object that contains the names, default values, and sources of the dimensions associated with an Amazon
         * CloudWatch event destination.
         * </p>
         * This is a convenience that creates an instance of the {@link CloudWatchDestination.Builder} avoiding the need
         * to create one manually via {@link CloudWatchDestination#builder()}.
         *
         * When the {@link Consumer} completes, {@link CloudWatchDestination.Builder#build()} is called immediately and
         * its result is passed to {@link #cloudWatchDestination(CloudWatchDestination)}.
         * 
         * @param cloudWatchDestination
         *        a consumer that will call methods on {@link CloudWatchDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchDestination(CloudWatchDestination)
         */
        default Builder cloudWatchDestination(Consumer<CloudWatchDestination.Builder> cloudWatchDestination) {
            return cloudWatchDestination(CloudWatchDestination.builder().applyMutation(cloudWatchDestination).build());
        }

        /**
         * <p>
         * An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon SNS)
         * event destination.
         * </p>
         * 
         * @param snsDestination
         *        An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon
         *        SNS) event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsDestination(SNSDestination snsDestination);

        /**
         * <p>
         * An object that contains the topic ARN associated with an Amazon Simple Notification Service (Amazon SNS)
         * event destination.
         * </p>
         * This is a convenience that creates an instance of the {@link SNSDestination.Builder} avoiding the need to
         * create one manually via {@link SNSDestination#builder()}.
         *
         * When the {@link Consumer} completes, {@link SNSDestination.Builder#build()} is called immediately and its
         * result is passed to {@link #snsDestination(SNSDestination)}.
         * 
         * @param snsDestination
         *        a consumer that will call methods on {@link SNSDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #snsDestination(SNSDestination)
         */
        default Builder snsDestination(Consumer<SNSDestination.Builder> snsDestination) {
            return snsDestination(SNSDestination.builder().applyMutation(snsDestination).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Boolean enabled;

        private List<String> matchingEventTypes = DefaultSdkAutoConstructList.getInstance();

        private KinesisFirehoseDestination kinesisFirehoseDestination;

        private CloudWatchDestination cloudWatchDestination;

        private SNSDestination snsDestination;

        private BuilderImpl() {
        }

        private BuilderImpl(EventDestination model) {
            name(model.name);
            enabled(model.enabled);
            matchingEventTypesWithStrings(model.matchingEventTypes);
            kinesisFirehoseDestination(model.kinesisFirehoseDestination);
            cloudWatchDestination(model.cloudWatchDestination);
            snsDestination(model.snsDestination);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final Collection<String> getMatchingEventTypesAsStrings() {
            return matchingEventTypes;
        }

        @Override
        public final Builder matchingEventTypesWithStrings(Collection<String> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copy(matchingEventTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchingEventTypesWithStrings(String... matchingEventTypes) {
            matchingEventTypesWithStrings(Arrays.asList(matchingEventTypes));
            return this;
        }

        @Override
        public final Builder matchingEventTypes(Collection<EventType> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copyEnumToString(matchingEventTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchingEventTypes(EventType... matchingEventTypes) {
            matchingEventTypes(Arrays.asList(matchingEventTypes));
            return this;
        }

        public final void setMatchingEventTypesWithStrings(Collection<String> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copy(matchingEventTypes);
        }

        public final KinesisFirehoseDestination.Builder getKinesisFirehoseDestination() {
            return kinesisFirehoseDestination != null ? kinesisFirehoseDestination.toBuilder() : null;
        }

        @Override
        public final Builder kinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination) {
            this.kinesisFirehoseDestination = kinesisFirehoseDestination;
            return this;
        }

        public final void setKinesisFirehoseDestination(KinesisFirehoseDestination.BuilderImpl kinesisFirehoseDestination) {
            this.kinesisFirehoseDestination = kinesisFirehoseDestination != null ? kinesisFirehoseDestination.build() : null;
        }

        public final CloudWatchDestination.Builder getCloudWatchDestination() {
            return cloudWatchDestination != null ? cloudWatchDestination.toBuilder() : null;
        }

        @Override
        public final Builder cloudWatchDestination(CloudWatchDestination cloudWatchDestination) {
            this.cloudWatchDestination = cloudWatchDestination;
            return this;
        }

        public final void setCloudWatchDestination(CloudWatchDestination.BuilderImpl cloudWatchDestination) {
            this.cloudWatchDestination = cloudWatchDestination != null ? cloudWatchDestination.build() : null;
        }

        public final SNSDestination.Builder getSnsDestination() {
            return snsDestination != null ? snsDestination.toBuilder() : null;
        }

        @Override
        public final Builder snsDestination(SNSDestination snsDestination) {
            this.snsDestination = snsDestination;
            return this;
        }

        public final void setSnsDestination(SNSDestination.BuilderImpl snsDestination) {
            this.snsDestination = snsDestination != null ? snsDestination.build() : null;
        }

        @Override
        public EventDestination build() {
            return new EventDestination(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
