/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the details of a configuration set. Configuration sets enable you to publish email sending events. For
 * information about using configuration sets, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/monitor-sending-activity.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeConfigurationSetResponse extends SesResponse implements
        ToCopyableBuilder<DescribeConfigurationSetResponse.Builder, DescribeConfigurationSetResponse> {
    private static final SdkField<ConfigurationSet> CONFIGURATION_SET_FIELD = SdkField
            .<ConfigurationSet> builder(MarshallingType.SDK_POJO)
            .getter(getter(DescribeConfigurationSetResponse::configurationSet)).setter(setter(Builder::configurationSet))
            .constructor(ConfigurationSet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSet").build()).build();

    private static final SdkField<List<EventDestination>> EVENT_DESTINATIONS_FIELD = SdkField
            .<List<EventDestination>> builder(MarshallingType.LIST)
            .getter(getter(DescribeConfigurationSetResponse::eventDestinations))
            .setter(setter(Builder::eventDestinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventDestinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EventDestination> builder(MarshallingType.SDK_POJO)
                                            .constructor(EventDestination::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TrackingOptions> TRACKING_OPTIONS_FIELD = SdkField
            .<TrackingOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(DescribeConfigurationSetResponse::trackingOptions)).setter(setter(Builder::trackingOptions))
            .constructor(TrackingOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrackingOptions").build()).build();

    private static final SdkField<DeliveryOptions> DELIVERY_OPTIONS_FIELD = SdkField
            .<DeliveryOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(DescribeConfigurationSetResponse::deliveryOptions)).setter(setter(Builder::deliveryOptions))
            .constructor(DeliveryOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliveryOptions").build()).build();

    private static final SdkField<ReputationOptions> REPUTATION_OPTIONS_FIELD = SdkField
            .<ReputationOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(DescribeConfigurationSetResponse::reputationOptions)).setter(setter(Builder::reputationOptions))
            .constructor(ReputationOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReputationOptions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIGURATION_SET_FIELD,
            EVENT_DESTINATIONS_FIELD, TRACKING_OPTIONS_FIELD, DELIVERY_OPTIONS_FIELD, REPUTATION_OPTIONS_FIELD));

    private final ConfigurationSet configurationSet;

    private final List<EventDestination> eventDestinations;

    private final TrackingOptions trackingOptions;

    private final DeliveryOptions deliveryOptions;

    private final ReputationOptions reputationOptions;

    private DescribeConfigurationSetResponse(BuilderImpl builder) {
        super(builder);
        this.configurationSet = builder.configurationSet;
        this.eventDestinations = builder.eventDestinations;
        this.trackingOptions = builder.trackingOptions;
        this.deliveryOptions = builder.deliveryOptions;
        this.reputationOptions = builder.reputationOptions;
    }

    /**
     * <p>
     * The configuration set object associated with the specified configuration set.
     * </p>
     * 
     * @return The configuration set object associated with the specified configuration set.
     */
    public ConfigurationSet configurationSet() {
        return configurationSet;
    }

    /**
     * Returns true if the EventDestinations property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEventDestinations() {
        return eventDestinations != null && !(eventDestinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of event destinations associated with the configuration set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEventDestinations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of event destinations associated with the configuration set.
     */
    public List<EventDestination> eventDestinations() {
        return eventDestinations;
    }

    /**
     * <p>
     * The name of the custom open and click tracking domain associated with the configuration set.
     * </p>
     * 
     * @return The name of the custom open and click tracking domain associated with the configuration set.
     */
    public TrackingOptions trackingOptions() {
        return trackingOptions;
    }

    /**
     * Returns the value of the DeliveryOptions property for this object.
     * 
     * @return The value of the DeliveryOptions property for this object.
     */
    public DeliveryOptions deliveryOptions() {
        return deliveryOptions;
    }

    /**
     * <p>
     * An object that represents the reputation settings for the configuration set.
     * </p>
     * 
     * @return An object that represents the reputation settings for the configuration set.
     */
    public ReputationOptions reputationOptions() {
        return reputationOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(configurationSet());
        hashCode = 31 * hashCode + Objects.hashCode(eventDestinations());
        hashCode = 31 * hashCode + Objects.hashCode(trackingOptions());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryOptions());
        hashCode = 31 * hashCode + Objects.hashCode(reputationOptions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeConfigurationSetResponse)) {
            return false;
        }
        DescribeConfigurationSetResponse other = (DescribeConfigurationSetResponse) obj;
        return Objects.equals(configurationSet(), other.configurationSet())
                && Objects.equals(eventDestinations(), other.eventDestinations())
                && Objects.equals(trackingOptions(), other.trackingOptions())
                && Objects.equals(deliveryOptions(), other.deliveryOptions())
                && Objects.equals(reputationOptions(), other.reputationOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DescribeConfigurationSetResponse").add("ConfigurationSet", configurationSet())
                .add("EventDestinations", eventDestinations()).add("TrackingOptions", trackingOptions())
                .add("DeliveryOptions", deliveryOptions()).add("ReputationOptions", reputationOptions()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConfigurationSet":
            return Optional.ofNullable(clazz.cast(configurationSet()));
        case "EventDestinations":
            return Optional.ofNullable(clazz.cast(eventDestinations()));
        case "TrackingOptions":
            return Optional.ofNullable(clazz.cast(trackingOptions()));
        case "DeliveryOptions":
            return Optional.ofNullable(clazz.cast(deliveryOptions()));
        case "ReputationOptions":
            return Optional.ofNullable(clazz.cast(reputationOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeConfigurationSetResponse, T> g) {
        return obj -> g.apply((DescribeConfigurationSetResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesResponse.Builder, SdkPojo, CopyableBuilder<Builder, DescribeConfigurationSetResponse> {
        /**
         * <p>
         * The configuration set object associated with the specified configuration set.
         * </p>
         * 
         * @param configurationSet
         *        The configuration set object associated with the specified configuration set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSet(ConfigurationSet configurationSet);

        /**
         * <p>
         * The configuration set object associated with the specified configuration set.
         * </p>
         * This is a convenience that creates an instance of the {@link ConfigurationSet.Builder} avoiding the need to
         * create one manually via {@link ConfigurationSet#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfigurationSet.Builder#build()} is called immediately and its
         * result is passed to {@link #configurationSet(ConfigurationSet)}.
         * 
         * @param configurationSet
         *        a consumer that will call methods on {@link ConfigurationSet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurationSet(ConfigurationSet)
         */
        default Builder configurationSet(Consumer<ConfigurationSet.Builder> configurationSet) {
            return configurationSet(ConfigurationSet.builder().applyMutation(configurationSet).build());
        }

        /**
         * <p>
         * A list of event destinations associated with the configuration set.
         * </p>
         * 
         * @param eventDestinations
         *        A list of event destinations associated with the configuration set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventDestinations(Collection<EventDestination> eventDestinations);

        /**
         * <p>
         * A list of event destinations associated with the configuration set.
         * </p>
         * 
         * @param eventDestinations
         *        A list of event destinations associated with the configuration set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventDestinations(EventDestination... eventDestinations);

        /**
         * <p>
         * A list of event destinations associated with the configuration set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EventDestination>.Builder} avoiding the
         * need to create one manually via {@link List<EventDestination>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EventDestination>.Builder#build()} is called immediately and
         * its result is passed to {@link #eventDestinations(List<EventDestination>)}.
         * 
         * @param eventDestinations
         *        a consumer that will call methods on {@link List<EventDestination>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eventDestinations(List<EventDestination>)
         */
        Builder eventDestinations(Consumer<EventDestination.Builder>... eventDestinations);

        /**
         * <p>
         * The name of the custom open and click tracking domain associated with the configuration set.
         * </p>
         * 
         * @param trackingOptions
         *        The name of the custom open and click tracking domain associated with the configuration set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trackingOptions(TrackingOptions trackingOptions);

        /**
         * <p>
         * The name of the custom open and click tracking domain associated with the configuration set.
         * </p>
         * This is a convenience that creates an instance of the {@link TrackingOptions.Builder} avoiding the need to
         * create one manually via {@link TrackingOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link TrackingOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #trackingOptions(TrackingOptions)}.
         * 
         * @param trackingOptions
         *        a consumer that will call methods on {@link TrackingOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trackingOptions(TrackingOptions)
         */
        default Builder trackingOptions(Consumer<TrackingOptions.Builder> trackingOptions) {
            return trackingOptions(TrackingOptions.builder().applyMutation(trackingOptions).build());
        }

        /**
         * Sets the value of the DeliveryOptions property for this object.
         *
         * @param deliveryOptions
         *        The new value for the DeliveryOptions property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryOptions(DeliveryOptions deliveryOptions);

        /**
         * Sets the value of the DeliveryOptions property for this object.
         *
         * This is a convenience that creates an instance of the {@link DeliveryOptions.Builder} avoiding the need to
         * create one manually via {@link DeliveryOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link DeliveryOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #deliveryOptions(DeliveryOptions)}.
         * 
         * @param deliveryOptions
         *        a consumer that will call methods on {@link DeliveryOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deliveryOptions(DeliveryOptions)
         */
        default Builder deliveryOptions(Consumer<DeliveryOptions.Builder> deliveryOptions) {
            return deliveryOptions(DeliveryOptions.builder().applyMutation(deliveryOptions).build());
        }

        /**
         * <p>
         * An object that represents the reputation settings for the configuration set.
         * </p>
         * 
         * @param reputationOptions
         *        An object that represents the reputation settings for the configuration set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reputationOptions(ReputationOptions reputationOptions);

        /**
         * <p>
         * An object that represents the reputation settings for the configuration set.
         * </p>
         * This is a convenience that creates an instance of the {@link ReputationOptions.Builder} avoiding the need to
         * create one manually via {@link ReputationOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link ReputationOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #reputationOptions(ReputationOptions)}.
         * 
         * @param reputationOptions
         *        a consumer that will call methods on {@link ReputationOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #reputationOptions(ReputationOptions)
         */
        default Builder reputationOptions(Consumer<ReputationOptions.Builder> reputationOptions) {
            return reputationOptions(ReputationOptions.builder().applyMutation(reputationOptions).build());
        }
    }

    static final class BuilderImpl extends SesResponse.BuilderImpl implements Builder {
        private ConfigurationSet configurationSet;

        private List<EventDestination> eventDestinations = DefaultSdkAutoConstructList.getInstance();

        private TrackingOptions trackingOptions;

        private DeliveryOptions deliveryOptions;

        private ReputationOptions reputationOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeConfigurationSetResponse model) {
            super(model);
            configurationSet(model.configurationSet);
            eventDestinations(model.eventDestinations);
            trackingOptions(model.trackingOptions);
            deliveryOptions(model.deliveryOptions);
            reputationOptions(model.reputationOptions);
        }

        public final ConfigurationSet.Builder getConfigurationSet() {
            return configurationSet != null ? configurationSet.toBuilder() : null;
        }

        @Override
        public final Builder configurationSet(ConfigurationSet configurationSet) {
            this.configurationSet = configurationSet;
            return this;
        }

        public final void setConfigurationSet(ConfigurationSet.BuilderImpl configurationSet) {
            this.configurationSet = configurationSet != null ? configurationSet.build() : null;
        }

        public final Collection<EventDestination.Builder> getEventDestinations() {
            return eventDestinations != null ? eventDestinations.stream().map(EventDestination::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder eventDestinations(Collection<EventDestination> eventDestinations) {
            this.eventDestinations = EventDestinationsCopier.copy(eventDestinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventDestinations(EventDestination... eventDestinations) {
            eventDestinations(Arrays.asList(eventDestinations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventDestinations(Consumer<EventDestination.Builder>... eventDestinations) {
            eventDestinations(Stream.of(eventDestinations).map(c -> EventDestination.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEventDestinations(Collection<EventDestination.BuilderImpl> eventDestinations) {
            this.eventDestinations = EventDestinationsCopier.copyFromBuilder(eventDestinations);
        }

        public final TrackingOptions.Builder getTrackingOptions() {
            return trackingOptions != null ? trackingOptions.toBuilder() : null;
        }

        @Override
        public final Builder trackingOptions(TrackingOptions trackingOptions) {
            this.trackingOptions = trackingOptions;
            return this;
        }

        public final void setTrackingOptions(TrackingOptions.BuilderImpl trackingOptions) {
            this.trackingOptions = trackingOptions != null ? trackingOptions.build() : null;
        }

        public final DeliveryOptions.Builder getDeliveryOptions() {
            return deliveryOptions != null ? deliveryOptions.toBuilder() : null;
        }

        @Override
        public final Builder deliveryOptions(DeliveryOptions deliveryOptions) {
            this.deliveryOptions = deliveryOptions;
            return this;
        }

        public final void setDeliveryOptions(DeliveryOptions.BuilderImpl deliveryOptions) {
            this.deliveryOptions = deliveryOptions != null ? deliveryOptions.build() : null;
        }

        public final ReputationOptions.Builder getReputationOptions() {
            return reputationOptions != null ? reputationOptions.toBuilder() : null;
        }

        @Override
        public final Builder reputationOptions(ReputationOptions reputationOptions) {
            this.reputationOptions = reputationOptions;
            return this;
        }

        public final void setReputationOptions(ReputationOptions.BuilderImpl reputationOptions) {
            this.reputationOptions = reputationOptions != null ? reputationOptions.build() : null;
        }

        @Override
        public DescribeConfigurationSetResponse build() {
            return new DescribeConfigurationSetResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
