/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to return the requested sending authorization policies for an identity. Sending authorization is
 * an Amazon SES feature that enables you to authorize other senders to use your identities. For information, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetIdentityPoliciesRequest extends SesRequest implements
        ToCopyableBuilder<GetIdentityPoliciesRequest.Builder, GetIdentityPoliciesRequest> {
    private static final SdkField<String> IDENTITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetIdentityPoliciesRequest::identity)).setter(setter(Builder::identity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Identity").build()).build();

    private static final SdkField<List<String>> POLICY_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(GetIdentityPoliciesRequest::policyNames))
            .setter(setter(Builder::policyNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTITY_FIELD,
            POLICY_NAMES_FIELD));

    private final String identity;

    private final List<String> policyNames;

    private GetIdentityPoliciesRequest(BuilderImpl builder) {
        super(builder);
        this.identity = builder.identity;
        this.policyNames = builder.policyNames;
    }

    /**
     * <p>
     * The identity for which the policies will be retrieved. You can specify an identity by using its name or by using
     * its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>, <code>example.com</code>,
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
     * </p>
     * <p>
     * To successfully call this API, you must own the identity.
     * </p>
     * 
     * @return The identity for which the policies will be retrieved. You can specify an identity by using its name or
     *         by using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>,
     *         <code>example.com</code>, <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.</p>
     *         <p>
     *         To successfully call this API, you must own the identity.
     */
    public String identity() {
        return identity;
    }

    /**
     * Returns true if the PolicyNames property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasPolicyNames() {
        return policyNames != null && !(policyNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time. If you do
     * not know the names of the policies that are attached to the identity, you can use
     * <code>ListIdentityPolicies</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPolicyNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time. If
     *         you do not know the names of the policies that are attached to the identity, you can use
     *         <code>ListIdentityPolicies</code>.
     */
    public List<String> policyNames() {
        return policyNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(identity());
        hashCode = 31 * hashCode + Objects.hashCode(policyNames());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetIdentityPoliciesRequest)) {
            return false;
        }
        GetIdentityPoliciesRequest other = (GetIdentityPoliciesRequest) obj;
        return Objects.equals(identity(), other.identity()) && Objects.equals(policyNames(), other.policyNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetIdentityPoliciesRequest").add("Identity", identity()).add("PolicyNames", policyNames())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Identity":
            return Optional.ofNullable(clazz.cast(identity()));
        case "PolicyNames":
            return Optional.ofNullable(clazz.cast(policyNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetIdentityPoliciesRequest, T> g) {
        return obj -> g.apply((GetIdentityPoliciesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetIdentityPoliciesRequest> {
        /**
         * <p>
         * The identity for which the policies will be retrieved. You can specify an identity by using its name or by
         * using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>, <code>example.com</code>,
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
         * </p>
         * <p>
         * To successfully call this API, you must own the identity.
         * </p>
         * 
         * @param identity
         *        The identity for which the policies will be retrieved. You can specify an identity by using its name
         *        or by using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>,
         *        <code>example.com</code>, <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.</p>
         *        <p>
         *        To successfully call this API, you must own the identity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identity(String identity);

        /**
         * <p>
         * A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time. If you
         * do not know the names of the policies that are attached to the identity, you can use
         * <code>ListIdentityPolicies</code>.
         * </p>
         * 
         * @param policyNames
         *        A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time.
         *        If you do not know the names of the policies that are attached to the identity, you can use
         *        <code>ListIdentityPolicies</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyNames(Collection<String> policyNames);

        /**
         * <p>
         * A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time. If you
         * do not know the names of the policies that are attached to the identity, you can use
         * <code>ListIdentityPolicies</code>.
         * </p>
         * 
         * @param policyNames
         *        A list of the names of policies to be retrieved. You can retrieve a maximum of 20 policies at a time.
         *        If you do not know the names of the policies that are attached to the identity, you can use
         *        <code>ListIdentityPolicies</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyNames(String... policyNames);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String identity;

        private List<String> policyNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetIdentityPoliciesRequest model) {
            super(model);
            identity(model.identity);
            policyNames(model.policyNames);
        }

        public final String getIdentity() {
            return identity;
        }

        @Override
        public final Builder identity(String identity) {
            this.identity = identity;
            return this;
        }

        public final void setIdentity(String identity) {
            this.identity = identity;
        }

        public final Collection<String> getPolicyNames() {
            return policyNames;
        }

        @Override
        public final Builder policyNames(Collection<String> policyNames) {
            this.policyNames = PolicyNameListCopier.copy(policyNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policyNames(String... policyNames) {
            policyNames(Arrays.asList(policyNames));
            return this;
        }

        public final void setPolicyNames(Collection<String> policyNames) {
            this.policyNames = PolicyNameListCopier.copy(policyNames);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetIdentityPoliciesRequest build() {
            return new GetIdentityPoliciesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
