/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to return the details of a configuration set. Configuration sets enable you to publish email
 * sending events. For information about using configuration sets, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/monitor-sending-activity.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeConfigurationSetRequest extends SesRequest implements
        ToCopyableBuilder<DescribeConfigurationSetRequest.Builder, DescribeConfigurationSetRequest> {
    private static final SdkField<String> CONFIGURATION_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeConfigurationSetRequest::configurationSetName)).setter(setter(Builder::configurationSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSetName").build())
            .build();

    private static final SdkField<List<String>> CONFIGURATION_SET_ATTRIBUTE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DescribeConfigurationSetRequest::configurationSetAttributeNamesAsStrings))
            .setter(setter(Builder::configurationSetAttributeNamesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSetAttributeNames")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIGURATION_SET_NAME_FIELD,
            CONFIGURATION_SET_ATTRIBUTE_NAMES_FIELD));

    private final String configurationSetName;

    private final List<String> configurationSetAttributeNames;

    private DescribeConfigurationSetRequest(BuilderImpl builder) {
        super(builder);
        this.configurationSetName = builder.configurationSetName;
        this.configurationSetAttributeNames = builder.configurationSetAttributeNames;
    }

    /**
     * <p>
     * The name of the configuration set to describe.
     * </p>
     * 
     * @return The name of the configuration set to describe.
     */
    public String configurationSetName() {
        return configurationSetName;
    }

    /**
     * <p>
     * A list of configuration set attributes to return.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConfigurationSetAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of configuration set attributes to return.
     */
    public List<ConfigurationSetAttribute> configurationSetAttributeNames() {
        return ConfigurationSetAttributeListCopier.copyStringToEnum(configurationSetAttributeNames);
    }

    /**
     * Returns true if the ConfigurationSetAttributeNames property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public boolean hasConfigurationSetAttributeNames() {
        return configurationSetAttributeNames != null && !(configurationSetAttributeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of configuration set attributes to return.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConfigurationSetAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of configuration set attributes to return.
     */
    public List<String> configurationSetAttributeNamesAsStrings() {
        return configurationSetAttributeNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetAttributeNamesAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeConfigurationSetRequest)) {
            return false;
        }
        DescribeConfigurationSetRequest other = (DescribeConfigurationSetRequest) obj;
        return Objects.equals(configurationSetName(), other.configurationSetName())
                && Objects.equals(configurationSetAttributeNamesAsStrings(), other.configurationSetAttributeNamesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DescribeConfigurationSetRequest").add("ConfigurationSetName", configurationSetName())
                .add("ConfigurationSetAttributeNames", configurationSetAttributeNamesAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConfigurationSetName":
            return Optional.ofNullable(clazz.cast(configurationSetName()));
        case "ConfigurationSetAttributeNames":
            return Optional.ofNullable(clazz.cast(configurationSetAttributeNamesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeConfigurationSetRequest, T> g) {
        return obj -> g.apply((DescribeConfigurationSetRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, DescribeConfigurationSetRequest> {
        /**
         * <p>
         * The name of the configuration set to describe.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to describe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);

        /**
         * <p>
         * A list of configuration set attributes to return.
         * </p>
         * 
         * @param configurationSetAttributeNames
         *        A list of configuration set attributes to return.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetAttributeNamesWithStrings(Collection<String> configurationSetAttributeNames);

        /**
         * <p>
         * A list of configuration set attributes to return.
         * </p>
         * 
         * @param configurationSetAttributeNames
         *        A list of configuration set attributes to return.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetAttributeNamesWithStrings(String... configurationSetAttributeNames);

        /**
         * <p>
         * A list of configuration set attributes to return.
         * </p>
         * 
         * @param configurationSetAttributeNames
         *        A list of configuration set attributes to return.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetAttributeNames(Collection<ConfigurationSetAttribute> configurationSetAttributeNames);

        /**
         * <p>
         * A list of configuration set attributes to return.
         * </p>
         * 
         * @param configurationSetAttributeNames
         *        A list of configuration set attributes to return.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetAttributeNames(ConfigurationSetAttribute... configurationSetAttributeNames);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String configurationSetName;

        private List<String> configurationSetAttributeNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeConfigurationSetRequest model) {
            super(model);
            configurationSetName(model.configurationSetName);
            configurationSetAttributeNamesWithStrings(model.configurationSetAttributeNames);
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        public final Collection<String> getConfigurationSetAttributeNamesAsStrings() {
            return configurationSetAttributeNames;
        }

        @Override
        public final Builder configurationSetAttributeNamesWithStrings(Collection<String> configurationSetAttributeNames) {
            this.configurationSetAttributeNames = ConfigurationSetAttributeListCopier.copy(configurationSetAttributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurationSetAttributeNamesWithStrings(String... configurationSetAttributeNames) {
            configurationSetAttributeNamesWithStrings(Arrays.asList(configurationSetAttributeNames));
            return this;
        }

        @Override
        public final Builder configurationSetAttributeNames(Collection<ConfigurationSetAttribute> configurationSetAttributeNames) {
            this.configurationSetAttributeNames = ConfigurationSetAttributeListCopier
                    .copyEnumToString(configurationSetAttributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurationSetAttributeNames(ConfigurationSetAttribute... configurationSetAttributeNames) {
            configurationSetAttributeNames(Arrays.asList(configurationSetAttributeNames));
            return this;
        }

        public final void setConfigurationSetAttributeNamesWithStrings(Collection<String> configurationSetAttributeNames) {
            this.configurationSetAttributeNames = ConfigurationSetAttributeListCopier.copy(configurationSetAttributeNames);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DescribeConfigurationSetRequest build() {
            return new DescribeConfigurationSetRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
