/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a single formatted email using Amazon SES. For more information, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-formatted.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendEmailRequest extends SesRequest implements ToCopyableBuilder<SendEmailRequest.Builder, SendEmailRequest> {
    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Source")
            .getter(getter(SendEmailRequest::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<Destination> DESTINATION_FIELD = SdkField.<Destination> builder(MarshallingType.SDK_POJO)
            .memberName("Destination").getter(getter(SendEmailRequest::destination)).setter(setter(Builder::destination))
            .constructor(Destination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destination").build()).build();

    private static final SdkField<Message> MESSAGE_FIELD = SdkField.<Message> builder(MarshallingType.SDK_POJO)
            .memberName("Message").getter(getter(SendEmailRequest::message)).setter(setter(Builder::message))
            .constructor(Message::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<List<String>> REPLY_TO_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ReplyToAddresses")
            .getter(getter(SendEmailRequest::replyToAddresses))
            .setter(setter(Builder::replyToAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplyToAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RETURN_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReturnPath").getter(getter(SendEmailRequest::returnPath)).setter(setter(Builder::returnPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReturnPath").build()).build();

    private static final SdkField<String> SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceArn").getter(getter(SendEmailRequest::sourceArn)).setter(setter(Builder::sourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceArn").build()).build();

    private static final SdkField<String> RETURN_PATH_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReturnPathArn").getter(getter(SendEmailRequest::returnPathArn)).setter(setter(Builder::returnPathArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReturnPathArn").build()).build();

    private static final SdkField<List<MessageTag>> TAGS_FIELD = SdkField
            .<List<MessageTag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(SendEmailRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MessageTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONFIGURATION_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationSetName").getter(getter(SendEmailRequest::configurationSetName))
            .setter(setter(Builder::configurationSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSetName").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_FIELD,
            DESTINATION_FIELD, MESSAGE_FIELD, REPLY_TO_ADDRESSES_FIELD, RETURN_PATH_FIELD, SOURCE_ARN_FIELD,
            RETURN_PATH_ARN_FIELD, TAGS_FIELD, CONFIGURATION_SET_NAME_FIELD));

    private final String source;

    private final Destination destination;

    private final Message message;

    private final List<String> replyToAddresses;

    private final String returnPath;

    private final String sourceArn;

    private final String returnPathArn;

    private final List<MessageTag> tags;

    private final String configurationSetName;

    private SendEmailRequest(BuilderImpl builder) {
        super(builder);
        this.source = builder.source;
        this.destination = builder.destination;
        this.message = builder.message;
        this.replyToAddresses = builder.replyToAddresses;
        this.returnPath = builder.returnPath;
        this.sourceArn = builder.sourceArn;
        this.returnPathArn = builder.returnPathArn;
        this.tags = builder.tags;
        this.configurationSetName = builder.configurationSetName;
    }

    /**
     * <p>
     * The email address that is sending the email. This email address must be either individually verified with Amazon
     * SES, or from a domain that has been verified with Amazon SES. For information about verifying identities, see the
     * <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
     * Developer Guide</a>.
     * </p>
     * <p>
     * If you are sending on behalf of another user and have been permitted to do so by a sending authorization policy,
     * then you must also specify the <code>SourceArn</code> parameter. For more information about sending
     * authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * <note>
     * <p>
     * Amazon SES does not support the SMTPUTF8 extension, as described in <a
     * href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a source email
     * address (the part of the email address that precedes the @ sign) may only contain <a
     * href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the <i>domain
     * part</i> of an address (the part after the @ sign) contains non-ASCII characters, they must be encoded using
     * Punycode, as described in <a href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>. The sender name (also
     * known as the <i>friendly name</i>) may contain non-ASCII characters. These characters must be encoded using MIME
     * encoded-word syntax, as described in <a href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>. MIME
     * encoded-word syntax uses the following form: <code>=?charset?encoding?encoded-text?=</code>.
     * </p>
     * </note>
     * 
     * @return The email address that is sending the email. This email address must be either individually verified with
     *         Amazon SES, or from a domain that has been verified with Amazon SES. For information about verifying
     *         identities, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
     *         Developer Guide</a>.</p>
     *         <p>
     *         If you are sending on behalf of another user and have been permitted to do so by a sending authorization
     *         policy, then you must also specify the <code>SourceArn</code> parameter. For more information about
     *         sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     *         </p>
     *         <note>
     *         <p>
     *         Amazon SES does not support the SMTPUTF8 extension, as described in <a
     *         href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a
     *         source email address (the part of the email address that precedes the @ sign) may only contain <a
     *         href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the
     *         <i>domain part</i> of an address (the part after the @ sign) contains non-ASCII characters, they must be
     *         encoded using Punycode, as described in <a href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>.
     *         The sender name (also known as the <i>friendly name</i>) may contain non-ASCII characters. These
     *         characters must be encoded using MIME encoded-word syntax, as described in <a
     *         href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>. MIME encoded-word syntax uses the following
     *         form: <code>=?charset?encoding?encoded-text?=</code>.
     *         </p>
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * The destination for this email, composed of To:, CC:, and BCC: fields.
     * </p>
     * 
     * @return The destination for this email, composed of To:, CC:, and BCC: fields.
     */
    public Destination destination() {
        return destination;
    }

    /**
     * <p>
     * The message to be sent.
     * </p>
     * 
     * @return The message to be sent.
     */
    public Message message() {
        return message;
    }

    /**
     * Returns true if the ReplyToAddresses property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasReplyToAddresses() {
        return replyToAddresses != null && !(replyToAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to address
     * will receive the reply.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasReplyToAddresses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
     *         address will receive the reply.
     */
    public List<String> replyToAddresses() {
        return replyToAddresses;
    }

    /**
     * <p>
     * The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled. If the
     * message cannot be delivered to the recipient, then an error message will be returned from the recipient's ISP;
     * this message will then be forwarded to the email address specified by the <code>ReturnPath</code> parameter. The
     * <code>ReturnPath</code> parameter is never overwritten. This email address must be either individually verified
     * with Amazon SES, or from a domain that has been verified with Amazon SES.
     * </p>
     * 
     * @return The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled.
     *         If the message cannot be delivered to the recipient, then an error message will be returned from the
     *         recipient's ISP; this message will then be forwarded to the email address specified by the
     *         <code>ReturnPath</code> parameter. The <code>ReturnPath</code> parameter is never overwritten. This email
     *         address must be either individually verified with Amazon SES, or from a domain that has been verified
     *         with Amazon SES.
     */
    public String returnPath() {
        return returnPath;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to send for the email address specified in the <code>Source</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
     * <code>user@example.com</code>.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to send for the email address specified in the
     *         <code>Source</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to send from <code>user@example.com</code>, then you would specify the
     *         <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
     *         the <code>Source</code> to be <code>user@example.com</code>.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the <code>ReturnPath</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
     * <code>feedback@example.com</code>.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>ReturnPath</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to use <code>feedback@example.com</code>, then you would specify the
     *         <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
     *         and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String returnPathArn() {
        return returnPathArn;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send using <code>SendEmail</code>.
     * Tags correspond to characteristics of the email that you define, so that you can publish email sending events.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send using
     *         <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
     *         publish email sending events.
     */
    public List<MessageTag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the configuration set to use when you send an email using <code>SendEmail</code>.
     * </p>
     * 
     * @return The name of the configuration set to use when you send an email using <code>SendEmail</code>.
     */
    public String configurationSetName() {
        return configurationSetName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(hasReplyToAddresses() ? replyToAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(returnPath());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(returnPathArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendEmailRequest)) {
            return false;
        }
        SendEmailRequest other = (SendEmailRequest) obj;
        return Objects.equals(source(), other.source()) && Objects.equals(destination(), other.destination())
                && Objects.equals(message(), other.message()) && hasReplyToAddresses() == other.hasReplyToAddresses()
                && Objects.equals(replyToAddresses(), other.replyToAddresses())
                && Objects.equals(returnPath(), other.returnPath()) && Objects.equals(sourceArn(), other.sourceArn())
                && Objects.equals(returnPathArn(), other.returnPathArn()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(configurationSetName(), other.configurationSetName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SendEmailRequest").add("Source", source()).add("Destination", destination())
                .add("Message", message()).add("ReplyToAddresses", hasReplyToAddresses() ? replyToAddresses() : null)
                .add("ReturnPath", returnPath()).add("SourceArn", sourceArn()).add("ReturnPathArn", returnPathArn())
                .add("Tags", hasTags() ? tags() : null).add("ConfigurationSetName", configurationSetName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "Destination":
            return Optional.ofNullable(clazz.cast(destination()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "ReplyToAddresses":
            return Optional.ofNullable(clazz.cast(replyToAddresses()));
        case "ReturnPath":
            return Optional.ofNullable(clazz.cast(returnPath()));
        case "SourceArn":
            return Optional.ofNullable(clazz.cast(sourceArn()));
        case "ReturnPathArn":
            return Optional.ofNullable(clazz.cast(returnPathArn()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "ConfigurationSetName":
            return Optional.ofNullable(clazz.cast(configurationSetName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SendEmailRequest, T> g) {
        return obj -> g.apply((SendEmailRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, SendEmailRequest> {
        /**
         * <p>
         * The email address that is sending the email. This email address must be either individually verified with
         * Amazon SES, or from a domain that has been verified with Amazon SES. For information about verifying
         * identities, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
         * Developer Guide</a>.
         * </p>
         * <p>
         * If you are sending on behalf of another user and have been permitted to do so by a sending authorization
         * policy, then you must also specify the <code>SourceArn</code> parameter. For more information about sending
         * authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * <note>
         * <p>
         * Amazon SES does not support the SMTPUTF8 extension, as described in <a
         * href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a source
         * email address (the part of the email address that precedes the @ sign) may only contain <a
         * href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the <i>domain
         * part</i> of an address (the part after the @ sign) contains non-ASCII characters, they must be encoded using
         * Punycode, as described in <a href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>. The sender name
         * (also known as the <i>friendly name</i>) may contain non-ASCII characters. These characters must be encoded
         * using MIME encoded-word syntax, as described in <a href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         * MIME encoded-word syntax uses the following form: <code>=?charset?encoding?encoded-text?=</code>.
         * </p>
         * </note>
         * 
         * @param source
         *        The email address that is sending the email. This email address must be either individually verified
         *        with Amazon SES, or from a domain that has been verified with Amazon SES. For information about
         *        verifying identities, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon
         *        SES Developer Guide</a>.</p>
         *        <p>
         *        If you are sending on behalf of another user and have been permitted to do so by a sending
         *        authorization policy, then you must also specify the <code>SourceArn</code> parameter. For more
         *        information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         *        </p>
         *        <note>
         *        <p>
         *        Amazon SES does not support the SMTPUTF8 extension, as described in <a
         *        href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a
         *        source email address (the part of the email address that precedes the @ sign) may only contain <a
         *        href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the
         *        <i>domain part</i> of an address (the part after the @ sign) contains non-ASCII characters, they must
         *        be encoded using Punycode, as described in <a
         *        href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>. The sender name (also known as the
         *        <i>friendly name</i>) may contain non-ASCII characters. These characters must be encoded using MIME
         *        encoded-word syntax, as described in <a href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>. MIME
         *        encoded-word syntax uses the following form: <code>=?charset?encoding?encoded-text?=</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * The destination for this email, composed of To:, CC:, and BCC: fields.
         * </p>
         * 
         * @param destination
         *        The destination for this email, composed of To:, CC:, and BCC: fields.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(Destination destination);

        /**
         * <p>
         * The destination for this email, composed of To:, CC:, and BCC: fields.
         * </p>
         * This is a convenience that creates an instance of the {@link Destination.Builder} avoiding the need to create
         * one manually via {@link Destination#builder()}.
         *
         * When the {@link Consumer} completes, {@link Destination.Builder#build()} is called immediately and its result
         * is passed to {@link #destination(Destination)}.
         * 
         * @param destination
         *        a consumer that will call methods on {@link Destination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destination(Destination)
         */
        default Builder destination(Consumer<Destination.Builder> destination) {
            return destination(Destination.builder().applyMutation(destination).build());
        }

        /**
         * <p>
         * The message to be sent.
         * </p>
         * 
         * @param message
         *        The message to be sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(Message message);

        /**
         * <p>
         * The message to be sent.
         * </p>
         * This is a convenience that creates an instance of the {@link Message.Builder} avoiding the need to create one
         * manually via {@link Message#builder()}.
         *
         * When the {@link Consumer} completes, {@link Message.Builder#build()} is called immediately and its result is
         * passed to {@link #message(Message)}.
         * 
         * @param message
         *        a consumer that will call methods on {@link Message.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #message(Message)
         */
        default Builder message(Consumer<Message.Builder> message) {
            return message(Message.builder().applyMutation(message).build());
        }

        /**
         * <p>
         * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         * address will receive the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         *        address will receive the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(Collection<String> replyToAddresses);

        /**
         * <p>
         * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         * address will receive the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         *        address will receive the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(String... replyToAddresses);

        /**
         * <p>
         * The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled. If
         * the message cannot be delivered to the recipient, then an error message will be returned from the recipient's
         * ISP; this message will then be forwarded to the email address specified by the <code>ReturnPath</code>
         * parameter. The <code>ReturnPath</code> parameter is never overwritten. This email address must be either
         * individually verified with Amazon SES, or from a domain that has been verified with Amazon SES.
         * </p>
         * 
         * @param returnPath
         *        The email address that bounces and complaints will be forwarded to when feedback forwarding is
         *        enabled. If the message cannot be delivered to the recipient, then an error message will be returned
         *        from the recipient's ISP; this message will then be forwarded to the email address specified by the
         *        <code>ReturnPath</code> parameter. The <code>ReturnPath</code> parameter is never overwritten. This
         *        email address must be either individually verified with Amazon SES, or from a domain that has been
         *        verified with Amazon SES.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPath(String returnPath);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to send for the email address specified in the
         * <code>Source</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
         * <code>user@example.com</code>.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param sourceArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to send for the email address
         *        specified in the <code>Source</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to send from <code>user@example.com</code>, then you would specify the
         *        <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
         *        the <code>Source</code> to be <code>user@example.com</code>.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>ReturnPath</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
         * <code>feedback@example.com</code>.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param returnPathArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>ReturnPath</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to use <code>feedback@example.com</code>, then you would specify the
         *        <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
         *        and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPathArn(String returnPathArn);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you
         *        can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<MessageTag> tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you
         *        can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(MessageTag... tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MessageTag>.Builder} avoiding the need to
         * create one manually via {@link List<MessageTag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MessageTag>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<MessageTag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<MessageTag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<MessageTag>)
         */
        Builder tags(Consumer<MessageTag.Builder>... tags);

        /**
         * <p>
         * The name of the configuration set to use when you send an email using <code>SendEmail</code>.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to use when you send an email using <code>SendEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String source;

        private Destination destination;

        private Message message;

        private List<String> replyToAddresses = DefaultSdkAutoConstructList.getInstance();

        private String returnPath;

        private String sourceArn;

        private String returnPathArn;

        private List<MessageTag> tags = DefaultSdkAutoConstructList.getInstance();

        private String configurationSetName;

        private BuilderImpl() {
        }

        private BuilderImpl(SendEmailRequest model) {
            super(model);
            source(model.source);
            destination(model.destination);
            message(model.message);
            replyToAddresses(model.replyToAddresses);
            returnPath(model.returnPath);
            sourceArn(model.sourceArn);
            returnPathArn(model.returnPathArn);
            tags(model.tags);
            configurationSetName(model.configurationSetName);
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final Destination.Builder getDestination() {
            return destination != null ? destination.toBuilder() : null;
        }

        @Override
        public final Builder destination(Destination destination) {
            this.destination = destination;
            return this;
        }

        public final void setDestination(Destination.BuilderImpl destination) {
            this.destination = destination != null ? destination.build() : null;
        }

        public final Message.Builder getMessage() {
            return message != null ? message.toBuilder() : null;
        }

        @Override
        public final Builder message(Message message) {
            this.message = message;
            return this;
        }

        public final void setMessage(Message.BuilderImpl message) {
            this.message = message != null ? message.build() : null;
        }

        public final Collection<String> getReplyToAddresses() {
            if (replyToAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return replyToAddresses;
        }

        @Override
        public final Builder replyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = AddressListCopier.copy(replyToAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replyToAddresses(String... replyToAddresses) {
            replyToAddresses(Arrays.asList(replyToAddresses));
            return this;
        }

        public final void setReplyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = AddressListCopier.copy(replyToAddresses);
        }

        public final String getReturnPath() {
            return returnPath;
        }

        @Override
        public final Builder returnPath(String returnPath) {
            this.returnPath = returnPath;
            return this;
        }

        public final void setReturnPath(String returnPath) {
            this.returnPath = returnPath;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        public final String getReturnPathArn() {
            return returnPathArn;
        }

        @Override
        public final Builder returnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
            return this;
        }

        public final void setReturnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
        }

        public final Collection<MessageTag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(MessageTag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<MessageTag> tags) {
            this.tags = MessageTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(MessageTag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<MessageTag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> MessageTag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<MessageTag.BuilderImpl> tags) {
            this.tags = MessageTagListCopier.copyFromBuilder(tags);
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SendEmailRequest build() {
            return new SendEmailRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
