/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Message-related information to include in the Delivery Status Notification (DSN) when an email that Amazon SES
 * receives on your behalf bounces.
 * </p>
 * <p>
 * For information about receiving email through Amazon SES, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MessageDsn implements SdkPojo, Serializable, ToCopyableBuilder<MessageDsn.Builder, MessageDsn> {
    private static final SdkField<String> REPORTING_MTA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReportingMta").getter(getter(MessageDsn::reportingMta)).setter(setter(Builder::reportingMta))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReportingMta").build()).build();

    private static final SdkField<Instant> ARRIVAL_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ArrivalDate").getter(getter(MessageDsn::arrivalDate)).setter(setter(Builder::arrivalDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ArrivalDate").build()).build();

    private static final SdkField<List<ExtensionField>> EXTENSION_FIELDS_FIELD = SdkField
            .<List<ExtensionField>> builder(MarshallingType.LIST)
            .memberName("ExtensionFields")
            .getter(getter(MessageDsn::extensionFields))
            .setter(setter(Builder::extensionFields))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExtensionFields").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ExtensionField> builder(MarshallingType.SDK_POJO)
                                            .constructor(ExtensionField::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPORTING_MTA_FIELD,
            ARRIVAL_DATE_FIELD, EXTENSION_FIELDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String reportingMta;

    private final Instant arrivalDate;

    private final List<ExtensionField> extensionFields;

    private MessageDsn(BuilderImpl builder) {
        this.reportingMta = builder.reportingMta;
        this.arrivalDate = builder.arrivalDate;
        this.extensionFields = builder.extensionFields;
    }

    /**
     * <p>
     * The reporting MTA that attempted to deliver the message, formatted as specified in <a
     * href="https://tools.ietf.org/html/rfc3464">RFC 3464</a> (<code>mta-name-type; mta-name</code>). The default value
     * is <code>dns; inbound-smtp.[region].amazonaws.com</code>.
     * </p>
     * 
     * @return The reporting MTA that attempted to deliver the message, formatted as specified in <a
     *         href="https://tools.ietf.org/html/rfc3464">RFC 3464</a> (<code>mta-name-type; mta-name</code>). The
     *         default value is <code>dns; inbound-smtp.[region].amazonaws.com</code>.
     */
    public String reportingMta() {
        return reportingMta;
    }

    /**
     * <p>
     * When the message was received by the reporting mail transfer agent (MTA), in <a
     * href="https://www.ietf.org/rfc/rfc0822.txt">RFC 822</a> date-time format.
     * </p>
     * 
     * @return When the message was received by the reporting mail transfer agent (MTA), in <a
     *         href="https://www.ietf.org/rfc/rfc0822.txt">RFC 822</a> date-time format.
     */
    public Instant arrivalDate() {
        return arrivalDate;
    }

    /**
     * Returns true if the ExtensionFields property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasExtensionFields() {
        return extensionFields != null && !(extensionFields instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Additional X-headers to include in the DSN.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExtensionFields()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Additional X-headers to include in the DSN.
     */
    public List<ExtensionField> extensionFields() {
        return extensionFields;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(reportingMta());
        hashCode = 31 * hashCode + Objects.hashCode(arrivalDate());
        hashCode = 31 * hashCode + Objects.hashCode(extensionFields());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageDsn)) {
            return false;
        }
        MessageDsn other = (MessageDsn) obj;
        return Objects.equals(reportingMta(), other.reportingMta()) && Objects.equals(arrivalDate(), other.arrivalDate())
                && Objects.equals(extensionFields(), other.extensionFields());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("MessageDsn").add("ReportingMta", reportingMta()).add("ArrivalDate", arrivalDate())
                .add("ExtensionFields", extensionFields()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReportingMta":
            return Optional.ofNullable(clazz.cast(reportingMta()));
        case "ArrivalDate":
            return Optional.ofNullable(clazz.cast(arrivalDate()));
        case "ExtensionFields":
            return Optional.ofNullable(clazz.cast(extensionFields()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MessageDsn, T> g) {
        return obj -> g.apply((MessageDsn) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MessageDsn> {
        /**
         * <p>
         * The reporting MTA that attempted to deliver the message, formatted as specified in <a
         * href="https://tools.ietf.org/html/rfc3464">RFC 3464</a> (<code>mta-name-type; mta-name</code>). The default
         * value is <code>dns; inbound-smtp.[region].amazonaws.com</code>.
         * </p>
         * 
         * @param reportingMta
         *        The reporting MTA that attempted to deliver the message, formatted as specified in <a
         *        href="https://tools.ietf.org/html/rfc3464">RFC 3464</a> (<code>mta-name-type; mta-name</code>). The
         *        default value is <code>dns; inbound-smtp.[region].amazonaws.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportingMta(String reportingMta);

        /**
         * <p>
         * When the message was received by the reporting mail transfer agent (MTA), in <a
         * href="https://www.ietf.org/rfc/rfc0822.txt">RFC 822</a> date-time format.
         * </p>
         * 
         * @param arrivalDate
         *        When the message was received by the reporting mail transfer agent (MTA), in <a
         *        href="https://www.ietf.org/rfc/rfc0822.txt">RFC 822</a> date-time format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arrivalDate(Instant arrivalDate);

        /**
         * <p>
         * Additional X-headers to include in the DSN.
         * </p>
         * 
         * @param extensionFields
         *        Additional X-headers to include in the DSN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extensionFields(Collection<ExtensionField> extensionFields);

        /**
         * <p>
         * Additional X-headers to include in the DSN.
         * </p>
         * 
         * @param extensionFields
         *        Additional X-headers to include in the DSN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extensionFields(ExtensionField... extensionFields);

        /**
         * <p>
         * Additional X-headers to include in the DSN.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ExtensionField>.Builder} avoiding the need
         * to create one manually via {@link List<ExtensionField>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ExtensionField>.Builder#build()} is called immediately and
         * its result is passed to {@link #extensionFields(List<ExtensionField>)}.
         * 
         * @param extensionFields
         *        a consumer that will call methods on {@link List<ExtensionField>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #extensionFields(List<ExtensionField>)
         */
        Builder extensionFields(Consumer<ExtensionField.Builder>... extensionFields);
    }

    static final class BuilderImpl implements Builder {
        private String reportingMta;

        private Instant arrivalDate;

        private List<ExtensionField> extensionFields = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MessageDsn model) {
            reportingMta(model.reportingMta);
            arrivalDate(model.arrivalDate);
            extensionFields(model.extensionFields);
        }

        public final String getReportingMta() {
            return reportingMta;
        }

        @Override
        public final Builder reportingMta(String reportingMta) {
            this.reportingMta = reportingMta;
            return this;
        }

        public final void setReportingMta(String reportingMta) {
            this.reportingMta = reportingMta;
        }

        public final Instant getArrivalDate() {
            return arrivalDate;
        }

        @Override
        public final Builder arrivalDate(Instant arrivalDate) {
            this.arrivalDate = arrivalDate;
            return this;
        }

        public final void setArrivalDate(Instant arrivalDate) {
            this.arrivalDate = arrivalDate;
        }

        public final Collection<ExtensionField.Builder> getExtensionFields() {
            return extensionFields != null ? extensionFields.stream().map(ExtensionField::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder extensionFields(Collection<ExtensionField> extensionFields) {
            this.extensionFields = ExtensionFieldListCopier.copy(extensionFields);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder extensionFields(ExtensionField... extensionFields) {
            extensionFields(Arrays.asList(extensionFields));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder extensionFields(Consumer<ExtensionField.Builder>... extensionFields) {
            extensionFields(Stream.of(extensionFields).map(c -> ExtensionField.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setExtensionFields(Collection<ExtensionField.BuilderImpl> extensionFields) {
            this.extensionFields = ExtensionFieldListCopier.copyFromBuilder(extensionFields);
        }

        @Override
        public MessageDsn build() {
            return new MessageDsn(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
