/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the raw data of the message.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RawMessage implements SdkPojo, Serializable, ToCopyableBuilder<RawMessage.Builder, RawMessage> {
    private static final SdkField<SdkBytes> DATA_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("Data").getter(getter(RawMessage::data)).setter(setter(Builder::data))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Data").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_FIELD));

    private static final long serialVersionUID = 1L;

    private final SdkBytes data;

    private RawMessage(BuilderImpl builder) {
        this.data = builder.data;
    }

    /**
     * <p>
     * The raw data of the message. This data needs to base64-encoded if you are accessing Amazon SES directly through
     * the HTTPS interface. If you are accessing Amazon SES using an AWS SDK, the SDK takes care of the base 64-encoding
     * for you. In all cases, the client must ensure that the message format complies with Internet email standards
     * regarding email header fields, MIME types, and MIME encoding.
     * </p>
     * <p>
     * The To:, CC:, and BCC: headers in the raw message can contain a group list.
     * </p>
     * <p>
     * If you are using <code>SendRawEmail</code> with sending authorization, you can include X-headers in the raw
     * message to specify the "Source," "From," and "Return-Path" addresses. For more information, see the documentation
     * for <code>SendRawEmail</code>.
     * </p>
     * <important>
     * <p>
     * Do not include these X-headers in the DKIM signature, because they are removed by Amazon SES before sending the
     * email.
     * </p>
     * </important>
     * <p>
     * For more information, go to the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-raw.html">Amazon SES Developer Guide</a>.
     * </p>
     * 
     * @return The raw data of the message. This data needs to base64-encoded if you are accessing Amazon SES directly
     *         through the HTTPS interface. If you are accessing Amazon SES using an AWS SDK, the SDK takes care of the
     *         base 64-encoding for you. In all cases, the client must ensure that the message format complies with
     *         Internet email standards regarding email header fields, MIME types, and MIME encoding.</p>
     *         <p>
     *         The To:, CC:, and BCC: headers in the raw message can contain a group list.
     *         </p>
     *         <p>
     *         If you are using <code>SendRawEmail</code> with sending authorization, you can include X-headers in the
     *         raw message to specify the "Source," "From," and "Return-Path" addresses. For more information, see the
     *         documentation for <code>SendRawEmail</code>.
     *         </p>
     *         <important>
     *         <p>
     *         Do not include these X-headers in the DKIM signature, because they are removed by Amazon SES before
     *         sending the email.
     *         </p>
     *         </important>
     *         <p>
     *         For more information, go to the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-raw.html">Amazon SES Developer
     *         Guide</a>.
     */
    public final SdkBytes data() {
        return data;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(data());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RawMessage)) {
            return false;
        }
        RawMessage other = (RawMessage) obj;
        return Objects.equals(data(), other.data());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RawMessage").add("Data", data()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RawMessage, T> g) {
        return obj -> g.apply((RawMessage) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RawMessage> {
        /**
         * <p>
         * The raw data of the message. This data needs to base64-encoded if you are accessing Amazon SES directly
         * through the HTTPS interface. If you are accessing Amazon SES using an AWS SDK, the SDK takes care of the base
         * 64-encoding for you. In all cases, the client must ensure that the message format complies with Internet
         * email standards regarding email header fields, MIME types, and MIME encoding.
         * </p>
         * <p>
         * The To:, CC:, and BCC: headers in the raw message can contain a group list.
         * </p>
         * <p>
         * If you are using <code>SendRawEmail</code> with sending authorization, you can include X-headers in the raw
         * message to specify the "Source," "From," and "Return-Path" addresses. For more information, see the
         * documentation for <code>SendRawEmail</code>.
         * </p>
         * <important>
         * <p>
         * Do not include these X-headers in the DKIM signature, because they are removed by Amazon SES before sending
         * the email.
         * </p>
         * </important>
         * <p>
         * For more information, go to the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-raw.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param data
         *        The raw data of the message. This data needs to base64-encoded if you are accessing Amazon SES
         *        directly through the HTTPS interface. If you are accessing Amazon SES using an AWS SDK, the SDK takes
         *        care of the base 64-encoding for you. In all cases, the client must ensure that the message format
         *        complies with Internet email standards regarding email header fields, MIME types, and MIME
         *        encoding.</p>
         *        <p>
         *        The To:, CC:, and BCC: headers in the raw message can contain a group list.
         *        </p>
         *        <p>
         *        If you are using <code>SendRawEmail</code> with sending authorization, you can include X-headers in
         *        the raw message to specify the "Source," "From," and "Return-Path" addresses. For more information,
         *        see the documentation for <code>SendRawEmail</code>.
         *        </p>
         *        <important>
         *        <p>
         *        Do not include these X-headers in the DKIM signature, because they are removed by Amazon SES before
         *        sending the email.
         *        </p>
         *        </important>
         *        <p>
         *        For more information, go to the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-raw.html">Amazon SES Developer
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(SdkBytes data);
    }

    static final class BuilderImpl implements Builder {
        private SdkBytes data;

        private BuilderImpl() {
        }

        private BuilderImpl(RawMessage model) {
            data(model.data);
        }

        public final ByteBuffer getData() {
            return data == null ? null : data.asByteBuffer();
        }

        @Override
        public final Builder data(SdkBytes data) {
            this.data = StandardMemberCopier.copy(data);
            return this;
        }

        public final void setData(ByteBuffer data) {
            data(data == null ? null : SdkBytes.fromByteBuffer(data));
        }

        @Override
        public RawMessage build() {
            return new RawMessage(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
