/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the destination of the message, consisting of To:, CC:, and BCC: fields.
 * </p>
 * <note>
 * <p>
 * Amazon SES does not support the SMTPUTF8 extension, as described in <a
 * href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a destination email
 * address (the part of the email address that precedes the @ sign) may only contain <a
 * href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the <i>domain part</i>
 * of an address (the part after the @ sign) contains non-ASCII characters, they must be encoded using Punycode, as
 * described in <a href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Destination implements SdkPojo, Serializable, ToCopyableBuilder<Destination.Builder, Destination> {
    private static final SdkField<List<String>> TO_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ToAddresses")
            .getter(getter(Destination::toAddresses))
            .setter(setter(Builder::toAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ToAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CC_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CcAddresses")
            .getter(getter(Destination::ccAddresses))
            .setter(setter(Builder::ccAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CcAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> BCC_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("BccAddresses")
            .getter(getter(Destination::bccAddresses))
            .setter(setter(Builder::bccAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BccAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TO_ADDRESSES_FIELD,
            CC_ADDRESSES_FIELD, BCC_ADDRESSES_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> toAddresses;

    private final List<String> ccAddresses;

    private final List<String> bccAddresses;

    private Destination(BuilderImpl builder) {
        this.toAddresses = builder.toAddresses;
        this.ccAddresses = builder.ccAddresses;
        this.bccAddresses = builder.bccAddresses;
    }

    /**
     * For responses, this returns true if the service returned a value for the ToAddresses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasToAddresses() {
        return toAddresses != null && !(toAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The recipients to place on the To: line of the message.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasToAddresses} method.
     * </p>
     * 
     * @return The recipients to place on the To: line of the message.
     */
    public final List<String> toAddresses() {
        return toAddresses;
    }

    /**
     * For responses, this returns true if the service returned a value for the CcAddresses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCcAddresses() {
        return ccAddresses != null && !(ccAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The recipients to place on the CC: line of the message.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCcAddresses} method.
     * </p>
     * 
     * @return The recipients to place on the CC: line of the message.
     */
    public final List<String> ccAddresses() {
        return ccAddresses;
    }

    /**
     * For responses, this returns true if the service returned a value for the BccAddresses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBccAddresses() {
        return bccAddresses != null && !(bccAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The recipients to place on the BCC: line of the message.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBccAddresses} method.
     * </p>
     * 
     * @return The recipients to place on the BCC: line of the message.
     */
    public final List<String> bccAddresses() {
        return bccAddresses;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasToAddresses() ? toAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCcAddresses() ? ccAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasBccAddresses() ? bccAddresses() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Destination)) {
            return false;
        }
        Destination other = (Destination) obj;
        return hasToAddresses() == other.hasToAddresses() && Objects.equals(toAddresses(), other.toAddresses())
                && hasCcAddresses() == other.hasCcAddresses() && Objects.equals(ccAddresses(), other.ccAddresses())
                && hasBccAddresses() == other.hasBccAddresses() && Objects.equals(bccAddresses(), other.bccAddresses());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Destination").add("ToAddresses", hasToAddresses() ? toAddresses() : null)
                .add("CcAddresses", hasCcAddresses() ? ccAddresses() : null)
                .add("BccAddresses", hasBccAddresses() ? bccAddresses() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ToAddresses":
            return Optional.ofNullable(clazz.cast(toAddresses()));
        case "CcAddresses":
            return Optional.ofNullable(clazz.cast(ccAddresses()));
        case "BccAddresses":
            return Optional.ofNullable(clazz.cast(bccAddresses()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Destination, T> g) {
        return obj -> g.apply((Destination) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Destination> {
        /**
         * <p>
         * The recipients to place on the To: line of the message.
         * </p>
         * 
         * @param toAddresses
         *        The recipients to place on the To: line of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder toAddresses(Collection<String> toAddresses);

        /**
         * <p>
         * The recipients to place on the To: line of the message.
         * </p>
         * 
         * @param toAddresses
         *        The recipients to place on the To: line of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder toAddresses(String... toAddresses);

        /**
         * <p>
         * The recipients to place on the CC: line of the message.
         * </p>
         * 
         * @param ccAddresses
         *        The recipients to place on the CC: line of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ccAddresses(Collection<String> ccAddresses);

        /**
         * <p>
         * The recipients to place on the CC: line of the message.
         * </p>
         * 
         * @param ccAddresses
         *        The recipients to place on the CC: line of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ccAddresses(String... ccAddresses);

        /**
         * <p>
         * The recipients to place on the BCC: line of the message.
         * </p>
         * 
         * @param bccAddresses
         *        The recipients to place on the BCC: line of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bccAddresses(Collection<String> bccAddresses);

        /**
         * <p>
         * The recipients to place on the BCC: line of the message.
         * </p>
         * 
         * @param bccAddresses
         *        The recipients to place on the BCC: line of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bccAddresses(String... bccAddresses);
    }

    static final class BuilderImpl implements Builder {
        private List<String> toAddresses = DefaultSdkAutoConstructList.getInstance();

        private List<String> ccAddresses = DefaultSdkAutoConstructList.getInstance();

        private List<String> bccAddresses = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Destination model) {
            toAddresses(model.toAddresses);
            ccAddresses(model.ccAddresses);
            bccAddresses(model.bccAddresses);
        }

        public final Collection<String> getToAddresses() {
            if (toAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return toAddresses;
        }

        public final void setToAddresses(Collection<String> toAddresses) {
            this.toAddresses = AddressListCopier.copy(toAddresses);
        }

        @Override
        @Transient
        public final Builder toAddresses(Collection<String> toAddresses) {
            this.toAddresses = AddressListCopier.copy(toAddresses);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder toAddresses(String... toAddresses) {
            toAddresses(Arrays.asList(toAddresses));
            return this;
        }

        public final Collection<String> getCcAddresses() {
            if (ccAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return ccAddresses;
        }

        public final void setCcAddresses(Collection<String> ccAddresses) {
            this.ccAddresses = AddressListCopier.copy(ccAddresses);
        }

        @Override
        @Transient
        public final Builder ccAddresses(Collection<String> ccAddresses) {
            this.ccAddresses = AddressListCopier.copy(ccAddresses);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder ccAddresses(String... ccAddresses) {
            ccAddresses(Arrays.asList(ccAddresses));
            return this;
        }

        public final Collection<String> getBccAddresses() {
            if (bccAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return bccAddresses;
        }

        public final void setBccAddresses(Collection<String> bccAddresses) {
            this.bccAddresses = AddressListCopier.copy(bccAddresses);
        }

        @Override
        @Transient
        public final Builder bccAddresses(Collection<String> bccAddresses) {
            this.bccAddresses = AddressListCopier.copy(bccAddresses);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder bccAddresses(String... bccAddresses) {
            bccAddresses(Arrays.asList(bccAddresses));
            return this;
        }

        @Override
        public Destination build() {
            return new Destination(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
