/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * When included in a receipt rule, this action calls an AWS Lambda function and, optionally, publishes a notification
 * to Amazon Simple Notification Service (Amazon SNS).
 * </p>
 * <p>
 * To enable Amazon SES to call your AWS Lambda function or to publish to an Amazon SNS topic of another account, Amazon
 * SES must have permission to access those resources. For information about giving permissions, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 * <p>
 * For information about using AWS Lambda actions in receipt rules, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LambdaAction implements SdkPojo, Serializable, ToCopyableBuilder<LambdaAction.Builder, LambdaAction> {
    private static final SdkField<String> TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TopicArn").getter(getter(LambdaAction::topicArn)).setter(setter(Builder::topicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopicArn").build()).build();

    private static final SdkField<String> FUNCTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FunctionArn").getter(getter(LambdaAction::functionArn)).setter(setter(Builder::functionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionArn").build()).build();

    private static final SdkField<String> INVOCATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InvocationType").getter(getter(LambdaAction::invocationTypeAsString))
            .setter(setter(Builder::invocationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InvocationType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TOPIC_ARN_FIELD,
            FUNCTION_ARN_FIELD, INVOCATION_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String topicArn;

    private final String functionArn;

    private final String invocationType;

    private LambdaAction(BuilderImpl builder) {
        this.topicArn = builder.topicArn;
        this.functionArn = builder.functionArn;
        this.invocationType = builder.invocationType;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon SNS topic to notify when the Lambda action is taken. An example of
     * an Amazon SNS topic ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more information about
     * Amazon SNS topics, see the <a href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS
     * Developer Guide</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon SNS topic to notify when the Lambda action is taken. An
     *         example of an Amazon SNS topic ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more
     *         information about Amazon SNS topics, see the <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
     */
    public final String topicArn() {
        return topicArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the AWS Lambda function. An example of an AWS Lambda function ARN is
     * <code>arn:aws:lambda:us-west-2:account-id:function:MyFunction</code>. For more information about AWS Lambda, see
     * the <a href="https://docs.aws.amazon.com/lambda/latest/dg/welcome.html">AWS Lambda Developer Guide</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the AWS Lambda function. An example of an AWS Lambda function ARN is
     *         <code>arn:aws:lambda:us-west-2:account-id:function:MyFunction</code>. For more information about AWS
     *         Lambda, see the <a href="https://docs.aws.amazon.com/lambda/latest/dg/welcome.html">AWS Lambda Developer
     *         Guide</a>.
     */
    public final String functionArn() {
        return functionArn;
    }

    /**
     * <p>
     * The invocation type of the AWS Lambda function. An invocation type of <code>RequestResponse</code> means that the
     * execution of the function will immediately result in a response, and a value of <code>Event</code> means that the
     * function will be invoked asynchronously. The default value is <code>Event</code>. For information about AWS
     * Lambda invocation types, see the <a href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">AWS
     * Lambda Developer Guide</a>.
     * </p>
     * <important>
     * <p>
     * There is a 30-second timeout on <code>RequestResponse</code> invocations. You should use <code>Event</code>
     * invocation in most cases. Use <code>RequestResponse</code> only when you want to make a mail flow decision, such
     * as whether to stop the receipt rule or the receipt rule set.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #invocationType}
     * will return {@link InvocationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #invocationTypeAsString}.
     * </p>
     * 
     * @return The invocation type of the AWS Lambda function. An invocation type of <code>RequestResponse</code> means
     *         that the execution of the function will immediately result in a response, and a value of
     *         <code>Event</code> means that the function will be invoked asynchronously. The default value is
     *         <code>Event</code>. For information about AWS Lambda invocation types, see the <a
     *         href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">AWS Lambda Developer Guide</a>.</p>
     *         <important>
     *         <p>
     *         There is a 30-second timeout on <code>RequestResponse</code> invocations. You should use
     *         <code>Event</code> invocation in most cases. Use <code>RequestResponse</code> only when you want to make
     *         a mail flow decision, such as whether to stop the receipt rule or the receipt rule set.
     *         </p>
     * @see InvocationType
     */
    public final InvocationType invocationType() {
        return InvocationType.fromValue(invocationType);
    }

    /**
     * <p>
     * The invocation type of the AWS Lambda function. An invocation type of <code>RequestResponse</code> means that the
     * execution of the function will immediately result in a response, and a value of <code>Event</code> means that the
     * function will be invoked asynchronously. The default value is <code>Event</code>. For information about AWS
     * Lambda invocation types, see the <a href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">AWS
     * Lambda Developer Guide</a>.
     * </p>
     * <important>
     * <p>
     * There is a 30-second timeout on <code>RequestResponse</code> invocations. You should use <code>Event</code>
     * invocation in most cases. Use <code>RequestResponse</code> only when you want to make a mail flow decision, such
     * as whether to stop the receipt rule or the receipt rule set.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #invocationType}
     * will return {@link InvocationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #invocationTypeAsString}.
     * </p>
     * 
     * @return The invocation type of the AWS Lambda function. An invocation type of <code>RequestResponse</code> means
     *         that the execution of the function will immediately result in a response, and a value of
     *         <code>Event</code> means that the function will be invoked asynchronously. The default value is
     *         <code>Event</code>. For information about AWS Lambda invocation types, see the <a
     *         href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">AWS Lambda Developer Guide</a>.</p>
     *         <important>
     *         <p>
     *         There is a 30-second timeout on <code>RequestResponse</code> invocations. You should use
     *         <code>Event</code> invocation in most cases. Use <code>RequestResponse</code> only when you want to make
     *         a mail flow decision, such as whether to stop the receipt rule or the receipt rule set.
     *         </p>
     * @see InvocationType
     */
    public final String invocationTypeAsString() {
        return invocationType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(topicArn());
        hashCode = 31 * hashCode + Objects.hashCode(functionArn());
        hashCode = 31 * hashCode + Objects.hashCode(invocationTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LambdaAction)) {
            return false;
        }
        LambdaAction other = (LambdaAction) obj;
        return Objects.equals(topicArn(), other.topicArn()) && Objects.equals(functionArn(), other.functionArn())
                && Objects.equals(invocationTypeAsString(), other.invocationTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LambdaAction").add("TopicArn", topicArn()).add("FunctionArn", functionArn())
                .add("InvocationType", invocationTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TopicArn":
            return Optional.ofNullable(clazz.cast(topicArn()));
        case "FunctionArn":
            return Optional.ofNullable(clazz.cast(functionArn()));
        case "InvocationType":
            return Optional.ofNullable(clazz.cast(invocationTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LambdaAction, T> g) {
        return obj -> g.apply((LambdaAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LambdaAction> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon SNS topic to notify when the Lambda action is taken. An example
         * of an Amazon SNS topic ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more information
         * about Amazon SNS topics, see the <a href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon
         * SNS Developer Guide</a>.
         * </p>
         * 
         * @param topicArn
         *        The Amazon Resource Name (ARN) of the Amazon SNS topic to notify when the Lambda action is taken. An
         *        example of an Amazon SNS topic ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For
         *        more information about Amazon SNS topics, see the <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topicArn(String topicArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the AWS Lambda function. An example of an AWS Lambda function ARN is
         * <code>arn:aws:lambda:us-west-2:account-id:function:MyFunction</code>. For more information about AWS Lambda,
         * see the <a href="https://docs.aws.amazon.com/lambda/latest/dg/welcome.html">AWS Lambda Developer Guide</a>.
         * </p>
         * 
         * @param functionArn
         *        The Amazon Resource Name (ARN) of the AWS Lambda function. An example of an AWS Lambda function ARN is
         *        <code>arn:aws:lambda:us-west-2:account-id:function:MyFunction</code>. For more information about AWS
         *        Lambda, see the <a href="https://docs.aws.amazon.com/lambda/latest/dg/welcome.html">AWS Lambda
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionArn(String functionArn);

        /**
         * <p>
         * The invocation type of the AWS Lambda function. An invocation type of <code>RequestResponse</code> means that
         * the execution of the function will immediately result in a response, and a value of <code>Event</code> means
         * that the function will be invoked asynchronously. The default value is <code>Event</code>. For information
         * about AWS Lambda invocation types, see the <a
         * href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">AWS Lambda Developer Guide</a>.
         * </p>
         * <important>
         * <p>
         * There is a 30-second timeout on <code>RequestResponse</code> invocations. You should use <code>Event</code>
         * invocation in most cases. Use <code>RequestResponse</code> only when you want to make a mail flow decision,
         * such as whether to stop the receipt rule or the receipt rule set.
         * </p>
         * </important>
         * 
         * @param invocationType
         *        The invocation type of the AWS Lambda function. An invocation type of <code>RequestResponse</code>
         *        means that the execution of the function will immediately result in a response, and a value of
         *        <code>Event</code> means that the function will be invoked asynchronously. The default value is
         *        <code>Event</code>. For information about AWS Lambda invocation types, see the <a
         *        href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">AWS Lambda Developer
         *        Guide</a>.</p> <important>
         *        <p>
         *        There is a 30-second timeout on <code>RequestResponse</code> invocations. You should use
         *        <code>Event</code> invocation in most cases. Use <code>RequestResponse</code> only when you want to
         *        make a mail flow decision, such as whether to stop the receipt rule or the receipt rule set.
         *        </p>
         * @see InvocationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InvocationType
         */
        Builder invocationType(String invocationType);

        /**
         * <p>
         * The invocation type of the AWS Lambda function. An invocation type of <code>RequestResponse</code> means that
         * the execution of the function will immediately result in a response, and a value of <code>Event</code> means
         * that the function will be invoked asynchronously. The default value is <code>Event</code>. For information
         * about AWS Lambda invocation types, see the <a
         * href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">AWS Lambda Developer Guide</a>.
         * </p>
         * <important>
         * <p>
         * There is a 30-second timeout on <code>RequestResponse</code> invocations. You should use <code>Event</code>
         * invocation in most cases. Use <code>RequestResponse</code> only when you want to make a mail flow decision,
         * such as whether to stop the receipt rule or the receipt rule set.
         * </p>
         * </important>
         * 
         * @param invocationType
         *        The invocation type of the AWS Lambda function. An invocation type of <code>RequestResponse</code>
         *        means that the execution of the function will immediately result in a response, and a value of
         *        <code>Event</code> means that the function will be invoked asynchronously. The default value is
         *        <code>Event</code>. For information about AWS Lambda invocation types, see the <a
         *        href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">AWS Lambda Developer
         *        Guide</a>.</p> <important>
         *        <p>
         *        There is a 30-second timeout on <code>RequestResponse</code> invocations. You should use
         *        <code>Event</code> invocation in most cases. Use <code>RequestResponse</code> only when you want to
         *        make a mail flow decision, such as whether to stop the receipt rule or the receipt rule set.
         *        </p>
         * @see InvocationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InvocationType
         */
        Builder invocationType(InvocationType invocationType);
    }

    static final class BuilderImpl implements Builder {
        private String topicArn;

        private String functionArn;

        private String invocationType;

        private BuilderImpl() {
        }

        private BuilderImpl(LambdaAction model) {
            topicArn(model.topicArn);
            functionArn(model.functionArn);
            invocationType(model.invocationType);
        }

        public final String getTopicArn() {
            return topicArn;
        }

        public final void setTopicArn(String topicArn) {
            this.topicArn = topicArn;
        }

        @Override
        @Transient
        public final Builder topicArn(String topicArn) {
            this.topicArn = topicArn;
            return this;
        }

        public final String getFunctionArn() {
            return functionArn;
        }

        public final void setFunctionArn(String functionArn) {
            this.functionArn = functionArn;
        }

        @Override
        @Transient
        public final Builder functionArn(String functionArn) {
            this.functionArn = functionArn;
            return this;
        }

        public final String getInvocationType() {
            return invocationType;
        }

        public final void setInvocationType(String invocationType) {
            this.invocationType = invocationType;
        }

        @Override
        @Transient
        public final Builder invocationType(String invocationType) {
            this.invocationType = invocationType;
            return this;
        }

        @Override
        @Transient
        public final Builder invocationType(InvocationType invocationType) {
            this.invocationType(invocationType == null ? null : invocationType.toString());
            return this;
        }

        @Override
        public LambdaAction build() {
            return new LambdaAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
