/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the DKIM attributes of a verified email address or a domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IdentityDkimAttributes implements SdkPojo, Serializable,
        ToCopyableBuilder<IdentityDkimAttributes.Builder, IdentityDkimAttributes> {
    private static final SdkField<Boolean> DKIM_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DkimEnabled").getter(getter(IdentityDkimAttributes::dkimEnabled)).setter(setter(Builder::dkimEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DkimEnabled").build()).build();

    private static final SdkField<String> DKIM_VERIFICATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DkimVerificationStatus").getter(getter(IdentityDkimAttributes::dkimVerificationStatusAsString))
            .setter(setter(Builder::dkimVerificationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DkimVerificationStatus").build())
            .build();

    private static final SdkField<List<String>> DKIM_TOKENS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DkimTokens")
            .getter(getter(IdentityDkimAttributes::dkimTokens))
            .setter(setter(Builder::dkimTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DkimTokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DKIM_ENABLED_FIELD,
            DKIM_VERIFICATION_STATUS_FIELD, DKIM_TOKENS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean dkimEnabled;

    private final String dkimVerificationStatus;

    private final List<String> dkimTokens;

    private IdentityDkimAttributes(BuilderImpl builder) {
        this.dkimEnabled = builder.dkimEnabled;
        this.dkimVerificationStatus = builder.dkimVerificationStatus;
        this.dkimTokens = builder.dkimTokens;
    }

    /**
     * <p>
     * Is true if DKIM signing is enabled for email sent from the identity. It's false otherwise. The default value is
     * true.
     * </p>
     * 
     * @return Is true if DKIM signing is enabled for email sent from the identity. It's false otherwise. The default
     *         value is true.
     */
    public final Boolean dkimEnabled() {
        return dkimEnabled;
    }

    /**
     * <p>
     * Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the domain
     * name's DNS. (This only applies to domain identities, not email address identities.)
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dkimVerificationStatus} will return {@link VerificationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dkimVerificationStatusAsString}.
     * </p>
     * 
     * @return Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the
     *         domain name's DNS. (This only applies to domain identities, not email address identities.)
     * @see VerificationStatus
     */
    public final VerificationStatus dkimVerificationStatus() {
        return VerificationStatus.fromValue(dkimVerificationStatus);
    }

    /**
     * <p>
     * Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the domain
     * name's DNS. (This only applies to domain identities, not email address identities.)
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dkimVerificationStatus} will return {@link VerificationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dkimVerificationStatusAsString}.
     * </p>
     * 
     * @return Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the
     *         domain name's DNS. (This only applies to domain identities, not email address identities.)
     * @see VerificationStatus
     */
    public final String dkimVerificationStatusAsString() {
        return dkimVerificationStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the DkimTokens property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDkimTokens() {
        return dkimTokens != null && !(dkimTokens instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of character strings that represent the domain's identity. Using these tokens, you need to create DNS CNAME
     * records that point to DKIM public keys that are hosted by Amazon SES. Amazon Web Services eventually detects that
     * you've updated your DNS records. This detection process might take up to 72 hours. After successful detection,
     * Amazon SES is able to DKIM-sign email originating from that domain. (This only applies to domain identities, not
     * email address identities.)
     * </p>
     * <p>
     * For more information about creating DNS records using DKIM tokens, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Amazon SES Developer Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDkimTokens} method.
     * </p>
     * 
     * @return A set of character strings that represent the domain's identity. Using these tokens, you need to create
     *         DNS CNAME records that point to DKIM public keys that are hosted by Amazon SES. Amazon Web Services
     *         eventually detects that you've updated your DNS records. This detection process might take up to 72
     *         hours. After successful detection, Amazon SES is able to DKIM-sign email originating from that domain.
     *         (This only applies to domain identities, not email address identities.)</p>
     *         <p>
     *         For more information about creating DNS records using DKIM tokens, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Amazon SES Developer
     *         Guide</a>.
     */
    public final List<String> dkimTokens() {
        return dkimTokens;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dkimEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(dkimVerificationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDkimTokens() ? dkimTokens() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IdentityDkimAttributes)) {
            return false;
        }
        IdentityDkimAttributes other = (IdentityDkimAttributes) obj;
        return Objects.equals(dkimEnabled(), other.dkimEnabled())
                && Objects.equals(dkimVerificationStatusAsString(), other.dkimVerificationStatusAsString())
                && hasDkimTokens() == other.hasDkimTokens() && Objects.equals(dkimTokens(), other.dkimTokens());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IdentityDkimAttributes").add("DkimEnabled", dkimEnabled())
                .add("DkimVerificationStatus", dkimVerificationStatusAsString())
                .add("DkimTokens", hasDkimTokens() ? dkimTokens() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DkimEnabled":
            return Optional.ofNullable(clazz.cast(dkimEnabled()));
        case "DkimVerificationStatus":
            return Optional.ofNullable(clazz.cast(dkimVerificationStatusAsString()));
        case "DkimTokens":
            return Optional.ofNullable(clazz.cast(dkimTokens()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IdentityDkimAttributes, T> g) {
        return obj -> g.apply((IdentityDkimAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IdentityDkimAttributes> {
        /**
         * <p>
         * Is true if DKIM signing is enabled for email sent from the identity. It's false otherwise. The default value
         * is true.
         * </p>
         * 
         * @param dkimEnabled
         *        Is true if DKIM signing is enabled for email sent from the identity. It's false otherwise. The default
         *        value is true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dkimEnabled(Boolean dkimEnabled);

        /**
         * <p>
         * Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the domain
         * name's DNS. (This only applies to domain identities, not email address identities.)
         * </p>
         * 
         * @param dkimVerificationStatus
         *        Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the
         *        domain name's DNS. (This only applies to domain identities, not email address identities.)
         * @see VerificationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerificationStatus
         */
        Builder dkimVerificationStatus(String dkimVerificationStatus);

        /**
         * <p>
         * Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the domain
         * name's DNS. (This only applies to domain identities, not email address identities.)
         * </p>
         * 
         * @param dkimVerificationStatus
         *        Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the
         *        domain name's DNS. (This only applies to domain identities, not email address identities.)
         * @see VerificationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerificationStatus
         */
        Builder dkimVerificationStatus(VerificationStatus dkimVerificationStatus);

        /**
         * <p>
         * A set of character strings that represent the domain's identity. Using these tokens, you need to create DNS
         * CNAME records that point to DKIM public keys that are hosted by Amazon SES. Amazon Web Services eventually
         * detects that you've updated your DNS records. This detection process might take up to 72 hours. After
         * successful detection, Amazon SES is able to DKIM-sign email originating from that domain. (This only applies
         * to domain identities, not email address identities.)
         * </p>
         * <p>
         * For more information about creating DNS records using DKIM tokens, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Amazon SES Developer Guide</a>.
         * </p>
         * 
         * @param dkimTokens
         *        A set of character strings that represent the domain's identity. Using these tokens, you need to
         *        create DNS CNAME records that point to DKIM public keys that are hosted by Amazon SES. Amazon Web
         *        Services eventually detects that you've updated your DNS records. This detection process might take up
         *        to 72 hours. After successful detection, Amazon SES is able to DKIM-sign email originating from that
         *        domain. (This only applies to domain identities, not email address identities.)</p>
         *        <p>
         *        For more information about creating DNS records using DKIM tokens, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Amazon SES Developer
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dkimTokens(Collection<String> dkimTokens);

        /**
         * <p>
         * A set of character strings that represent the domain's identity. Using these tokens, you need to create DNS
         * CNAME records that point to DKIM public keys that are hosted by Amazon SES. Amazon Web Services eventually
         * detects that you've updated your DNS records. This detection process might take up to 72 hours. After
         * successful detection, Amazon SES is able to DKIM-sign email originating from that domain. (This only applies
         * to domain identities, not email address identities.)
         * </p>
         * <p>
         * For more information about creating DNS records using DKIM tokens, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Amazon SES Developer Guide</a>.
         * </p>
         * 
         * @param dkimTokens
         *        A set of character strings that represent the domain's identity. Using these tokens, you need to
         *        create DNS CNAME records that point to DKIM public keys that are hosted by Amazon SES. Amazon Web
         *        Services eventually detects that you've updated your DNS records. This detection process might take up
         *        to 72 hours. After successful detection, Amazon SES is able to DKIM-sign email originating from that
         *        domain. (This only applies to domain identities, not email address identities.)</p>
         *        <p>
         *        For more information about creating DNS records using DKIM tokens, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Amazon SES Developer
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dkimTokens(String... dkimTokens);
    }

    static final class BuilderImpl implements Builder {
        private Boolean dkimEnabled;

        private String dkimVerificationStatus;

        private List<String> dkimTokens = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IdentityDkimAttributes model) {
            dkimEnabled(model.dkimEnabled);
            dkimVerificationStatus(model.dkimVerificationStatus);
            dkimTokens(model.dkimTokens);
        }

        public final Boolean getDkimEnabled() {
            return dkimEnabled;
        }

        public final void setDkimEnabled(Boolean dkimEnabled) {
            this.dkimEnabled = dkimEnabled;
        }

        @Override
        @Transient
        public final Builder dkimEnabled(Boolean dkimEnabled) {
            this.dkimEnabled = dkimEnabled;
            return this;
        }

        public final String getDkimVerificationStatus() {
            return dkimVerificationStatus;
        }

        public final void setDkimVerificationStatus(String dkimVerificationStatus) {
            this.dkimVerificationStatus = dkimVerificationStatus;
        }

        @Override
        @Transient
        public final Builder dkimVerificationStatus(String dkimVerificationStatus) {
            this.dkimVerificationStatus = dkimVerificationStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder dkimVerificationStatus(VerificationStatus dkimVerificationStatus) {
            this.dkimVerificationStatus(dkimVerificationStatus == null ? null : dkimVerificationStatus.toString());
            return this;
        }

        public final Collection<String> getDkimTokens() {
            if (dkimTokens instanceof SdkAutoConstructList) {
                return null;
            }
            return dkimTokens;
        }

        public final void setDkimTokens(Collection<String> dkimTokens) {
            this.dkimTokens = VerificationTokenListCopier.copy(dkimTokens);
        }

        @Override
        @Transient
        public final Builder dkimTokens(Collection<String> dkimTokens) {
            this.dkimTokens = VerificationTokenListCopier.copy(dkimTokens);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder dkimTokens(String... dkimTokens) {
            dkimTokens(Arrays.asList(dkimTokens));
            return this;
        }

        @Override
        public IdentityDkimAttributes build() {
            return new IdentityDkimAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
