/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the reputation settings for a configuration set.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReputationOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<ReputationOptions.Builder, ReputationOptions> {
    private static final SdkField<Boolean> SENDING_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SendingEnabled").getter(getter(ReputationOptions::sendingEnabled))
            .setter(setter(Builder::sendingEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SendingEnabled").build()).build();

    private static final SdkField<Boolean> REPUTATION_METRICS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ReputationMetricsEnabled").getter(getter(ReputationOptions::reputationMetricsEnabled))
            .setter(setter(Builder::reputationMetricsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReputationMetricsEnabled").build())
            .build();

    private static final SdkField<Instant> LAST_FRESH_START_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastFreshStart").getter(getter(ReputationOptions::lastFreshStart))
            .setter(setter(Builder::lastFreshStart))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastFreshStart").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SENDING_ENABLED_FIELD,
            REPUTATION_METRICS_ENABLED_FIELD, LAST_FRESH_START_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("SendingEnabled", SENDING_ENABLED_FIELD);
                    put("ReputationMetricsEnabled", REPUTATION_METRICS_ENABLED_FIELD);
                    put("LastFreshStart", LAST_FRESH_START_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final Boolean sendingEnabled;

    private final Boolean reputationMetricsEnabled;

    private final Instant lastFreshStart;

    private ReputationOptions(BuilderImpl builder) {
        this.sendingEnabled = builder.sendingEnabled;
        this.reputationMetricsEnabled = builder.reputationMetricsEnabled;
        this.lastFreshStart = builder.lastFreshStart;
    }

    /**
     * <p>
     * Describes whether email sending is enabled or disabled for the configuration set. If the value is
     * <code>true</code>, then Amazon SES sends emails that use the configuration set. If the value is
     * <code>false</code>, Amazon SES does not send emails that use the configuration set. The default value is
     * <code>true</code>. You can change this setting using <a>UpdateConfigurationSetSendingEnabled</a>.
     * </p>
     * 
     * @return Describes whether email sending is enabled or disabled for the configuration set. If the value is
     *         <code>true</code>, then Amazon SES sends emails that use the configuration set. If the value is
     *         <code>false</code>, Amazon SES does not send emails that use the configuration set. The default value is
     *         <code>true</code>. You can change this setting using <a>UpdateConfigurationSetSendingEnabled</a>.
     */
    public final Boolean sendingEnabled() {
        return sendingEnabled;
    }

    /**
     * <p>
     * Describes whether or not Amazon SES publishes reputation metrics for the configuration set, such as bounce and
     * complaint rates, to Amazon CloudWatch.
     * </p>
     * <p>
     * If the value is <code>true</code>, reputation metrics are published. If the value is <code>false</code>,
     * reputation metrics are not published. The default value is <code>false</code>.
     * </p>
     * 
     * @return Describes whether or not Amazon SES publishes reputation metrics for the configuration set, such as
     *         bounce and complaint rates, to Amazon CloudWatch.</p>
     *         <p>
     *         If the value is <code>true</code>, reputation metrics are published. If the value is <code>false</code>,
     *         reputation metrics are not published. The default value is <code>false</code>.
     */
    public final Boolean reputationMetricsEnabled() {
        return reputationMetricsEnabled;
    }

    /**
     * <p>
     * The date and time at which the reputation metrics for the configuration set were last reset. Resetting these
     * metrics is known as a <i>fresh start</i>.
     * </p>
     * <p>
     * When you disable email sending for a configuration set using <a>UpdateConfigurationSetSendingEnabled</a> and
     * later re-enable it, the reputation metrics for the configuration set (but not for the entire Amazon SES account)
     * are reset.
     * </p>
     * <p>
     * If email sending for the configuration set has never been disabled and later re-enabled, the value of this
     * attribute is <code>null</code>.
     * </p>
     * 
     * @return The date and time at which the reputation metrics for the configuration set were last reset. Resetting
     *         these metrics is known as a <i>fresh start</i>.</p>
     *         <p>
     *         When you disable email sending for a configuration set using <a>UpdateConfigurationSetSendingEnabled</a>
     *         and later re-enable it, the reputation metrics for the configuration set (but not for the entire Amazon
     *         SES account) are reset.
     *         </p>
     *         <p>
     *         If email sending for the configuration set has never been disabled and later re-enabled, the value of
     *         this attribute is <code>null</code>.
     */
    public final Instant lastFreshStart() {
        return lastFreshStart;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sendingEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(reputationMetricsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(lastFreshStart());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReputationOptions)) {
            return false;
        }
        ReputationOptions other = (ReputationOptions) obj;
        return Objects.equals(sendingEnabled(), other.sendingEnabled())
                && Objects.equals(reputationMetricsEnabled(), other.reputationMetricsEnabled())
                && Objects.equals(lastFreshStart(), other.lastFreshStart());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReputationOptions").add("SendingEnabled", sendingEnabled())
                .add("ReputationMetricsEnabled", reputationMetricsEnabled()).add("LastFreshStart", lastFreshStart()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SendingEnabled":
            return Optional.ofNullable(clazz.cast(sendingEnabled()));
        case "ReputationMetricsEnabled":
            return Optional.ofNullable(clazz.cast(reputationMetricsEnabled()));
        case "LastFreshStart":
            return Optional.ofNullable(clazz.cast(lastFreshStart()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ReputationOptions, T> g) {
        return obj -> g.apply((ReputationOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReputationOptions> {
        /**
         * <p>
         * Describes whether email sending is enabled or disabled for the configuration set. If the value is
         * <code>true</code>, then Amazon SES sends emails that use the configuration set. If the value is
         * <code>false</code>, Amazon SES does not send emails that use the configuration set. The default value is
         * <code>true</code>. You can change this setting using <a>UpdateConfigurationSetSendingEnabled</a>.
         * </p>
         * 
         * @param sendingEnabled
         *        Describes whether email sending is enabled or disabled for the configuration set. If the value is
         *        <code>true</code>, then Amazon SES sends emails that use the configuration set. If the value is
         *        <code>false</code>, Amazon SES does not send emails that use the configuration set. The default value
         *        is <code>true</code>. You can change this setting using <a>UpdateConfigurationSetSendingEnabled</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sendingEnabled(Boolean sendingEnabled);

        /**
         * <p>
         * Describes whether or not Amazon SES publishes reputation metrics for the configuration set, such as bounce
         * and complaint rates, to Amazon CloudWatch.
         * </p>
         * <p>
         * If the value is <code>true</code>, reputation metrics are published. If the value is <code>false</code>,
         * reputation metrics are not published. The default value is <code>false</code>.
         * </p>
         * 
         * @param reputationMetricsEnabled
         *        Describes whether or not Amazon SES publishes reputation metrics for the configuration set, such as
         *        bounce and complaint rates, to Amazon CloudWatch.</p>
         *        <p>
         *        If the value is <code>true</code>, reputation metrics are published. If the value is
         *        <code>false</code>, reputation metrics are not published. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reputationMetricsEnabled(Boolean reputationMetricsEnabled);

        /**
         * <p>
         * The date and time at which the reputation metrics for the configuration set were last reset. Resetting these
         * metrics is known as a <i>fresh start</i>.
         * </p>
         * <p>
         * When you disable email sending for a configuration set using <a>UpdateConfigurationSetSendingEnabled</a> and
         * later re-enable it, the reputation metrics for the configuration set (but not for the entire Amazon SES
         * account) are reset.
         * </p>
         * <p>
         * If email sending for the configuration set has never been disabled and later re-enabled, the value of this
         * attribute is <code>null</code>.
         * </p>
         * 
         * @param lastFreshStart
         *        The date and time at which the reputation metrics for the configuration set were last reset. Resetting
         *        these metrics is known as a <i>fresh start</i>.</p>
         *        <p>
         *        When you disable email sending for a configuration set using
         *        <a>UpdateConfigurationSetSendingEnabled</a> and later re-enable it, the reputation metrics for the
         *        configuration set (but not for the entire Amazon SES account) are reset.
         *        </p>
         *        <p>
         *        If email sending for the configuration set has never been disabled and later re-enabled, the value of
         *        this attribute is <code>null</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastFreshStart(Instant lastFreshStart);
    }

    static final class BuilderImpl implements Builder {
        private Boolean sendingEnabled;

        private Boolean reputationMetricsEnabled;

        private Instant lastFreshStart;

        private BuilderImpl() {
        }

        private BuilderImpl(ReputationOptions model) {
            sendingEnabled(model.sendingEnabled);
            reputationMetricsEnabled(model.reputationMetricsEnabled);
            lastFreshStart(model.lastFreshStart);
        }

        public final Boolean getSendingEnabled() {
            return sendingEnabled;
        }

        public final void setSendingEnabled(Boolean sendingEnabled) {
            this.sendingEnabled = sendingEnabled;
        }

        @Override
        public final Builder sendingEnabled(Boolean sendingEnabled) {
            this.sendingEnabled = sendingEnabled;
            return this;
        }

        public final Boolean getReputationMetricsEnabled() {
            return reputationMetricsEnabled;
        }

        public final void setReputationMetricsEnabled(Boolean reputationMetricsEnabled) {
            this.reputationMetricsEnabled = reputationMetricsEnabled;
        }

        @Override
        public final Builder reputationMetricsEnabled(Boolean reputationMetricsEnabled) {
            this.reputationMetricsEnabled = reputationMetricsEnabled;
            return this;
        }

        public final Instant getLastFreshStart() {
            return lastFreshStart;
        }

        public final void setLastFreshStart(Instant lastFreshStart) {
            this.lastFreshStart = lastFreshStart;
        }

        @Override
        public final Builder lastFreshStart(Instant lastFreshStart) {
            this.lastFreshStart = lastFreshStart;
            return this;
        }

        @Override
        public ReputationOptions build() {
            return new ReputationOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
