/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Receipt rules enable you to specify which actions Amazon SES should take when it receives mail on behalf of one or
 * more email addresses or domains that you own.
 * </p>
 * <p>
 * Each receipt rule defines a set of email addresses or domains that it applies to. If the email addresses or domains
 * match at least one recipient address of the message, Amazon SES executes all of the receipt rule's actions on the
 * message.
 * </p>
 * <p>
 * For information about setting up receipt rules, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-receipt-rules.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReceiptRule implements SdkPojo, Serializable, ToCopyableBuilder<ReceiptRule.Builder, ReceiptRule> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReceiptRule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ReceiptRule::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<String> TLS_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReceiptRule::tlsPolicyAsString)).setter(setter(Builder::tlsPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TlsPolicy").build()).build();

    private static final SdkField<List<String>> RECIPIENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ReceiptRule::recipients))
            .setter(setter(Builder::recipients))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recipients").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ReceiptAction>> ACTIONS_FIELD = SdkField
            .<List<ReceiptAction>> builder(MarshallingType.LIST)
            .getter(getter(ReceiptRule::actions))
            .setter(setter(Builder::actions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ReceiptAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(ReceiptAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> SCAN_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ReceiptRule::scanEnabled)).setter(setter(Builder::scanEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScanEnabled").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ENABLED_FIELD,
            TLS_POLICY_FIELD, RECIPIENTS_FIELD, ACTIONS_FIELD, SCAN_ENABLED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Boolean enabled;

    private final String tlsPolicy;

    private final List<String> recipients;

    private final List<ReceiptAction> actions;

    private final Boolean scanEnabled;

    private ReceiptRule(BuilderImpl builder) {
        this.name = builder.name;
        this.enabled = builder.enabled;
        this.tlsPolicy = builder.tlsPolicy;
        this.recipients = builder.recipients;
        this.actions = builder.actions;
        this.scanEnabled = builder.scanEnabled;
    }

    /**
     * <p>
     * The name of the receipt rule. The name must:
     * </p>
     * <ul>
     * <li>
     * <p>
     * This value can only contain ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     * </p>
     * </li>
     * <li>
     * <p>
     * Start and end with a letter or number.
     * </p>
     * </li>
     * <li>
     * <p>
     * Contain less than 64 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the receipt rule. The name must:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         This value can only contain ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start and end with a letter or number.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Contain less than 64 characters.
     *         </p>
     *         </li>
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * If <code>true</code>, the receipt rule is active. The default value is <code>false</code>.
     * </p>
     * 
     * @return If <code>true</code>, the receipt rule is active. The default value is <code>false</code>.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted with
     * Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will bounce emails
     * that are not received over TLS. The default is <code>Optional</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tlsPolicy} will
     * return {@link TlsPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tlsPolicyAsString}.
     * </p>
     * 
     * @return Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted
     *         with Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will
     *         bounce emails that are not received over TLS. The default is <code>Optional</code>.
     * @see TlsPolicy
     */
    public TlsPolicy tlsPolicy() {
        return TlsPolicy.fromValue(tlsPolicy);
    }

    /**
     * <p>
     * Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted with
     * Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will bounce emails
     * that are not received over TLS. The default is <code>Optional</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tlsPolicy} will
     * return {@link TlsPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tlsPolicyAsString}.
     * </p>
     * 
     * @return Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted
     *         with Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will
     *         bounce emails that are not received over TLS. The default is <code>Optional</code>.
     * @see TlsPolicy
     */
    public String tlsPolicyAsString() {
        return tlsPolicy;
    }

    /**
     * <p>
     * The recipient domains and email addresses that the receipt rule applies to. If this field is not specified, this
     * rule will match all recipients under all verified domains.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The recipient domains and email addresses that the receipt rule applies to. If this field is not
     *         specified, this rule will match all recipients under all verified domains.
     */
    public List<String> recipients() {
        return recipients;
    }

    /**
     * <p>
     * An ordered list of actions to perform on messages that match at least one of the recipient email addresses or
     * domains specified in the receipt rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An ordered list of actions to perform on messages that match at least one of the recipient email
     *         addresses or domains specified in the receipt rule.
     */
    public List<ReceiptAction> actions() {
        return actions;
    }

    /**
     * <p>
     * If <code>true</code>, then messages that this receipt rule applies to are scanned for spam and viruses. The
     * default value is <code>false</code>.
     * </p>
     * 
     * @return If <code>true</code>, then messages that this receipt rule applies to are scanned for spam and viruses.
     *         The default value is <code>false</code>.
     */
    public Boolean scanEnabled() {
        return scanEnabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(tlsPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(recipients());
        hashCode = 31 * hashCode + Objects.hashCode(actions());
        hashCode = 31 * hashCode + Objects.hashCode(scanEnabled());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReceiptRule)) {
            return false;
        }
        ReceiptRule other = (ReceiptRule) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(enabled(), other.enabled())
                && Objects.equals(tlsPolicyAsString(), other.tlsPolicyAsString())
                && Objects.equals(recipients(), other.recipients()) && Objects.equals(actions(), other.actions())
                && Objects.equals(scanEnabled(), other.scanEnabled());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ReceiptRule").add("Name", name()).add("Enabled", enabled())
                .add("TlsPolicy", tlsPolicyAsString()).add("Recipients", recipients()).add("Actions", actions())
                .add("ScanEnabled", scanEnabled()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "TlsPolicy":
            return Optional.ofNullable(clazz.cast(tlsPolicyAsString()));
        case "Recipients":
            return Optional.ofNullable(clazz.cast(recipients()));
        case "Actions":
            return Optional.ofNullable(clazz.cast(actions()));
        case "ScanEnabled":
            return Optional.ofNullable(clazz.cast(scanEnabled()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReceiptRule, T> g) {
        return obj -> g.apply((ReceiptRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReceiptRule> {
        /**
         * <p>
         * The name of the receipt rule. The name must:
         * </p>
         * <ul>
         * <li>
         * <p>
         * This value can only contain ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Start and end with a letter or number.
         * </p>
         * </li>
         * <li>
         * <p>
         * Contain less than 64 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the receipt rule. The name must:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        This value can only contain ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Start and end with a letter or number.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Contain less than 64 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If <code>true</code>, the receipt rule is active. The default value is <code>false</code>.
         * </p>
         * 
         * @param enabled
         *        If <code>true</code>, the receipt rule is active. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted with
         * Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will bounce
         * emails that are not received over TLS. The default is <code>Optional</code>.
         * </p>
         * 
         * @param tlsPolicy
         *        Specifies whether Amazon SES should require that incoming email is delivered over a connection
         *        encrypted with Transport Layer Security (TLS). If this parameter is set to <code>Require</code>,
         *        Amazon SES will bounce emails that are not received over TLS. The default is <code>Optional</code>.
         * @see TlsPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TlsPolicy
         */
        Builder tlsPolicy(String tlsPolicy);

        /**
         * <p>
         * Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted with
         * Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will bounce
         * emails that are not received over TLS. The default is <code>Optional</code>.
         * </p>
         * 
         * @param tlsPolicy
         *        Specifies whether Amazon SES should require that incoming email is delivered over a connection
         *        encrypted with Transport Layer Security (TLS). If this parameter is set to <code>Require</code>,
         *        Amazon SES will bounce emails that are not received over TLS. The default is <code>Optional</code>.
         * @see TlsPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TlsPolicy
         */
        Builder tlsPolicy(TlsPolicy tlsPolicy);

        /**
         * <p>
         * The recipient domains and email addresses that the receipt rule applies to. If this field is not specified,
         * this rule will match all recipients under all verified domains.
         * </p>
         * 
         * @param recipients
         *        The recipient domains and email addresses that the receipt rule applies to. If this field is not
         *        specified, this rule will match all recipients under all verified domains.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipients(Collection<String> recipients);

        /**
         * <p>
         * The recipient domains and email addresses that the receipt rule applies to. If this field is not specified,
         * this rule will match all recipients under all verified domains.
         * </p>
         * 
         * @param recipients
         *        The recipient domains and email addresses that the receipt rule applies to. If this field is not
         *        specified, this rule will match all recipients under all verified domains.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipients(String... recipients);

        /**
         * <p>
         * An ordered list of actions to perform on messages that match at least one of the recipient email addresses or
         * domains specified in the receipt rule.
         * </p>
         * 
         * @param actions
         *        An ordered list of actions to perform on messages that match at least one of the recipient email
         *        addresses or domains specified in the receipt rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<ReceiptAction> actions);

        /**
         * <p>
         * An ordered list of actions to perform on messages that match at least one of the recipient email addresses or
         * domains specified in the receipt rule.
         * </p>
         * 
         * @param actions
         *        An ordered list of actions to perform on messages that match at least one of the recipient email
         *        addresses or domains specified in the receipt rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(ReceiptAction... actions);

        /**
         * <p>
         * An ordered list of actions to perform on messages that match at least one of the recipient email addresses or
         * domains specified in the receipt rule.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ReceiptAction>.Builder} avoiding the need
         * to create one manually via {@link List<ReceiptAction>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ReceiptAction>.Builder#build()} is called immediately and
         * its result is passed to {@link #actions(List<ReceiptAction>)}.
         * 
         * @param actions
         *        a consumer that will call methods on {@link List<ReceiptAction>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actions(List<ReceiptAction>)
         */
        Builder actions(Consumer<ReceiptAction.Builder>... actions);

        /**
         * <p>
         * If <code>true</code>, then messages that this receipt rule applies to are scanned for spam and viruses. The
         * default value is <code>false</code>.
         * </p>
         * 
         * @param scanEnabled
         *        If <code>true</code>, then messages that this receipt rule applies to are scanned for spam and
         *        viruses. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanEnabled(Boolean scanEnabled);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Boolean enabled;

        private String tlsPolicy;

        private List<String> recipients = DefaultSdkAutoConstructList.getInstance();

        private List<ReceiptAction> actions = DefaultSdkAutoConstructList.getInstance();

        private Boolean scanEnabled;

        private BuilderImpl() {
        }

        private BuilderImpl(ReceiptRule model) {
            name(model.name);
            enabled(model.enabled);
            tlsPolicy(model.tlsPolicy);
            recipients(model.recipients);
            actions(model.actions);
            scanEnabled(model.scanEnabled);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final String getTlsPolicyAsString() {
            return tlsPolicy;
        }

        @Override
        public final Builder tlsPolicy(String tlsPolicy) {
            this.tlsPolicy = tlsPolicy;
            return this;
        }

        @Override
        public final Builder tlsPolicy(TlsPolicy tlsPolicy) {
            this.tlsPolicy(tlsPolicy.toString());
            return this;
        }

        public final void setTlsPolicy(String tlsPolicy) {
            this.tlsPolicy = tlsPolicy;
        }

        public final Collection<String> getRecipients() {
            return recipients;
        }

        @Override
        public final Builder recipients(Collection<String> recipients) {
            this.recipients = RecipientsListCopier.copy(recipients);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recipients(String... recipients) {
            recipients(Arrays.asList(recipients));
            return this;
        }

        public final void setRecipients(Collection<String> recipients) {
            this.recipients = RecipientsListCopier.copy(recipients);
        }

        public final Collection<ReceiptAction.Builder> getActions() {
            return actions != null ? actions.stream().map(ReceiptAction::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder actions(Collection<ReceiptAction> actions) {
            this.actions = ReceiptActionsListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(ReceiptAction... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Consumer<ReceiptAction.Builder>... actions) {
            actions(Stream.of(actions).map(c -> ReceiptAction.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setActions(Collection<ReceiptAction.BuilderImpl> actions) {
            this.actions = ReceiptActionsListCopier.copyFromBuilder(actions);
        }

        public final Boolean getScanEnabled() {
            return scanEnabled;
        }

        @Override
        public final Builder scanEnabled(Boolean scanEnabled) {
            this.scanEnabled = scanEnabled;
            return this;
        }

        public final void setScanEnabled(Boolean scanEnabled) {
            this.scanEnabled = scanEnabled;
        }

        @Override
        public ReceiptRule build() {
            return new ReceiptRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
