/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * When included in a receipt rule, this action saves the received message to an Amazon Simple Storage Service (Amazon
 * S3) bucket and, optionally, publishes a notification to Amazon Simple Notification Service (Amazon SNS).
 * </p>
 * <p>
 * To enable Amazon SES to write emails to your Amazon S3 bucket, use an AWS KMS key to encrypt your emails, or publish
 * to an Amazon SNS topic of another account, Amazon SES must have permission to access those resources. For information
 * about giving permissions, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 * <note>
 * <p>
 * When you save your emails to an Amazon S3 bucket, the maximum email size (including headers) is 30 MB. Emails larger
 * than that will bounce.
 * </p>
 * </note>
 * <p>
 * For information about specifying Amazon S3 actions in receipt rules, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-s3.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Action implements SdkPojo, Serializable, ToCopyableBuilder<S3Action.Builder, S3Action> {
    private static final SdkField<String> TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Action::topicArn)).setter(setter(Builder::topicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopicArn").build()).build();

    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Action::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<String> OBJECT_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Action::objectKeyPrefix)).setter(setter(Builder::objectKeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectKeyPrefix").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Action::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TOPIC_ARN_FIELD,
            BUCKET_NAME_FIELD, OBJECT_KEY_PREFIX_FIELD, KMS_KEY_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String topicArn;

    private final String bucketName;

    private final String objectKeyPrefix;

    private final String kmsKeyArn;

    private S3Action(BuilderImpl builder) {
        this.topicArn = builder.topicArn;
        this.bucketName = builder.bucketName;
        this.objectKeyPrefix = builder.objectKeyPrefix;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The ARN of the Amazon SNS topic to notify when the message is saved to the Amazon S3 bucket. An example of an
     * Amazon SNS topic ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more information about
     * Amazon SNS topics, see the <a href="http://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS
     * Developer Guide</a>.
     * </p>
     * 
     * @return The ARN of the Amazon SNS topic to notify when the message is saved to the Amazon S3 bucket. An example
     *         of an Amazon SNS topic ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more
     *         information about Amazon SNS topics, see the <a
     *         href="http://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
     */
    public String topicArn() {
        return topicArn;
    }

    /**
     * <p>
     * The name of the Amazon S3 bucket that incoming email will be saved to.
     * </p>
     * 
     * @return The name of the Amazon S3 bucket that incoming email will be saved to.
     */
    public String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * The key prefix of the Amazon S3 bucket. The key prefix is similar to a directory name that enables you to store
     * similar data under the same directory in a bucket.
     * </p>
     * 
     * @return The key prefix of the Amazon S3 bucket. The key prefix is similar to a directory name that enables you to
     *         store similar data under the same directory in a bucket.
     */
    public String objectKeyPrefix() {
        return objectKeyPrefix;
    }

    /**
     * <p>
     * The customer master key that Amazon SES should use to encrypt your emails before saving them to the Amazon S3
     * bucket. You can use the default master key or a custom master key you created in AWS KMS as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To use the default master key, provide an ARN in the form of
     * <code>arn:aws:kms:REGION:ACCOUNT-ID-WITHOUT-HYPHENS:alias/aws/ses</code>. For example, if your AWS account ID is
     * 123456789012 and you want to use the default master key in the US West (Oregon) region, the ARN of the default
     * master key would be <code>arn:aws:kms:us-west-2:123456789012:alias/aws/ses</code>. If you use the default master
     * key, you don't need to perform any extra steps to give Amazon SES permission to use the key.
     * </p>
     * </li>
     * <li>
     * <p>
     * To use a custom master key you created in AWS KMS, provide the ARN of the master key and ensure that you add a
     * statement to your key's policy to give Amazon SES permission to use it. For more information about giving
     * permissions, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about key policies, see the <a
     * href="http://docs.aws.amazon.com/kms/latest/developerguide/concepts.html">AWS KMS Developer Guide</a>. If you do
     * not specify a master key, Amazon SES will not encrypt your emails.
     * </p>
     * <important>
     * <p>
     * Your mail is encrypted by Amazon SES using the Amazon S3 encryption client before the mail is submitted to Amazon
     * S3 for storage. It is not encrypted using Amazon S3 server-side encryption. This means that you must use the
     * Amazon S3 encryption client to decrypt the email after retrieving it from Amazon S3, as the service has no access
     * to use your AWS KMS keys for decryption. This encryption client is currently available with the <a
     * href="http://aws.amazon.com/sdk-for-java/">AWS SDK for Java</a> and <a
     * href="http://aws.amazon.com/sdk-for-ruby/">AWS SDK for Ruby</a> only. For more information about client-side
     * encryption using AWS KMS master keys, see the <a
     * href="http://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html">Amazon S3 Developer
     * Guide</a>.
     * </p>
     * </important>
     * 
     * @return The customer master key that Amazon SES should use to encrypt your emails before saving them to the
     *         Amazon S3 bucket. You can use the default master key or a custom master key you created in AWS KMS as
     *         follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To use the default master key, provide an ARN in the form of
     *         <code>arn:aws:kms:REGION:ACCOUNT-ID-WITHOUT-HYPHENS:alias/aws/ses</code>. For example, if your AWS
     *         account ID is 123456789012 and you want to use the default master key in the US West (Oregon) region, the
     *         ARN of the default master key would be <code>arn:aws:kms:us-west-2:123456789012:alias/aws/ses</code>. If
     *         you use the default master key, you don't need to perform any extra steps to give Amazon SES permission
     *         to use the key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To use a custom master key you created in AWS KMS, provide the ARN of the master key and ensure that you
     *         add a statement to your key's policy to give Amazon SES permission to use it. For more information about
     *         giving permissions, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html">Amazon SES
     *         Developer Guide</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about key policies, see the <a
     *         href="http://docs.aws.amazon.com/kms/latest/developerguide/concepts.html">AWS KMS Developer Guide</a>. If
     *         you do not specify a master key, Amazon SES will not encrypt your emails.
     *         </p>
     *         <important>
     *         <p>
     *         Your mail is encrypted by Amazon SES using the Amazon S3 encryption client before the mail is submitted
     *         to Amazon S3 for storage. It is not encrypted using Amazon S3 server-side encryption. This means that you
     *         must use the Amazon S3 encryption client to decrypt the email after retrieving it from Amazon S3, as the
     *         service has no access to use your AWS KMS keys for decryption. This encryption client is currently
     *         available with the <a href="http://aws.amazon.com/sdk-for-java/">AWS SDK for Java</a> and <a
     *         href="http://aws.amazon.com/sdk-for-ruby/">AWS SDK for Ruby</a> only. For more information about
     *         client-side encryption using AWS KMS master keys, see the <a
     *         href="http://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html">Amazon S3 Developer
     *         Guide</a>.
     *         </p>
     */
    public String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(topicArn());
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(objectKeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Action)) {
            return false;
        }
        S3Action other = (S3Action) obj;
        return Objects.equals(topicArn(), other.topicArn()) && Objects.equals(bucketName(), other.bucketName())
                && Objects.equals(objectKeyPrefix(), other.objectKeyPrefix()) && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3Action").add("TopicArn", topicArn()).add("BucketName", bucketName())
                .add("ObjectKeyPrefix", objectKeyPrefix()).add("KmsKeyArn", kmsKeyArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TopicArn":
            return Optional.ofNullable(clazz.cast(topicArn()));
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "ObjectKeyPrefix":
            return Optional.ofNullable(clazz.cast(objectKeyPrefix()));
        case "KmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3Action, T> g) {
        return obj -> g.apply((S3Action) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Action> {
        /**
         * <p>
         * The ARN of the Amazon SNS topic to notify when the message is saved to the Amazon S3 bucket. An example of an
         * Amazon SNS topic ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more information about
         * Amazon SNS topics, see the <a href="http://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS
         * Developer Guide</a>.
         * </p>
         * 
         * @param topicArn
         *        The ARN of the Amazon SNS topic to notify when the message is saved to the Amazon S3 bucket. An
         *        example of an Amazon SNS topic ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For
         *        more information about Amazon SNS topics, see the <a
         *        href="http://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topicArn(String topicArn);

        /**
         * <p>
         * The name of the Amazon S3 bucket that incoming email will be saved to.
         * </p>
         * 
         * @param bucketName
         *        The name of the Amazon S3 bucket that incoming email will be saved to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * The key prefix of the Amazon S3 bucket. The key prefix is similar to a directory name that enables you to
         * store similar data under the same directory in a bucket.
         * </p>
         * 
         * @param objectKeyPrefix
         *        The key prefix of the Amazon S3 bucket. The key prefix is similar to a directory name that enables you
         *        to store similar data under the same directory in a bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectKeyPrefix(String objectKeyPrefix);

        /**
         * <p>
         * The customer master key that Amazon SES should use to encrypt your emails before saving them to the Amazon S3
         * bucket. You can use the default master key or a custom master key you created in AWS KMS as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To use the default master key, provide an ARN in the form of
         * <code>arn:aws:kms:REGION:ACCOUNT-ID-WITHOUT-HYPHENS:alias/aws/ses</code>. For example, if your AWS account ID
         * is 123456789012 and you want to use the default master key in the US West (Oregon) region, the ARN of the
         * default master key would be <code>arn:aws:kms:us-west-2:123456789012:alias/aws/ses</code>. If you use the
         * default master key, you don't need to perform any extra steps to give Amazon SES permission to use the key.
         * </p>
         * </li>
         * <li>
         * <p>
         * To use a custom master key you created in AWS KMS, provide the ARN of the master key and ensure that you add
         * a statement to your key's policy to give Amazon SES permission to use it. For more information about giving
         * permissions, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html">Amazon SES
         * Developer Guide</a>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about key policies, see the <a
         * href="http://docs.aws.amazon.com/kms/latest/developerguide/concepts.html">AWS KMS Developer Guide</a>. If you
         * do not specify a master key, Amazon SES will not encrypt your emails.
         * </p>
         * <important>
         * <p>
         * Your mail is encrypted by Amazon SES using the Amazon S3 encryption client before the mail is submitted to
         * Amazon S3 for storage. It is not encrypted using Amazon S3 server-side encryption. This means that you must
         * use the Amazon S3 encryption client to decrypt the email after retrieving it from Amazon S3, as the service
         * has no access to use your AWS KMS keys for decryption. This encryption client is currently available with the
         * <a href="http://aws.amazon.com/sdk-for-java/">AWS SDK for Java</a> and <a
         * href="http://aws.amazon.com/sdk-for-ruby/">AWS SDK for Ruby</a> only. For more information about client-side
         * encryption using AWS KMS master keys, see the <a
         * href="http://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html">Amazon S3 Developer
         * Guide</a>.
         * </p>
         * </important>
         * 
         * @param kmsKeyArn
         *        The customer master key that Amazon SES should use to encrypt your emails before saving them to the
         *        Amazon S3 bucket. You can use the default master key or a custom master key you created in AWS KMS as
         *        follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To use the default master key, provide an ARN in the form of
         *        <code>arn:aws:kms:REGION:ACCOUNT-ID-WITHOUT-HYPHENS:alias/aws/ses</code>. For example, if your AWS
         *        account ID is 123456789012 and you want to use the default master key in the US West (Oregon) region,
         *        the ARN of the default master key would be
         *        <code>arn:aws:kms:us-west-2:123456789012:alias/aws/ses</code>. If you use the default master key, you
         *        don't need to perform any extra steps to give Amazon SES permission to use the key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To use a custom master key you created in AWS KMS, provide the ARN of the master key and ensure that
         *        you add a statement to your key's policy to give Amazon SES permission to use it. For more information
         *        about giving permissions, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html">Amazon
         *        SES Developer Guide</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about key policies, see the <a
         *        href="http://docs.aws.amazon.com/kms/latest/developerguide/concepts.html">AWS KMS Developer Guide</a>.
         *        If you do not specify a master key, Amazon SES will not encrypt your emails.
         *        </p>
         *        <important>
         *        <p>
         *        Your mail is encrypted by Amazon SES using the Amazon S3 encryption client before the mail is
         *        submitted to Amazon S3 for storage. It is not encrypted using Amazon S3 server-side encryption. This
         *        means that you must use the Amazon S3 encryption client to decrypt the email after retrieving it from
         *        Amazon S3, as the service has no access to use your AWS KMS keys for decryption. This encryption
         *        client is currently available with the <a href="http://aws.amazon.com/sdk-for-java/">AWS SDK for
         *        Java</a> and <a href="http://aws.amazon.com/sdk-for-ruby/">AWS SDK for Ruby</a> only. For more
         *        information about client-side encryption using AWS KMS master keys, see the <a
         *        href="http://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html">Amazon S3
         *        Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String topicArn;

        private String bucketName;

        private String objectKeyPrefix;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Action model) {
            topicArn(model.topicArn);
            bucketName(model.bucketName);
            objectKeyPrefix(model.objectKeyPrefix);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getTopicArn() {
            return topicArn;
        }

        @Override
        public final Builder topicArn(String topicArn) {
            this.topicArn = topicArn;
            return this;
        }

        public final void setTopicArn(String topicArn) {
            this.topicArn = topicArn;
        }

        public final String getBucketName() {
            return bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        public final String getObjectKeyPrefix() {
            return objectKeyPrefix;
        }

        @Override
        public final Builder objectKeyPrefix(String objectKeyPrefix) {
            this.objectKeyPrefix = objectKeyPrefix;
            return this;
        }

        public final void setObjectKeyPrefix(String objectKeyPrefix) {
            this.objectKeyPrefix = objectKeyPrefix;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public S3Action build() {
            return new S3Action(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
