/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a templated email to multiple destinations using Amazon SES. For more information, see
 * the <a href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-personalized-email-api.html">Amazon SES
 * Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendBulkTemplatedEmailRequest extends SesRequest implements
        ToCopyableBuilder<SendBulkTemplatedEmailRequest.Builder, SendBulkTemplatedEmailRequest> {
    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendBulkTemplatedEmailRequest::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<String> SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendBulkTemplatedEmailRequest::sourceArn)).setter(setter(Builder::sourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceArn").build()).build();

    private static final SdkField<List<String>> REPLY_TO_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(SendBulkTemplatedEmailRequest::replyToAddresses))
            .setter(setter(Builder::replyToAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplyToAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RETURN_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendBulkTemplatedEmailRequest::returnPath)).setter(setter(Builder::returnPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReturnPath").build()).build();

    private static final SdkField<String> RETURN_PATH_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendBulkTemplatedEmailRequest::returnPathArn)).setter(setter(Builder::returnPathArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReturnPathArn").build()).build();

    private static final SdkField<String> CONFIGURATION_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendBulkTemplatedEmailRequest::configurationSetName)).setter(setter(Builder::configurationSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSetName").build())
            .build();

    private static final SdkField<List<MessageTag>> DEFAULT_TAGS_FIELD = SdkField
            .<List<MessageTag>> builder(MarshallingType.LIST)
            .getter(getter(SendBulkTemplatedEmailRequest::defaultTags))
            .setter(setter(Builder::defaultTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MessageTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TEMPLATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendBulkTemplatedEmailRequest::template)).setter(setter(Builder::template))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Template").build()).build();

    private static final SdkField<String> TEMPLATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendBulkTemplatedEmailRequest::templateArn)).setter(setter(Builder::templateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateArn").build()).build();

    private static final SdkField<String> DEFAULT_TEMPLATE_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SendBulkTemplatedEmailRequest::defaultTemplateData)).setter(setter(Builder::defaultTemplateData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultTemplateData").build())
            .build();

    private static final SdkField<List<BulkEmailDestination>> DESTINATIONS_FIELD = SdkField
            .<List<BulkEmailDestination>> builder(MarshallingType.LIST)
            .getter(getter(SendBulkTemplatedEmailRequest::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BulkEmailDestination> builder(MarshallingType.SDK_POJO)
                                            .constructor(BulkEmailDestination::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_FIELD,
            SOURCE_ARN_FIELD, REPLY_TO_ADDRESSES_FIELD, RETURN_PATH_FIELD, RETURN_PATH_ARN_FIELD, CONFIGURATION_SET_NAME_FIELD,
            DEFAULT_TAGS_FIELD, TEMPLATE_FIELD, TEMPLATE_ARN_FIELD, DEFAULT_TEMPLATE_DATA_FIELD, DESTINATIONS_FIELD));

    private final String source;

    private final String sourceArn;

    private final List<String> replyToAddresses;

    private final String returnPath;

    private final String returnPathArn;

    private final String configurationSetName;

    private final List<MessageTag> defaultTags;

    private final String template;

    private final String templateArn;

    private final String defaultTemplateData;

    private final List<BulkEmailDestination> destinations;

    private SendBulkTemplatedEmailRequest(BuilderImpl builder) {
        super(builder);
        this.source = builder.source;
        this.sourceArn = builder.sourceArn;
        this.replyToAddresses = builder.replyToAddresses;
        this.returnPath = builder.returnPath;
        this.returnPathArn = builder.returnPathArn;
        this.configurationSetName = builder.configurationSetName;
        this.defaultTags = builder.defaultTags;
        this.template = builder.template;
        this.templateArn = builder.templateArn;
        this.defaultTemplateData = builder.defaultTemplateData;
        this.destinations = builder.destinations;
    }

    /**
     * <p>
     * The email address that is sending the email. This email address must be either individually verified with Amazon
     * SES, or from a domain that has been verified with Amazon SES. For information about verifying identities, see the
     * <a href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
     * Developer Guide</a>.
     * </p>
     * <p>
     * If you are sending on behalf of another user and have been permitted to do so by a sending authorization policy,
     * then you must also specify the <code>SourceArn</code> parameter. For more information about sending
     * authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * <note>
     * <p>
     * Amazon SES does not support the SMTPUTF8 extension, as described in <a
     * href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a source email
     * address (the part of the email address that precedes the @ sign) may only contain <a
     * href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the <i>domain
     * part</i> of an address (the part after the @ sign) contains non-ASCII characters, they must be encoded using
     * Punycode, as described in <a href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>. The sender name (also
     * known as the <i>friendly name</i>) may contain non-ASCII characters. These characters must be encoded using MIME
     * encoded-word syntax, as described in <a href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>. MIME
     * encoded-word syntax uses the following form: <code>=?charset?encoding?encoded-text?=</code>.
     * </p>
     * </note>
     * 
     * @return The email address that is sending the email. This email address must be either individually verified with
     *         Amazon SES, or from a domain that has been verified with Amazon SES. For information about verifying
     *         identities, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
     *         Developer Guide</a>.</p>
     *         <p>
     *         If you are sending on behalf of another user and have been permitted to do so by a sending authorization
     *         policy, then you must also specify the <code>SourceArn</code> parameter. For more information about
     *         sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     *         </p>
     *         <note>
     *         <p>
     *         Amazon SES does not support the SMTPUTF8 extension, as described in <a
     *         href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a
     *         source email address (the part of the email address that precedes the @ sign) may only contain <a
     *         href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the
     *         <i>domain part</i> of an address (the part after the @ sign) contains non-ASCII characters, they must be
     *         encoded using Punycode, as described in <a href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>.
     *         The sender name (also known as the <i>friendly name</i>) may contain non-ASCII characters. These
     *         characters must be encoded using MIME encoded-word syntax, as described in <a
     *         href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>. MIME encoded-word syntax uses the following
     *         form: <code>=?charset?encoding?encoded-text?=</code>.
     *         </p>
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to send for the email address specified in the <code>Source</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
     * <code>user@example.com</code>.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to send for the email address specified in the
     *         <code>Source</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to send from <code>user@example.com</code>, then you would specify the
     *         <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
     *         the <code>Source</code> to be <code>user@example.com</code>.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to address
     * will receive the reply.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
     *         address will receive the reply.
     */
    public List<String> replyToAddresses() {
        return replyToAddresses;
    }

    /**
     * <p>
     * The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled. If the
     * message cannot be delivered to the recipient, then an error message will be returned from the recipient's ISP;
     * this message will then be forwarded to the email address specified by the <code>ReturnPath</code> parameter. The
     * <code>ReturnPath</code> parameter is never overwritten. This email address must be either individually verified
     * with Amazon SES, or from a domain that has been verified with Amazon SES.
     * </p>
     * 
     * @return The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled.
     *         If the message cannot be delivered to the recipient, then an error message will be returned from the
     *         recipient's ISP; this message will then be forwarded to the email address specified by the
     *         <code>ReturnPath</code> parameter. The <code>ReturnPath</code> parameter is never overwritten. This email
     *         address must be either individually verified with Amazon SES, or from a domain that has been verified
     *         with Amazon SES.
     */
    public String returnPath() {
        return returnPath;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the <code>ReturnPath</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
     * <code>feedback@example.com</code>.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>ReturnPath</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to use <code>feedback@example.com</code>, then you would specify the
     *         <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
     *         and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String returnPathArn() {
        return returnPathArn;
    }

    /**
     * <p>
     * The name of the configuration set to use when you send an email using <code>SendBulkTemplatedEmail</code>.
     * </p>
     * 
     * @return The name of the configuration set to use when you send an email using <code>SendBulkTemplatedEmail</code>
     *         .
     */
    public String configurationSetName() {
        return configurationSetName;
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send to a destination using
     * <code>SendBulkTemplatedEmail</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send to a destination
     *         using <code>SendBulkTemplatedEmail</code>.
     */
    public List<MessageTag> defaultTags() {
        return defaultTags;
    }

    /**
     * <p>
     * The template to use when sending this email.
     * </p>
     * 
     * @return The template to use when sending this email.
     */
    public String template() {
        return template;
    }

    /**
     * <p>
     * The ARN of the template to use when sending this email.
     * </p>
     * 
     * @return The ARN of the template to use when sending this email.
     */
    public String templateArn() {
        return templateArn;
    }

    /**
     * <p>
     * A list of replacement values to apply to the template when replacement data is not specified in a Destination
     * object. These values act as a default or fallback option when no other data is available.
     * </p>
     * <p>
     * The template data is a JSON object, typically consisting of key-value pairs in which the keys correspond to
     * replacement tags in the email template.
     * </p>
     * 
     * @return A list of replacement values to apply to the template when replacement data is not specified in a
     *         Destination object. These values act as a default or fallback option when no other data is available.</p>
     *         <p>
     *         The template data is a JSON object, typically consisting of key-value pairs in which the keys correspond
     *         to replacement tags in the email template.
     */
    public String defaultTemplateData() {
        return defaultTemplateData;
    }

    /**
     * <p>
     * One or more <code>Destination</code> objects. All of the recipients in a <code>Destination</code> will receive
     * the same version of the email. You can specify up to 50 <code>Destination</code> objects within a
     * <code>Destinations</code> array.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more <code>Destination</code> objects. All of the recipients in a <code>Destination</code> will
     *         receive the same version of the email. You can specify up to 50 <code>Destination</code> objects within a
     *         <code>Destinations</code> array.
     */
    public List<BulkEmailDestination> destinations() {
        return destinations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(replyToAddresses());
        hashCode = 31 * hashCode + Objects.hashCode(returnPath());
        hashCode = 31 * hashCode + Objects.hashCode(returnPathArn());
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        hashCode = 31 * hashCode + Objects.hashCode(defaultTags());
        hashCode = 31 * hashCode + Objects.hashCode(template());
        hashCode = 31 * hashCode + Objects.hashCode(templateArn());
        hashCode = 31 * hashCode + Objects.hashCode(defaultTemplateData());
        hashCode = 31 * hashCode + Objects.hashCode(destinations());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendBulkTemplatedEmailRequest)) {
            return false;
        }
        SendBulkTemplatedEmailRequest other = (SendBulkTemplatedEmailRequest) obj;
        return Objects.equals(source(), other.source()) && Objects.equals(sourceArn(), other.sourceArn())
                && Objects.equals(replyToAddresses(), other.replyToAddresses())
                && Objects.equals(returnPath(), other.returnPath()) && Objects.equals(returnPathArn(), other.returnPathArn())
                && Objects.equals(configurationSetName(), other.configurationSetName())
                && Objects.equals(defaultTags(), other.defaultTags()) && Objects.equals(template(), other.template())
                && Objects.equals(templateArn(), other.templateArn())
                && Objects.equals(defaultTemplateData(), other.defaultTemplateData())
                && Objects.equals(destinations(), other.destinations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SendBulkTemplatedEmailRequest").add("Source", source()).add("SourceArn", sourceArn())
                .add("ReplyToAddresses", replyToAddresses()).add("ReturnPath", returnPath())
                .add("ReturnPathArn", returnPathArn()).add("ConfigurationSetName", configurationSetName())
                .add("DefaultTags", defaultTags()).add("Template", template()).add("TemplateArn", templateArn())
                .add("DefaultTemplateData", defaultTemplateData()).add("Destinations", destinations()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "SourceArn":
            return Optional.ofNullable(clazz.cast(sourceArn()));
        case "ReplyToAddresses":
            return Optional.ofNullable(clazz.cast(replyToAddresses()));
        case "ReturnPath":
            return Optional.ofNullable(clazz.cast(returnPath()));
        case "ReturnPathArn":
            return Optional.ofNullable(clazz.cast(returnPathArn()));
        case "ConfigurationSetName":
            return Optional.ofNullable(clazz.cast(configurationSetName()));
        case "DefaultTags":
            return Optional.ofNullable(clazz.cast(defaultTags()));
        case "Template":
            return Optional.ofNullable(clazz.cast(template()));
        case "TemplateArn":
            return Optional.ofNullable(clazz.cast(templateArn()));
        case "DefaultTemplateData":
            return Optional.ofNullable(clazz.cast(defaultTemplateData()));
        case "Destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SendBulkTemplatedEmailRequest, T> g) {
        return obj -> g.apply((SendBulkTemplatedEmailRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, SendBulkTemplatedEmailRequest> {
        /**
         * <p>
         * The email address that is sending the email. This email address must be either individually verified with
         * Amazon SES, or from a domain that has been verified with Amazon SES. For information about verifying
         * identities, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
         * Developer Guide</a>.
         * </p>
         * <p>
         * If you are sending on behalf of another user and have been permitted to do so by a sending authorization
         * policy, then you must also specify the <code>SourceArn</code> parameter. For more information about sending
         * authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * <note>
         * <p>
         * Amazon SES does not support the SMTPUTF8 extension, as described in <a
         * href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a source
         * email address (the part of the email address that precedes the @ sign) may only contain <a
         * href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the <i>domain
         * part</i> of an address (the part after the @ sign) contains non-ASCII characters, they must be encoded using
         * Punycode, as described in <a href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>. The sender name
         * (also known as the <i>friendly name</i>) may contain non-ASCII characters. These characters must be encoded
         * using MIME encoded-word syntax, as described in <a href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         * MIME encoded-word syntax uses the following form: <code>=?charset?encoding?encoded-text?=</code>.
         * </p>
         * </note>
         * 
         * @param source
         *        The email address that is sending the email. This email address must be either individually verified
         *        with Amazon SES, or from a domain that has been verified with Amazon SES. For information about
         *        verifying identities, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon
         *        SES Developer Guide</a>.</p>
         *        <p>
         *        If you are sending on behalf of another user and have been permitted to do so by a sending
         *        authorization policy, then you must also specify the <code>SourceArn</code> parameter. For more
         *        information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         *        </p>
         *        <note>
         *        <p>
         *        Amazon SES does not support the SMTPUTF8 extension, as described in <a
         *        href="https://tools.ietf.org/html/rfc6531">RFC6531</a>. For this reason, the <i>local part</i> of a
         *        source email address (the part of the email address that precedes the @ sign) may only contain <a
         *        href="https://en.wikipedia.org/wiki/Email_address#Local-part">7-bit ASCII characters</a>. If the
         *        <i>domain part</i> of an address (the part after the @ sign) contains non-ASCII characters, they must
         *        be encoded using Punycode, as described in <a
         *        href="https://tools.ietf.org/html/rfc3492.html">RFC3492</a>. The sender name (also known as the
         *        <i>friendly name</i>) may contain non-ASCII characters. These characters must be encoded using MIME
         *        encoded-word syntax, as described in <a href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>. MIME
         *        encoded-word syntax uses the following form: <code>=?charset?encoding?encoded-text?=</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to send for the email address specified in the
         * <code>Source</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
         * <code>user@example.com</code>.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param sourceArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to send for the email address
         *        specified in the <code>Source</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to send from <code>user@example.com</code>, then you would specify the
         *        <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
         *        the <code>Source</code> to be <code>user@example.com</code>.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         * address will receive the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         *        address will receive the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(Collection<String> replyToAddresses);

        /**
         * <p>
         * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         * address will receive the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         *        address will receive the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(String... replyToAddresses);

        /**
         * <p>
         * The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled. If
         * the message cannot be delivered to the recipient, then an error message will be returned from the recipient's
         * ISP; this message will then be forwarded to the email address specified by the <code>ReturnPath</code>
         * parameter. The <code>ReturnPath</code> parameter is never overwritten. This email address must be either
         * individually verified with Amazon SES, or from a domain that has been verified with Amazon SES.
         * </p>
         * 
         * @param returnPath
         *        The email address that bounces and complaints will be forwarded to when feedback forwarding is
         *        enabled. If the message cannot be delivered to the recipient, then an error message will be returned
         *        from the recipient's ISP; this message will then be forwarded to the email address specified by the
         *        <code>ReturnPath</code> parameter. The <code>ReturnPath</code> parameter is never overwritten. This
         *        email address must be either individually verified with Amazon SES, or from a domain that has been
         *        verified with Amazon SES.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPath(String returnPath);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>ReturnPath</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
         * <code>feedback@example.com</code>.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param returnPathArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>ReturnPath</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to use <code>feedback@example.com</code>, then you would specify the
         *        <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
         *        and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPathArn(String returnPathArn);

        /**
         * <p>
         * The name of the configuration set to use when you send an email using <code>SendBulkTemplatedEmail</code>.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to use when you send an email using
         *        <code>SendBulkTemplatedEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send to a destination using
         * <code>SendBulkTemplatedEmail</code>.
         * </p>
         * 
         * @param defaultTags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send to a destination
         *        using <code>SendBulkTemplatedEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultTags(Collection<MessageTag> defaultTags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send to a destination using
         * <code>SendBulkTemplatedEmail</code>.
         * </p>
         * 
         * @param defaultTags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send to a destination
         *        using <code>SendBulkTemplatedEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultTags(MessageTag... defaultTags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send to a destination using
         * <code>SendBulkTemplatedEmail</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MessageTag>.Builder} avoiding the need to
         * create one manually via {@link List<MessageTag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MessageTag>.Builder#build()} is called immediately and its
         * result is passed to {@link #defaultTags(List<MessageTag>)}.
         * 
         * @param defaultTags
         *        a consumer that will call methods on {@link List<MessageTag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultTags(List<MessageTag>)
         */
        Builder defaultTags(Consumer<MessageTag.Builder>... defaultTags);

        /**
         * <p>
         * The template to use when sending this email.
         * </p>
         * 
         * @param template
         *        The template to use when sending this email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder template(String template);

        /**
         * <p>
         * The ARN of the template to use when sending this email.
         * </p>
         * 
         * @param templateArn
         *        The ARN of the template to use when sending this email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateArn(String templateArn);

        /**
         * <p>
         * A list of replacement values to apply to the template when replacement data is not specified in a Destination
         * object. These values act as a default or fallback option when no other data is available.
         * </p>
         * <p>
         * The template data is a JSON object, typically consisting of key-value pairs in which the keys correspond to
         * replacement tags in the email template.
         * </p>
         * 
         * @param defaultTemplateData
         *        A list of replacement values to apply to the template when replacement data is not specified in a
         *        Destination object. These values act as a default or fallback option when no other data is
         *        available.</p>
         *        <p>
         *        The template data is a JSON object, typically consisting of key-value pairs in which the keys
         *        correspond to replacement tags in the email template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultTemplateData(String defaultTemplateData);

        /**
         * <p>
         * One or more <code>Destination</code> objects. All of the recipients in a <code>Destination</code> will
         * receive the same version of the email. You can specify up to 50 <code>Destination</code> objects within a
         * <code>Destinations</code> array.
         * </p>
         * 
         * @param destinations
         *        One or more <code>Destination</code> objects. All of the recipients in a <code>Destination</code> will
         *        receive the same version of the email. You can specify up to 50 <code>Destination</code> objects
         *        within a <code>Destinations</code> array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<BulkEmailDestination> destinations);

        /**
         * <p>
         * One or more <code>Destination</code> objects. All of the recipients in a <code>Destination</code> will
         * receive the same version of the email. You can specify up to 50 <code>Destination</code> objects within a
         * <code>Destinations</code> array.
         * </p>
         * 
         * @param destinations
         *        One or more <code>Destination</code> objects. All of the recipients in a <code>Destination</code> will
         *        receive the same version of the email. You can specify up to 50 <code>Destination</code> objects
         *        within a <code>Destinations</code> array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(BulkEmailDestination... destinations);

        /**
         * <p>
         * One or more <code>Destination</code> objects. All of the recipients in a <code>Destination</code> will
         * receive the same version of the email. You can specify up to 50 <code>Destination</code> objects within a
         * <code>Destinations</code> array.
         * </p>
         * This is a convenience that creates an instance of the {@link List<BulkEmailDestination>.Builder} avoiding the
         * need to create one manually via {@link List<BulkEmailDestination>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<BulkEmailDestination>.Builder#build()} is called immediately
         * and its result is passed to {@link #destinations(List<BulkEmailDestination>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on {@link List<BulkEmailDestination>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(List<BulkEmailDestination>)
         */
        Builder destinations(Consumer<BulkEmailDestination.Builder>... destinations);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String source;

        private String sourceArn;

        private List<String> replyToAddresses = DefaultSdkAutoConstructList.getInstance();

        private String returnPath;

        private String returnPathArn;

        private String configurationSetName;

        private List<MessageTag> defaultTags = DefaultSdkAutoConstructList.getInstance();

        private String template;

        private String templateArn;

        private String defaultTemplateData;

        private List<BulkEmailDestination> destinations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SendBulkTemplatedEmailRequest model) {
            super(model);
            source(model.source);
            sourceArn(model.sourceArn);
            replyToAddresses(model.replyToAddresses);
            returnPath(model.returnPath);
            returnPathArn(model.returnPathArn);
            configurationSetName(model.configurationSetName);
            defaultTags(model.defaultTags);
            template(model.template);
            templateArn(model.templateArn);
            defaultTemplateData(model.defaultTemplateData);
            destinations(model.destinations);
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        public final Collection<String> getReplyToAddresses() {
            return replyToAddresses;
        }

        @Override
        public final Builder replyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = AddressListCopier.copy(replyToAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replyToAddresses(String... replyToAddresses) {
            replyToAddresses(Arrays.asList(replyToAddresses));
            return this;
        }

        public final void setReplyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = AddressListCopier.copy(replyToAddresses);
        }

        public final String getReturnPath() {
            return returnPath;
        }

        @Override
        public final Builder returnPath(String returnPath) {
            this.returnPath = returnPath;
            return this;
        }

        public final void setReturnPath(String returnPath) {
            this.returnPath = returnPath;
        }

        public final String getReturnPathArn() {
            return returnPathArn;
        }

        @Override
        public final Builder returnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
            return this;
        }

        public final void setReturnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        public final Collection<MessageTag.Builder> getDefaultTags() {
            return defaultTags != null ? defaultTags.stream().map(MessageTag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder defaultTags(Collection<MessageTag> defaultTags) {
            this.defaultTags = MessageTagListCopier.copy(defaultTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder defaultTags(MessageTag... defaultTags) {
            defaultTags(Arrays.asList(defaultTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder defaultTags(Consumer<MessageTag.Builder>... defaultTags) {
            defaultTags(Stream.of(defaultTags).map(c -> MessageTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDefaultTags(Collection<MessageTag.BuilderImpl> defaultTags) {
            this.defaultTags = MessageTagListCopier.copyFromBuilder(defaultTags);
        }

        public final String getTemplate() {
            return template;
        }

        @Override
        public final Builder template(String template) {
            this.template = template;
            return this;
        }

        public final void setTemplate(String template) {
            this.template = template;
        }

        public final String getTemplateArn() {
            return templateArn;
        }

        @Override
        public final Builder templateArn(String templateArn) {
            this.templateArn = templateArn;
            return this;
        }

        public final void setTemplateArn(String templateArn) {
            this.templateArn = templateArn;
        }

        public final String getDefaultTemplateData() {
            return defaultTemplateData;
        }

        @Override
        public final Builder defaultTemplateData(String defaultTemplateData) {
            this.defaultTemplateData = defaultTemplateData;
            return this;
        }

        public final void setDefaultTemplateData(String defaultTemplateData) {
            this.defaultTemplateData = defaultTemplateData;
        }

        public final Collection<BulkEmailDestination.Builder> getDestinations() {
            return destinations != null ? destinations.stream().map(BulkEmailDestination::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder destinations(Collection<BulkEmailDestination> destinations) {
            this.destinations = BulkEmailDestinationListCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(BulkEmailDestination... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(Consumer<BulkEmailDestination.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> BulkEmailDestination.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDestinations(Collection<BulkEmailDestination.BuilderImpl> destinations) {
            this.destinations = BulkEmailDestinationListCopier.copyFromBuilder(destinations);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SendBulkTemplatedEmailRequest build() {
            return new SendBulkTemplatedEmailRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
