/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sesv2;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sesv2.model.AccountSuspendedException;
import software.amazon.awssdk.services.sesv2.model.AlreadyExistsException;
import software.amazon.awssdk.services.sesv2.model.BadRequestException;
import software.amazon.awssdk.services.sesv2.model.ConcurrentModificationException;
import software.amazon.awssdk.services.sesv2.model.CreateConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.sesv2.model.CreateConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.sesv2.model.CreateConfigurationSetRequest;
import software.amazon.awssdk.services.sesv2.model.CreateConfigurationSetResponse;
import software.amazon.awssdk.services.sesv2.model.CreateDedicatedIpPoolRequest;
import software.amazon.awssdk.services.sesv2.model.CreateDedicatedIpPoolResponse;
import software.amazon.awssdk.services.sesv2.model.CreateDeliverabilityTestReportRequest;
import software.amazon.awssdk.services.sesv2.model.CreateDeliverabilityTestReportResponse;
import software.amazon.awssdk.services.sesv2.model.CreateEmailIdentityRequest;
import software.amazon.awssdk.services.sesv2.model.CreateEmailIdentityResponse;
import software.amazon.awssdk.services.sesv2.model.DeleteConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.sesv2.model.DeleteConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.sesv2.model.DeleteConfigurationSetRequest;
import software.amazon.awssdk.services.sesv2.model.DeleteConfigurationSetResponse;
import software.amazon.awssdk.services.sesv2.model.DeleteDedicatedIpPoolRequest;
import software.amazon.awssdk.services.sesv2.model.DeleteDedicatedIpPoolResponse;
import software.amazon.awssdk.services.sesv2.model.DeleteEmailIdentityRequest;
import software.amazon.awssdk.services.sesv2.model.DeleteEmailIdentityResponse;
import software.amazon.awssdk.services.sesv2.model.GetAccountRequest;
import software.amazon.awssdk.services.sesv2.model.GetAccountResponse;
import software.amazon.awssdk.services.sesv2.model.GetBlacklistReportsRequest;
import software.amazon.awssdk.services.sesv2.model.GetBlacklistReportsResponse;
import software.amazon.awssdk.services.sesv2.model.GetConfigurationSetEventDestinationsRequest;
import software.amazon.awssdk.services.sesv2.model.GetConfigurationSetEventDestinationsResponse;
import software.amazon.awssdk.services.sesv2.model.GetConfigurationSetRequest;
import software.amazon.awssdk.services.sesv2.model.GetConfigurationSetResponse;
import software.amazon.awssdk.services.sesv2.model.GetDedicatedIpRequest;
import software.amazon.awssdk.services.sesv2.model.GetDedicatedIpResponse;
import software.amazon.awssdk.services.sesv2.model.GetDedicatedIpsRequest;
import software.amazon.awssdk.services.sesv2.model.GetDedicatedIpsResponse;
import software.amazon.awssdk.services.sesv2.model.GetDeliverabilityDashboardOptionsRequest;
import software.amazon.awssdk.services.sesv2.model.GetDeliverabilityDashboardOptionsResponse;
import software.amazon.awssdk.services.sesv2.model.GetDeliverabilityTestReportRequest;
import software.amazon.awssdk.services.sesv2.model.GetDeliverabilityTestReportResponse;
import software.amazon.awssdk.services.sesv2.model.GetDomainDeliverabilityCampaignRequest;
import software.amazon.awssdk.services.sesv2.model.GetDomainDeliverabilityCampaignResponse;
import software.amazon.awssdk.services.sesv2.model.GetDomainStatisticsReportRequest;
import software.amazon.awssdk.services.sesv2.model.GetDomainStatisticsReportResponse;
import software.amazon.awssdk.services.sesv2.model.GetEmailIdentityRequest;
import software.amazon.awssdk.services.sesv2.model.GetEmailIdentityResponse;
import software.amazon.awssdk.services.sesv2.model.LimitExceededException;
import software.amazon.awssdk.services.sesv2.model.ListConfigurationSetsRequest;
import software.amazon.awssdk.services.sesv2.model.ListConfigurationSetsResponse;
import software.amazon.awssdk.services.sesv2.model.ListDedicatedIpPoolsRequest;
import software.amazon.awssdk.services.sesv2.model.ListDedicatedIpPoolsResponse;
import software.amazon.awssdk.services.sesv2.model.ListDeliverabilityTestReportsRequest;
import software.amazon.awssdk.services.sesv2.model.ListDeliverabilityTestReportsResponse;
import software.amazon.awssdk.services.sesv2.model.ListDomainDeliverabilityCampaignsRequest;
import software.amazon.awssdk.services.sesv2.model.ListDomainDeliverabilityCampaignsResponse;
import software.amazon.awssdk.services.sesv2.model.ListEmailIdentitiesRequest;
import software.amazon.awssdk.services.sesv2.model.ListEmailIdentitiesResponse;
import software.amazon.awssdk.services.sesv2.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.sesv2.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.sesv2.model.MailFromDomainNotVerifiedException;
import software.amazon.awssdk.services.sesv2.model.MessageRejectedException;
import software.amazon.awssdk.services.sesv2.model.NotFoundException;
import software.amazon.awssdk.services.sesv2.model.PutAccountDedicatedIpWarmupAttributesRequest;
import software.amazon.awssdk.services.sesv2.model.PutAccountDedicatedIpWarmupAttributesResponse;
import software.amazon.awssdk.services.sesv2.model.PutAccountSendingAttributesRequest;
import software.amazon.awssdk.services.sesv2.model.PutAccountSendingAttributesResponse;
import software.amazon.awssdk.services.sesv2.model.PutConfigurationSetDeliveryOptionsRequest;
import software.amazon.awssdk.services.sesv2.model.PutConfigurationSetDeliveryOptionsResponse;
import software.amazon.awssdk.services.sesv2.model.PutConfigurationSetReputationOptionsRequest;
import software.amazon.awssdk.services.sesv2.model.PutConfigurationSetReputationOptionsResponse;
import software.amazon.awssdk.services.sesv2.model.PutConfigurationSetSendingOptionsRequest;
import software.amazon.awssdk.services.sesv2.model.PutConfigurationSetSendingOptionsResponse;
import software.amazon.awssdk.services.sesv2.model.PutConfigurationSetTrackingOptionsRequest;
import software.amazon.awssdk.services.sesv2.model.PutConfigurationSetTrackingOptionsResponse;
import software.amazon.awssdk.services.sesv2.model.PutDedicatedIpInPoolRequest;
import software.amazon.awssdk.services.sesv2.model.PutDedicatedIpInPoolResponse;
import software.amazon.awssdk.services.sesv2.model.PutDedicatedIpWarmupAttributesRequest;
import software.amazon.awssdk.services.sesv2.model.PutDedicatedIpWarmupAttributesResponse;
import software.amazon.awssdk.services.sesv2.model.PutDeliverabilityDashboardOptionRequest;
import software.amazon.awssdk.services.sesv2.model.PutDeliverabilityDashboardOptionResponse;
import software.amazon.awssdk.services.sesv2.model.PutEmailIdentityDkimAttributesRequest;
import software.amazon.awssdk.services.sesv2.model.PutEmailIdentityDkimAttributesResponse;
import software.amazon.awssdk.services.sesv2.model.PutEmailIdentityFeedbackAttributesRequest;
import software.amazon.awssdk.services.sesv2.model.PutEmailIdentityFeedbackAttributesResponse;
import software.amazon.awssdk.services.sesv2.model.PutEmailIdentityMailFromAttributesRequest;
import software.amazon.awssdk.services.sesv2.model.PutEmailIdentityMailFromAttributesResponse;
import software.amazon.awssdk.services.sesv2.model.SendEmailRequest;
import software.amazon.awssdk.services.sesv2.model.SendEmailResponse;
import software.amazon.awssdk.services.sesv2.model.SendingPausedException;
import software.amazon.awssdk.services.sesv2.model.SesV2Exception;
import software.amazon.awssdk.services.sesv2.model.SesV2Request;
import software.amazon.awssdk.services.sesv2.model.TagResourceRequest;
import software.amazon.awssdk.services.sesv2.model.TagResourceResponse;
import software.amazon.awssdk.services.sesv2.model.TooManyRequestsException;
import software.amazon.awssdk.services.sesv2.model.UntagResourceRequest;
import software.amazon.awssdk.services.sesv2.model.UntagResourceResponse;
import software.amazon.awssdk.services.sesv2.model.UpdateConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.sesv2.model.UpdateConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.sesv2.paginators.GetDedicatedIpsPublisher;
import software.amazon.awssdk.services.sesv2.paginators.ListConfigurationSetsPublisher;
import software.amazon.awssdk.services.sesv2.paginators.ListDedicatedIpPoolsPublisher;
import software.amazon.awssdk.services.sesv2.paginators.ListDeliverabilityTestReportsPublisher;
import software.amazon.awssdk.services.sesv2.paginators.ListDomainDeliverabilityCampaignsPublisher;
import software.amazon.awssdk.services.sesv2.paginators.ListEmailIdentitiesPublisher;
import software.amazon.awssdk.services.sesv2.transform.CreateConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.CreateConfigurationSetRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.CreateDedicatedIpPoolRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.CreateDeliverabilityTestReportRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.CreateEmailIdentityRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.DeleteConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.DeleteConfigurationSetRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.DeleteDedicatedIpPoolRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.DeleteEmailIdentityRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetAccountRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetBlacklistReportsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetConfigurationSetEventDestinationsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetConfigurationSetRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetDedicatedIpRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetDedicatedIpsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetDeliverabilityDashboardOptionsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetDeliverabilityTestReportRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetDomainDeliverabilityCampaignRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetDomainStatisticsReportRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.GetEmailIdentityRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.ListConfigurationSetsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.ListDedicatedIpPoolsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.ListDeliverabilityTestReportsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.ListDomainDeliverabilityCampaignsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.ListEmailIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutAccountDedicatedIpWarmupAttributesRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutAccountSendingAttributesRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutConfigurationSetDeliveryOptionsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutConfigurationSetReputationOptionsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutConfigurationSetSendingOptionsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutConfigurationSetTrackingOptionsRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutDedicatedIpInPoolRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutDedicatedIpWarmupAttributesRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutDeliverabilityDashboardOptionRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutEmailIdentityDkimAttributesRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutEmailIdentityFeedbackAttributesRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.PutEmailIdentityMailFromAttributesRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.SendEmailRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.sesv2.transform.UpdateConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SesV2AsyncClient}.
 *
 * @see SesV2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSesV2AsyncClient implements SesV2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSesV2AsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSesV2AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Create a configuration set. <i>Configuration sets</i> are groups of rules that you can apply to the emails that
     * you send. You apply a configuration set to an email by specifying the name of the configuration set when you call
     * the Amazon SES API v2. When you apply a configuration set to an email, all of the rules in that configuration set
     * are applied to the email.
     * </p>
     *
     * @param createConfigurationSetRequest
     *        A request to create a configuration set.
     * @return A Java Future containing the result of the CreateConfigurationSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource specified in your request already exists.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.CreateConfigurationSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/CreateConfigurationSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationSetResponse> createConfigurationSet(
            CreateConfigurationSetRequest createConfigurationSetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConfigurationSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationSetRequest, CreateConfigurationSetResponse>()
                            .withOperationName("CreateConfigurationSet")
                            .withMarshaller(new CreateConfigurationSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createConfigurationSetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an event destination. <i>Events</i> include message sends, deliveries, opens, clicks, bounces, and
     * complaints. <i>Event destinations</i> are places that you can send information about these events to. For
     * example, you can send event data to Amazon SNS to receive notifications when you receive bounces or complaints,
     * or you can use Amazon Kinesis Data Firehose to stream data to Amazon S3 for long-term storage.
     * </p>
     * <p>
     * A single configuration set can include more than one event destination.
     * </p>
     *
     * @param createConfigurationSetEventDestinationRequest
     *        A request to add an event destination to a configuration set.
     * @return A Java Future containing the result of the CreateConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>AlreadyExistsException The resource specified in your request already exists.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.CreateConfigurationSetEventDestination
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/CreateConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationSetEventDestinationResponse> createConfigurationSetEventDestination(
            CreateConfigurationSetEventDestinationRequest createConfigurationSetEventDestinationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationSetEventDestinationRequest, CreateConfigurationSetEventDestinationResponse>()
                            .withOperationName("CreateConfigurationSetEventDestination")
                            .withMarshaller(new CreateConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createConfigurationSetEventDestinationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new pool of dedicated IP addresses. A pool can include one or more dedicated IP addresses that are
     * associated with your AWS account. You can associate a pool with a configuration set. When you send an email that
     * uses that configuration set, the message is sent from one of the addresses in the associated pool.
     * </p>
     *
     * @param createDedicatedIpPoolRequest
     *        A request to create a new dedicated IP pool.
     * @return A Java Future containing the result of the CreateDedicatedIpPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource specified in your request already exists.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.CreateDedicatedIpPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/CreateDedicatedIpPool" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDedicatedIpPoolResponse> createDedicatedIpPool(
            CreateDedicatedIpPoolRequest createDedicatedIpPoolRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDedicatedIpPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDedicatedIpPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDedicatedIpPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDedicatedIpPoolRequest, CreateDedicatedIpPoolResponse>()
                            .withOperationName("CreateDedicatedIpPool")
                            .withMarshaller(new CreateDedicatedIpPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDedicatedIpPoolRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new predictive inbox placement test. Predictive inbox placement tests can help you predict how your
     * messages will be handled by various email providers around the world. When you perform a predictive inbox
     * placement test, you provide a sample message that contains the content that you plan to send to your customers.
     * Amazon SES API v2 then sends that message to special email addresses spread across several major email providers.
     * After about 24 hours, the test is complete, and you can use the <code>GetDeliverabilityTestReport</code>
     * operation to view the results of the test.
     * </p>
     *
     * @param createDeliverabilityTestReportRequest
     *        A request to perform a predictive inbox placement test. Predictive inbox placement tests can help you
     *        predict how your messages will be handled by various email providers around the world. When you perform a
     *        predictive inbox placement test, you provide a sample message that contains the content that you plan to
     *        send to your customers. We send that message to special email addresses spread across several major email
     *        providers around the world. The test takes about 24 hours to complete. When the test is complete, you can
     *        use the <code>GetDeliverabilityTestReport</code> operation to view the results of the test.
     * @return A Java Future containing the result of the CreateDeliverabilityTestReport operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccountSuspendedException The message can't be sent because the account's ability to send email has
     *         been permanently restricted.</li>
     *         <li>SendingPausedException The message can't be sent because the account's ability to send email is
     *         currently paused.</li>
     *         <li>MessageRejectedException The message can't be sent because it contains invalid content.</li>
     *         <li>MailFromDomainNotVerifiedException The message can't be sent because the sending domain isn't
     *         verified.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.CreateDeliverabilityTestReport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/CreateDeliverabilityTestReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeliverabilityTestReportResponse> createDeliverabilityTestReport(
            CreateDeliverabilityTestReportRequest createDeliverabilityTestReportRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeliverabilityTestReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeliverabilityTestReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDeliverabilityTestReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeliverabilityTestReportRequest, CreateDeliverabilityTestReportResponse>()
                            .withOperationName("CreateDeliverabilityTestReport")
                            .withMarshaller(new CreateDeliverabilityTestReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDeliverabilityTestReportRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the process of verifying an email identity. An <i>identity</i> is an email address or domain that you use
     * when you send email. Before you can use an identity to send email, you first have to verify it. By verifying an
     * identity, you demonstrate that you're the owner of the identity, and that you've given Amazon SES API v2
     * permission to send email from the identity.
     * </p>
     * <p>
     * When you verify an email address, Amazon SES sends an email to the address. Your email address is verified as
     * soon as you follow the link in the verification email.
     * </p>
     * <p>
     * When you verify a domain, this operation provides a set of DKIM tokens, which you can convert into CNAME tokens.
     * You add these CNAME tokens to the DNS configuration for your domain. Your domain is verified when Amazon SES
     * detects these records in the DNS configuration for your domain. For some DNS providers, it can take 72 hours or
     * more to complete the domain verification process.
     * </p>
     *
     * @param createEmailIdentityRequest
     *        A request to begin the verification process for an email identity (an email address or domain).
     * @return A Java Future containing the result of the CreateEmailIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.CreateEmailIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/CreateEmailIdentity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEmailIdentityResponse> createEmailIdentity(
            CreateEmailIdentityRequest createEmailIdentityRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEmailIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEmailIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEmailIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEmailIdentityRequest, CreateEmailIdentityResponse>()
                            .withOperationName("CreateEmailIdentity")
                            .withMarshaller(new CreateEmailIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createEmailIdentityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an existing configuration set.
     * </p>
     * <p>
     * <i>Configuration sets</i> are groups of rules that you can apply to the emails you send. You apply a
     * configuration set to an email by including a reference to the configuration set in the headers of the email. When
     * you apply a configuration set to an email, all of the rules in that configuration set are applied to the email.
     * </p>
     *
     * @param deleteConfigurationSetRequest
     *        A request to delete a configuration set.
     * @return A Java Future containing the result of the DeleteConfigurationSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.DeleteConfigurationSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/DeleteConfigurationSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationSetResponse> deleteConfigurationSet(
            DeleteConfigurationSetRequest deleteConfigurationSetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConfigurationSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationSetRequest, DeleteConfigurationSetResponse>()
                            .withOperationName("DeleteConfigurationSet")
                            .withMarshaller(new DeleteConfigurationSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteConfigurationSetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an event destination.
     * </p>
     * <p>
     * <i>Events</i> include message sends, deliveries, opens, clicks, bounces, and complaints. <i>Event
     * destinations</i> are places that you can send information about these events to. For example, you can send event
     * data to Amazon SNS to receive notifications when you receive bounces or complaints, or you can use Amazon Kinesis
     * Data Firehose to stream data to Amazon S3 for long-term storage.
     * </p>
     *
     * @param deleteConfigurationSetEventDestinationRequest
     *        A request to delete an event destination from a configuration set.
     * @return A Java Future containing the result of the DeleteConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.DeleteConfigurationSetEventDestination
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/DeleteConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationSetEventDestinationResponse> deleteConfigurationSetEventDestination(
            DeleteConfigurationSetEventDestinationRequest deleteConfigurationSetEventDestinationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationSetEventDestinationRequest, DeleteConfigurationSetEventDestinationResponse>()
                            .withOperationName("DeleteConfigurationSetEventDestination")
                            .withMarshaller(new DeleteConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteConfigurationSetEventDestinationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a dedicated IP pool.
     * </p>
     *
     * @param deleteDedicatedIpPoolRequest
     *        A request to delete a dedicated IP pool.
     * @return A Java Future containing the result of the DeleteDedicatedIpPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.DeleteDedicatedIpPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/DeleteDedicatedIpPool" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDedicatedIpPoolResponse> deleteDedicatedIpPool(
            DeleteDedicatedIpPoolRequest deleteDedicatedIpPoolRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDedicatedIpPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDedicatedIpPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDedicatedIpPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDedicatedIpPoolRequest, DeleteDedicatedIpPoolResponse>()
                            .withOperationName("DeleteDedicatedIpPool")
                            .withMarshaller(new DeleteDedicatedIpPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDedicatedIpPoolRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an email identity. An identity can be either an email address or a domain name.
     * </p>
     *
     * @param deleteEmailIdentityRequest
     *        A request to delete an existing email identity. When you delete an identity, you lose the ability to send
     *        email from that identity. You can restore your ability to send email by completing the verification
     *        process for the identity again.
     * @return A Java Future containing the result of the DeleteEmailIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.DeleteEmailIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/DeleteEmailIdentity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEmailIdentityResponse> deleteEmailIdentity(
            DeleteEmailIdentityRequest deleteEmailIdentityRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEmailIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEmailIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEmailIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEmailIdentityRequest, DeleteEmailIdentityResponse>()
                            .withOperationName("DeleteEmailIdentity")
                            .withMarshaller(new DeleteEmailIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteEmailIdentityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtain information about the email-sending status and capabilities of your Amazon SES account in the current AWS
     * Region.
     * </p>
     *
     * @param getAccountRequest
     *        A request to obtain information about the email-sending capabilities of your Amazon SES account.
     * @return A Java Future containing the result of the GetAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountResponse> getAccount(GetAccountRequest getAccountRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountRequest, GetAccountResponse>().withOperationName("GetAccount")
                            .withMarshaller(new GetAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getAccountRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve a list of the blacklists that your dedicated IP addresses appear on.
     * </p>
     *
     * @param getBlacklistReportsRequest
     *        A request to retrieve a list of the blacklists that your dedicated IP addresses appear on.
     * @return A Java Future containing the result of the GetBlacklistReports operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetBlacklistReports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetBlacklistReports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBlacklistReportsResponse> getBlacklistReports(
            GetBlacklistReportsRequest getBlacklistReportsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBlacklistReportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBlacklistReportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBlacklistReportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBlacklistReportsRequest, GetBlacklistReportsResponse>()
                            .withOperationName("GetBlacklistReports")
                            .withMarshaller(new GetBlacklistReportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getBlacklistReportsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get information about an existing configuration set, including the dedicated IP pool that it's associated with,
     * whether or not it's enabled for sending email, and more.
     * </p>
     * <p>
     * <i>Configuration sets</i> are groups of rules that you can apply to the emails you send. You apply a
     * configuration set to an email by including a reference to the configuration set in the headers of the email. When
     * you apply a configuration set to an email, all of the rules in that configuration set are applied to the email.
     * </p>
     *
     * @param getConfigurationSetRequest
     *        A request to obtain information about a configuration set.
     * @return A Java Future containing the result of the GetConfigurationSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetConfigurationSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetConfigurationSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationSetResponse> getConfigurationSet(
            GetConfigurationSetRequest getConfigurationSetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigurationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfigurationSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationSetRequest, GetConfigurationSetResponse>()
                            .withOperationName("GetConfigurationSet")
                            .withMarshaller(new GetConfigurationSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getConfigurationSetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve a list of event destinations that are associated with a configuration set.
     * </p>
     * <p>
     * <i>Events</i> include message sends, deliveries, opens, clicks, bounces, and complaints. <i>Event
     * destinations</i> are places that you can send information about these events to. For example, you can send event
     * data to Amazon SNS to receive notifications when you receive bounces or complaints, or you can use Amazon Kinesis
     * Data Firehose to stream data to Amazon S3 for long-term storage.
     * </p>
     *
     * @param getConfigurationSetEventDestinationsRequest
     *        A request to obtain information about the event destinations for a configuration set.
     * @return A Java Future containing the result of the GetConfigurationSetEventDestinations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetConfigurationSetEventDestinations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetConfigurationSetEventDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationSetEventDestinationsResponse> getConfigurationSetEventDestinations(
            GetConfigurationSetEventDestinationsRequest getConfigurationSetEventDestinationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigurationSetEventDestinationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetConfigurationSetEventDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationSetEventDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationSetEventDestinationsRequest, GetConfigurationSetEventDestinationsResponse>()
                            .withOperationName("GetConfigurationSetEventDestinations")
                            .withMarshaller(new GetConfigurationSetEventDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getConfigurationSetEventDestinationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get information about a dedicated IP address, including the name of the dedicated IP pool that it's associated
     * with, as well information about the automatic warm-up process for the address.
     * </p>
     *
     * @param getDedicatedIpRequest
     *        A request to obtain more information about a dedicated IP address.
     * @return A Java Future containing the result of the GetDedicatedIp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetDedicatedIp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetDedicatedIp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDedicatedIpResponse> getDedicatedIp(GetDedicatedIpRequest getDedicatedIpRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDedicatedIpResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDedicatedIpResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDedicatedIpResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDedicatedIpRequest, GetDedicatedIpResponse>()
                            .withOperationName("GetDedicatedIp")
                            .withMarshaller(new GetDedicatedIpRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDedicatedIpRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the dedicated IP addresses that are associated with your AWS account.
     * </p>
     *
     * @param getDedicatedIpsRequest
     *        A request to obtain more information about dedicated IP pools.
     * @return A Java Future containing the result of the GetDedicatedIps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetDedicatedIps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetDedicatedIps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDedicatedIpsResponse> getDedicatedIps(GetDedicatedIpsRequest getDedicatedIpsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDedicatedIpsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDedicatedIpsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDedicatedIpsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDedicatedIpsRequest, GetDedicatedIpsResponse>()
                            .withOperationName("GetDedicatedIps")
                            .withMarshaller(new GetDedicatedIpsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDedicatedIpsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the dedicated IP addresses that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getDedicatedIps(software.amazon.awssdk.services.sesv2.model.GetDedicatedIpsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.GetDedicatedIpsPublisher publisher = client.getDedicatedIpsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.GetDedicatedIpsPublisher publisher = client.getDedicatedIpsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sesv2.model.GetDedicatedIpsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sesv2.model.GetDedicatedIpsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getDedicatedIps(software.amazon.awssdk.services.sesv2.model.GetDedicatedIpsRequest)} operation.</b>
     * </p>
     *
     * @param getDedicatedIpsRequest
     *        A request to obtain more information about dedicated IP pools.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetDedicatedIps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetDedicatedIps" target="_top">AWS API
     *      Documentation</a>
     */
    public GetDedicatedIpsPublisher getDedicatedIpsPaginator(GetDedicatedIpsRequest getDedicatedIpsRequest) {
        return new GetDedicatedIpsPublisher(this, applyPaginatorUserAgent(getDedicatedIpsRequest));
    }

    /**
     * <p>
     * Retrieve information about the status of the Deliverability dashboard for your account. When the Deliverability
     * dashboard is enabled, you gain access to reputation, deliverability, and other metrics for the domains that you
     * use to send email. You also gain the ability to perform predictive inbox placement tests.
     * </p>
     * <p>
     * When you use the Deliverability dashboard, you pay a monthly subscription charge, in addition to any other fees
     * that you accrue by using Amazon SES and other AWS services. For more information about the features and cost of a
     * Deliverability dashboard subscription, see <a href="http://aws.amazon.com/pinpoint/pricing/">Amazon Pinpoint
     * Pricing</a>.
     * </p>
     *
     * @param getDeliverabilityDashboardOptionsRequest
     *        Retrieve information about the status of the Deliverability dashboard for your AWS account. When the
     *        Deliverability dashboard is enabled, you gain access to reputation, deliverability, and other metrics for
     *        your domains. You also gain the ability to perform predictive inbox placement tests.</p>
     *        <p>
     *        When you use the Deliverability dashboard, you pay a monthly subscription charge, in addition to any other
     *        fees that you accrue by using Amazon SES and other AWS services. For more information about the features
     *        and cost of a Deliverability dashboard subscription, see <a
     *        href="http://aws.amazon.com/pinpoint/pricing/">Amazon Pinpoint Pricing</a>.
     * @return A Java Future containing the result of the GetDeliverabilityDashboardOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li> <li>
     *         LimitExceededException There are too many instances of the specified resource type.</li> <li>
     *         BadRequestException The input you provided is invalid.</li> <li>SdkException Base class for all
     *         exceptions that can be thrown by the SDK (both service and client). Can be used for catch all scenarios.
     *         </li> <li>SdkClientException If any client side error occurs such as an IO related failure, failure to
     *         get credentials, etc.</li> <li>SesV2Exception Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetDeliverabilityDashboardOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetDeliverabilityDashboardOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeliverabilityDashboardOptionsResponse> getDeliverabilityDashboardOptions(
            GetDeliverabilityDashboardOptionsRequest getDeliverabilityDashboardOptionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeliverabilityDashboardOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetDeliverabilityDashboardOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeliverabilityDashboardOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeliverabilityDashboardOptionsRequest, GetDeliverabilityDashboardOptionsResponse>()
                            .withOperationName("GetDeliverabilityDashboardOptions")
                            .withMarshaller(new GetDeliverabilityDashboardOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDeliverabilityDashboardOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the results of a predictive inbox placement test.
     * </p>
     *
     * @param getDeliverabilityTestReportRequest
     *        A request to retrieve the results of a predictive inbox placement test.
     * @return A Java Future containing the result of the GetDeliverabilityTestReport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetDeliverabilityTestReport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetDeliverabilityTestReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeliverabilityTestReportResponse> getDeliverabilityTestReport(
            GetDeliverabilityTestReportRequest getDeliverabilityTestReportRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeliverabilityTestReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDeliverabilityTestReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeliverabilityTestReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeliverabilityTestReportRequest, GetDeliverabilityTestReportResponse>()
                            .withOperationName("GetDeliverabilityTestReport")
                            .withMarshaller(new GetDeliverabilityTestReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDeliverabilityTestReportRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve all the deliverability data for a specific campaign. This data is available for a campaign only if the
     * campaign sent email by using a domain that the Deliverability dashboard is enabled for.
     * </p>
     *
     * @param getDomainDeliverabilityCampaignRequest
     *        Retrieve all the deliverability data for a specific campaign. This data is available for a campaign only
     *        if the campaign sent email by using a domain that the Deliverability dashboard is enabled for (
     *        <code>PutDeliverabilityDashboardOption</code> operation).
     * @return A Java Future containing the result of the GetDomainDeliverabilityCampaign operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetDomainDeliverabilityCampaign
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetDomainDeliverabilityCampaign"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainDeliverabilityCampaignResponse> getDomainDeliverabilityCampaign(
            GetDomainDeliverabilityCampaignRequest getDomainDeliverabilityCampaignRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainDeliverabilityCampaignResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainDeliverabilityCampaignResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainDeliverabilityCampaignResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainDeliverabilityCampaignRequest, GetDomainDeliverabilityCampaignResponse>()
                            .withOperationName("GetDomainDeliverabilityCampaign")
                            .withMarshaller(new GetDomainDeliverabilityCampaignRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDomainDeliverabilityCampaignRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve inbox placement and engagement rates for the domains that you use to send email.
     * </p>
     *
     * @param getDomainStatisticsReportRequest
     *        A request to obtain deliverability metrics for a domain.
     * @return A Java Future containing the result of the GetDomainStatisticsReport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetDomainStatisticsReport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetDomainStatisticsReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainStatisticsReportResponse> getDomainStatisticsReport(
            GetDomainStatisticsReportRequest getDomainStatisticsReportRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainStatisticsReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainStatisticsReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainStatisticsReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainStatisticsReportRequest, GetDomainStatisticsReportResponse>()
                            .withOperationName("GetDomainStatisticsReport")
                            .withMarshaller(new GetDomainStatisticsReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDomainStatisticsReportRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a specific identity, including the identity's verification status, its DKIM
     * authentication status, and its custom Mail-From settings.
     * </p>
     *
     * @param getEmailIdentityRequest
     *        A request to return details about an email identity.
     * @return A Java Future containing the result of the GetEmailIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.GetEmailIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/GetEmailIdentity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetEmailIdentityResponse> getEmailIdentity(GetEmailIdentityRequest getEmailIdentityRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEmailIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEmailIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEmailIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEmailIdentityRequest, GetEmailIdentityResponse>()
                            .withOperationName("GetEmailIdentity")
                            .withMarshaller(new GetEmailIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getEmailIdentityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all of the configuration sets associated with your account in the current region.
     * </p>
     * <p>
     * <i>Configuration sets</i> are groups of rules that you can apply to the emails you send. You apply a
     * configuration set to an email by including a reference to the configuration set in the headers of the email. When
     * you apply a configuration set to an email, all of the rules in that configuration set are applied to the email.
     * </p>
     *
     * @param listConfigurationSetsRequest
     *        A request to obtain a list of configuration sets for your Amazon SES account in the current AWS Region.
     * @return A Java Future containing the result of the ListConfigurationSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListConfigurationSets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListConfigurationSets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationSetsResponse> listConfigurationSets(
            ListConfigurationSetsRequest listConfigurationSetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationSetsRequest, ListConfigurationSetsResponse>()
                            .withOperationName("ListConfigurationSets")
                            .withMarshaller(new ListConfigurationSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listConfigurationSetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all of the configuration sets associated with your account in the current region.
     * </p>
     * <p>
     * <i>Configuration sets</i> are groups of rules that you can apply to the emails you send. You apply a
     * configuration set to an email by including a reference to the configuration set in the headers of the email. When
     * you apply a configuration set to an email, all of the rules in that configuration set are applied to the email.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConfigurationSets(software.amazon.awssdk.services.sesv2.model.ListConfigurationSetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListConfigurationSetsPublisher publisher = client.listConfigurationSetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListConfigurationSetsPublisher publisher = client.listConfigurationSetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sesv2.model.ListConfigurationSetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sesv2.model.ListConfigurationSetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConfigurationSets(software.amazon.awssdk.services.sesv2.model.ListConfigurationSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listConfigurationSetsRequest
     *        A request to obtain a list of configuration sets for your Amazon SES account in the current AWS Region.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListConfigurationSets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListConfigurationSets" target="_top">AWS
     *      API Documentation</a>
     */
    public ListConfigurationSetsPublisher listConfigurationSetsPaginator(ListConfigurationSetsRequest listConfigurationSetsRequest) {
        return new ListConfigurationSetsPublisher(this, applyPaginatorUserAgent(listConfigurationSetsRequest));
    }

    /**
     * <p>
     * List all of the dedicated IP pools that exist in your AWS account in the current Region.
     * </p>
     *
     * @param listDedicatedIpPoolsRequest
     *        A request to obtain a list of dedicated IP pools.
     * @return A Java Future containing the result of the ListDedicatedIpPools operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListDedicatedIpPools
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListDedicatedIpPools" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDedicatedIpPoolsResponse> listDedicatedIpPools(
            ListDedicatedIpPoolsRequest listDedicatedIpPoolsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDedicatedIpPoolsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDedicatedIpPoolsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDedicatedIpPoolsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDedicatedIpPoolsRequest, ListDedicatedIpPoolsResponse>()
                            .withOperationName("ListDedicatedIpPools")
                            .withMarshaller(new ListDedicatedIpPoolsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDedicatedIpPoolsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all of the dedicated IP pools that exist in your AWS account in the current Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDedicatedIpPools(software.amazon.awssdk.services.sesv2.model.ListDedicatedIpPoolsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListDedicatedIpPoolsPublisher publisher = client.listDedicatedIpPoolsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListDedicatedIpPoolsPublisher publisher = client.listDedicatedIpPoolsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sesv2.model.ListDedicatedIpPoolsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sesv2.model.ListDedicatedIpPoolsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDedicatedIpPools(software.amazon.awssdk.services.sesv2.model.ListDedicatedIpPoolsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDedicatedIpPoolsRequest
     *        A request to obtain a list of dedicated IP pools.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListDedicatedIpPools
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListDedicatedIpPools" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDedicatedIpPoolsPublisher listDedicatedIpPoolsPaginator(ListDedicatedIpPoolsRequest listDedicatedIpPoolsRequest) {
        return new ListDedicatedIpPoolsPublisher(this, applyPaginatorUserAgent(listDedicatedIpPoolsRequest));
    }

    /**
     * <p>
     * Show a list of the predictive inbox placement tests that you've performed, regardless of their statuses. For
     * predictive inbox placement tests that are complete, you can use the <code>GetDeliverabilityTestReport</code>
     * operation to view the results.
     * </p>
     *
     * @param listDeliverabilityTestReportsRequest
     *        A request to list all of the predictive inbox placement tests that you've performed.
     * @return A Java Future containing the result of the ListDeliverabilityTestReports operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListDeliverabilityTestReports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListDeliverabilityTestReports"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeliverabilityTestReportsResponse> listDeliverabilityTestReports(
            ListDeliverabilityTestReportsRequest listDeliverabilityTestReportsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeliverabilityTestReportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeliverabilityTestReportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeliverabilityTestReportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeliverabilityTestReportsRequest, ListDeliverabilityTestReportsResponse>()
                            .withOperationName("ListDeliverabilityTestReports")
                            .withMarshaller(new ListDeliverabilityTestReportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDeliverabilityTestReportsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Show a list of the predictive inbox placement tests that you've performed, regardless of their statuses. For
     * predictive inbox placement tests that are complete, you can use the <code>GetDeliverabilityTestReport</code>
     * operation to view the results.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDeliverabilityTestReports(software.amazon.awssdk.services.sesv2.model.ListDeliverabilityTestReportsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListDeliverabilityTestReportsPublisher publisher = client.listDeliverabilityTestReportsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListDeliverabilityTestReportsPublisher publisher = client.listDeliverabilityTestReportsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sesv2.model.ListDeliverabilityTestReportsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sesv2.model.ListDeliverabilityTestReportsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeliverabilityTestReports(software.amazon.awssdk.services.sesv2.model.ListDeliverabilityTestReportsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDeliverabilityTestReportsRequest
     *        A request to list all of the predictive inbox placement tests that you've performed.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListDeliverabilityTestReports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListDeliverabilityTestReports"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDeliverabilityTestReportsPublisher listDeliverabilityTestReportsPaginator(
            ListDeliverabilityTestReportsRequest listDeliverabilityTestReportsRequest) {
        return new ListDeliverabilityTestReportsPublisher(this, applyPaginatorUserAgent(listDeliverabilityTestReportsRequest));
    }

    /**
     * <p>
     * Retrieve deliverability data for all the campaigns that used a specific domain to send email during a specified
     * time range. This data is available for a domain only if you enabled the Deliverability dashboard for the domain.
     * </p>
     *
     * @param listDomainDeliverabilityCampaignsRequest
     *        Retrieve deliverability data for all the campaigns that used a specific domain to send email during a
     *        specified time range. This data is available for a domain only if you enabled the Deliverability
     *        dashboard.
     * @return A Java Future containing the result of the ListDomainDeliverabilityCampaigns operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListDomainDeliverabilityCampaigns
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListDomainDeliverabilityCampaigns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainDeliverabilityCampaignsResponse> listDomainDeliverabilityCampaigns(
            ListDomainDeliverabilityCampaignsRequest listDomainDeliverabilityCampaignsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainDeliverabilityCampaignsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListDomainDeliverabilityCampaignsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainDeliverabilityCampaignsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainDeliverabilityCampaignsRequest, ListDomainDeliverabilityCampaignsResponse>()
                            .withOperationName("ListDomainDeliverabilityCampaigns")
                            .withMarshaller(new ListDomainDeliverabilityCampaignsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDomainDeliverabilityCampaignsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve deliverability data for all the campaigns that used a specific domain to send email during a specified
     * time range. This data is available for a domain only if you enabled the Deliverability dashboard for the domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDomainDeliverabilityCampaigns(software.amazon.awssdk.services.sesv2.model.ListDomainDeliverabilityCampaignsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListDomainDeliverabilityCampaignsPublisher publisher = client.listDomainDeliverabilityCampaignsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListDomainDeliverabilityCampaignsPublisher publisher = client.listDomainDeliverabilityCampaignsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sesv2.model.ListDomainDeliverabilityCampaignsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sesv2.model.ListDomainDeliverabilityCampaignsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomainDeliverabilityCampaigns(software.amazon.awssdk.services.sesv2.model.ListDomainDeliverabilityCampaignsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDomainDeliverabilityCampaignsRequest
     *        Retrieve deliverability data for all the campaigns that used a specific domain to send email during a
     *        specified time range. This data is available for a domain only if you enabled the Deliverability
     *        dashboard.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListDomainDeliverabilityCampaigns
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListDomainDeliverabilityCampaigns"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDomainDeliverabilityCampaignsPublisher listDomainDeliverabilityCampaignsPaginator(
            ListDomainDeliverabilityCampaignsRequest listDomainDeliverabilityCampaignsRequest) {
        return new ListDomainDeliverabilityCampaignsPublisher(this,
                applyPaginatorUserAgent(listDomainDeliverabilityCampaignsRequest));
    }

    /**
     * <p>
     * Returns a list of all of the email identities that are associated with your AWS account. An identity can be
     * either an email address or a domain. This operation returns identities that are verified as well as those that
     * aren't. This operation returns identities that are associated with Amazon SES and Amazon Pinpoint.
     * </p>
     *
     * @param listEmailIdentitiesRequest
     *        A request to list all of the email identities associated with your AWS account. This list includes
     *        identities that you've already verified, identities that are unverified, and identities that were verified
     *        in the past, but are no longer verified.
     * @return A Java Future containing the result of the ListEmailIdentities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListEmailIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListEmailIdentities" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEmailIdentitiesResponse> listEmailIdentities(
            ListEmailIdentitiesRequest listEmailIdentitiesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEmailIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEmailIdentitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEmailIdentitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEmailIdentitiesRequest, ListEmailIdentitiesResponse>()
                            .withOperationName("ListEmailIdentities")
                            .withMarshaller(new ListEmailIdentitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEmailIdentitiesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all of the email identities that are associated with your AWS account. An identity can be
     * either an email address or a domain. This operation returns identities that are verified as well as those that
     * aren't. This operation returns identities that are associated with Amazon SES and Amazon Pinpoint.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEmailIdentities(software.amazon.awssdk.services.sesv2.model.ListEmailIdentitiesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListEmailIdentitiesPublisher publisher = client.listEmailIdentitiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sesv2.paginators.ListEmailIdentitiesPublisher publisher = client.listEmailIdentitiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sesv2.model.ListEmailIdentitiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sesv2.model.ListEmailIdentitiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEmailIdentities(software.amazon.awssdk.services.sesv2.model.ListEmailIdentitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEmailIdentitiesRequest
     *        A request to list all of the email identities associated with your AWS account. This list includes
     *        identities that you've already verified, identities that are unverified, and identities that were verified
     *        in the past, but are no longer verified.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListEmailIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListEmailIdentities" target="_top">AWS API
     *      Documentation</a>
     */
    public ListEmailIdentitiesPublisher listEmailIdentitiesPaginator(ListEmailIdentitiesRequest listEmailIdentitiesRequest) {
        return new ListEmailIdentitiesPublisher(this, applyPaginatorUserAgent(listEmailIdentitiesRequest));
    }

    /**
     * <p>
     * Retrieve a list of the tags (keys and values) that are associated with a specified resource. A <i>tag</i> is a
     * label that you optionally define and associate with a resource. Each tag consists of a required <i>tag
     * key</i> and an optional associated <i>tag value</i>. A tag key is a general label that acts as a category for
     * more specific tag values. A tag value acts as a descriptor within a tag key.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable the automatic warm-up feature for dedicated IP addresses.
     * </p>
     *
     * @param putAccountDedicatedIpWarmupAttributesRequest
     *        A request to enable or disable the automatic IP address warm-up feature.
     * @return A Java Future containing the result of the PutAccountDedicatedIpWarmupAttributes operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutAccountDedicatedIpWarmupAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutAccountDedicatedIpWarmupAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccountDedicatedIpWarmupAttributesResponse> putAccountDedicatedIpWarmupAttributes(
            PutAccountDedicatedIpWarmupAttributesRequest putAccountDedicatedIpWarmupAttributesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAccountDedicatedIpWarmupAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutAccountDedicatedIpWarmupAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAccountDedicatedIpWarmupAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccountDedicatedIpWarmupAttributesRequest, PutAccountDedicatedIpWarmupAttributesResponse>()
                            .withOperationName("PutAccountDedicatedIpWarmupAttributes")
                            .withMarshaller(new PutAccountDedicatedIpWarmupAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putAccountDedicatedIpWarmupAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable the ability of your account to send email.
     * </p>
     *
     * @param putAccountSendingAttributesRequest
     *        A request to change the ability of your account to send email.
     * @return A Java Future containing the result of the PutAccountSendingAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutAccountSendingAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutAccountSendingAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccountSendingAttributesResponse> putAccountSendingAttributes(
            PutAccountSendingAttributesRequest putAccountSendingAttributesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAccountSendingAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAccountSendingAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAccountSendingAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccountSendingAttributesRequest, PutAccountSendingAttributesResponse>()
                            .withOperationName("PutAccountSendingAttributes")
                            .withMarshaller(new PutAccountSendingAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putAccountSendingAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associate a configuration set with a dedicated IP pool. You can use dedicated IP pools to create groups of
     * dedicated IP addresses for sending specific types of email.
     * </p>
     *
     * @param putConfigurationSetDeliveryOptionsRequest
     *        A request to associate a configuration set with a dedicated IP pool.
     * @return A Java Future containing the result of the PutConfigurationSetDeliveryOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutConfigurationSetDeliveryOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutConfigurationSetDeliveryOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutConfigurationSetDeliveryOptionsResponse> putConfigurationSetDeliveryOptions(
            PutConfigurationSetDeliveryOptionsRequest putConfigurationSetDeliveryOptionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutConfigurationSetDeliveryOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutConfigurationSetDeliveryOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutConfigurationSetDeliveryOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutConfigurationSetDeliveryOptionsRequest, PutConfigurationSetDeliveryOptionsResponse>()
                            .withOperationName("PutConfigurationSetDeliveryOptions")
                            .withMarshaller(new PutConfigurationSetDeliveryOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putConfigurationSetDeliveryOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable collection of reputation metrics for emails that you send using a particular configuration set
     * in a specific AWS Region.
     * </p>
     *
     * @param putConfigurationSetReputationOptionsRequest
     *        A request to enable or disable tracking of reputation metrics for a configuration set.
     * @return A Java Future containing the result of the PutConfigurationSetReputationOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutConfigurationSetReputationOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutConfigurationSetReputationOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutConfigurationSetReputationOptionsResponse> putConfigurationSetReputationOptions(
            PutConfigurationSetReputationOptionsRequest putConfigurationSetReputationOptionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutConfigurationSetReputationOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutConfigurationSetReputationOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutConfigurationSetReputationOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutConfigurationSetReputationOptionsRequest, PutConfigurationSetReputationOptionsResponse>()
                            .withOperationName("PutConfigurationSetReputationOptions")
                            .withMarshaller(new PutConfigurationSetReputationOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putConfigurationSetReputationOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable email sending for messages that use a particular configuration set in a specific AWS Region.
     * </p>
     *
     * @param putConfigurationSetSendingOptionsRequest
     *        A request to enable or disable the ability of Amazon SES to send emails that use a specific configuration
     *        set.
     * @return A Java Future containing the result of the PutConfigurationSetSendingOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutConfigurationSetSendingOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutConfigurationSetSendingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutConfigurationSetSendingOptionsResponse> putConfigurationSetSendingOptions(
            PutConfigurationSetSendingOptionsRequest putConfigurationSetSendingOptionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutConfigurationSetSendingOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutConfigurationSetSendingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutConfigurationSetSendingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutConfigurationSetSendingOptionsRequest, PutConfigurationSetSendingOptionsResponse>()
                            .withOperationName("PutConfigurationSetSendingOptions")
                            .withMarshaller(new PutConfigurationSetSendingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putConfigurationSetSendingOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Specify a custom domain to use for open and click tracking elements in email that you send.
     * </p>
     *
     * @param putConfigurationSetTrackingOptionsRequest
     *        A request to add a custom domain for tracking open and click events to a configuration set.
     * @return A Java Future containing the result of the PutConfigurationSetTrackingOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutConfigurationSetTrackingOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutConfigurationSetTrackingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutConfigurationSetTrackingOptionsResponse> putConfigurationSetTrackingOptions(
            PutConfigurationSetTrackingOptionsRequest putConfigurationSetTrackingOptionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutConfigurationSetTrackingOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutConfigurationSetTrackingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutConfigurationSetTrackingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutConfigurationSetTrackingOptionsRequest, PutConfigurationSetTrackingOptionsResponse>()
                            .withOperationName("PutConfigurationSetTrackingOptions")
                            .withMarshaller(new PutConfigurationSetTrackingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putConfigurationSetTrackingOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Move a dedicated IP address to an existing dedicated IP pool.
     * </p>
     * <note>
     * <p>
     * The dedicated IP address that you specify must already exist, and must be associated with your AWS account.
     * </p>
     * <p>
     * The dedicated IP pool you specify must already exist. You can create a new pool by using the
     * <code>CreateDedicatedIpPool</code> operation.
     * </p>
     * </note>
     *
     * @param putDedicatedIpInPoolRequest
     *        A request to move a dedicated IP address to a dedicated IP pool.
     * @return A Java Future containing the result of the PutDedicatedIpInPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutDedicatedIpInPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutDedicatedIpInPool" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutDedicatedIpInPoolResponse> putDedicatedIpInPool(
            PutDedicatedIpInPoolRequest putDedicatedIpInPoolRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDedicatedIpInPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDedicatedIpInPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDedicatedIpInPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDedicatedIpInPoolRequest, PutDedicatedIpInPoolResponse>()
                            .withOperationName("PutDedicatedIpInPool")
                            .withMarshaller(new PutDedicatedIpInPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putDedicatedIpInPoolRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p/>
     *
     * @param putDedicatedIpWarmupAttributesRequest
     *        A request to change the warm-up attributes for a dedicated IP address. This operation is useful when you
     *        want to resume the warm-up process for an existing IP address.
     * @return A Java Future containing the result of the PutDedicatedIpWarmupAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li> <li>
     *         TooManyRequestsException Too many requests have been made to the operation.</li> <li>BadRequestException
     *         The input you provided is invalid.</li> <li>SdkException Base class for all exceptions that can be thrown
     *         by the SDK (both service and client). Can be used for catch all scenarios.</li> <li>SdkClientException If
     *         any client side error occurs such as an IO related failure, failure to get credentials, etc.</li> <li>
     *         SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an instance of
     *         this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutDedicatedIpWarmupAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutDedicatedIpWarmupAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDedicatedIpWarmupAttributesResponse> putDedicatedIpWarmupAttributes(
            PutDedicatedIpWarmupAttributesRequest putDedicatedIpWarmupAttributesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDedicatedIpWarmupAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDedicatedIpWarmupAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDedicatedIpWarmupAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDedicatedIpWarmupAttributesRequest, PutDedicatedIpWarmupAttributesResponse>()
                            .withOperationName("PutDedicatedIpWarmupAttributes")
                            .withMarshaller(new PutDedicatedIpWarmupAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putDedicatedIpWarmupAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable the Deliverability dashboard. When you enable the Deliverability dashboard, you gain access to
     * reputation, deliverability, and other metrics for the domains that you use to send email. You also gain the
     * ability to perform predictive inbox placement tests.
     * </p>
     * <p>
     * When you use the Deliverability dashboard, you pay a monthly subscription charge, in addition to any other fees
     * that you accrue by using Amazon SES and other AWS services. For more information about the features and cost of a
     * Deliverability dashboard subscription, see <a href="http://aws.amazon.com/pinpoint/pricing/">Amazon Pinpoint
     * Pricing</a>.
     * </p>
     *
     * @param putDeliverabilityDashboardOptionRequest
     *        Enable or disable the Deliverability dashboard. When you enable the Deliverability dashboard, you gain
     *        access to reputation, deliverability, and other metrics for the domains that you use to send email using
     *        Amazon SES API v2. You also gain the ability to perform predictive inbox placement tests.</p>
     *        <p>
     *        When you use the Deliverability dashboard, you pay a monthly subscription charge, in addition to any other
     *        fees that you accrue by using Amazon SES and other AWS services. For more information about the features
     *        and cost of a Deliverability dashboard subscription, see <a
     *        href="http://aws.amazon.com/pinpoint/pricing/">Amazon Pinpoint Pricing</a>.
     * @return A Java Future containing the result of the PutDeliverabilityDashboardOption operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource specified in your request already exists.</li> <li>
     *         NotFoundException The resource you attempted to access doesn't exist.</li> <li>TooManyRequestsException
     *         Too many requests have been made to the operation.</li> <li>LimitExceededException There are too many
     *         instances of the specified resource type.</li> <li>BadRequestException The input you provided is invalid.
     *         </li> <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li> <li>SdkClientException If any client side error occurs
     *         such as an IO related failure, failure to get credentials, etc.</li> <li>SesV2Exception Base class for
     *         all service exceptions. Unknown exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutDeliverabilityDashboardOption
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutDeliverabilityDashboardOption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDeliverabilityDashboardOptionResponse> putDeliverabilityDashboardOption(
            PutDeliverabilityDashboardOptionRequest putDeliverabilityDashboardOptionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDeliverabilityDashboardOptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutDeliverabilityDashboardOptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDeliverabilityDashboardOptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDeliverabilityDashboardOptionRequest, PutDeliverabilityDashboardOptionResponse>()
                            .withOperationName("PutDeliverabilityDashboardOption")
                            .withMarshaller(new PutDeliverabilityDashboardOptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putDeliverabilityDashboardOptionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to enable or disable DKIM authentication for an email identity.
     * </p>
     *
     * @param putEmailIdentityDkimAttributesRequest
     *        A request to enable or disable DKIM signing of email that you send from an email identity.
     * @return A Java Future containing the result of the PutEmailIdentityDkimAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutEmailIdentityDkimAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutEmailIdentityDkimAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutEmailIdentityDkimAttributesResponse> putEmailIdentityDkimAttributes(
            PutEmailIdentityDkimAttributesRequest putEmailIdentityDkimAttributesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEmailIdentityDkimAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutEmailIdentityDkimAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEmailIdentityDkimAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEmailIdentityDkimAttributesRequest, PutEmailIdentityDkimAttributesResponse>()
                            .withOperationName("PutEmailIdentityDkimAttributes")
                            .withMarshaller(new PutEmailIdentityDkimAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putEmailIdentityDkimAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to enable or disable feedback forwarding for an identity. This setting determines what happens when an
     * identity is used to send an email that results in a bounce or complaint event.
     * </p>
     * <p>
     * If the value is <code>true</code>, you receive email notifications when bounce or complaint events occur. These
     * notifications are sent to the address that you specified in the <code>Return-Path</code> header of the original
     * email.
     * </p>
     * <p>
     * You're required to have a method of tracking bounces and complaints. If you haven't set up another mechanism for
     * receiving bounce or complaint notifications (for example, by setting up an event destination), you receive an
     * email notification when these events occur (even if this setting is disabled).
     * </p>
     *
     * @param putEmailIdentityFeedbackAttributesRequest
     *        A request to set the attributes that control how bounce and complaint events are processed.
     * @return A Java Future containing the result of the PutEmailIdentityFeedbackAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutEmailIdentityFeedbackAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutEmailIdentityFeedbackAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutEmailIdentityFeedbackAttributesResponse> putEmailIdentityFeedbackAttributes(
            PutEmailIdentityFeedbackAttributesRequest putEmailIdentityFeedbackAttributesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEmailIdentityFeedbackAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutEmailIdentityFeedbackAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEmailIdentityFeedbackAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEmailIdentityFeedbackAttributesRequest, PutEmailIdentityFeedbackAttributesResponse>()
                            .withOperationName("PutEmailIdentityFeedbackAttributes")
                            .withMarshaller(new PutEmailIdentityFeedbackAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putEmailIdentityFeedbackAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to enable or disable the custom Mail-From domain configuration for an email identity.
     * </p>
     *
     * @param putEmailIdentityMailFromAttributesRequest
     *        A request to configure the custom MAIL FROM domain for a verified identity.
     * @return A Java Future containing the result of the PutEmailIdentityMailFromAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.PutEmailIdentityMailFromAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/PutEmailIdentityMailFromAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutEmailIdentityMailFromAttributesResponse> putEmailIdentityMailFromAttributes(
            PutEmailIdentityMailFromAttributesRequest putEmailIdentityMailFromAttributesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEmailIdentityMailFromAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutEmailIdentityMailFromAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEmailIdentityMailFromAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEmailIdentityMailFromAttributesRequest, PutEmailIdentityMailFromAttributesResponse>()
                            .withOperationName("PutEmailIdentityMailFromAttributes")
                            .withMarshaller(new PutEmailIdentityMailFromAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putEmailIdentityMailFromAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends an email message. You can use the Amazon SES API v2 to send two types of messages:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Simple</b> – A standard email message. When you create this type of message, you specify the sender, the
     * recipient, and the message body, and the Amazon SES API v2 assembles the message for you.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Raw</b> – A raw, MIME-formatted email message. When you send this type of email, you have to specify all of
     * the message headers, as well as the message body. You can use this message type to send messages that contain
     * attachments. The message that you specify has to be a valid MIME message.
     * </p>
     * </li>
     * </ul>
     *
     * @param sendEmailRequest
     *        A request to send an email message.
     * @return A Java Future containing the result of the SendEmail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>AccountSuspendedException The message can't be sent because the account's ability to send email has
     *         been permanently restricted.</li>
     *         <li>SendingPausedException The message can't be sent because the account's ability to send email is
     *         currently paused.</li>
     *         <li>MessageRejectedException The message can't be sent because it contains invalid content.</li>
     *         <li>MailFromDomainNotVerifiedException The message can't be sent because the sending domain isn't
     *         verified.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.SendEmail
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/SendEmail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SendEmailResponse> sendEmail(SendEmailRequest sendEmailRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendEmailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendEmailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendEmailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendEmailRequest, SendEmailResponse>().withOperationName("SendEmail")
                            .withMarshaller(new SendEmailRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(sendEmailRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add one or more tags (keys and values) to a specified resource. A <i>tag</i> is a label that you optionally
     * define and associate with a resource. Tags can help you categorize and manage resources in different ways, such
     * as by purpose, owner, environment, or other criteria. A resource can have as many as 50 tags.
     * </p>
     * <p>
     * Each tag consists of a required <i>tag key</i> and an associated <i>tag value</i>, both of which you define. A
     * tag key is a general label that acts as a category for more specific tag values. A tag value acts as a descriptor
     * within a tag key.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove one or more tags (keys and values) from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the configuration of an event destination for a configuration set.
     * </p>
     * <p>
     * <i>Events</i> include message sends, deliveries, opens, clicks, bounces, and complaints. <i>Event
     * destinations</i> are places that you can send information about these events to. For example, you can send event
     * data to Amazon SNS to receive notifications when you receive bounces or complaints, or you can use Amazon Kinesis
     * Data Firehose to stream data to Amazon S3 for long-term storage.
     * </p>
     *
     * @param updateConfigurationSetEventDestinationRequest
     *        A request to change the settings for an event destination for a configuration set.
     * @return A Java Future containing the result of the UpdateConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SesV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SesV2AsyncClient.UpdateConfigurationSetEventDestination
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/UpdateConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigurationSetEventDestinationResponse> updateConfigurationSetEventDestination(
            UpdateConfigurationSetEventDestinationRequest updateConfigurationSetEventDestinationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationSetEventDestinationRequest, UpdateConfigurationSetEventDestinationResponse>()
                            .withOperationName("UpdateConfigurationSetEventDestination")
                            .withMarshaller(new UpdateConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateConfigurationSetEventDestinationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SesV2Exception::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MessageRejected")
                                .exceptionBuilderSupplier(MessageRejectedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SendingPausedException")
                                .exceptionBuilderSupplier(SendingPausedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountSuspendedException")
                                .exceptionBuilderSupplier(AccountSuspendedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MailFromDomainNotVerifiedException")
                                .exceptionBuilderSupplier(MailFromDomainNotVerifiedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private <T extends SesV2Request> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
