/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sesv2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send email messages to multiple destinations using Amazon SES. For more information, see the
 * <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-personalized-email-api.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendBulkEmailRequest extends SesV2Request implements
        ToCopyableBuilder<SendBulkEmailRequest.Builder, SendBulkEmailRequest> {
    private static final SdkField<String> FROM_EMAIL_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FromEmailAddress").getter(getter(SendBulkEmailRequest::fromEmailAddress))
            .setter(setter(Builder::fromEmailAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FromEmailAddress").build()).build();

    private static final SdkField<String> FROM_EMAIL_ADDRESS_IDENTITY_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FromEmailAddressIdentityArn")
            .getter(getter(SendBulkEmailRequest::fromEmailAddressIdentityArn))
            .setter(setter(Builder::fromEmailAddressIdentityArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FromEmailAddressIdentityArn")
                    .build()).build();

    private static final SdkField<List<String>> REPLY_TO_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ReplyToAddresses")
            .getter(getter(SendBulkEmailRequest::replyToAddresses))
            .setter(setter(Builder::replyToAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplyToAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FEEDBACK_FORWARDING_EMAIL_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FeedbackForwardingEmailAddress")
            .getter(getter(SendBulkEmailRequest::feedbackForwardingEmailAddress))
            .setter(setter(Builder::feedbackForwardingEmailAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeedbackForwardingEmailAddress")
                    .build()).build();

    private static final SdkField<String> FEEDBACK_FORWARDING_EMAIL_ADDRESS_IDENTITY_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FeedbackForwardingEmailAddressIdentityArn")
            .getter(getter(SendBulkEmailRequest::feedbackForwardingEmailAddressIdentityArn))
            .setter(setter(Builder::feedbackForwardingEmailAddressIdentityArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("FeedbackForwardingEmailAddressIdentityArn").build()).build();

    private static final SdkField<List<MessageTag>> DEFAULT_EMAIL_TAGS_FIELD = SdkField
            .<List<MessageTag>> builder(MarshallingType.LIST)
            .memberName("DefaultEmailTags")
            .getter(getter(SendBulkEmailRequest::defaultEmailTags))
            .setter(setter(Builder::defaultEmailTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultEmailTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MessageTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<BulkEmailContent> DEFAULT_CONTENT_FIELD = SdkField
            .<BulkEmailContent> builder(MarshallingType.SDK_POJO).memberName("DefaultContent")
            .getter(getter(SendBulkEmailRequest::defaultContent)).setter(setter(Builder::defaultContent))
            .constructor(BulkEmailContent::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultContent").build()).build();

    private static final SdkField<List<BulkEmailEntry>> BULK_EMAIL_ENTRIES_FIELD = SdkField
            .<List<BulkEmailEntry>> builder(MarshallingType.LIST)
            .memberName("BulkEmailEntries")
            .getter(getter(SendBulkEmailRequest::bulkEmailEntries))
            .setter(setter(Builder::bulkEmailEntries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BulkEmailEntries").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BulkEmailEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(BulkEmailEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONFIGURATION_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationSetName").getter(getter(SendBulkEmailRequest::configurationSetName))
            .setter(setter(Builder::configurationSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSetName").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FROM_EMAIL_ADDRESS_FIELD,
            FROM_EMAIL_ADDRESS_IDENTITY_ARN_FIELD, REPLY_TO_ADDRESSES_FIELD, FEEDBACK_FORWARDING_EMAIL_ADDRESS_FIELD,
            FEEDBACK_FORWARDING_EMAIL_ADDRESS_IDENTITY_ARN_FIELD, DEFAULT_EMAIL_TAGS_FIELD, DEFAULT_CONTENT_FIELD,
            BULK_EMAIL_ENTRIES_FIELD, CONFIGURATION_SET_NAME_FIELD));

    private final String fromEmailAddress;

    private final String fromEmailAddressIdentityArn;

    private final List<String> replyToAddresses;

    private final String feedbackForwardingEmailAddress;

    private final String feedbackForwardingEmailAddressIdentityArn;

    private final List<MessageTag> defaultEmailTags;

    private final BulkEmailContent defaultContent;

    private final List<BulkEmailEntry> bulkEmailEntries;

    private final String configurationSetName;

    private SendBulkEmailRequest(BuilderImpl builder) {
        super(builder);
        this.fromEmailAddress = builder.fromEmailAddress;
        this.fromEmailAddressIdentityArn = builder.fromEmailAddressIdentityArn;
        this.replyToAddresses = builder.replyToAddresses;
        this.feedbackForwardingEmailAddress = builder.feedbackForwardingEmailAddress;
        this.feedbackForwardingEmailAddressIdentityArn = builder.feedbackForwardingEmailAddressIdentityArn;
        this.defaultEmailTags = builder.defaultEmailTags;
        this.defaultContent = builder.defaultContent;
        this.bulkEmailEntries = builder.bulkEmailEntries;
        this.configurationSetName = builder.configurationSetName;
    }

    /**
     * <p>
     * The email address that you want to use as the "From" address for the email. The address that you specify has to
     * be verified.
     * </p>
     * 
     * @return The email address that you want to use as the "From" address for the email. The address that you specify
     *         has to be verified.
     */
    public String fromEmailAddress() {
        return fromEmailAddress;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the
     * <code>FromEmailAddress</code> parameter.
     * </p>
     * <p>
     * For example, if the owner of example.com (which has ARN arn:aws:ses:us-east-1:123456789012:identity/example.com)
     * attaches a policy to it that authorizes you to use sender@example.com, then you would specify the
     * <code>FromEmailAddressIdentityArn</code> to be arn:aws:ses:us-east-1:123456789012:identity/example.com, and the
     * <code>FromEmailAddress</code> to be sender@example.com.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>FromEmailAddress</code> parameter.</p>
     *         <p>
     *         For example, if the owner of example.com (which has ARN
     *         arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you to
     *         use sender@example.com, then you would specify the <code>FromEmailAddressIdentityArn</code> to be
     *         arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FromEmailAddress</code> to be
     *         sender@example.com.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String fromEmailAddressIdentityArn() {
        return fromEmailAddressIdentityArn;
    }

    /**
     * Returns true if the ReplyToAddresses property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasReplyToAddresses() {
        return replyToAddresses != null && !(replyToAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The "Reply-to" email addresses for the message. When the recipient replies to the message, each Reply-to address
     * receives the reply.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasReplyToAddresses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The "Reply-to" email addresses for the message. When the recipient replies to the message, each Reply-to
     *         address receives the reply.
     */
    public List<String> replyToAddresses() {
        return replyToAddresses;
    }

    /**
     * <p>
     * The address that you want bounce and complaint notifications to be sent to.
     * </p>
     * 
     * @return The address that you want bounce and complaint notifications to be sent to.
     */
    public String feedbackForwardingEmailAddress() {
        return feedbackForwardingEmailAddress;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the
     * <code>FeedbackForwardingEmailAddress</code> parameter.
     * </p>
     * <p>
     * For example, if the owner of example.com (which has ARN arn:aws:ses:us-east-1:123456789012:identity/example.com)
     * attaches a policy to it that authorizes you to use feedback@example.com, then you would specify the
     * <code>FeedbackForwardingEmailAddressIdentityArn</code> to be
     * arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FeedbackForwardingEmailAddress</code> to
     * be feedback@example.com.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>FeedbackForwardingEmailAddress</code> parameter.</p>
     *         <p>
     *         For example, if the owner of example.com (which has ARN
     *         arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you to
     *         use feedback@example.com, then you would specify the
     *         <code>FeedbackForwardingEmailAddressIdentityArn</code> to be
     *         arn:aws:ses:us-east-1:123456789012:identity/example.com, and the
     *         <code>FeedbackForwardingEmailAddress</code> to be feedback@example.com.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String feedbackForwardingEmailAddressIdentityArn() {
        return feedbackForwardingEmailAddressIdentityArn;
    }

    /**
     * Returns true if the DefaultEmailTags property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasDefaultEmailTags() {
        return defaultEmailTags != null && !(defaultEmailTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send using the
     * <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so that you
     * can publish email sending events.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDefaultEmailTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send using the
     *         <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so
     *         that you can publish email sending events.
     */
    public List<MessageTag> defaultEmailTags() {
        return defaultEmailTags;
    }

    /**
     * <p>
     * An object that contains the body of the message. You can specify a template message.
     * </p>
     * 
     * @return An object that contains the body of the message. You can specify a template message.
     */
    public BulkEmailContent defaultContent() {
        return defaultContent;
    }

    /**
     * Returns true if the BulkEmailEntries property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasBulkEmailEntries() {
        return bulkEmailEntries != null && !(bulkEmailEntries instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of bulk email entry objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasBulkEmailEntries()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of bulk email entry objects.
     */
    public List<BulkEmailEntry> bulkEmailEntries() {
        return bulkEmailEntries;
    }

    /**
     * <p>
     * The name of the configuration set that you want to use when sending the email.
     * </p>
     * 
     * @return The name of the configuration set that you want to use when sending the email.
     */
    public String configurationSetName() {
        return configurationSetName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(fromEmailAddress());
        hashCode = 31 * hashCode + Objects.hashCode(fromEmailAddressIdentityArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasReplyToAddresses() ? replyToAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(feedbackForwardingEmailAddress());
        hashCode = 31 * hashCode + Objects.hashCode(feedbackForwardingEmailAddressIdentityArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasDefaultEmailTags() ? defaultEmailTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(defaultContent());
        hashCode = 31 * hashCode + Objects.hashCode(hasBulkEmailEntries() ? bulkEmailEntries() : null);
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendBulkEmailRequest)) {
            return false;
        }
        SendBulkEmailRequest other = (SendBulkEmailRequest) obj;
        return Objects.equals(fromEmailAddress(), other.fromEmailAddress())
                && Objects.equals(fromEmailAddressIdentityArn(), other.fromEmailAddressIdentityArn())
                && hasReplyToAddresses() == other.hasReplyToAddresses()
                && Objects.equals(replyToAddresses(), other.replyToAddresses())
                && Objects.equals(feedbackForwardingEmailAddress(), other.feedbackForwardingEmailAddress())
                && Objects.equals(feedbackForwardingEmailAddressIdentityArn(), other.feedbackForwardingEmailAddressIdentityArn())
                && hasDefaultEmailTags() == other.hasDefaultEmailTags()
                && Objects.equals(defaultEmailTags(), other.defaultEmailTags())
                && Objects.equals(defaultContent(), other.defaultContent())
                && hasBulkEmailEntries() == other.hasBulkEmailEntries()
                && Objects.equals(bulkEmailEntries(), other.bulkEmailEntries())
                && Objects.equals(configurationSetName(), other.configurationSetName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SendBulkEmailRequest").add("FromEmailAddress", fromEmailAddress())
                .add("FromEmailAddressIdentityArn", fromEmailAddressIdentityArn())
                .add("ReplyToAddresses", hasReplyToAddresses() ? replyToAddresses() : null)
                .add("FeedbackForwardingEmailAddress", feedbackForwardingEmailAddress())
                .add("FeedbackForwardingEmailAddressIdentityArn", feedbackForwardingEmailAddressIdentityArn())
                .add("DefaultEmailTags", hasDefaultEmailTags() ? defaultEmailTags() : null)
                .add("DefaultContent", defaultContent())
                .add("BulkEmailEntries", hasBulkEmailEntries() ? bulkEmailEntries() : null)
                .add("ConfigurationSetName", configurationSetName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FromEmailAddress":
            return Optional.ofNullable(clazz.cast(fromEmailAddress()));
        case "FromEmailAddressIdentityArn":
            return Optional.ofNullable(clazz.cast(fromEmailAddressIdentityArn()));
        case "ReplyToAddresses":
            return Optional.ofNullable(clazz.cast(replyToAddresses()));
        case "FeedbackForwardingEmailAddress":
            return Optional.ofNullable(clazz.cast(feedbackForwardingEmailAddress()));
        case "FeedbackForwardingEmailAddressIdentityArn":
            return Optional.ofNullable(clazz.cast(feedbackForwardingEmailAddressIdentityArn()));
        case "DefaultEmailTags":
            return Optional.ofNullable(clazz.cast(defaultEmailTags()));
        case "DefaultContent":
            return Optional.ofNullable(clazz.cast(defaultContent()));
        case "BulkEmailEntries":
            return Optional.ofNullable(clazz.cast(bulkEmailEntries()));
        case "ConfigurationSetName":
            return Optional.ofNullable(clazz.cast(configurationSetName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SendBulkEmailRequest, T> g) {
        return obj -> g.apply((SendBulkEmailRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesV2Request.Builder, SdkPojo, CopyableBuilder<Builder, SendBulkEmailRequest> {
        /**
         * <p>
         * The email address that you want to use as the "From" address for the email. The address that you specify has
         * to be verified.
         * </p>
         * 
         * @param fromEmailAddress
         *        The email address that you want to use as the "From" address for the email. The address that you
         *        specify has to be verified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromEmailAddress(String fromEmailAddress);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>FromEmailAddress</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of example.com (which has ARN
         * arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you to use
         * sender@example.com, then you would specify the <code>FromEmailAddressIdentityArn</code> to be
         * arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FromEmailAddress</code> to be
         * sender@example.com.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param fromEmailAddressIdentityArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>FromEmailAddress</code> parameter.</p>
         *        <p>
         *        For example, if the owner of example.com (which has ARN
         *        arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you
         *        to use sender@example.com, then you would specify the <code>FromEmailAddressIdentityArn</code> to be
         *        arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FromEmailAddress</code> to be
         *        sender@example.com.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromEmailAddressIdentityArn(String fromEmailAddressIdentityArn);

        /**
         * <p>
         * The "Reply-to" email addresses for the message. When the recipient replies to the message, each Reply-to
         * address receives the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The "Reply-to" email addresses for the message. When the recipient replies to the message, each
         *        Reply-to address receives the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(Collection<String> replyToAddresses);

        /**
         * <p>
         * The "Reply-to" email addresses for the message. When the recipient replies to the message, each Reply-to
         * address receives the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The "Reply-to" email addresses for the message. When the recipient replies to the message, each
         *        Reply-to address receives the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(String... replyToAddresses);

        /**
         * <p>
         * The address that you want bounce and complaint notifications to be sent to.
         * </p>
         * 
         * @param feedbackForwardingEmailAddress
         *        The address that you want bounce and complaint notifications to be sent to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder feedbackForwardingEmailAddress(String feedbackForwardingEmailAddress);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>FeedbackForwardingEmailAddress</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of example.com (which has ARN
         * arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you to use
         * feedback@example.com, then you would specify the <code>FeedbackForwardingEmailAddressIdentityArn</code> to be
         * arn:aws:ses:us-east-1:123456789012:identity/example.com, and the <code>FeedbackForwardingEmailAddress</code>
         * to be feedback@example.com.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param feedbackForwardingEmailAddressIdentityArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>FeedbackForwardingEmailAddress</code> parameter.</p>
         *        <p>
         *        For example, if the owner of example.com (which has ARN
         *        arn:aws:ses:us-east-1:123456789012:identity/example.com) attaches a policy to it that authorizes you
         *        to use feedback@example.com, then you would specify the
         *        <code>FeedbackForwardingEmailAddressIdentityArn</code> to be
         *        arn:aws:ses:us-east-1:123456789012:identity/example.com, and the
         *        <code>FeedbackForwardingEmailAddress</code> to be feedback@example.com.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder feedbackForwardingEmailAddressIdentityArn(String feedbackForwardingEmailAddressIdentityArn);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         * <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so that
         * you can publish email sending events.
         * </p>
         * 
         * @param defaultEmailTags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         *        <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so
         *        that you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultEmailTags(Collection<MessageTag> defaultEmailTags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         * <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so that
         * you can publish email sending events.
         * </p>
         * 
         * @param defaultEmailTags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         *        <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so
         *        that you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultEmailTags(MessageTag... defaultEmailTags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using the
         * <code>SendEmail</code> operation. Tags correspond to characteristics of the email that you define, so that
         * you can publish email sending events.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MessageTag>.Builder} avoiding the need to
         * create one manually via {@link List<MessageTag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MessageTag>.Builder#build()} is called immediately and its
         * result is passed to {@link #defaultEmailTags(List<MessageTag>)}.
         * 
         * @param defaultEmailTags
         *        a consumer that will call methods on {@link List<MessageTag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultEmailTags(List<MessageTag>)
         */
        Builder defaultEmailTags(Consumer<MessageTag.Builder>... defaultEmailTags);

        /**
         * <p>
         * An object that contains the body of the message. You can specify a template message.
         * </p>
         * 
         * @param defaultContent
         *        An object that contains the body of the message. You can specify a template message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultContent(BulkEmailContent defaultContent);

        /**
         * <p>
         * An object that contains the body of the message. You can specify a template message.
         * </p>
         * This is a convenience that creates an instance of the {@link BulkEmailContent.Builder} avoiding the need to
         * create one manually via {@link BulkEmailContent#builder()}.
         *
         * When the {@link Consumer} completes, {@link BulkEmailContent.Builder#build()} is called immediately and its
         * result is passed to {@link #defaultContent(BulkEmailContent)}.
         * 
         * @param defaultContent
         *        a consumer that will call methods on {@link BulkEmailContent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultContent(BulkEmailContent)
         */
        default Builder defaultContent(Consumer<BulkEmailContent.Builder> defaultContent) {
            return defaultContent(BulkEmailContent.builder().applyMutation(defaultContent).build());
        }

        /**
         * <p>
         * The list of bulk email entry objects.
         * </p>
         * 
         * @param bulkEmailEntries
         *        The list of bulk email entry objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bulkEmailEntries(Collection<BulkEmailEntry> bulkEmailEntries);

        /**
         * <p>
         * The list of bulk email entry objects.
         * </p>
         * 
         * @param bulkEmailEntries
         *        The list of bulk email entry objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bulkEmailEntries(BulkEmailEntry... bulkEmailEntries);

        /**
         * <p>
         * The list of bulk email entry objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<BulkEmailEntry>.Builder} avoiding the need
         * to create one manually via {@link List<BulkEmailEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<BulkEmailEntry>.Builder#build()} is called immediately and
         * its result is passed to {@link #bulkEmailEntries(List<BulkEmailEntry>)}.
         * 
         * @param bulkEmailEntries
         *        a consumer that will call methods on {@link List<BulkEmailEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bulkEmailEntries(List<BulkEmailEntry>)
         */
        Builder bulkEmailEntries(Consumer<BulkEmailEntry.Builder>... bulkEmailEntries);

        /**
         * <p>
         * The name of the configuration set that you want to use when sending the email.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set that you want to use when sending the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesV2Request.BuilderImpl implements Builder {
        private String fromEmailAddress;

        private String fromEmailAddressIdentityArn;

        private List<String> replyToAddresses = DefaultSdkAutoConstructList.getInstance();

        private String feedbackForwardingEmailAddress;

        private String feedbackForwardingEmailAddressIdentityArn;

        private List<MessageTag> defaultEmailTags = DefaultSdkAutoConstructList.getInstance();

        private BulkEmailContent defaultContent;

        private List<BulkEmailEntry> bulkEmailEntries = DefaultSdkAutoConstructList.getInstance();

        private String configurationSetName;

        private BuilderImpl() {
        }

        private BuilderImpl(SendBulkEmailRequest model) {
            super(model);
            fromEmailAddress(model.fromEmailAddress);
            fromEmailAddressIdentityArn(model.fromEmailAddressIdentityArn);
            replyToAddresses(model.replyToAddresses);
            feedbackForwardingEmailAddress(model.feedbackForwardingEmailAddress);
            feedbackForwardingEmailAddressIdentityArn(model.feedbackForwardingEmailAddressIdentityArn);
            defaultEmailTags(model.defaultEmailTags);
            defaultContent(model.defaultContent);
            bulkEmailEntries(model.bulkEmailEntries);
            configurationSetName(model.configurationSetName);
        }

        public final String getFromEmailAddress() {
            return fromEmailAddress;
        }

        @Override
        public final Builder fromEmailAddress(String fromEmailAddress) {
            this.fromEmailAddress = fromEmailAddress;
            return this;
        }

        public final void setFromEmailAddress(String fromEmailAddress) {
            this.fromEmailAddress = fromEmailAddress;
        }

        public final String getFromEmailAddressIdentityArn() {
            return fromEmailAddressIdentityArn;
        }

        @Override
        public final Builder fromEmailAddressIdentityArn(String fromEmailAddressIdentityArn) {
            this.fromEmailAddressIdentityArn = fromEmailAddressIdentityArn;
            return this;
        }

        public final void setFromEmailAddressIdentityArn(String fromEmailAddressIdentityArn) {
            this.fromEmailAddressIdentityArn = fromEmailAddressIdentityArn;
        }

        public final Collection<String> getReplyToAddresses() {
            if (replyToAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return replyToAddresses;
        }

        @Override
        public final Builder replyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = EmailAddressListCopier.copy(replyToAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replyToAddresses(String... replyToAddresses) {
            replyToAddresses(Arrays.asList(replyToAddresses));
            return this;
        }

        public final void setReplyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = EmailAddressListCopier.copy(replyToAddresses);
        }

        public final String getFeedbackForwardingEmailAddress() {
            return feedbackForwardingEmailAddress;
        }

        @Override
        public final Builder feedbackForwardingEmailAddress(String feedbackForwardingEmailAddress) {
            this.feedbackForwardingEmailAddress = feedbackForwardingEmailAddress;
            return this;
        }

        public final void setFeedbackForwardingEmailAddress(String feedbackForwardingEmailAddress) {
            this.feedbackForwardingEmailAddress = feedbackForwardingEmailAddress;
        }

        public final String getFeedbackForwardingEmailAddressIdentityArn() {
            return feedbackForwardingEmailAddressIdentityArn;
        }

        @Override
        public final Builder feedbackForwardingEmailAddressIdentityArn(String feedbackForwardingEmailAddressIdentityArn) {
            this.feedbackForwardingEmailAddressIdentityArn = feedbackForwardingEmailAddressIdentityArn;
            return this;
        }

        public final void setFeedbackForwardingEmailAddressIdentityArn(String feedbackForwardingEmailAddressIdentityArn) {
            this.feedbackForwardingEmailAddressIdentityArn = feedbackForwardingEmailAddressIdentityArn;
        }

        public final Collection<MessageTag.Builder> getDefaultEmailTags() {
            if (defaultEmailTags instanceof SdkAutoConstructList) {
                return null;
            }
            return defaultEmailTags != null ? defaultEmailTags.stream().map(MessageTag::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder defaultEmailTags(Collection<MessageTag> defaultEmailTags) {
            this.defaultEmailTags = MessageTagListCopier.copy(defaultEmailTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder defaultEmailTags(MessageTag... defaultEmailTags) {
            defaultEmailTags(Arrays.asList(defaultEmailTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder defaultEmailTags(Consumer<MessageTag.Builder>... defaultEmailTags) {
            defaultEmailTags(Stream.of(defaultEmailTags).map(c -> MessageTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDefaultEmailTags(Collection<MessageTag.BuilderImpl> defaultEmailTags) {
            this.defaultEmailTags = MessageTagListCopier.copyFromBuilder(defaultEmailTags);
        }

        public final BulkEmailContent.Builder getDefaultContent() {
            return defaultContent != null ? defaultContent.toBuilder() : null;
        }

        @Override
        public final Builder defaultContent(BulkEmailContent defaultContent) {
            this.defaultContent = defaultContent;
            return this;
        }

        public final void setDefaultContent(BulkEmailContent.BuilderImpl defaultContent) {
            this.defaultContent = defaultContent != null ? defaultContent.build() : null;
        }

        public final Collection<BulkEmailEntry.Builder> getBulkEmailEntries() {
            if (bulkEmailEntries instanceof SdkAutoConstructList) {
                return null;
            }
            return bulkEmailEntries != null ? bulkEmailEntries.stream().map(BulkEmailEntry::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder bulkEmailEntries(Collection<BulkEmailEntry> bulkEmailEntries) {
            this.bulkEmailEntries = BulkEmailEntryListCopier.copy(bulkEmailEntries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bulkEmailEntries(BulkEmailEntry... bulkEmailEntries) {
            bulkEmailEntries(Arrays.asList(bulkEmailEntries));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bulkEmailEntries(Consumer<BulkEmailEntry.Builder>... bulkEmailEntries) {
            bulkEmailEntries(Stream.of(bulkEmailEntries).map(c -> BulkEmailEntry.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setBulkEmailEntries(Collection<BulkEmailEntry.BuilderImpl> bulkEmailEntries) {
            this.bulkEmailEntries = BulkEmailEntryListCopier.copyFromBuilder(bulkEmailEntries);
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SendBulkEmailRequest build() {
            return new SendBulkEmailRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
