/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sesv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that contains information about the DKIM authentication status for an email identity.
 * </p>
 * <p>
 * Amazon SES determines the authentication status by searching for specific records in the DNS configuration for the
 * domain. If you used <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a> to
 * set up DKIM authentication, Amazon SES tries to find three unique CNAME records in the DNS configuration for your
 * domain. If you provided a public key to perform DKIM authentication, Amazon SES tries to find a TXT record that uses
 * the selector that you specified. The value of the TXT record must be a public key that's paired with the private key
 * that you specified in the process of creating the identity
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DkimAttributes implements SdkPojo, Serializable, ToCopyableBuilder<DkimAttributes.Builder, DkimAttributes> {
    private static final SdkField<Boolean> SIGNING_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SigningEnabled").getter(getter(DkimAttributes::signingEnabled)).setter(setter(Builder::signingEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningEnabled").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DkimAttributes::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<String>> TOKENS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Tokens")
            .getter(getter(DkimAttributes::tokens))
            .setter(setter(Builder::tokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SIGNING_ATTRIBUTES_ORIGIN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SigningAttributesOrigin").getter(getter(DkimAttributes::signingAttributesOriginAsString))
            .setter(setter(Builder::signingAttributesOrigin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningAttributesOrigin").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SIGNING_ENABLED_FIELD,
            STATUS_FIELD, TOKENS_FIELD, SIGNING_ATTRIBUTES_ORIGIN_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean signingEnabled;

    private final String status;

    private final List<String> tokens;

    private final String signingAttributesOrigin;

    private DkimAttributes(BuilderImpl builder) {
        this.signingEnabled = builder.signingEnabled;
        this.status = builder.status;
        this.tokens = builder.tokens;
        this.signingAttributesOrigin = builder.signingAttributesOrigin;
    }

    /**
     * <p>
     * If the value is <code>true</code>, then the messages that you send from the identity are signed using DKIM. If
     * the value is <code>false</code>, then the messages that you send from the identity aren't DKIM-signed.
     * </p>
     * 
     * @return If the value is <code>true</code>, then the messages that you send from the identity are signed using
     *         DKIM. If the value is <code>false</code>, then the messages that you send from the identity aren't
     *         DKIM-signed.
     */
    public final Boolean signingEnabled() {
        return signingEnabled;
    }

    /**
     * <p>
     * Describes whether or not Amazon SES has successfully located the DKIM records in the DNS records for the domain.
     * The status can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> – The verification process was initiated, but Amazon SES hasn't yet detected the DKIM
     * records in the DNS configuration for the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCESS</code> – The verification process completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> – The verification process failed. This typically occurs when Amazon SES fails to find the
     * DKIM records in the DNS configuration of the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TEMPORARY_FAILURE</code> – A temporary issue is preventing Amazon SES from determining the DKIM
     * authentication status of the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_STARTED</code> – The DKIM verification process hasn't been initiated for the domain.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DkimStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Describes whether or not Amazon SES has successfully located the DKIM records in the DNS records for the
     *         domain. The status can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> – The verification process was initiated, but Amazon SES hasn't yet detected the
     *         DKIM records in the DNS configuration for the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCESS</code> – The verification process completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> – The verification process failed. This typically occurs when Amazon SES fails to
     *         find the DKIM records in the DNS configuration of the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TEMPORARY_FAILURE</code> – A temporary issue is preventing Amazon SES from determining the DKIM
     *         authentication status of the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_STARTED</code> – The DKIM verification process hasn't been initiated for the domain.
     *         </p>
     *         </li>
     * @see DkimStatus
     */
    public final DkimStatus status() {
        return DkimStatus.fromValue(status);
    }

    /**
     * <p>
     * Describes whether or not Amazon SES has successfully located the DKIM records in the DNS records for the domain.
     * The status can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> – The verification process was initiated, but Amazon SES hasn't yet detected the DKIM
     * records in the DNS configuration for the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCESS</code> – The verification process completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> – The verification process failed. This typically occurs when Amazon SES fails to find the
     * DKIM records in the DNS configuration of the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TEMPORARY_FAILURE</code> – A temporary issue is preventing Amazon SES from determining the DKIM
     * authentication status of the domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_STARTED</code> – The DKIM verification process hasn't been initiated for the domain.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DkimStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Describes whether or not Amazon SES has successfully located the DKIM records in the DNS records for the
     *         domain. The status can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> – The verification process was initiated, but Amazon SES hasn't yet detected the
     *         DKIM records in the DNS configuration for the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCESS</code> – The verification process completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> – The verification process failed. This typically occurs when Amazon SES fails to
     *         find the DKIM records in the DNS configuration of the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TEMPORARY_FAILURE</code> – A temporary issue is preventing Amazon SES from determining the DKIM
     *         authentication status of the domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_STARTED</code> – The DKIM verification process hasn't been initiated for the domain.
     *         </p>
     *         </li>
     * @see DkimStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * Returns true if the Tokens property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTokens() {
        return tokens != null && !(tokens instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If you used <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a> to
     * configure DKIM authentication for the domain, then this object contains a set of unique strings that you use to
     * create a set of CNAME records that you add to the DNS configuration for your domain. When Amazon SES detects
     * these records in the DNS configuration for your domain, the DKIM authentication process is complete.
     * </p>
     * <p>
     * If you configured DKIM authentication for the domain by providing your own public-private key pair, then this
     * object contains the selector for the public key.
     * </p>
     * <p>
     * Regardless of the DKIM authentication method you use, Amazon SES searches for the appropriate records in the DNS
     * configuration of the domain for up to 72 hours.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTokens()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If you used <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>
     *         to configure DKIM authentication for the domain, then this object contains a set of unique strings that
     *         you use to create a set of CNAME records that you add to the DNS configuration for your domain. When
     *         Amazon SES detects these records in the DNS configuration for your domain, the DKIM authentication
     *         process is complete.</p>
     *         <p>
     *         If you configured DKIM authentication for the domain by providing your own public-private key pair, then
     *         this object contains the selector for the public key.
     *         </p>
     *         <p>
     *         Regardless of the DKIM authentication method you use, Amazon SES searches for the appropriate records in
     *         the DNS configuration of the domain for up to 72 hours.
     */
    public final List<String> tokens() {
        return tokens;
    }

    /**
     * <p>
     * A string that indicates how DKIM was configured for the identity. There are two possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS_SES</code> – Indicates that DKIM was configured for the identity by using <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EXTERNAL</code> – Indicates that DKIM was configured for the identity by using Bring Your Own DKIM
     * (BYODKIM).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #signingAttributesOrigin} will return {@link DkimSigningAttributesOrigin#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #signingAttributesOriginAsString}.
     * </p>
     * 
     * @return A string that indicates how DKIM was configured for the identity. There are two possible values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS_SES</code> – Indicates that DKIM was configured for the identity by using <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EXTERNAL</code> – Indicates that DKIM was configured for the identity by using Bring Your Own DKIM
     *         (BYODKIM).
     *         </p>
     *         </li>
     * @see DkimSigningAttributesOrigin
     */
    public final DkimSigningAttributesOrigin signingAttributesOrigin() {
        return DkimSigningAttributesOrigin.fromValue(signingAttributesOrigin);
    }

    /**
     * <p>
     * A string that indicates how DKIM was configured for the identity. There are two possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS_SES</code> – Indicates that DKIM was configured for the identity by using <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EXTERNAL</code> – Indicates that DKIM was configured for the identity by using Bring Your Own DKIM
     * (BYODKIM).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #signingAttributesOrigin} will return {@link DkimSigningAttributesOrigin#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #signingAttributesOriginAsString}.
     * </p>
     * 
     * @return A string that indicates how DKIM was configured for the identity. There are two possible values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS_SES</code> – Indicates that DKIM was configured for the identity by using <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EXTERNAL</code> – Indicates that DKIM was configured for the identity by using Bring Your Own DKIM
     *         (BYODKIM).
     *         </p>
     *         </li>
     * @see DkimSigningAttributesOrigin
     */
    public final String signingAttributesOriginAsString() {
        return signingAttributesOrigin;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(signingEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTokens() ? tokens() : null);
        hashCode = 31 * hashCode + Objects.hashCode(signingAttributesOriginAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DkimAttributes)) {
            return false;
        }
        DkimAttributes other = (DkimAttributes) obj;
        return Objects.equals(signingEnabled(), other.signingEnabled())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasTokens() == other.hasTokens()
                && Objects.equals(tokens(), other.tokens())
                && Objects.equals(signingAttributesOriginAsString(), other.signingAttributesOriginAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DkimAttributes").add("SigningEnabled", signingEnabled()).add("Status", statusAsString())
                .add("Tokens", hasTokens() ? tokens() : null).add("SigningAttributesOrigin", signingAttributesOriginAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SigningEnabled":
            return Optional.ofNullable(clazz.cast(signingEnabled()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Tokens":
            return Optional.ofNullable(clazz.cast(tokens()));
        case "SigningAttributesOrigin":
            return Optional.ofNullable(clazz.cast(signingAttributesOriginAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DkimAttributes, T> g) {
        return obj -> g.apply((DkimAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DkimAttributes> {
        /**
         * <p>
         * If the value is <code>true</code>, then the messages that you send from the identity are signed using DKIM.
         * If the value is <code>false</code>, then the messages that you send from the identity aren't DKIM-signed.
         * </p>
         * 
         * @param signingEnabled
         *        If the value is <code>true</code>, then the messages that you send from the identity are signed using
         *        DKIM. If the value is <code>false</code>, then the messages that you send from the identity aren't
         *        DKIM-signed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingEnabled(Boolean signingEnabled);

        /**
         * <p>
         * Describes whether or not Amazon SES has successfully located the DKIM records in the DNS records for the
         * domain. The status can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> – The verification process was initiated, but Amazon SES hasn't yet detected the DKIM
         * records in the DNS configuration for the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCESS</code> – The verification process completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> – The verification process failed. This typically occurs when Amazon SES fails to find
         * the DKIM records in the DNS configuration of the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TEMPORARY_FAILURE</code> – A temporary issue is preventing Amazon SES from determining the DKIM
         * authentication status of the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_STARTED</code> – The DKIM verification process hasn't been initiated for the domain.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Describes whether or not Amazon SES has successfully located the DKIM records in the DNS records for
         *        the domain. The status can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> – The verification process was initiated, but Amazon SES hasn't yet detected the
         *        DKIM records in the DNS configuration for the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCESS</code> – The verification process completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> – The verification process failed. This typically occurs when Amazon SES fails to
         *        find the DKIM records in the DNS configuration of the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TEMPORARY_FAILURE</code> – A temporary issue is preventing Amazon SES from determining the DKIM
         *        authentication status of the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_STARTED</code> – The DKIM verification process hasn't been initiated for the domain.
         *        </p>
         *        </li>
         * @see DkimStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DkimStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Describes whether or not Amazon SES has successfully located the DKIM records in the DNS records for the
         * domain. The status can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> – The verification process was initiated, but Amazon SES hasn't yet detected the DKIM
         * records in the DNS configuration for the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCESS</code> – The verification process completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> – The verification process failed. This typically occurs when Amazon SES fails to find
         * the DKIM records in the DNS configuration of the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TEMPORARY_FAILURE</code> – A temporary issue is preventing Amazon SES from determining the DKIM
         * authentication status of the domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_STARTED</code> – The DKIM verification process hasn't been initiated for the domain.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Describes whether or not Amazon SES has successfully located the DKIM records in the DNS records for
         *        the domain. The status can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> – The verification process was initiated, but Amazon SES hasn't yet detected the
         *        DKIM records in the DNS configuration for the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCESS</code> – The verification process completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> – The verification process failed. This typically occurs when Amazon SES fails to
         *        find the DKIM records in the DNS configuration of the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TEMPORARY_FAILURE</code> – A temporary issue is preventing Amazon SES from determining the DKIM
         *        authentication status of the domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_STARTED</code> – The DKIM verification process hasn't been initiated for the domain.
         *        </p>
         *        </li>
         * @see DkimStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DkimStatus
         */
        Builder status(DkimStatus status);

        /**
         * <p>
         * If you used <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a> to
         * configure DKIM authentication for the domain, then this object contains a set of unique strings that you use
         * to create a set of CNAME records that you add to the DNS configuration for your domain. When Amazon SES
         * detects these records in the DNS configuration for your domain, the DKIM authentication process is complete.
         * </p>
         * <p>
         * If you configured DKIM authentication for the domain by providing your own public-private key pair, then this
         * object contains the selector for the public key.
         * </p>
         * <p>
         * Regardless of the DKIM authentication method you use, Amazon SES searches for the appropriate records in the
         * DNS configuration of the domain for up to 72 hours.
         * </p>
         * 
         * @param tokens
         *        If you used <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy
         *        DKIM</a> to configure DKIM authentication for the domain, then this object contains a set of unique
         *        strings that you use to create a set of CNAME records that you add to the DNS configuration for your
         *        domain. When Amazon SES detects these records in the DNS configuration for your domain, the DKIM
         *        authentication process is complete.</p>
         *        <p>
         *        If you configured DKIM authentication for the domain by providing your own public-private key pair,
         *        then this object contains the selector for the public key.
         *        </p>
         *        <p>
         *        Regardless of the DKIM authentication method you use, Amazon SES searches for the appropriate records
         *        in the DNS configuration of the domain for up to 72 hours.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokens(Collection<String> tokens);

        /**
         * <p>
         * If you used <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a> to
         * configure DKIM authentication for the domain, then this object contains a set of unique strings that you use
         * to create a set of CNAME records that you add to the DNS configuration for your domain. When Amazon SES
         * detects these records in the DNS configuration for your domain, the DKIM authentication process is complete.
         * </p>
         * <p>
         * If you configured DKIM authentication for the domain by providing your own public-private key pair, then this
         * object contains the selector for the public key.
         * </p>
         * <p>
         * Regardless of the DKIM authentication method you use, Amazon SES searches for the appropriate records in the
         * DNS configuration of the domain for up to 72 hours.
         * </p>
         * 
         * @param tokens
         *        If you used <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy
         *        DKIM</a> to configure DKIM authentication for the domain, then this object contains a set of unique
         *        strings that you use to create a set of CNAME records that you add to the DNS configuration for your
         *        domain. When Amazon SES detects these records in the DNS configuration for your domain, the DKIM
         *        authentication process is complete.</p>
         *        <p>
         *        If you configured DKIM authentication for the domain by providing your own public-private key pair,
         *        then this object contains the selector for the public key.
         *        </p>
         *        <p>
         *        Regardless of the DKIM authentication method you use, Amazon SES searches for the appropriate records
         *        in the DNS configuration of the domain for up to 72 hours.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokens(String... tokens);

        /**
         * <p>
         * A string that indicates how DKIM was configured for the identity. There are two possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS_SES</code> – Indicates that DKIM was configured for the identity by using <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EXTERNAL</code> – Indicates that DKIM was configured for the identity by using Bring Your Own DKIM
         * (BYODKIM).
         * </p>
         * </li>
         * </ul>
         * 
         * @param signingAttributesOrigin
         *        A string that indicates how DKIM was configured for the identity. There are two possible values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS_SES</code> – Indicates that DKIM was configured for the identity by using <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EXTERNAL</code> – Indicates that DKIM was configured for the identity by using Bring Your Own
         *        DKIM (BYODKIM).
         *        </p>
         *        </li>
         * @see DkimSigningAttributesOrigin
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DkimSigningAttributesOrigin
         */
        Builder signingAttributesOrigin(String signingAttributesOrigin);

        /**
         * <p>
         * A string that indicates how DKIM was configured for the identity. There are two possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS_SES</code> – Indicates that DKIM was configured for the identity by using <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EXTERNAL</code> – Indicates that DKIM was configured for the identity by using Bring Your Own DKIM
         * (BYODKIM).
         * </p>
         * </li>
         * </ul>
         * 
         * @param signingAttributesOrigin
         *        A string that indicates how DKIM was configured for the identity. There are two possible values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS_SES</code> – Indicates that DKIM was configured for the identity by using <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EXTERNAL</code> – Indicates that DKIM was configured for the identity by using Bring Your Own
         *        DKIM (BYODKIM).
         *        </p>
         *        </li>
         * @see DkimSigningAttributesOrigin
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DkimSigningAttributesOrigin
         */
        Builder signingAttributesOrigin(DkimSigningAttributesOrigin signingAttributesOrigin);
    }

    static final class BuilderImpl implements Builder {
        private Boolean signingEnabled;

        private String status;

        private List<String> tokens = DefaultSdkAutoConstructList.getInstance();

        private String signingAttributesOrigin;

        private BuilderImpl() {
        }

        private BuilderImpl(DkimAttributes model) {
            signingEnabled(model.signingEnabled);
            status(model.status);
            tokens(model.tokens);
            signingAttributesOrigin(model.signingAttributesOrigin);
        }

        public final Boolean getSigningEnabled() {
            return signingEnabled;
        }

        @Override
        public final Builder signingEnabled(Boolean signingEnabled) {
            this.signingEnabled = signingEnabled;
            return this;
        }

        public final void setSigningEnabled(Boolean signingEnabled) {
            this.signingEnabled = signingEnabled;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DkimStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Collection<String> getTokens() {
            if (tokens instanceof SdkAutoConstructList) {
                return null;
            }
            return tokens;
        }

        @Override
        public final Builder tokens(Collection<String> tokens) {
            this.tokens = DnsTokenListCopier.copy(tokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tokens(String... tokens) {
            tokens(Arrays.asList(tokens));
            return this;
        }

        public final void setTokens(Collection<String> tokens) {
            this.tokens = DnsTokenListCopier.copy(tokens);
        }

        public final String getSigningAttributesOrigin() {
            return signingAttributesOrigin;
        }

        @Override
        public final Builder signingAttributesOrigin(String signingAttributesOrigin) {
            this.signingAttributesOrigin = signingAttributesOrigin;
            return this;
        }

        @Override
        public final Builder signingAttributesOrigin(DkimSigningAttributesOrigin signingAttributesOrigin) {
            this.signingAttributesOrigin(signingAttributesOrigin == null ? null : signingAttributesOrigin.toString());
            return this;
        }

        public final void setSigningAttributesOrigin(String signingAttributesOrigin) {
            this.signingAttributesOrigin = signingAttributesOrigin;
        }

        @Override
        public DkimAttributes build() {
            return new DkimAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
