/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sesv2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A request to begin the verification process for an email identity (an email address or domain).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateEmailIdentityRequest extends SesV2Request implements
        ToCopyableBuilder<CreateEmailIdentityRequest.Builder, CreateEmailIdentityRequest> {
    private static final SdkField<String> EMAIL_IDENTITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EmailIdentity").getter(getter(CreateEmailIdentityRequest::emailIdentity))
            .setter(setter(Builder::emailIdentity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmailIdentity").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateEmailIdentityRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DkimSigningAttributes> DKIM_SIGNING_ATTRIBUTES_FIELD = SdkField
            .<DkimSigningAttributes> builder(MarshallingType.SDK_POJO).memberName("DkimSigningAttributes")
            .getter(getter(CreateEmailIdentityRequest::dkimSigningAttributes)).setter(setter(Builder::dkimSigningAttributes))
            .constructor(DkimSigningAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DkimSigningAttributes").build())
            .build();

    private static final SdkField<String> CONFIGURATION_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationSetName").getter(getter(CreateEmailIdentityRequest::configurationSetName))
            .setter(setter(Builder::configurationSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSetName").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EMAIL_IDENTITY_FIELD,
            TAGS_FIELD, DKIM_SIGNING_ATTRIBUTES_FIELD, CONFIGURATION_SET_NAME_FIELD));

    private final String emailIdentity;

    private final List<Tag> tags;

    private final DkimSigningAttributes dkimSigningAttributes;

    private final String configurationSetName;

    private CreateEmailIdentityRequest(BuilderImpl builder) {
        super(builder);
        this.emailIdentity = builder.emailIdentity;
        this.tags = builder.tags;
        this.dkimSigningAttributes = builder.dkimSigningAttributes;
        this.configurationSetName = builder.configurationSetName;
    }

    /**
     * <p>
     * The email address or domain that you want to verify.
     * </p>
     * 
     * @return The email address or domain that you want to verify.
     */
    public final String emailIdentity() {
        return emailIdentity;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that define the tags (keys and values) that you want to associate with the email identity.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that define the tags (keys and values) that you want to associate with the email
     *         identity.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * If your request includes this object, Amazon SES configures the identity to use Bring Your Own DKIM (BYODKIM) for
     * DKIM authentication purposes, as opposed to the default method, <a
     * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
     * </p>
     * <p>
     * You can only specify this object if the email identity is a domain, as opposed to an address.
     * </p>
     * 
     * @return If your request includes this object, Amazon SES configures the identity to use Bring Your Own DKIM
     *         (BYODKIM) for DKIM authentication purposes, as opposed to the default method, <a
     *         href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.</p>
     *         <p>
     *         You can only specify this object if the email identity is a domain, as opposed to an address.
     */
    public final DkimSigningAttributes dkimSigningAttributes() {
        return dkimSigningAttributes;
    }

    /**
     * <p>
     * The configuration set to use by default when sending from this identity. Note that any configuration set defined
     * in the email sending request takes precedence.
     * </p>
     * 
     * @return The configuration set to use by default when sending from this identity. Note that any configuration set
     *         defined in the email sending request takes precedence.
     */
    public final String configurationSetName() {
        return configurationSetName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(emailIdentity());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dkimSigningAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateEmailIdentityRequest)) {
            return false;
        }
        CreateEmailIdentityRequest other = (CreateEmailIdentityRequest) obj;
        return Objects.equals(emailIdentity(), other.emailIdentity()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(dkimSigningAttributes(), other.dkimSigningAttributes())
                && Objects.equals(configurationSetName(), other.configurationSetName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateEmailIdentityRequest").add("EmailIdentity", emailIdentity())
                .add("Tags", hasTags() ? tags() : null).add("DkimSigningAttributes", dkimSigningAttributes())
                .add("ConfigurationSetName", configurationSetName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EmailIdentity":
            return Optional.ofNullable(clazz.cast(emailIdentity()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "DkimSigningAttributes":
            return Optional.ofNullable(clazz.cast(dkimSigningAttributes()));
        case "ConfigurationSetName":
            return Optional.ofNullable(clazz.cast(configurationSetName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateEmailIdentityRequest, T> g) {
        return obj -> g.apply((CreateEmailIdentityRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesV2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateEmailIdentityRequest> {
        /**
         * <p>
         * The email address or domain that you want to verify.
         * </p>
         * 
         * @param emailIdentity
         *        The email address or domain that you want to verify.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailIdentity(String emailIdentity);

        /**
         * <p>
         * An array of objects that define the tags (keys and values) that you want to associate with the email
         * identity.
         * </p>
         * 
         * @param tags
         *        An array of objects that define the tags (keys and values) that you want to associate with the email
         *        identity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * An array of objects that define the tags (keys and values) that you want to associate with the email
         * identity.
         * </p>
         * 
         * @param tags
         *        An array of objects that define the tags (keys and values) that you want to associate with the email
         *        identity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * An array of objects that define the tags (keys and values) that you want to associate with the email
         * identity.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * If your request includes this object, Amazon SES configures the identity to use Bring Your Own DKIM (BYODKIM)
         * for DKIM authentication purposes, as opposed to the default method, <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
         * </p>
         * <p>
         * You can only specify this object if the email identity is a domain, as opposed to an address.
         * </p>
         * 
         * @param dkimSigningAttributes
         *        If your request includes this object, Amazon SES configures the identity to use Bring Your Own DKIM
         *        (BYODKIM) for DKIM authentication purposes, as opposed to the default method, <a
         *        href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.</p>
         *        <p>
         *        You can only specify this object if the email identity is a domain, as opposed to an address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dkimSigningAttributes(DkimSigningAttributes dkimSigningAttributes);

        /**
         * <p>
         * If your request includes this object, Amazon SES configures the identity to use Bring Your Own DKIM (BYODKIM)
         * for DKIM authentication purposes, as opposed to the default method, <a
         * href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Easy DKIM</a>.
         * </p>
         * <p>
         * You can only specify this object if the email identity is a domain, as opposed to an address.
         * </p>
         * This is a convenience that creates an instance of the {@link DkimSigningAttributes.Builder} avoiding the need
         * to create one manually via {@link DkimSigningAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link DkimSigningAttributes.Builder#build()} is called immediately and
         * its result is passed to {@link #dkimSigningAttributes(DkimSigningAttributes)}.
         * 
         * @param dkimSigningAttributes
         *        a consumer that will call methods on {@link DkimSigningAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dkimSigningAttributes(DkimSigningAttributes)
         */
        default Builder dkimSigningAttributes(Consumer<DkimSigningAttributes.Builder> dkimSigningAttributes) {
            return dkimSigningAttributes(DkimSigningAttributes.builder().applyMutation(dkimSigningAttributes).build());
        }

        /**
         * <p>
         * The configuration set to use by default when sending from this identity. Note that any configuration set
         * defined in the email sending request takes precedence.
         * </p>
         * 
         * @param configurationSetName
         *        The configuration set to use by default when sending from this identity. Note that any configuration
         *        set defined in the email sending request takes precedence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesV2Request.BuilderImpl implements Builder {
        private String emailIdentity;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private DkimSigningAttributes dkimSigningAttributes;

        private String configurationSetName;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateEmailIdentityRequest model) {
            super(model);
            emailIdentity(model.emailIdentity);
            tags(model.tags);
            dkimSigningAttributes(model.dkimSigningAttributes);
            configurationSetName(model.configurationSetName);
        }

        public final String getEmailIdentity() {
            return emailIdentity;
        }

        @Override
        public final Builder emailIdentity(String emailIdentity) {
            this.emailIdentity = emailIdentity;
            return this;
        }

        public final void setEmailIdentity(String emailIdentity) {
            this.emailIdentity = emailIdentity;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final DkimSigningAttributes.Builder getDkimSigningAttributes() {
            return dkimSigningAttributes != null ? dkimSigningAttributes.toBuilder() : null;
        }

        @Override
        public final Builder dkimSigningAttributes(DkimSigningAttributes dkimSigningAttributes) {
            this.dkimSigningAttributes = dkimSigningAttributes;
            return this;
        }

        public final void setDkimSigningAttributes(DkimSigningAttributes.BuilderImpl dkimSigningAttributes) {
            this.dkimSigningAttributes = dkimSigningAttributes != null ? dkimSigningAttributes.build() : null;
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateEmailIdentityRequest build() {
            return new CreateEmailIdentityRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
