/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sesv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that defines the email template to use for an email message, and the values to use for any message
 * variables in that template. An <i>email template</i> is a type of message template that contains content that you
 * want to reuse in email messages that you send. You can specifiy the email template by providing the name or ARN of an
 * <i>email template</i> previously saved in your Amazon SES account or by providing the full template content.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Template implements SdkPojo, Serializable, ToCopyableBuilder<Template.Builder, Template> {
    private static final SdkField<String> TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateName").getter(getter(Template::templateName)).setter(setter(Builder::templateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateName").build()).build();

    private static final SdkField<String> TEMPLATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateArn").getter(getter(Template::templateArn)).setter(setter(Builder::templateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateArn").build()).build();

    private static final SdkField<EmailTemplateContent> TEMPLATE_CONTENT_FIELD = SdkField
            .<EmailTemplateContent> builder(MarshallingType.SDK_POJO).memberName("TemplateContent")
            .getter(getter(Template::templateContent)).setter(setter(Builder::templateContent))
            .constructor(EmailTemplateContent::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateContent").build()).build();

    private static final SdkField<String> TEMPLATE_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateData").getter(getter(Template::templateData)).setter(setter(Builder::templateData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateData").build()).build();

    private static final SdkField<List<MessageHeader>> HEADERS_FIELD = SdkField
            .<List<MessageHeader>> builder(MarshallingType.LIST)
            .memberName("Headers")
            .getter(getter(Template::headers))
            .setter(setter(Builder::headers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Headers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MessageHeader> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageHeader::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Attachment>> ATTACHMENTS_FIELD = SdkField
            .<List<Attachment>> builder(MarshallingType.LIST)
            .memberName("Attachments")
            .getter(getter(Template::attachments))
            .setter(setter(Builder::attachments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attachments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Attachment> builder(MarshallingType.SDK_POJO)
                                            .constructor(Attachment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TEMPLATE_NAME_FIELD,
            TEMPLATE_ARN_FIELD, TEMPLATE_CONTENT_FIELD, TEMPLATE_DATA_FIELD, HEADERS_FIELD, ATTACHMENTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String templateName;

    private final String templateArn;

    private final EmailTemplateContent templateContent;

    private final String templateData;

    private final List<MessageHeader> headers;

    private final List<Attachment> attachments;

    private Template(BuilderImpl builder) {
        this.templateName = builder.templateName;
        this.templateArn = builder.templateArn;
        this.templateContent = builder.templateContent;
        this.templateData = builder.templateData;
        this.headers = builder.headers;
        this.attachments = builder.attachments;
    }

    /**
     * <p>
     * The name of the template. You will refer to this name when you send email using the
     * <code>SendTemplatedEmail</code> or <code>SendBulkTemplatedEmail</code> operations.
     * </p>
     * 
     * @return The name of the template. You will refer to this name when you send email using the
     *         <code>SendTemplatedEmail</code> or <code>SendBulkTemplatedEmail</code> operations.
     */
    public final String templateName() {
        return templateName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the template.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the template.
     */
    public final String templateArn() {
        return templateArn;
    }

    /**
     * <p>
     * The content of the template.
     * </p>
     * <note>
     * <p>
     * Amazon SES supports only simple substitions when you send email using the <code>SendEmail</code> or
     * <code>SendBulkEmail</code> operations and you provide the full template content in the request.
     * </p>
     * </note>
     * 
     * @return The content of the template.</p> <note>
     *         <p>
     *         Amazon SES supports only simple substitions when you send email using the <code>SendEmail</code> or
     *         <code>SendBulkEmail</code> operations and you provide the full template content in the request.
     *         </p>
     */
    public final EmailTemplateContent templateContent() {
        return templateContent;
    }

    /**
     * <p>
     * An object that defines the values to use for message variables in the template. This object is a set of key-value
     * pairs. Each key defines a message variable in the template. The corresponding value defines the value to use for
     * that variable.
     * </p>
     * 
     * @return An object that defines the values to use for message variables in the template. This object is a set of
     *         key-value pairs. Each key defines a message variable in the template. The corresponding value defines the
     *         value to use for that variable.
     */
    public final String templateData() {
        return templateData;
    }

    /**
     * For responses, this returns true if the service returned a value for the Headers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasHeaders() {
        return headers != null && !(headers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of message headers that will be added to the email message.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHeaders} method.
     * </p>
     * 
     * @return The list of message headers that will be added to the email message.
     */
    public final List<MessageHeader> headers() {
        return headers;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attachments property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAttachments() {
        return attachments != null && !(attachments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The List of attachments to include in your email. All recipients will receive the same attachments.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttachments} method.
     * </p>
     * 
     * @return The List of attachments to include in your email. All recipients will receive the same attachments.
     */
    public final List<Attachment> attachments() {
        return attachments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(templateName());
        hashCode = 31 * hashCode + Objects.hashCode(templateArn());
        hashCode = 31 * hashCode + Objects.hashCode(templateContent());
        hashCode = 31 * hashCode + Objects.hashCode(templateData());
        hashCode = 31 * hashCode + Objects.hashCode(hasHeaders() ? headers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAttachments() ? attachments() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Template)) {
            return false;
        }
        Template other = (Template) obj;
        return Objects.equals(templateName(), other.templateName()) && Objects.equals(templateArn(), other.templateArn())
                && Objects.equals(templateContent(), other.templateContent())
                && Objects.equals(templateData(), other.templateData()) && hasHeaders() == other.hasHeaders()
                && Objects.equals(headers(), other.headers()) && hasAttachments() == other.hasAttachments()
                && Objects.equals(attachments(), other.attachments());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Template").add("TemplateName", templateName()).add("TemplateArn", templateArn())
                .add("TemplateContent", templateContent()).add("TemplateData", templateData())
                .add("Headers", hasHeaders() ? headers() : null).add("Attachments", hasAttachments() ? attachments() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TemplateName":
            return Optional.ofNullable(clazz.cast(templateName()));
        case "TemplateArn":
            return Optional.ofNullable(clazz.cast(templateArn()));
        case "TemplateContent":
            return Optional.ofNullable(clazz.cast(templateContent()));
        case "TemplateData":
            return Optional.ofNullable(clazz.cast(templateData()));
        case "Headers":
            return Optional.ofNullable(clazz.cast(headers()));
        case "Attachments":
            return Optional.ofNullable(clazz.cast(attachments()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("TemplateName", TEMPLATE_NAME_FIELD);
        map.put("TemplateArn", TEMPLATE_ARN_FIELD);
        map.put("TemplateContent", TEMPLATE_CONTENT_FIELD);
        map.put("TemplateData", TEMPLATE_DATA_FIELD);
        map.put("Headers", HEADERS_FIELD);
        map.put("Attachments", ATTACHMENTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Template, T> g) {
        return obj -> g.apply((Template) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Template> {
        /**
         * <p>
         * The name of the template. You will refer to this name when you send email using the
         * <code>SendTemplatedEmail</code> or <code>SendBulkTemplatedEmail</code> operations.
         * </p>
         * 
         * @param templateName
         *        The name of the template. You will refer to this name when you send email using the
         *        <code>SendTemplatedEmail</code> or <code>SendBulkTemplatedEmail</code> operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateName(String templateName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the template.
         * </p>
         * 
         * @param templateArn
         *        The Amazon Resource Name (ARN) of the template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateArn(String templateArn);

        /**
         * <p>
         * The content of the template.
         * </p>
         * <note>
         * <p>
         * Amazon SES supports only simple substitions when you send email using the <code>SendEmail</code> or
         * <code>SendBulkEmail</code> operations and you provide the full template content in the request.
         * </p>
         * </note>
         * 
         * @param templateContent
         *        The content of the template.</p> <note>
         *        <p>
         *        Amazon SES supports only simple substitions when you send email using the <code>SendEmail</code> or
         *        <code>SendBulkEmail</code> operations and you provide the full template content in the request.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateContent(EmailTemplateContent templateContent);

        /**
         * <p>
         * The content of the template.
         * </p>
         * <note>
         * <p>
         * Amazon SES supports only simple substitions when you send email using the <code>SendEmail</code> or
         * <code>SendBulkEmail</code> operations and you provide the full template content in the request.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link EmailTemplateContent.Builder}
         * avoiding the need to create one manually via {@link EmailTemplateContent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EmailTemplateContent.Builder#build()} is called immediately and
         * its result is passed to {@link #templateContent(EmailTemplateContent)}.
         * 
         * @param templateContent
         *        a consumer that will call methods on {@link EmailTemplateContent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #templateContent(EmailTemplateContent)
         */
        default Builder templateContent(Consumer<EmailTemplateContent.Builder> templateContent) {
            return templateContent(EmailTemplateContent.builder().applyMutation(templateContent).build());
        }

        /**
         * <p>
         * An object that defines the values to use for message variables in the template. This object is a set of
         * key-value pairs. Each key defines a message variable in the template. The corresponding value defines the
         * value to use for that variable.
         * </p>
         * 
         * @param templateData
         *        An object that defines the values to use for message variables in the template. This object is a set
         *        of key-value pairs. Each key defines a message variable in the template. The corresponding value
         *        defines the value to use for that variable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateData(String templateData);

        /**
         * <p>
         * The list of message headers that will be added to the email message.
         * </p>
         * 
         * @param headers
         *        The list of message headers that will be added to the email message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headers(Collection<MessageHeader> headers);

        /**
         * <p>
         * The list of message headers that will be added to the email message.
         * </p>
         * 
         * @param headers
         *        The list of message headers that will be added to the email message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headers(MessageHeader... headers);

        /**
         * <p>
         * The list of message headers that will be added to the email message.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sesv2.model.MessageHeader.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.sesv2.model.MessageHeader#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sesv2.model.MessageHeader.Builder#build()} is called immediately and
         * its result is passed to {@link #headers(List<MessageHeader>)}.
         * 
         * @param headers
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sesv2.model.MessageHeader.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #headers(java.util.Collection<MessageHeader>)
         */
        Builder headers(Consumer<MessageHeader.Builder>... headers);

        /**
         * <p>
         * The List of attachments to include in your email. All recipients will receive the same attachments.
         * </p>
         * 
         * @param attachments
         *        The List of attachments to include in your email. All recipients will receive the same attachments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachments(Collection<Attachment> attachments);

        /**
         * <p>
         * The List of attachments to include in your email. All recipients will receive the same attachments.
         * </p>
         * 
         * @param attachments
         *        The List of attachments to include in your email. All recipients will receive the same attachments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachments(Attachment... attachments);

        /**
         * <p>
         * The List of attachments to include in your email. All recipients will receive the same attachments.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sesv2.model.Attachment.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.sesv2.model.Attachment#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sesv2.model.Attachment.Builder#build()} is called immediately and its
         * result is passed to {@link #attachments(List<Attachment>)}.
         * 
         * @param attachments
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sesv2.model.Attachment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachments(java.util.Collection<Attachment>)
         */
        Builder attachments(Consumer<Attachment.Builder>... attachments);
    }

    static final class BuilderImpl implements Builder {
        private String templateName;

        private String templateArn;

        private EmailTemplateContent templateContent;

        private String templateData;

        private List<MessageHeader> headers = DefaultSdkAutoConstructList.getInstance();

        private List<Attachment> attachments = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Template model) {
            templateName(model.templateName);
            templateArn(model.templateArn);
            templateContent(model.templateContent);
            templateData(model.templateData);
            headers(model.headers);
            attachments(model.attachments);
        }

        public final String getTemplateName() {
            return templateName;
        }

        public final void setTemplateName(String templateName) {
            this.templateName = templateName;
        }

        @Override
        public final Builder templateName(String templateName) {
            this.templateName = templateName;
            return this;
        }

        public final String getTemplateArn() {
            return templateArn;
        }

        public final void setTemplateArn(String templateArn) {
            this.templateArn = templateArn;
        }

        @Override
        public final Builder templateArn(String templateArn) {
            this.templateArn = templateArn;
            return this;
        }

        public final EmailTemplateContent.Builder getTemplateContent() {
            return templateContent != null ? templateContent.toBuilder() : null;
        }

        public final void setTemplateContent(EmailTemplateContent.BuilderImpl templateContent) {
            this.templateContent = templateContent != null ? templateContent.build() : null;
        }

        @Override
        public final Builder templateContent(EmailTemplateContent templateContent) {
            this.templateContent = templateContent;
            return this;
        }

        public final String getTemplateData() {
            return templateData;
        }

        public final void setTemplateData(String templateData) {
            this.templateData = templateData;
        }

        @Override
        public final Builder templateData(String templateData) {
            this.templateData = templateData;
            return this;
        }

        public final List<MessageHeader.Builder> getHeaders() {
            List<MessageHeader.Builder> result = MessageHeaderListCopier.copyToBuilder(this.headers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setHeaders(Collection<MessageHeader.BuilderImpl> headers) {
            this.headers = MessageHeaderListCopier.copyFromBuilder(headers);
        }

        @Override
        public final Builder headers(Collection<MessageHeader> headers) {
            this.headers = MessageHeaderListCopier.copy(headers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder headers(MessageHeader... headers) {
            headers(Arrays.asList(headers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder headers(Consumer<MessageHeader.Builder>... headers) {
            headers(Stream.of(headers).map(c -> MessageHeader.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Attachment.Builder> getAttachments() {
            List<Attachment.Builder> result = AttachmentListCopier.copyToBuilder(this.attachments);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttachments(Collection<Attachment.BuilderImpl> attachments) {
            this.attachments = AttachmentListCopier.copyFromBuilder(attachments);
        }

        @Override
        public final Builder attachments(Collection<Attachment> attachments) {
            this.attachments = AttachmentListCopier.copy(attachments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachments(Attachment... attachments) {
            attachments(Arrays.asList(attachments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachments(Consumer<Attachment.Builder>... attachments) {
            attachments(Stream.of(attachments).map(c -> Attachment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Template build() {
            return new Template(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
