/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sfn;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sfn.model.ActivityDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.ActivityLimitExceededException;
import software.amazon.awssdk.services.sfn.model.ActivityWorkerLimitExceededException;
import software.amazon.awssdk.services.sfn.model.CreateActivityRequest;
import software.amazon.awssdk.services.sfn.model.CreateActivityResponse;
import software.amazon.awssdk.services.sfn.model.CreateStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.CreateStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.DeleteActivityRequest;
import software.amazon.awssdk.services.sfn.model.DeleteActivityResponse;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.DescribeActivityRequest;
import software.amazon.awssdk.services.sfn.model.DescribeActivityResponse;
import software.amazon.awssdk.services.sfn.model.DescribeExecutionRequest;
import software.amazon.awssdk.services.sfn.model.DescribeExecutionResponse;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineForExecutionRequest;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineForExecutionResponse;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.ExecutionAlreadyExistsException;
import software.amazon.awssdk.services.sfn.model.ExecutionDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.ExecutionLimitExceededException;
import software.amazon.awssdk.services.sfn.model.GetActivityTaskRequest;
import software.amazon.awssdk.services.sfn.model.GetActivityTaskResponse;
import software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest;
import software.amazon.awssdk.services.sfn.model.GetExecutionHistoryResponse;
import software.amazon.awssdk.services.sfn.model.InvalidArnException;
import software.amazon.awssdk.services.sfn.model.InvalidDefinitionException;
import software.amazon.awssdk.services.sfn.model.InvalidExecutionInputException;
import software.amazon.awssdk.services.sfn.model.InvalidLoggingConfigurationException;
import software.amazon.awssdk.services.sfn.model.InvalidNameException;
import software.amazon.awssdk.services.sfn.model.InvalidOutputException;
import software.amazon.awssdk.services.sfn.model.InvalidTokenException;
import software.amazon.awssdk.services.sfn.model.ListActivitiesRequest;
import software.amazon.awssdk.services.sfn.model.ListActivitiesResponse;
import software.amazon.awssdk.services.sfn.model.ListExecutionsRequest;
import software.amazon.awssdk.services.sfn.model.ListExecutionsResponse;
import software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest;
import software.amazon.awssdk.services.sfn.model.ListStateMachinesResponse;
import software.amazon.awssdk.services.sfn.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.sfn.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.sfn.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.sfn.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sfn.model.SendTaskFailureRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskFailureResponse;
import software.amazon.awssdk.services.sfn.model.SendTaskHeartbeatRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskHeartbeatResponse;
import software.amazon.awssdk.services.sfn.model.SendTaskSuccessRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskSuccessResponse;
import software.amazon.awssdk.services.sfn.model.SfnException;
import software.amazon.awssdk.services.sfn.model.SfnRequest;
import software.amazon.awssdk.services.sfn.model.StartExecutionRequest;
import software.amazon.awssdk.services.sfn.model.StartExecutionResponse;
import software.amazon.awssdk.services.sfn.model.StateMachineAlreadyExistsException;
import software.amazon.awssdk.services.sfn.model.StateMachineDeletingException;
import software.amazon.awssdk.services.sfn.model.StateMachineDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.StateMachineLimitExceededException;
import software.amazon.awssdk.services.sfn.model.StateMachineTypeNotSupportedException;
import software.amazon.awssdk.services.sfn.model.StopExecutionRequest;
import software.amazon.awssdk.services.sfn.model.StopExecutionResponse;
import software.amazon.awssdk.services.sfn.model.TagResourceRequest;
import software.amazon.awssdk.services.sfn.model.TagResourceResponse;
import software.amazon.awssdk.services.sfn.model.TaskDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.TaskTimedOutException;
import software.amazon.awssdk.services.sfn.model.TooManyTagsException;
import software.amazon.awssdk.services.sfn.model.UntagResourceRequest;
import software.amazon.awssdk.services.sfn.model.UntagResourceResponse;
import software.amazon.awssdk.services.sfn.model.UpdateStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.UpdateStateMachineResponse;
import software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable;
import software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable;
import software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable;
import software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable;
import software.amazon.awssdk.services.sfn.transform.CreateActivityRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.CreateStateMachineRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DeleteActivityRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DeleteStateMachineRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeActivityRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeStateMachineForExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeStateMachineRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.GetActivityTaskRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.GetExecutionHistoryRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListActivitiesRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListExecutionsRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListStateMachinesRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.SendTaskFailureRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.SendTaskHeartbeatRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.SendTaskSuccessRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.StartExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.StopExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.UpdateStateMachineRequestMarshaller;

/**
 * Internal implementation of {@link SfnClient}.
 *
 * @see SfnClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSfnClient implements SfnClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSfnClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an activity. An activity is a task that you write in any programming language and host on any machine
     * that has access to AWS Step Functions. Activities must poll Step Functions using the <code>GetActivityTask</code>
     * API action and respond using <code>SendTask*</code> API actions. This function lets Step Functions know the
     * existence of your activity and returns an identifier for use in a state machine and when polling from the
     * activity.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note> <note>
     * <p>
     * <code>CreateActivity</code> is an idempotent API. Subsequent requests won’t create a duplicate resource if it was
     * already created. <code>CreateActivity</code>'s idempotency check is based on the activity <code>name</code>. If a
     * following request has different <code>tags</code> values, Step Functions will ignore these differences and treat
     * it as an idempotent request of the previous. In this case, <code>tags</code> will not be updated, even if they
     * are different.
     * </p>
     * </note>
     *
     * @param createActivityRequest
     * @return Result of the CreateActivity operation returned by the service.
     * @throws ActivityLimitExceededException
     *         The maximum number of activities has been reached. Existing activities must be deleted before a new
     *         activity can be created.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.CreateActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateActivity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateActivityResponse createActivity(CreateActivityRequest createActivityRequest)
            throws ActivityLimitExceededException, InvalidNameException, TooManyTagsException, AwsServiceException,
            SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateActivityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateActivityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createActivityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateActivity");

            return clientHandler.execute(new ClientExecutionParams<CreateActivityRequest, CreateActivityResponse>()
                    .withOperationName("CreateActivity").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createActivityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateActivityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a state machine. A state machine consists of a collection of states that can do work (<code>Task</code>
     * states), determine to which states to transition next (<code>Choice</code> states), stop an execution with an
     * error (<code>Fail</code> states), and so on. State machines are specified using a JSON-based, structured
     * language. For more information, see <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html">Amazon States
     * Language</a> in the AWS Step Functions User Guide.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note> <note>
     * <p>
     * <code>CreateStateMachine</code> is an idempotent API. Subsequent requests won’t create a duplicate resource if it
     * was already created. <code>CreateStateMachine</code>'s idempotency check is based on the state machine
     * <code>name</code>, <code>definition</code>, <code>type</code>, and <code>LoggingConfiguration</code>. If a
     * following request has a different <code>roleArn</code> or <code>tags</code>, Step Functions will ignore these
     * differences and treat it as an idempotent request of the previous. In this case, <code>roleArn</code> and
     * <code>tags</code> will not be updated, even if they are different.
     * </p>
     * </note>
     *
     * @param createStateMachineRequest
     * @return Result of the CreateStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidDefinitionException
     *         The provided Amazon States Language definition is invalid.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws InvalidLoggingConfigurationException
     * @throws StateMachineAlreadyExistsException
     *         A state machine with the same name but a different definition or role ARN already exists.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws StateMachineLimitExceededException
     *         The maximum number of state machines has been reached. Existing state machines must be deleted before a
     *         new state machine can be created.
     * @throws StateMachineTypeNotSupportedException
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.CreateStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateStateMachineResponse createStateMachine(CreateStateMachineRequest createStateMachineRequest)
            throws InvalidArnException, InvalidDefinitionException, InvalidNameException, InvalidLoggingConfigurationException,
            StateMachineAlreadyExistsException, StateMachineDeletingException, StateMachineLimitExceededException,
            StateMachineTypeNotSupportedException, TooManyTagsException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStateMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateStateMachineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStateMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStateMachine");

            return clientHandler.execute(new ClientExecutionParams<CreateStateMachineRequest, CreateStateMachineResponse>()
                    .withOperationName("CreateStateMachine").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createStateMachineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateStateMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an activity.
     * </p>
     *
     * @param deleteActivityRequest
     * @return Result of the DeleteActivity operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DeleteActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteActivity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteActivityResponse deleteActivity(DeleteActivityRequest deleteActivityRequest) throws InvalidArnException,
            AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteActivityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteActivityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteActivityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteActivity");

            return clientHandler.execute(new ClientExecutionParams<DeleteActivityRequest, DeleteActivityResponse>()
                    .withOperationName("DeleteActivity").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteActivityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteActivityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a state machine. This is an asynchronous operation: It sets the state machine's status to
     * <code>DELETING</code> and begins the deletion process.
     * </p>
     * <note>
     * <p>
     * For <code>EXPRESS</code>state machines, the deletion will happen eventually (usually less than a minute). Running
     * executions may emit logs after <code>DeleteStateMachine</code> API is called.
     * </p>
     * </note>
     *
     * @param deleteStateMachineRequest
     * @return Result of the DeleteStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DeleteStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteStateMachineResponse deleteStateMachine(DeleteStateMachineRequest deleteStateMachineRequest)
            throws InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStateMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteStateMachineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStateMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStateMachine");

            return clientHandler.execute(new ClientExecutionParams<DeleteStateMachineRequest, DeleteStateMachineResponse>()
                    .withOperationName("DeleteStateMachine").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteStateMachineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteStateMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an activity.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param describeActivityRequest
     * @return Result of the DescribeActivity operation returned by the service.
     * @throws ActivityDoesNotExistException
     *         The specified activity does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeActivity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeActivityResponse describeActivity(DescribeActivityRequest describeActivityRequest)
            throws ActivityDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeActivityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeActivityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeActivityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeActivity");

            return clientHandler.execute(new ClientExecutionParams<DescribeActivityRequest, DescribeActivityResponse>()
                    .withOperationName("DescribeActivity").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeActivityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeActivityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an execution.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param describeExecutionRequest
     * @return Result of the DescribeExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeExecutionResponse describeExecution(DescribeExecutionRequest describeExecutionRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeExecutionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExecution");

            return clientHandler.execute(new ClientExecutionParams<DescribeExecutionRequest, DescribeExecutionResponse>()
                    .withOperationName("DescribeExecution").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeExecutionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a state machine.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param describeStateMachineRequest
     * @return Result of the DescribeStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachine" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeStateMachineResponse describeStateMachine(DescribeStateMachineRequest describeStateMachineRequest)
            throws InvalidArnException, StateMachineDoesNotExistException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStateMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeStateMachineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStateMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStateMachine");

            return clientHandler.execute(new ClientExecutionParams<DescribeStateMachineRequest, DescribeStateMachineResponse>()
                    .withOperationName("DescribeStateMachine").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeStateMachineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeStateMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the state machine associated with a specific execution.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param describeStateMachineForExecutionRequest
     * @return Result of the DescribeStateMachineForExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeStateMachineForExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachineForExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStateMachineForExecutionResponse describeStateMachineForExecution(
            DescribeStateMachineForExecutionRequest describeStateMachineForExecutionRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStateMachineForExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeStateMachineForExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeStateMachineForExecutionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStateMachineForExecution");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeStateMachineForExecutionRequest, DescribeStateMachineForExecutionResponse>()
                            .withOperationName("DescribeStateMachineForExecution").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeStateMachineForExecutionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeStateMachineForExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Used by workers to retrieve a task (with the specified activity ARN) which has been scheduled for execution by a
     * running state machine. This initiates a long poll, where the service holds the HTTP connection open and responds
     * as soon as a task becomes available (i.e. an execution of a task of this type is needed.) The maximum time the
     * service holds on to the request before responding is 60 seconds. If no task is available within 60 seconds, the
     * poll returns a <code>taskToken</code> with a null string.
     * </p>
     * <important>
     * <p>
     * Workers should set their client side socket timeout to at least 65 seconds (5 seconds higher than the maximum
     * time the service may hold the poll request).
     * </p>
     * <p>
     * Polling with <code>GetActivityTask</code> can cause latency in some implementations. See <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/bp-activity-pollers.html">Avoid Latency When Polling
     * for Activity Tasks</a> in the Step Functions Developer Guide.
     * </p>
     * </important>
     *
     * @param getActivityTaskRequest
     * @return Result of the GetActivityTask operation returned by the service.
     * @throws ActivityDoesNotExistException
     *         The specified activity does not exist.
     * @throws ActivityWorkerLimitExceededException
     *         The maximum number of workers concurrently polling for activity tasks has been reached.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetActivityTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetActivityTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetActivityTaskResponse getActivityTask(GetActivityTaskRequest getActivityTaskRequest)
            throws ActivityDoesNotExistException, ActivityWorkerLimitExceededException, InvalidArnException, AwsServiceException,
            SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetActivityTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetActivityTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getActivityTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetActivityTask");

            return clientHandler.execute(new ClientExecutionParams<GetActivityTaskRequest, GetActivityTaskResponse>()
                    .withOperationName("GetActivityTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getActivityTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetActivityTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the history of the specified execution as a list of events. By default, the results are returned in
     * ascending order of the <code>timeStamp</code> of the events. Use the <code>reverseOrder</code> parameter to get
     * the latest events first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param getExecutionHistoryRequest
     * @return Result of the GetExecutionHistory operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetExecutionHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetExecutionHistory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExecutionHistoryResponse getExecutionHistory(GetExecutionHistoryRequest getExecutionHistoryRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, InvalidTokenException, AwsServiceException,
            SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExecutionHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetExecutionHistoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExecutionHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExecutionHistory");

            return clientHandler.execute(new ClientExecutionParams<GetExecutionHistoryRequest, GetExecutionHistoryResponse>()
                    .withOperationName("GetExecutionHistory").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getExecutionHistoryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExecutionHistoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the history of the specified execution as a list of events. By default, the results are returned in
     * ascending order of the <code>timeStamp</code> of the events. Use the <code>reverseOrder</code> parameter to get
     * the latest events first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getExecutionHistory(software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client.getExecutionHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client
     *             .getExecutionHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.GetExecutionHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client.getExecutionHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getExecutionHistory(software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest)} operation.</b>
     * </p>
     *
     * @param getExecutionHistoryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetExecutionHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetExecutionHistory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExecutionHistoryIterable getExecutionHistoryPaginator(GetExecutionHistoryRequest getExecutionHistoryRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, InvalidTokenException, AwsServiceException,
            SdkClientException, SfnException {
        return new GetExecutionHistoryIterable(this, applyPaginatorUserAgent(getExecutionHistoryRequest));
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param listActivitiesRequest
     * @return Result of the ListActivities operation returned by the service.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListActivitiesResponse listActivities(ListActivitiesRequest listActivitiesRequest) throws InvalidTokenException,
            AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListActivitiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListActivitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listActivitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListActivities");

            return clientHandler.execute(new ClientExecutionParams<ListActivitiesRequest, ListActivitiesResponse>()
                    .withOperationName("ListActivities").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listActivitiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListActivitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of {@link #listActivities(software.amazon.awssdk.services.sfn.model.ListActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listActivities(software.amazon.awssdk.services.sfn.model.ListActivitiesRequest)} operation.</b>
     * </p>
     *
     * @param listActivitiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListActivitiesIterable listActivitiesPaginator(ListActivitiesRequest listActivitiesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        return new ListActivitiesIterable(this, applyPaginatorUserAgent(listActivitiesRequest));
    }

    /**
     * <p>
     * Lists the executions of a state machine that meet the filtering criteria. Results are sorted by time, with the
     * most recent execution first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param listExecutionsRequest
     * @return Result of the ListExecutions operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineTypeNotSupportedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListExecutionsResponse listExecutions(ListExecutionsRequest listExecutionsRequest) throws InvalidArnException,
            InvalidTokenException, StateMachineDoesNotExistException, StateMachineTypeNotSupportedException, AwsServiceException,
            SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExecutionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExecutions");

            return clientHandler.execute(new ClientExecutionParams<ListExecutionsRequest, ListExecutionsResponse>()
                    .withOperationName("ListExecutions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listExecutionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExecutionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the executions of a state machine that meet the filtering criteria. Results are sorted by time, with the
     * most recent execution first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listExecutions(software.amazon.awssdk.services.sfn.model.ListExecutionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListExecutionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExecutions(software.amazon.awssdk.services.sfn.model.ListExecutionsRequest)} operation.</b>
     * </p>
     *
     * @param listExecutionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineTypeNotSupportedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListExecutionsIterable listExecutionsPaginator(ListExecutionsRequest listExecutionsRequest)
            throws InvalidArnException, InvalidTokenException, StateMachineDoesNotExistException,
            StateMachineTypeNotSupportedException, AwsServiceException, SdkClientException, SfnException {
        return new ListExecutionsIterable(this, applyPaginatorUserAgent(listExecutionsRequest));
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param listStateMachinesRequest
     * @return Result of the ListStateMachines operation returned by the service.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListStateMachines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStateMachinesResponse listStateMachines(ListStateMachinesRequest listStateMachinesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStateMachinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStateMachinesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStateMachinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStateMachines");

            return clientHandler.execute(new ClientExecutionParams<ListStateMachinesRequest, ListStateMachinesResponse>()
                    .withOperationName("ListStateMachines").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listStateMachinesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStateMachinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listStateMachines(software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client.listStateMachinesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client
     *             .listStateMachinesPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListStateMachinesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client.listStateMachinesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStateMachines(software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest)} operation.</b>
     * </p>
     *
     * @param listStateMachinesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListStateMachines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStateMachinesIterable listStateMachinesPaginator(ListStateMachinesRequest listStateMachinesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        return new ListStateMachinesIterable(this, applyPaginatorUserAgent(listStateMachinesRequest));
    }

    /**
     * <p>
     * List tags for a given resource.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidArnException, ResourceNotFoundException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report that the task identified by the <code>taskToken</code> failed.
     * </p>
     *
     * @param sendTaskFailureRequest
     * @return Result of the SendTaskFailure operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskFailure
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskFailure" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SendTaskFailureResponse sendTaskFailure(SendTaskFailureRequest sendTaskFailureRequest)
            throws TaskDoesNotExistException, InvalidTokenException, TaskTimedOutException, AwsServiceException,
            SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendTaskFailureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SendTaskFailureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendTaskFailureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendTaskFailure");

            return clientHandler.execute(new ClientExecutionParams<SendTaskFailureRequest, SendTaskFailureResponse>()
                    .withOperationName("SendTaskFailure").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(sendTaskFailureRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SendTaskFailureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report to Step Functions that the task represented by the specified
     * <code>taskToken</code> is still making progress. This action resets the <code>Heartbeat</code> clock. The
     * <code>Heartbeat</code> threshold is specified in the state machine's Amazon States Language definition (
     * <code>HeartbeatSeconds</code>). This action does not in itself create an event in the execution history. However,
     * if the task times out, the execution history contains an <code>ActivityTimedOut</code> entry for activities, or a
     * <code>TaskTimedOut</code> entry for for tasks using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-sync">job run</a> or
     * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token">
     * callback</a> pattern.
     * </p>
     * <note>
     * <p>
     * The <code>Timeout</code> of a task, defined in the state machine's Amazon States Language definition, is its
     * maximum allowed duration, regardless of the number of <a>SendTaskHeartbeat</a> requests received. Use
     * <code>HeartbeatSeconds</code> to configure the timeout interval for heartbeats.
     * </p>
     * </note>
     *
     * @param sendTaskHeartbeatRequest
     * @return Result of the SendTaskHeartbeat operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskHeartbeat
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskHeartbeat" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SendTaskHeartbeatResponse sendTaskHeartbeat(SendTaskHeartbeatRequest sendTaskHeartbeatRequest)
            throws TaskDoesNotExistException, InvalidTokenException, TaskTimedOutException, AwsServiceException,
            SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendTaskHeartbeatResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SendTaskHeartbeatResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendTaskHeartbeatRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendTaskHeartbeat");

            return clientHandler.execute(new ClientExecutionParams<SendTaskHeartbeatRequest, SendTaskHeartbeatResponse>()
                    .withOperationName("SendTaskHeartbeat").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(sendTaskHeartbeatRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SendTaskHeartbeatRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report that the task identified by the <code>taskToken</code> completed successfully.
     * </p>
     *
     * @param sendTaskSuccessRequest
     * @return Result of the SendTaskSuccess operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidOutputException
     *         The provided JSON output data is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskSuccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskSuccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SendTaskSuccessResponse sendTaskSuccess(SendTaskSuccessRequest sendTaskSuccessRequest)
            throws TaskDoesNotExistException, InvalidOutputException, InvalidTokenException, TaskTimedOutException,
            AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendTaskSuccessResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SendTaskSuccessResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendTaskSuccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendTaskSuccess");

            return clientHandler.execute(new ClientExecutionParams<SendTaskSuccessRequest, SendTaskSuccessResponse>()
                    .withOperationName("SendTaskSuccess").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(sendTaskSuccessRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SendTaskSuccessRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a state machine execution.
     * </p>
     * <note>
     * <p>
     * <code>StartExecution</code> is idempotent. If <code>StartExecution</code> is called with the same name and input
     * as a running execution, the call will succeed and return the same response as the original request. If the
     * execution is closed or if the input is different, it will return a 400 <code>ExecutionAlreadyExists</code> error.
     * Names can be reused after 90 days.
     * </p>
     * </note>
     *
     * @param startExecutionRequest
     * @return Result of the StartExecution operation returned by the service.
     * @throws ExecutionLimitExceededException
     *         The maximum number of running executions has been reached. Running executions must end or be stopped
     *         before a new execution can be started.
     * @throws ExecutionAlreadyExistsException
     *         The execution has the same <code>name</code> as another execution (but a different <code>input</code>
     *         ).</p> <note>
     *         <p>
     *         Executions with the same <code>name</code> and <code>input</code> are considered idempotent.
     *         </p>
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidExecutionInputException
     *         The provided JSON input data is invalid.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.StartExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StartExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartExecutionResponse startExecution(StartExecutionRequest startExecutionRequest)
            throws ExecutionLimitExceededException, ExecutionAlreadyExistsException, InvalidArnException,
            InvalidExecutionInputException, InvalidNameException, StateMachineDoesNotExistException,
            StateMachineDeletingException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartExecutionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartExecution");

            return clientHandler.execute(new ClientExecutionParams<StartExecutionRequest, StartExecutionResponse>()
                    .withOperationName("StartExecution").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startExecutionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an execution.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param stopExecutionRequest
     * @return Result of the StopExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.StopExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StopExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopExecutionResponse stopExecution(StopExecutionRequest stopExecutionRequest) throws ExecutionDoesNotExistException,
            InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopExecutionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopExecution");

            return clientHandler.execute(new ClientExecutionParams<StopExecutionRequest, StopExecutionResponse>()
                    .withOperationName("StopExecution").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopExecutionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Add a tag to a Step Functions resource.
     * </p>
     * <p>
     * An array of key-value pairs. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>AWS Billing and Cost Management User Guide</i>, and <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
     * Tags</a>.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidArnException,
            ResourceNotFoundException, TooManyTagsException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Remove a tag from a Step Functions resource
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidArnException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing state machine by modifying its <code>definition</code>, <code>roleArn</code>, or
     * <code>loggingConfiguration</code>. Running executions will continue to use the previous <code>definition</code>
     * and <code>roleArn</code>. You must include at least one of <code>definition</code> or <code>roleArn</code> or you
     * will receive a <code>MissingRequiredParameter</code> error.
     * </p>
     * <note>
     * <p>
     * All <code>StartExecution</code> calls within a few seconds will use the updated <code>definition</code> and
     * <code>roleArn</code>. Executions started immediately after calling <code>UpdateStateMachine</code> may use the
     * previous state machine <code>definition</code> and <code>roleArn</code>.
     * </p>
     * </note>
     *
     * @param updateStateMachineRequest
     * @return Result of the UpdateStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidDefinitionException
     *         The provided Amazon States Language definition is invalid.
     * @throws InvalidLoggingConfigurationException
     * @throws MissingRequiredParameterException
     *         Request is missing a required parameter. This error occurs if both <code>definition</code> and
     *         <code>roleArn</code> are not specified.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.UpdateStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UpdateStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateStateMachineResponse updateStateMachine(UpdateStateMachineRequest updateStateMachineRequest)
            throws InvalidArnException, InvalidDefinitionException, InvalidLoggingConfigurationException,
            MissingRequiredParameterException, StateMachineDeletingException, StateMachineDoesNotExistException,
            AwsServiceException, SdkClientException, SfnException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStateMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateStateMachineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStateMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStateMachine");

            return clientHandler.execute(new ClientExecutionParams<UpdateStateMachineRequest, UpdateStateMachineResponse>()
                    .withOperationName("UpdateStateMachine").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateStateMachineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateStateMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SfnException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActivityDoesNotExist")
                                .exceptionBuilderSupplier(ActivityDoesNotExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineTypeNotSupported")
                                .exceptionBuilderSupplier(StateMachineTypeNotSupportedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDefinition")
                                .exceptionBuilderSupplier(InvalidDefinitionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidName")
                                .exceptionBuilderSupplier(InvalidNameException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLoggingConfiguration")
                                .exceptionBuilderSupplier(InvalidLoggingConfigurationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineAlreadyExists")
                                .exceptionBuilderSupplier(StateMachineAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecutionLimitExceeded")
                                .exceptionBuilderSupplier(ExecutionLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArn")
                                .exceptionBuilderSupplier(InvalidArnException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineDeleting")
                                .exceptionBuilderSupplier(StateMachineDeletingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActivityWorkerLimitExceeded")
                                .exceptionBuilderSupplier(ActivityWorkerLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TaskTimedOut")
                                .exceptionBuilderSupplier(TaskTimedOutException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecutionAlreadyExists")
                                .exceptionBuilderSupplier(ExecutionAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameter")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecutionDoesNotExist")
                                .exceptionBuilderSupplier(ExecutionDoesNotExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineLimitExceeded")
                                .exceptionBuilderSupplier(StateMachineLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidToken")
                                .exceptionBuilderSupplier(InvalidTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTags")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOutput")
                                .exceptionBuilderSupplier(InvalidOutputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActivityLimitExceeded")
                                .exceptionBuilderSupplier(ActivityLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidExecutionInput")
                                .exceptionBuilderSupplier(InvalidExecutionInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TaskDoesNotExist")
                                .exceptionBuilderSupplier(TaskDoesNotExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineDoesNotExist")
                                .exceptionBuilderSupplier(StateMachineDoesNotExistException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SfnRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
