/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sfn.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Settings to configure server-side encryption.
 * </p>
 * <p>
 * For additional control over security, you can encrypt your data using a <b>customer-managed key</b> for Step
 * Functions state machines and activities. You can configure a symmetric KMS key and data key reuse period when
 * creating or updating a <b>State Machine</b>, and when creating an <b>Activity</b>. The execution history and state
 * machine definition will be encrypted with the key applied to the State Machine. Activity inputs will be encrypted
 * with the key applied to the Activity.
 * </p>
 * <note>
 * <p>
 * Step Functions automatically enables encryption at rest using Amazon Web Services owned keys at no charge. However,
 * KMS charges apply when using a customer managed key. For more information about pricing, see <a
 * href="https://aws.amazon.com/kms/pricing/">Key Management Service pricing</a>.
 * </p>
 * </note>
 * <p>
 * For more information on KMS, see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/overview.html">What
 * is Key Management Service?</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EncryptionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<EncryptionConfiguration.Builder, EncryptionConfiguration> {
    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyId").getter(getter(EncryptionConfiguration::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyId").build()).build();

    private static final SdkField<Integer> KMS_DATA_KEY_REUSE_PERIOD_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("kmsDataKeyReusePeriodSeconds")
            .getter(getter(EncryptionConfiguration::kmsDataKeyReusePeriodSeconds))
            .setter(setter(Builder::kmsDataKeyReusePeriodSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsDataKeyReusePeriodSeconds")
                    .build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(EncryptionConfiguration::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KMS_KEY_ID_FIELD,
            KMS_DATA_KEY_REUSE_PERIOD_SECONDS_FIELD, TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String kmsKeyId;

    private final Integer kmsDataKeyReusePeriodSeconds;

    private final String type;

    private EncryptionConfiguration(BuilderImpl builder) {
        this.kmsKeyId = builder.kmsKeyId;
        this.kmsDataKeyReusePeriodSeconds = builder.kmsDataKeyReusePeriodSeconds;
        this.type = builder.type;
    }

    /**
     * <p>
     * An alias, alias ARN, key ID, or key ARN of a symmetric encryption KMS key to encrypt data. To specify a KMS key
     * in a different Amazon Web Services account, you must use the key ARN or alias ARN.
     * </p>
     * 
     * @return An alias, alias ARN, key ID, or key ARN of a symmetric encryption KMS key to encrypt data. To specify a
     *         KMS key in a different Amazon Web Services account, you must use the key ARN or alias ARN.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * Maximum duration that Step Functions will reuse data keys. When the period expires, Step Functions will call
     * <code>GenerateDataKey</code>. Only applies to customer managed keys.
     * </p>
     * 
     * @return Maximum duration that Step Functions will reuse data keys. When the period expires, Step Functions will
     *         call <code>GenerateDataKey</code>. Only applies to customer managed keys.
     */
    public final Integer kmsDataKeyReusePeriodSeconds() {
        return kmsDataKeyReusePeriodSeconds;
    }

    /**
     * <p>
     * Encryption type
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Encryption type
     * @see EncryptionType
     */
    public final EncryptionType type() {
        return EncryptionType.fromValue(type);
    }

    /**
     * <p>
     * Encryption type
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Encryption type
     * @see EncryptionType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(kmsDataKeyReusePeriodSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EncryptionConfiguration)) {
            return false;
        }
        EncryptionConfiguration other = (EncryptionConfiguration) obj;
        return Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(kmsDataKeyReusePeriodSeconds(), other.kmsDataKeyReusePeriodSeconds())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EncryptionConfiguration").add("KmsKeyId", kmsKeyId())
                .add("KmsDataKeyReusePeriodSeconds", kmsDataKeyReusePeriodSeconds()).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "kmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "kmsDataKeyReusePeriodSeconds":
            return Optional.ofNullable(clazz.cast(kmsDataKeyReusePeriodSeconds()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("kmsKeyId", KMS_KEY_ID_FIELD);
        map.put("kmsDataKeyReusePeriodSeconds", KMS_DATA_KEY_REUSE_PERIOD_SECONDS_FIELD);
        map.put("type", TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EncryptionConfiguration, T> g) {
        return obj -> g.apply((EncryptionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EncryptionConfiguration> {
        /**
         * <p>
         * An alias, alias ARN, key ID, or key ARN of a symmetric encryption KMS key to encrypt data. To specify a KMS
         * key in a different Amazon Web Services account, you must use the key ARN or alias ARN.
         * </p>
         * 
         * @param kmsKeyId
         *        An alias, alias ARN, key ID, or key ARN of a symmetric encryption KMS key to encrypt data. To specify
         *        a KMS key in a different Amazon Web Services account, you must use the key ARN or alias ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * Maximum duration that Step Functions will reuse data keys. When the period expires, Step Functions will call
         * <code>GenerateDataKey</code>. Only applies to customer managed keys.
         * </p>
         * 
         * @param kmsDataKeyReusePeriodSeconds
         *        Maximum duration that Step Functions will reuse data keys. When the period expires, Step Functions
         *        will call <code>GenerateDataKey</code>. Only applies to customer managed keys.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsDataKeyReusePeriodSeconds(Integer kmsDataKeyReusePeriodSeconds);

        /**
         * <p>
         * Encryption type
         * </p>
         * 
         * @param type
         *        Encryption type
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder type(String type);

        /**
         * <p>
         * Encryption type
         * </p>
         * 
         * @param type
         *        Encryption type
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder type(EncryptionType type);
    }

    static final class BuilderImpl implements Builder {
        private String kmsKeyId;

        private Integer kmsDataKeyReusePeriodSeconds;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(EncryptionConfiguration model) {
            kmsKeyId(model.kmsKeyId);
            kmsDataKeyReusePeriodSeconds(model.kmsDataKeyReusePeriodSeconds);
            type(model.type);
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final Integer getKmsDataKeyReusePeriodSeconds() {
            return kmsDataKeyReusePeriodSeconds;
        }

        public final void setKmsDataKeyReusePeriodSeconds(Integer kmsDataKeyReusePeriodSeconds) {
            this.kmsDataKeyReusePeriodSeconds = kmsDataKeyReusePeriodSeconds;
        }

        @Override
        public final Builder kmsDataKeyReusePeriodSeconds(Integer kmsDataKeyReusePeriodSeconds) {
            this.kmsDataKeyReusePeriodSeconds = kmsDataKeyReusePeriodSeconds;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(EncryptionType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public EncryptionConfiguration build() {
            return new EncryptionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
