/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.shield;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.shield.model.AccessDeniedException;
import software.amazon.awssdk.services.shield.model.AccessDeniedForDependencyException;
import software.amazon.awssdk.services.shield.model.AssociateDrtLogBucketRequest;
import software.amazon.awssdk.services.shield.model.AssociateDrtLogBucketResponse;
import software.amazon.awssdk.services.shield.model.AssociateDrtRoleRequest;
import software.amazon.awssdk.services.shield.model.AssociateDrtRoleResponse;
import software.amazon.awssdk.services.shield.model.AssociateHealthCheckRequest;
import software.amazon.awssdk.services.shield.model.AssociateHealthCheckResponse;
import software.amazon.awssdk.services.shield.model.AssociateProactiveEngagementDetailsRequest;
import software.amazon.awssdk.services.shield.model.AssociateProactiveEngagementDetailsResponse;
import software.amazon.awssdk.services.shield.model.CreateProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.CreateProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.CreateProtectionRequest;
import software.amazon.awssdk.services.shield.model.CreateProtectionResponse;
import software.amazon.awssdk.services.shield.model.CreateSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.CreateSubscriptionResponse;
import software.amazon.awssdk.services.shield.model.DeleteProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.DeleteProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.DeleteProtectionRequest;
import software.amazon.awssdk.services.shield.model.DeleteProtectionResponse;
import software.amazon.awssdk.services.shield.model.DescribeAttackRequest;
import software.amazon.awssdk.services.shield.model.DescribeAttackResponse;
import software.amazon.awssdk.services.shield.model.DescribeAttackStatisticsRequest;
import software.amazon.awssdk.services.shield.model.DescribeAttackStatisticsResponse;
import software.amazon.awssdk.services.shield.model.DescribeDrtAccessRequest;
import software.amazon.awssdk.services.shield.model.DescribeDrtAccessResponse;
import software.amazon.awssdk.services.shield.model.DescribeEmergencyContactSettingsRequest;
import software.amazon.awssdk.services.shield.model.DescribeEmergencyContactSettingsResponse;
import software.amazon.awssdk.services.shield.model.DescribeProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.DescribeProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.DescribeProtectionRequest;
import software.amazon.awssdk.services.shield.model.DescribeProtectionResponse;
import software.amazon.awssdk.services.shield.model.DescribeSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.DescribeSubscriptionResponse;
import software.amazon.awssdk.services.shield.model.DisableProactiveEngagementRequest;
import software.amazon.awssdk.services.shield.model.DisableProactiveEngagementResponse;
import software.amazon.awssdk.services.shield.model.DisassociateDrtLogBucketRequest;
import software.amazon.awssdk.services.shield.model.DisassociateDrtLogBucketResponse;
import software.amazon.awssdk.services.shield.model.DisassociateDrtRoleRequest;
import software.amazon.awssdk.services.shield.model.DisassociateDrtRoleResponse;
import software.amazon.awssdk.services.shield.model.DisassociateHealthCheckRequest;
import software.amazon.awssdk.services.shield.model.DisassociateHealthCheckResponse;
import software.amazon.awssdk.services.shield.model.EnableProactiveEngagementRequest;
import software.amazon.awssdk.services.shield.model.EnableProactiveEngagementResponse;
import software.amazon.awssdk.services.shield.model.GetSubscriptionStateRequest;
import software.amazon.awssdk.services.shield.model.GetSubscriptionStateResponse;
import software.amazon.awssdk.services.shield.model.InternalErrorException;
import software.amazon.awssdk.services.shield.model.InvalidOperationException;
import software.amazon.awssdk.services.shield.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.shield.model.InvalidParameterException;
import software.amazon.awssdk.services.shield.model.InvalidResourceException;
import software.amazon.awssdk.services.shield.model.LimitsExceededException;
import software.amazon.awssdk.services.shield.model.ListAttacksRequest;
import software.amazon.awssdk.services.shield.model.ListAttacksResponse;
import software.amazon.awssdk.services.shield.model.ListProtectionGroupsRequest;
import software.amazon.awssdk.services.shield.model.ListProtectionGroupsResponse;
import software.amazon.awssdk.services.shield.model.ListProtectionsRequest;
import software.amazon.awssdk.services.shield.model.ListProtectionsResponse;
import software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.shield.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.shield.model.LockedSubscriptionException;
import software.amazon.awssdk.services.shield.model.NoAssociatedRoleException;
import software.amazon.awssdk.services.shield.model.OptimisticLockException;
import software.amazon.awssdk.services.shield.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.shield.model.ResourceNotFoundException;
import software.amazon.awssdk.services.shield.model.ShieldException;
import software.amazon.awssdk.services.shield.model.ShieldRequest;
import software.amazon.awssdk.services.shield.model.TagResourceRequest;
import software.amazon.awssdk.services.shield.model.TagResourceResponse;
import software.amazon.awssdk.services.shield.model.UntagResourceRequest;
import software.amazon.awssdk.services.shield.model.UntagResourceResponse;
import software.amazon.awssdk.services.shield.model.UpdateEmergencyContactSettingsRequest;
import software.amazon.awssdk.services.shield.model.UpdateEmergencyContactSettingsResponse;
import software.amazon.awssdk.services.shield.model.UpdateProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.UpdateProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.UpdateSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.UpdateSubscriptionResponse;
import software.amazon.awssdk.services.shield.paginators.ListAttacksIterable;
import software.amazon.awssdk.services.shield.paginators.ListProtectionGroupsIterable;
import software.amazon.awssdk.services.shield.paginators.ListProtectionsIterable;
import software.amazon.awssdk.services.shield.paginators.ListResourcesInProtectionGroupIterable;
import software.amazon.awssdk.services.shield.transform.AssociateDrtLogBucketRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateDrtRoleRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateHealthCheckRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateProactiveEngagementDetailsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DeleteProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DeleteProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeAttackRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeAttackStatisticsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeDrtAccessRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeEmergencyContactSettingsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisableProactiveEngagementRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateDrtLogBucketRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateDrtRoleRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateHealthCheckRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.EnableProactiveEngagementRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.GetSubscriptionStateRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListAttacksRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListProtectionGroupsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListProtectionsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListResourcesInProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateEmergencyContactSettingsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateSubscriptionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ShieldClient}.
 *
 * @see ShieldClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultShieldClient implements ShieldClient {
    private static final Logger log = Logger.loggerFor(DefaultShieldClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultShieldClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Authorizes the DDoS Response Team (DRT) to access the specified Amazon S3 bucket containing your AWS WAF logs.
     * You can associate up to 10 Amazon S3 buckets with your subscription.
     * </p>
     * <p>
     * To use the services of the DRT and make an <code>AssociateDRTLogBucket</code> request, you must be subscribed to
     * the <a href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     *
     * @param associateDrtLogBucketRequest
     * @return Result of the AssociateDRTLogBucket operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws NoAssociatedRoleException
     *         The ARN of the role that you specifed does not exist.
     * @throws LimitsExceededException
     *         Exception that indicates that the operation would exceed a limit.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws AccessDeniedForDependencyException
     *         In order to grant the necessary access to the DDoS Response Team (DRT), the user submitting the request
     *         must have the <code>iam:PassRole</code> permission. This error indicates the user did not have the
     *         appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.AssociateDRTLogBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateDRTLogBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AssociateDrtLogBucketResponse associateDRTLogBucket(AssociateDrtLogBucketRequest associateDrtLogBucketRequest)
            throws InternalErrorException, InvalidOperationException, NoAssociatedRoleException, LimitsExceededException,
            InvalidParameterException, AccessDeniedForDependencyException, OptimisticLockException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateDrtLogBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateDrtLogBucketResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateDrtLogBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDRTLogBucket");

            return clientHandler.execute(new ClientExecutionParams<AssociateDrtLogBucketRequest, AssociateDrtLogBucketResponse>()
                    .withOperationName("AssociateDRTLogBucket").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateDrtLogBucketRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateDrtLogBucketRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Authorizes the DDoS Response Team (DRT), using the specified role, to access your AWS account to assist with DDoS
     * attack mitigation during potential attacks. This enables the DRT to inspect your AWS WAF configuration and create
     * or update AWS WAF rules and web ACLs.
     * </p>
     * <p>
     * You can associate only one <code>RoleArn</code> with your subscription. If you submit an
     * <code>AssociateDRTRole</code> request for an account that already has an associated role, the new
     * <code>RoleArn</code> will replace the existing <code>RoleArn</code>.
     * </p>
     * <p>
     * Prior to making the <code>AssociateDRTRole</code> request, you must attach the <a href=
     * "https://console.aws.amazon.com/iam/home?#/policies/arn:aws:iam::aws:policy/service-role/AWSShieldDRTAccessPolicy"
     * >AWSShieldDRTAccessPolicy</a> managed policy to the role you will specify in the request. For more information
     * see <a
     * href=" https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_manage-attach-detach.html">Attaching and
     * Detaching IAM Policies</a>. The role must also trust the service principal <code> drt.shield.amazonaws.com</code>
     * . For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html">IAM JSON
     * Policy Elements: Principal</a>.
     * </p>
     * <p>
     * The DRT will have access only to your AWS WAF and Shield resources. By submitting this request, you authorize the
     * DRT to inspect your AWS WAF and Shield configuration and create and update AWS WAF rules and web ACLs on your
     * behalf. The DRT takes these actions only if explicitly authorized by you.
     * </p>
     * <p>
     * You must have the <code>iam:PassRole</code> permission to make an <code>AssociateDRTRole</code> request. For more
     * information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a
     * User Permissions to Pass a Role to an AWS Service</a>.
     * </p>
     * <p>
     * To use the services of the DRT and make an <code>AssociateDRTRole</code> request, you must be subscribed to the
     * <a href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     *
     * @param associateDrtRoleRequest
     * @return Result of the AssociateDRTRole operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws AccessDeniedForDependencyException
     *         In order to grant the necessary access to the DDoS Response Team (DRT), the user submitting the request
     *         must have the <code>iam:PassRole</code> permission. This error indicates the user did not have the
     *         appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.AssociateDRTRole
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateDRTRole" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AssociateDrtRoleResponse associateDRTRole(AssociateDrtRoleRequest associateDrtRoleRequest)
            throws InternalErrorException, InvalidOperationException, InvalidParameterException,
            AccessDeniedForDependencyException, OptimisticLockException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateDrtRoleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateDrtRoleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateDrtRoleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDRTRole");

            return clientHandler.execute(new ClientExecutionParams<AssociateDrtRoleRequest, AssociateDrtRoleResponse>()
                    .withOperationName("AssociateDRTRole").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateDrtRoleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateDrtRoleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds health-based detection to the Shield Advanced protection for a resource. Shield Advanced health-based
     * detection uses the health of your AWS resource to improve responsiveness and accuracy in attack detection and
     * mitigation.
     * </p>
     * <p>
     * You define the health check in Route 53 and then associate it with your Shield Advanced protection. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/ddos-overview.html#ddos-advanced-health-check-option"
     * >Shield Advanced Health-Based Detection</a> in the <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/">AWS WAF and AWS Shield Developer Guide</a>.
     * </p>
     *
     * @param associateHealthCheckRequest
     * @return Result of the AssociateHealthCheck operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws LimitsExceededException
     *         Exception that indicates that the operation would exceed a limit.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.AssociateHealthCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateHealthCheck" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AssociateHealthCheckResponse associateHealthCheck(AssociateHealthCheckRequest associateHealthCheckRequest)
            throws InternalErrorException, LimitsExceededException, ResourceNotFoundException, InvalidParameterException,
            OptimisticLockException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateHealthCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateHealthCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateHealthCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateHealthCheck");

            return clientHandler.execute(new ClientExecutionParams<AssociateHealthCheckRequest, AssociateHealthCheckResponse>()
                    .withOperationName("AssociateHealthCheck").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateHealthCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateHealthCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initializes proactive engagement and sets the list of contacts for the DDoS Response Team (DRT) to use. You must
     * provide at least one phone number in the emergency contact list.
     * </p>
     * <p>
     * After you have initialized proactive engagement using this call, to disable or enable proactive engagement, use
     * the calls <code>DisableProactiveEngagement</code> and <code>EnableProactiveEngagement</code>.
     * </p>
     * <note>
     * <p>
     * This call defines the list of email addresses and phone numbers that the DDoS Response Team (DRT) can use to
     * contact you for escalations to the DRT and to initiate proactive customer support.
     * </p>
     * <p>
     * The contacts that you provide in the request replace any contacts that were already defined. If you already have
     * contacts defined and want to use them, retrieve the list using <code>DescribeEmergencyContactSettings</code> and
     * then provide it to this call.
     * </p>
     * </note>
     *
     * @param associateProactiveEngagementDetailsRequest
     * @return Result of the AssociateProactiveEngagementDetails operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.AssociateProactiveEngagementDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateProactiveEngagementDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateProactiveEngagementDetailsResponse associateProactiveEngagementDetails(
            AssociateProactiveEngagementDetailsRequest associateProactiveEngagementDetailsRequest) throws InternalErrorException,
            InvalidOperationException, InvalidParameterException, ResourceNotFoundException, OptimisticLockException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateProactiveEngagementDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateProactiveEngagementDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateProactiveEngagementDetailsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateProactiveEngagementDetails");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateProactiveEngagementDetailsRequest, AssociateProactiveEngagementDetailsResponse>()
                            .withOperationName("AssociateProactiveEngagementDetails").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateProactiveEngagementDetailsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateProactiveEngagementDetailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables AWS Shield Advanced for a specific AWS resource. The resource can be an Amazon CloudFront distribution,
     * Elastic Load Balancing load balancer, AWS Global Accelerator accelerator, Elastic IP Address, or an Amazon Route
     * 53 hosted zone.
     * </p>
     * <p>
     * You can add protection to only a single resource with each CreateProtection request. If you want to add
     * protection to multiple resources at once, use the <a href="https://console.aws.amazon.com/waf/">AWS WAF
     * console</a>. For more information see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/getting-started-ddos.html">Getting Started with AWS
     * Shield Advanced</a> and <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/configure-new-protection.html">Add AWS Shield
     * Advanced Protection to more AWS Resources</a>.
     * </p>
     *
     * @param createProtectionRequest
     * @return Result of the CreateProtection operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidResourceException
     *         Exception that indicates that the resource is invalid. You might not have access to the resource, or the
     *         resource might not exist.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws LimitsExceededException
     *         Exception that indicates that the operation would exceed a limit.
     * @throws ResourceAlreadyExistsException
     *         Exception indicating the specified resource already exists. If available, this exception includes details
     *         in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.CreateProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateProtectionResponse createProtection(CreateProtectionRequest createProtectionRequest)
            throws InternalErrorException, InvalidResourceException, InvalidOperationException, LimitsExceededException,
            ResourceAlreadyExistsException, OptimisticLockException, ResourceNotFoundException, InvalidParameterException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProtectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProtection");

            return clientHandler.execute(new ClientExecutionParams<CreateProtectionRequest, CreateProtectionResponse>()
                    .withOperationName("CreateProtection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createProtectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProtectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a grouping of protected resources so they can be handled as a collective. This resource grouping improves
     * the accuracy of detection and reduces false positives.
     * </p>
     *
     * @param createProtectionGroupRequest
     * @return Result of the CreateProtectionGroup operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceAlreadyExistsException
     *         Exception indicating the specified resource already exists. If available, this exception includes details
     *         in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws LimitsExceededException
     *         Exception that indicates that the operation would exceed a limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.CreateProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateProtectionGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateProtectionGroupResponse createProtectionGroup(CreateProtectionGroupRequest createProtectionGroupRequest)
            throws InternalErrorException, ResourceAlreadyExistsException, OptimisticLockException, ResourceNotFoundException,
            InvalidParameterException, LimitsExceededException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateProtectionGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProtectionGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProtectionGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateProtectionGroupRequest, CreateProtectionGroupResponse>()
                    .withOperationName("CreateProtectionGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createProtectionGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProtectionGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Activates AWS Shield Advanced for an account.
     * </p>
     * <p>
     * When you initally create a subscription, your subscription is set to be automatically renewed at the end of the
     * existing subscription period. You can change this by submitting an <code>UpdateSubscription</code> request.
     * </p>
     *
     * @param createSubscriptionRequest
     * @return Result of the CreateSubscription operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceAlreadyExistsException
     *         Exception indicating the specified resource already exists. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.CreateSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSubscriptionResponse createSubscription(CreateSubscriptionRequest createSubscriptionRequest)
            throws InternalErrorException, ResourceAlreadyExistsException, AwsServiceException, SdkClientException,
            ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscription");

            return clientHandler.execute(new ClientExecutionParams<CreateSubscriptionRequest, CreateSubscriptionResponse>()
                    .withOperationName("CreateSubscription").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSubscriptionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an AWS Shield Advanced <a>Protection</a>.
     * </p>
     *
     * @param deleteProtectionRequest
     * @return Result of the DeleteProtection operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DeleteProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DeleteProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProtectionResponse deleteProtection(DeleteProtectionRequest deleteProtectionRequest)
            throws InternalErrorException, ResourceNotFoundException, OptimisticLockException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProtectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProtection");

            return clientHandler.execute(new ClientExecutionParams<DeleteProtectionRequest, DeleteProtectionResponse>()
                    .withOperationName("DeleteProtection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteProtectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProtectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified protection group.
     * </p>
     *
     * @param deleteProtectionGroupRequest
     * @return Result of the DeleteProtectionGroup operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DeleteProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DeleteProtectionGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteProtectionGroupResponse deleteProtectionGroup(DeleteProtectionGroupRequest deleteProtectionGroupRequest)
            throws InternalErrorException, OptimisticLockException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteProtectionGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProtectionGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProtectionGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteProtectionGroupRequest, DeleteProtectionGroupResponse>()
                    .withOperationName("DeleteProtectionGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteProtectionGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProtectionGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the details of a DDoS attack.
     * </p>
     *
     * @param describeAttackRequest
     * @return Result of the DescribeAttack operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws AccessDeniedException
     *         Exception that indicates the specified <code>AttackId</code> does not exist, or the requester does not
     *         have the appropriate permissions to access the <code>AttackId</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeAttack
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeAttack" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAttackResponse describeAttack(DescribeAttackRequest describeAttackRequest) throws InternalErrorException,
            AccessDeniedException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAttackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAttackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAttackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAttack");

            return clientHandler.execute(new ClientExecutionParams<DescribeAttackRequest, DescribeAttackResponse>()
                    .withOperationName("DescribeAttack").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAttackRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAttackRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the number and type of attacks AWS Shield has detected in the last year for all
     * resources that belong to your account, regardless of whether you've defined Shield protections for them. This
     * operation is available to Shield customers as well as to Shield Advanced customers.
     * </p>
     * <p>
     * The operation returns data for the time range of midnight UTC, one year ago, to midnight UTC, today. For example,
     * if the current time is <code>2020-10-26 15:39:32 PDT</code>, equal to <code>2020-10-26 22:39:32 UTC</code>, then
     * the time range for the attack data returned is from <code>2019-10-26 00:00:00 UTC</code> to
     * <code>2020-10-26 00:00:00 UTC</code>.
     * </p>
     * <p>
     * The time range indicates the period covered by the attack statistics data items.
     * </p>
     *
     * @param describeAttackStatisticsRequest
     * @return Result of the DescribeAttackStatistics operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeAttackStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeAttackStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAttackStatisticsResponse describeAttackStatistics(
            DescribeAttackStatisticsRequest describeAttackStatisticsRequest) throws InternalErrorException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAttackStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAttackStatisticsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAttackStatisticsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAttackStatistics");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAttackStatisticsRequest, DescribeAttackStatisticsResponse>()
                            .withOperationName("DescribeAttackStatistics").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAttackStatisticsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAttackStatisticsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current role and list of Amazon S3 log buckets used by the DDoS Response Team (DRT) to access your
     * AWS account while assisting with attack mitigation.
     * </p>
     *
     * @param describeDrtAccessRequest
     * @return Result of the DescribeDRTAccess operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeDRTAccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeDRTAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDrtAccessResponse describeDRTAccess(DescribeDrtAccessRequest describeDrtAccessRequest)
            throws InternalErrorException, ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDrtAccessResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDrtAccessResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDrtAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDRTAccess");

            return clientHandler.execute(new ClientExecutionParams<DescribeDrtAccessRequest, DescribeDrtAccessResponse>()
                    .withOperationName("DescribeDRTAccess").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDrtAccessRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDrtAccessRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A list of email addresses and phone numbers that the DDoS Response Team (DRT) can use to contact you if you have
     * proactive engagement enabled, for escalations to the DRT and to initiate proactive customer support.
     * </p>
     *
     * @param describeEmergencyContactSettingsRequest
     * @return Result of the DescribeEmergencyContactSettings operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeEmergencyContactSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeEmergencyContactSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEmergencyContactSettingsResponse describeEmergencyContactSettings(
            DescribeEmergencyContactSettingsRequest describeEmergencyContactSettingsRequest) throws InternalErrorException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEmergencyContactSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEmergencyContactSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeEmergencyContactSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEmergencyContactSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEmergencyContactSettingsRequest, DescribeEmergencyContactSettingsResponse>()
                            .withOperationName("DescribeEmergencyContactSettings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeEmergencyContactSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEmergencyContactSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the details of a <a>Protection</a> object.
     * </p>
     *
     * @param describeProtectionRequest
     * @return Result of the DescribeProtection operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeProtectionResponse describeProtection(DescribeProtectionRequest describeProtectionRequest)
            throws InternalErrorException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProtection");

            return clientHandler.execute(new ClientExecutionParams<DescribeProtectionRequest, DescribeProtectionResponse>()
                    .withOperationName("DescribeProtection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeProtectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeProtectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the specification for the specified protection group.
     * </p>
     *
     * @param describeProtectionGroupRequest
     * @return Result of the DescribeProtectionGroup operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeProtectionGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeProtectionGroupResponse describeProtectionGroup(DescribeProtectionGroupRequest describeProtectionGroupRequest)
            throws InternalErrorException, ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeProtectionGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProtectionGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProtectionGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeProtectionGroupRequest, DescribeProtectionGroupResponse>()
                            .withOperationName("DescribeProtectionGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeProtectionGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeProtectionGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides details about the AWS Shield Advanced subscription for an account.
     * </p>
     *
     * @param describeSubscriptionRequest
     * @return Result of the DescribeSubscription operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeSubscription" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeSubscriptionResponse describeSubscription(DescribeSubscriptionRequest describeSubscriptionRequest)
            throws InternalErrorException, ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSubscription");

            return clientHandler.execute(new ClientExecutionParams<DescribeSubscriptionRequest, DescribeSubscriptionResponse>()
                    .withOperationName("DescribeSubscription").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeSubscriptionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes authorization from the DDoS Response Team (DRT) to notify contacts about escalations to the DRT and to
     * initiate proactive customer support.
     * </p>
     *
     * @param disableProactiveEngagementRequest
     * @return Result of the DisableProactiveEngagement operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DisableProactiveEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisableProactiveEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableProactiveEngagementResponse disableProactiveEngagement(
            DisableProactiveEngagementRequest disableProactiveEngagementRequest) throws InternalErrorException,
            InvalidOperationException, InvalidParameterException, ResourceNotFoundException, OptimisticLockException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableProactiveEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableProactiveEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableProactiveEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableProactiveEngagement");

            return clientHandler
                    .execute(new ClientExecutionParams<DisableProactiveEngagementRequest, DisableProactiveEngagementResponse>()
                            .withOperationName("DisableProactiveEngagement").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disableProactiveEngagementRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisableProactiveEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the DDoS Response Team's (DRT) access to the specified Amazon S3 bucket containing your AWS WAF logs.
     * </p>
     * <p>
     * To make a <code>DisassociateDRTLogBucket</code> request, you must be subscribed to the <a
     * href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>. However, if you are
     * not subscribed to one of these support plans, but had been previously and had granted the DRT access to your
     * account, you can submit a <code>DisassociateDRTLogBucket</code> request to remove this access.
     * </p>
     *
     * @param disassociateDrtLogBucketRequest
     * @return Result of the DisassociateDRTLogBucket operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws NoAssociatedRoleException
     *         The ARN of the role that you specifed does not exist.
     * @throws AccessDeniedForDependencyException
     *         In order to grant the necessary access to the DDoS Response Team (DRT), the user submitting the request
     *         must have the <code>iam:PassRole</code> permission. This error indicates the user did not have the
     *         appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DisassociateDRTLogBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateDRTLogBucket"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateDrtLogBucketResponse disassociateDRTLogBucket(
            DisassociateDrtLogBucketRequest disassociateDrtLogBucketRequest) throws InternalErrorException,
            InvalidOperationException, NoAssociatedRoleException, AccessDeniedForDependencyException, OptimisticLockException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateDrtLogBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateDrtLogBucketResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateDrtLogBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDRTLogBucket");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateDrtLogBucketRequest, DisassociateDrtLogBucketResponse>()
                            .withOperationName("DisassociateDRTLogBucket").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateDrtLogBucketRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateDrtLogBucketRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the DDoS Response Team's (DRT) access to your AWS account.
     * </p>
     * <p>
     * To make a <code>DisassociateDRTRole</code> request, you must be subscribed to the <a
     * href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>. However, if you are
     * not subscribed to one of these support plans, but had been previously and had granted the DRT access to your
     * account, you can submit a <code>DisassociateDRTRole</code> request to remove this access.
     * </p>
     *
     * @param disassociateDrtRoleRequest
     * @return Result of the DisassociateDRTRole operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DisassociateDRTRole
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateDRTRole" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisassociateDrtRoleResponse disassociateDRTRole(DisassociateDrtRoleRequest disassociateDrtRoleRequest)
            throws InternalErrorException, InvalidOperationException, OptimisticLockException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateDrtRoleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateDrtRoleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateDrtRoleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDRTRole");

            return clientHandler.execute(new ClientExecutionParams<DisassociateDrtRoleRequest, DisassociateDrtRoleResponse>()
                    .withOperationName("DisassociateDRTRole").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disassociateDrtRoleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateDrtRoleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes health-based detection from the Shield Advanced protection for a resource. Shield Advanced health-based
     * detection uses the health of your AWS resource to improve responsiveness and accuracy in attack detection and
     * mitigation.
     * </p>
     * <p>
     * You define the health check in Route 53 and then associate or disassociate it with your Shield Advanced
     * protection. For more information, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/ddos-overview.html#ddos-advanced-health-check-option"
     * >Shield Advanced Health-Based Detection</a> in the <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/">AWS WAF and AWS Shield Developer Guide</a>.
     * </p>
     *
     * @param disassociateHealthCheckRequest
     * @return Result of the DisassociateHealthCheck operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DisassociateHealthCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateHealthCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateHealthCheckResponse disassociateHealthCheck(DisassociateHealthCheckRequest disassociateHealthCheckRequest)
            throws InternalErrorException, InvalidParameterException, ResourceNotFoundException, OptimisticLockException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateHealthCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateHealthCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateHealthCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateHealthCheck");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateHealthCheckRequest, DisassociateHealthCheckResponse>()
                            .withOperationName("DisassociateHealthCheck").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateHealthCheckRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateHealthCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Authorizes the DDoS Response Team (DRT) to use email and phone to notify contacts about escalations to the DRT
     * and to initiate proactive customer support.
     * </p>
     *
     * @param enableProactiveEngagementRequest
     * @return Result of the EnableProactiveEngagement operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.EnableProactiveEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/EnableProactiveEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableProactiveEngagementResponse enableProactiveEngagement(
            EnableProactiveEngagementRequest enableProactiveEngagementRequest) throws InternalErrorException,
            InvalidOperationException, InvalidParameterException, ResourceNotFoundException, OptimisticLockException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableProactiveEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableProactiveEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableProactiveEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableProactiveEngagement");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableProactiveEngagementRequest, EnableProactiveEngagementResponse>()
                            .withOperationName("EnableProactiveEngagement").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(enableProactiveEngagementRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableProactiveEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the <code>SubscriptionState</code>, either <code>Active</code> or <code>Inactive</code>.
     * </p>
     *
     * @param getSubscriptionStateRequest
     * @return Result of the GetSubscriptionState operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.GetSubscriptionState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/GetSubscriptionState" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSubscriptionStateResponse getSubscriptionState(GetSubscriptionStateRequest getSubscriptionStateRequest)
            throws InternalErrorException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSubscriptionStateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSubscriptionStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSubscriptionStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSubscriptionState");

            return clientHandler.execute(new ClientExecutionParams<GetSubscriptionStateRequest, GetSubscriptionStateResponse>()
                    .withOperationName("GetSubscriptionState").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSubscriptionStateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSubscriptionStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all ongoing DDoS attacks or all DDoS attacks during a specified time period.
     * </p>
     *
     * @param listAttacksRequest
     * @return Result of the ListAttacks operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListAttacks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListAttacks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAttacksResponse listAttacks(ListAttacksRequest listAttacksRequest) throws InternalErrorException,
            InvalidParameterException, InvalidOperationException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAttacksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAttacksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAttacksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAttacks");

            return clientHandler.execute(new ClientExecutionParams<ListAttacksRequest, ListAttacksResponse>()
                    .withOperationName("ListAttacks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAttacksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAttacksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all ongoing DDoS attacks or all DDoS attacks during a specified time period.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAttacks(software.amazon.awssdk.services.shield.model.ListAttacksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListAttacksIterable responses = client.listAttacksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.shield.paginators.ListAttacksIterable responses = client.listAttacksPaginator(request);
     *     for (software.amazon.awssdk.services.shield.model.ListAttacksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListAttacksIterable responses = client.listAttacksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAttacks(software.amazon.awssdk.services.shield.model.ListAttacksRequest)} operation.</b>
     * </p>
     *
     * @param listAttacksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListAttacks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListAttacks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAttacksIterable listAttacksPaginator(ListAttacksRequest listAttacksRequest) throws InternalErrorException,
            InvalidParameterException, InvalidOperationException, AwsServiceException, SdkClientException, ShieldException {
        return new ListAttacksIterable(this, applyPaginatorUserAgent(listAttacksRequest));
    }

    /**
     * <p>
     * Retrieves the <a>ProtectionGroup</a> objects for the account.
     * </p>
     *
     * @param listProtectionGroupsRequest
     * @return Result of the ListProtectionGroups operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidPaginationTokenException
     *         Exception that indicates that the NextToken specified in the request is invalid. Submit the request using
     *         the NextToken value that was returned in the response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListProtectionGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtectionGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListProtectionGroupsResponse listProtectionGroups(ListProtectionGroupsRequest listProtectionGroupsRequest)
            throws InternalErrorException, ResourceNotFoundException, InvalidPaginationTokenException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProtectionGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProtectionGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtectionGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtectionGroups");

            return clientHandler.execute(new ClientExecutionParams<ListProtectionGroupsRequest, ListProtectionGroupsResponse>()
                    .withOperationName("ListProtectionGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listProtectionGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProtectionGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the <a>ProtectionGroup</a> objects for the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProtectionGroups(software.amazon.awssdk.services.shield.model.ListProtectionGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListProtectionGroupsIterable responses = client.listProtectionGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.shield.paginators.ListProtectionGroupsIterable responses = client
     *             .listProtectionGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.shield.model.ListProtectionGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListProtectionGroupsIterable responses = client.listProtectionGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProtectionGroups(software.amazon.awssdk.services.shield.model.ListProtectionGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProtectionGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidPaginationTokenException
     *         Exception that indicates that the NextToken specified in the request is invalid. Submit the request using
     *         the NextToken value that was returned in the response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListProtectionGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtectionGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListProtectionGroupsIterable listProtectionGroupsPaginator(ListProtectionGroupsRequest listProtectionGroupsRequest)
            throws InternalErrorException, ResourceNotFoundException, InvalidPaginationTokenException, AwsServiceException,
            SdkClientException, ShieldException {
        return new ListProtectionGroupsIterable(this, applyPaginatorUserAgent(listProtectionGroupsRequest));
    }

    /**
     * <p>
     * Lists all <a>Protection</a> objects for the account.
     * </p>
     *
     * @param listProtectionsRequest
     * @return Result of the ListProtections operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidPaginationTokenException
     *         Exception that indicates that the NextToken specified in the request is invalid. Submit the request using
     *         the NextToken value that was returned in the response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListProtections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProtectionsResponse listProtections(ListProtectionsRequest listProtectionsRequest) throws InternalErrorException,
            ResourceNotFoundException, InvalidPaginationTokenException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProtectionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProtectionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtections");

            return clientHandler.execute(new ClientExecutionParams<ListProtectionsRequest, ListProtectionsResponse>()
                    .withOperationName("ListProtections").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listProtectionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProtectionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all <a>Protection</a> objects for the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProtections(software.amazon.awssdk.services.shield.model.ListProtectionsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListProtectionsIterable responses = client.listProtectionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.shield.paginators.ListProtectionsIterable responses = client
     *             .listProtectionsPaginator(request);
     *     for (software.amazon.awssdk.services.shield.model.ListProtectionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListProtectionsIterable responses = client.listProtectionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProtections(software.amazon.awssdk.services.shield.model.ListProtectionsRequest)} operation.</b>
     * </p>
     *
     * @param listProtectionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidPaginationTokenException
     *         Exception that indicates that the NextToken specified in the request is invalid. Submit the request using
     *         the NextToken value that was returned in the response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListProtections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProtectionsIterable listProtectionsPaginator(ListProtectionsRequest listProtectionsRequest)
            throws InternalErrorException, ResourceNotFoundException, InvalidPaginationTokenException, AwsServiceException,
            SdkClientException, ShieldException {
        return new ListProtectionsIterable(this, applyPaginatorUserAgent(listProtectionsRequest));
    }

    /**
     * <p>
     * Retrieves the resources that are included in the protection group.
     * </p>
     *
     * @param listResourcesInProtectionGroupRequest
     * @return Result of the ListResourcesInProtectionGroup operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidPaginationTokenException
     *         Exception that indicates that the NextToken specified in the request is invalid. Submit the request using
     *         the NextToken value that was returned in the response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListResourcesInProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListResourcesInProtectionGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourcesInProtectionGroupResponse listResourcesInProtectionGroup(
            ListResourcesInProtectionGroupRequest listResourcesInProtectionGroupRequest) throws InternalErrorException,
            ResourceNotFoundException, InvalidPaginationTokenException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourcesInProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResourcesInProtectionGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listResourcesInProtectionGroupRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourcesInProtectionGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResourcesInProtectionGroupRequest, ListResourcesInProtectionGroupResponse>()
                            .withOperationName("ListResourcesInProtectionGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listResourcesInProtectionGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResourcesInProtectionGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resources that are included in the protection group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResourcesInProtectionGroup(software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListResourcesInProtectionGroupIterable responses = client.listResourcesInProtectionGroupPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.shield.paginators.ListResourcesInProtectionGroupIterable responses = client
     *             .listResourcesInProtectionGroupPaginator(request);
     *     for (software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListResourcesInProtectionGroupIterable responses = client.listResourcesInProtectionGroupPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResourcesInProtectionGroup(software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResourcesInProtectionGroupRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidPaginationTokenException
     *         Exception that indicates that the NextToken specified in the request is invalid. Submit the request using
     *         the NextToken value that was returned in the response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListResourcesInProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListResourcesInProtectionGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourcesInProtectionGroupIterable listResourcesInProtectionGroupPaginator(
            ListResourcesInProtectionGroupRequest listResourcesInProtectionGroupRequest) throws InternalErrorException,
            ResourceNotFoundException, InvalidPaginationTokenException, AwsServiceException, SdkClientException, ShieldException {
        return new ListResourcesInProtectionGroupIterable(this, applyPaginatorUserAgent(listResourcesInProtectionGroupRequest));
    }

    /**
     * <p>
     * Gets information about AWS tags for a specified Amazon Resource Name (ARN) in AWS Shield.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidResourceException
     *         Exception that indicates that the resource is invalid. You might not have access to the resource, or the
     *         resource might not exist.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalErrorException, InvalidResourceException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates tags for a resource in AWS Shield.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidResourceException
     *         Exception that indicates that the resource is invalid. You might not have access to the resource, or the
     *         resource might not exist.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalErrorException,
            InvalidResourceException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a resource in AWS Shield.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidResourceException
     *         Exception that indicates that the resource is invalid. You might not have access to the resource, or the
     *         resource might not exist.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalErrorException,
            InvalidResourceException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the details of the list of email addresses and phone numbers that the DDoS Response Team (DRT) can use to
     * contact you if you have proactive engagement enabled, for escalations to the DRT and to initiate proactive
     * customer support.
     * </p>
     *
     * @param updateEmergencyContactSettingsRequest
     * @return Result of the UpdateEmergencyContactSettings operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.UpdateEmergencyContactSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateEmergencyContactSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEmergencyContactSettingsResponse updateEmergencyContactSettings(
            UpdateEmergencyContactSettingsRequest updateEmergencyContactSettingsRequest) throws InternalErrorException,
            InvalidParameterException, OptimisticLockException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEmergencyContactSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEmergencyContactSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateEmergencyContactSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEmergencyContactSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateEmergencyContactSettingsRequest, UpdateEmergencyContactSettingsResponse>()
                            .withOperationName("UpdateEmergencyContactSettings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateEmergencyContactSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateEmergencyContactSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing protection group. A protection group is a grouping of protected resources so they can be
     * handled as a collective. This resource grouping improves the accuracy of detection and reduces false positives.
     * </p>
     *
     * @param updateProtectionGroupRequest
     * @return Result of the UpdateProtectionGroup operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.UpdateProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateProtectionGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateProtectionGroupResponse updateProtectionGroup(UpdateProtectionGroupRequest updateProtectionGroupRequest)
            throws InternalErrorException, ResourceNotFoundException, OptimisticLockException, InvalidParameterException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateProtectionGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProtectionGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProtectionGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateProtectionGroupRequest, UpdateProtectionGroupResponse>()
                    .withOperationName("UpdateProtectionGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateProtectionGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateProtectionGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the details of an existing subscription. Only enter values for parameters you want to change. Empty
     * parameters are not updated.
     * </p>
     *
     * @param updateSubscriptionRequest
     * @return Result of the UpdateSubscription operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws LockedSubscriptionException
     *         You are trying to update a subscription that has not yet completed the 1-year commitment. You can change
     *         the <code>AutoRenew</code> parameter during the last 30 days of your subscription. This exception
     *         indicates that you are attempting to change <code>AutoRenew</code> prior to that period.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist. If available, this exception includes details
     *         in additional properties.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid. If available, this exception
     *         includes details in additional properties.
     * @throws OptimisticLockException
     *         Exception that indicates that the resource state has been modified by another client. Retrieve the
     *         resource and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.UpdateSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSubscriptionResponse updateSubscription(UpdateSubscriptionRequest updateSubscriptionRequest)
            throws InternalErrorException, LockedSubscriptionException, ResourceNotFoundException, InvalidParameterException,
            OptimisticLockException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscription");

            return clientHandler.execute(new ClientExecutionParams<UpdateSubscriptionRequest, UpdateSubscriptionResponse>()
                    .withOperationName("UpdateSubscription").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSubscriptionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ShieldException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptimisticLockException")
                                .exceptionBuilderSupplier(OptimisticLockException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoAssociatedRoleException")
                                .exceptionBuilderSupplier(NoAssociatedRoleException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedForDependencyException")
                                .exceptionBuilderSupplier(AccessDeniedForDependencyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LockedSubscriptionException")
                                .exceptionBuilderSupplier(LockedSubscriptionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceException")
                                .exceptionBuilderSupplier(InvalidResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitsExceededException")
                                .exceptionBuilderSupplier(LimitsExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ShieldRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
