/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.shield.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents a resource that is under DDoS protection.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Protection implements SdkPojo, Serializable, ToCopyableBuilder<Protection.Builder, Protection> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Protection::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Protection::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(Protection::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<List<String>> HEALTH_CHECK_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("HealthCheckIds")
            .getter(getter(Protection::healthCheckIds))
            .setter(setter(Builder::healthCheckIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheckIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PROTECTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProtectionArn").getter(getter(Protection::protectionArn)).setter(setter(Builder::protectionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProtectionArn").build()).build();

    private static final SdkField<ApplicationLayerAutomaticResponseConfiguration> APPLICATION_LAYER_AUTOMATIC_RESPONSE_CONFIGURATION_FIELD = SdkField
            .<ApplicationLayerAutomaticResponseConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("ApplicationLayerAutomaticResponseConfiguration")
            .getter(getter(Protection::applicationLayerAutomaticResponseConfiguration))
            .setter(setter(Builder::applicationLayerAutomaticResponseConfiguration))
            .constructor(ApplicationLayerAutomaticResponseConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("ApplicationLayerAutomaticResponseConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            RESOURCE_ARN_FIELD, HEALTH_CHECK_IDS_FIELD, PROTECTION_ARN_FIELD,
            APPLICATION_LAYER_AUTOMATIC_RESPONSE_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String resourceArn;

    private final List<String> healthCheckIds;

    private final String protectionArn;

    private final ApplicationLayerAutomaticResponseConfiguration applicationLayerAutomaticResponseConfiguration;

    private Protection(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.resourceArn = builder.resourceArn;
        this.healthCheckIds = builder.healthCheckIds;
        this.protectionArn = builder.protectionArn;
        this.applicationLayerAutomaticResponseConfiguration = builder.applicationLayerAutomaticResponseConfiguration;
    }

    /**
     * <p>
     * The unique identifier (ID) of the protection.
     * </p>
     * 
     * @return The unique identifier (ID) of the protection.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the protection. For example, <code>My CloudFront distributions</code>.
     * </p>
     * 
     * @return The name of the protection. For example, <code>My CloudFront distributions</code>.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) of the Amazon Web Services resource that is protected.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) of the Amazon Web Services resource that is protected.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the HealthCheckIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasHealthCheckIds() {
        return healthCheckIds != null && !(healthCheckIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The unique identifier (ID) for the Route 53 health check that's associated with the protection.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHealthCheckIds} method.
     * </p>
     * 
     * @return The unique identifier (ID) for the Route 53 health check that's associated with the protection.
     */
    public final List<String> healthCheckIds() {
        return healthCheckIds;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) of the protection.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) of the protection.
     */
    public final String protectionArn() {
        return protectionArn;
    }

    /**
     * <p>
     * The automatic application layer DDoS mitigation settings for the protection. This configuration determines
     * whether Shield Advanced automatically manages rules in the web ACL in order to respond to application layer
     * events that Shield Advanced determines to be DDoS attacks.
     * </p>
     * 
     * @return The automatic application layer DDoS mitigation settings for the protection. This configuration
     *         determines whether Shield Advanced automatically manages rules in the web ACL in order to respond to
     *         application layer events that Shield Advanced determines to be DDoS attacks.
     */
    public final ApplicationLayerAutomaticResponseConfiguration applicationLayerAutomaticResponseConfiguration() {
        return applicationLayerAutomaticResponseConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasHealthCheckIds() ? healthCheckIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(protectionArn());
        hashCode = 31 * hashCode + Objects.hashCode(applicationLayerAutomaticResponseConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Protection)) {
            return false;
        }
        Protection other = (Protection) obj;
        return Objects.equals(id(), other.id())
                && Objects.equals(name(), other.name())
                && Objects.equals(resourceArn(), other.resourceArn())
                && hasHealthCheckIds() == other.hasHealthCheckIds()
                && Objects.equals(healthCheckIds(), other.healthCheckIds())
                && Objects.equals(protectionArn(), other.protectionArn())
                && Objects.equals(applicationLayerAutomaticResponseConfiguration(),
                        other.applicationLayerAutomaticResponseConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Protection").add("Id", id()).add("Name", name()).add("ResourceArn", resourceArn())
                .add("HealthCheckIds", hasHealthCheckIds() ? healthCheckIds() : null).add("ProtectionArn", protectionArn())
                .add("ApplicationLayerAutomaticResponseConfiguration", applicationLayerAutomaticResponseConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "HealthCheckIds":
            return Optional.ofNullable(clazz.cast(healthCheckIds()));
        case "ProtectionArn":
            return Optional.ofNullable(clazz.cast(protectionArn()));
        case "ApplicationLayerAutomaticResponseConfiguration":
            return Optional.ofNullable(clazz.cast(applicationLayerAutomaticResponseConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Protection, T> g) {
        return obj -> g.apply((Protection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Protection> {
        /**
         * <p>
         * The unique identifier (ID) of the protection.
         * </p>
         * 
         * @param id
         *        The unique identifier (ID) of the protection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the protection. For example, <code>My CloudFront distributions</code>.
         * </p>
         * 
         * @param name
         *        The name of the protection. For example, <code>My CloudFront distributions</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN (Amazon Resource Name) of the Amazon Web Services resource that is protected.
         * </p>
         * 
         * @param resourceArn
         *        The ARN (Amazon Resource Name) of the Amazon Web Services resource that is protected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The unique identifier (ID) for the Route 53 health check that's associated with the protection.
         * </p>
         * 
         * @param healthCheckIds
         *        The unique identifier (ID) for the Route 53 health check that's associated with the protection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckIds(Collection<String> healthCheckIds);

        /**
         * <p>
         * The unique identifier (ID) for the Route 53 health check that's associated with the protection.
         * </p>
         * 
         * @param healthCheckIds
         *        The unique identifier (ID) for the Route 53 health check that's associated with the protection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckIds(String... healthCheckIds);

        /**
         * <p>
         * The ARN (Amazon Resource Name) of the protection.
         * </p>
         * 
         * @param protectionArn
         *        The ARN (Amazon Resource Name) of the protection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protectionArn(String protectionArn);

        /**
         * <p>
         * The automatic application layer DDoS mitigation settings for the protection. This configuration determines
         * whether Shield Advanced automatically manages rules in the web ACL in order to respond to application layer
         * events that Shield Advanced determines to be DDoS attacks.
         * </p>
         * 
         * @param applicationLayerAutomaticResponseConfiguration
         *        The automatic application layer DDoS mitigation settings for the protection. This configuration
         *        determines whether Shield Advanced automatically manages rules in the web ACL in order to respond to
         *        application layer events that Shield Advanced determines to be DDoS attacks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationLayerAutomaticResponseConfiguration(
                ApplicationLayerAutomaticResponseConfiguration applicationLayerAutomaticResponseConfiguration);

        /**
         * <p>
         * The automatic application layer DDoS mitigation settings for the protection. This configuration determines
         * whether Shield Advanced automatically manages rules in the web ACL in order to respond to application layer
         * events that Shield Advanced determines to be DDoS attacks.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link ApplicationLayerAutomaticResponseConfiguration.Builder} avoiding the need to create one manually via
         * {@link ApplicationLayerAutomaticResponseConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ApplicationLayerAutomaticResponseConfiguration.Builder#build()}
         * is called immediately and its result is passed to
         * {@link #applicationLayerAutomaticResponseConfiguration(ApplicationLayerAutomaticResponseConfiguration)}.
         * 
         * @param applicationLayerAutomaticResponseConfiguration
         *        a consumer that will call methods on {@link ApplicationLayerAutomaticResponseConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #applicationLayerAutomaticResponseConfiguration(ApplicationLayerAutomaticResponseConfiguration)
         */
        default Builder applicationLayerAutomaticResponseConfiguration(
                Consumer<ApplicationLayerAutomaticResponseConfiguration.Builder> applicationLayerAutomaticResponseConfiguration) {
            return applicationLayerAutomaticResponseConfiguration(ApplicationLayerAutomaticResponseConfiguration.builder()
                    .applyMutation(applicationLayerAutomaticResponseConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String resourceArn;

        private List<String> healthCheckIds = DefaultSdkAutoConstructList.getInstance();

        private String protectionArn;

        private ApplicationLayerAutomaticResponseConfiguration applicationLayerAutomaticResponseConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(Protection model) {
            id(model.id);
            name(model.name);
            resourceArn(model.resourceArn);
            healthCheckIds(model.healthCheckIds);
            protectionArn(model.protectionArn);
            applicationLayerAutomaticResponseConfiguration(model.applicationLayerAutomaticResponseConfiguration);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        @Transient
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        @Transient
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final Collection<String> getHealthCheckIds() {
            if (healthCheckIds instanceof SdkAutoConstructList) {
                return null;
            }
            return healthCheckIds;
        }

        public final void setHealthCheckIds(Collection<String> healthCheckIds) {
            this.healthCheckIds = HealthCheckIdsCopier.copy(healthCheckIds);
        }

        @Override
        @Transient
        public final Builder healthCheckIds(Collection<String> healthCheckIds) {
            this.healthCheckIds = HealthCheckIdsCopier.copy(healthCheckIds);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder healthCheckIds(String... healthCheckIds) {
            healthCheckIds(Arrays.asList(healthCheckIds));
            return this;
        }

        public final String getProtectionArn() {
            return protectionArn;
        }

        public final void setProtectionArn(String protectionArn) {
            this.protectionArn = protectionArn;
        }

        @Override
        @Transient
        public final Builder protectionArn(String protectionArn) {
            this.protectionArn = protectionArn;
            return this;
        }

        public final ApplicationLayerAutomaticResponseConfiguration.Builder getApplicationLayerAutomaticResponseConfiguration() {
            return applicationLayerAutomaticResponseConfiguration != null ? applicationLayerAutomaticResponseConfiguration
                    .toBuilder() : null;
        }

        public final void setApplicationLayerAutomaticResponseConfiguration(
                ApplicationLayerAutomaticResponseConfiguration.BuilderImpl applicationLayerAutomaticResponseConfiguration) {
            this.applicationLayerAutomaticResponseConfiguration = applicationLayerAutomaticResponseConfiguration != null ? applicationLayerAutomaticResponseConfiguration
                    .build() : null;
        }

        @Override
        @Transient
        public final Builder applicationLayerAutomaticResponseConfiguration(
                ApplicationLayerAutomaticResponseConfiguration applicationLayerAutomaticResponseConfiguration) {
            this.applicationLayerAutomaticResponseConfiguration = applicationLayerAutomaticResponseConfiguration;
            return this;
        }

        @Override
        public Protection build() {
            return new Protection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
