/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.signer;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.signer.model.AccessDeniedException;
import software.amazon.awssdk.services.signer.model.CancelSigningProfileRequest;
import software.amazon.awssdk.services.signer.model.CancelSigningProfileResponse;
import software.amazon.awssdk.services.signer.model.DescribeSigningJobRequest;
import software.amazon.awssdk.services.signer.model.DescribeSigningJobResponse;
import software.amazon.awssdk.services.signer.model.GetSigningPlatformRequest;
import software.amazon.awssdk.services.signer.model.GetSigningPlatformResponse;
import software.amazon.awssdk.services.signer.model.GetSigningProfileRequest;
import software.amazon.awssdk.services.signer.model.GetSigningProfileResponse;
import software.amazon.awssdk.services.signer.model.InternalServiceErrorException;
import software.amazon.awssdk.services.signer.model.ListSigningJobsRequest;
import software.amazon.awssdk.services.signer.model.ListSigningJobsResponse;
import software.amazon.awssdk.services.signer.model.ListSigningPlatformsRequest;
import software.amazon.awssdk.services.signer.model.ListSigningPlatformsResponse;
import software.amazon.awssdk.services.signer.model.ListSigningProfilesRequest;
import software.amazon.awssdk.services.signer.model.ListSigningProfilesResponse;
import software.amazon.awssdk.services.signer.model.PutSigningProfileRequest;
import software.amazon.awssdk.services.signer.model.PutSigningProfileResponse;
import software.amazon.awssdk.services.signer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.signer.model.SignerException;
import software.amazon.awssdk.services.signer.model.SignerRequest;
import software.amazon.awssdk.services.signer.model.StartSigningJobRequest;
import software.amazon.awssdk.services.signer.model.StartSigningJobResponse;
import software.amazon.awssdk.services.signer.model.ThrottlingException;
import software.amazon.awssdk.services.signer.model.ValidationException;
import software.amazon.awssdk.services.signer.paginators.ListSigningJobsPublisher;
import software.amazon.awssdk.services.signer.paginators.ListSigningPlatformsPublisher;
import software.amazon.awssdk.services.signer.paginators.ListSigningProfilesPublisher;
import software.amazon.awssdk.services.signer.transform.CancelSigningProfileRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.DescribeSigningJobRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.GetSigningPlatformRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.GetSigningProfileRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.ListSigningJobsRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.ListSigningPlatformsRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.ListSigningProfilesRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.PutSigningProfileRequestMarshaller;
import software.amazon.awssdk.services.signer.transform.StartSigningJobRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SignerAsyncClient}.
 *
 * @see SignerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSignerAsyncClient implements SignerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSignerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSignerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Changes the state of an <code>ACTIVE</code> signing profile to <code>CANCELED</code>. A canceled profile is still
     * viewable with the <code>ListSigningProfiles</code> operation, but it cannot perform new signing jobs, and is
     * deleted two years after cancelation.
     * </p>
     *
     * @param cancelSigningProfileRequest
     * @return A Java Future containing the result of the CancelSigningProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A specified resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.CancelSigningProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/CancelSigningProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelSigningProfileResponse> cancelSigningProfile(
            CancelSigningProfileRequest cancelSigningProfileRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelSigningProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelSigningProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<CancelSigningProfileRequest, CancelSigningProfileResponse>()
                    .withMarshaller(new CancelSigningProfileRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(cancelSigningProfileRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specific code signing job. You specify the job by using the <code>jobId</code> value
     * that is returned by the <a>StartSigningJob</a> operation.
     * </p>
     *
     * @param describeSigningJobRequest
     * @return A Java Future containing the result of the DescribeSigningJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A specified resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.DescribeSigningJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/DescribeSigningJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSigningJobResponse> describeSigningJob(DescribeSigningJobRequest describeSigningJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSigningJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSigningJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DescribeSigningJobRequest, DescribeSigningJobResponse>()
                    .withMarshaller(new DescribeSigningJobRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(describeSigningJobRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information on a specific signing platform.
     * </p>
     *
     * @param getSigningPlatformRequest
     * @return A Java Future containing the result of the GetSigningPlatform operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A specified resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.GetSigningPlatform
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/GetSigningPlatform" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSigningPlatformResponse> getSigningPlatform(GetSigningPlatformRequest getSigningPlatformRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSigningPlatformResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSigningPlatformResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetSigningPlatformRequest, GetSigningPlatformResponse>()
                    .withMarshaller(new GetSigningPlatformRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getSigningPlatformRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information on a specific signing profile.
     * </p>
     *
     * @param getSigningProfileRequest
     * @return A Java Future containing the result of the GetSigningProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A specified resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.GetSigningProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/GetSigningProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSigningProfileResponse> getSigningProfile(GetSigningProfileRequest getSigningProfileRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSigningProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSigningProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetSigningProfileRequest, GetSigningProfileResponse>()
                    .withMarshaller(new GetSigningProfileRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSigningProfileRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all your signing jobs. You can use the <code>maxResults</code> parameter to limit the number of signing
     * jobs that are returned in the response. If additional jobs remain to be listed, AWS Signer returns a
     * <code>nextToken</code> value. Use this value in subsequent calls to <code>ListSigningJobs</code> to fetch the
     * remaining values. You can continue calling <code>ListSigningJobs</code> with your <code>maxResults</code>
     * parameter and with new values that AWS Signer returns in the <code>nextToken</code> parameter until all of your
     * signing jobs have been returned.
     * </p>
     *
     * @param listSigningJobsRequest
     * @return A Java Future containing the result of the ListSigningJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException You signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.ListSigningJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSigningJobsResponse> listSigningJobs(ListSigningJobsRequest listSigningJobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSigningJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSigningJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListSigningJobsRequest, ListSigningJobsResponse>()
                    .withMarshaller(new ListSigningJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSigningJobsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all your signing jobs. You can use the <code>maxResults</code> parameter to limit the number of signing
     * jobs that are returned in the response. If additional jobs remain to be listed, AWS Signer returns a
     * <code>nextToken</code> value. Use this value in subsequent calls to <code>ListSigningJobs</code> to fetch the
     * remaining values. You can continue calling <code>ListSigningJobs</code> with your <code>maxResults</code>
     * parameter and with new values that AWS Signer returns in the <code>nextToken</code> parameter until all of your
     * signing jobs have been returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSigningJobs(software.amazon.awssdk.services.signer.model.ListSigningJobsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.signer.paginators.ListSigningJobsPublisher publisher = client.listSigningJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.signer.paginators.ListSigningJobsPublisher publisher = client.listSigningJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.signer.model.ListSigningJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.signer.model.ListSigningJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSigningJobs(software.amazon.awssdk.services.signer.model.ListSigningJobsRequest)} operation.</b>
     * </p>
     *
     * @param listSigningJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException You signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.ListSigningJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSigningJobsPublisher listSigningJobsPaginator(ListSigningJobsRequest listSigningJobsRequest) {
        return new ListSigningJobsPublisher(this, applyPaginatorUserAgent(listSigningJobsRequest));
    }

    /**
     * <p>
     * Lists all signing platforms available in AWS Signer that match the request parameters. If additional jobs remain
     * to be listed, AWS Signer returns a <code>nextToken</code> value. Use this value in subsequent calls to
     * <code>ListSigningJobs</code> to fetch the remaining values. You can continue calling <code>ListSigningJobs</code>
     * with your <code>maxResults</code> parameter and with new values that AWS Signer returns in the
     * <code>nextToken</code> parameter until all of your signing jobs have been returned.
     * </p>
     *
     * @param listSigningPlatformsRequest
     * @return A Java Future containing the result of the ListSigningPlatforms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException You signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.ListSigningPlatforms
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningPlatforms" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSigningPlatformsResponse> listSigningPlatforms(
            ListSigningPlatformsRequest listSigningPlatformsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSigningPlatformsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSigningPlatformsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListSigningPlatformsRequest, ListSigningPlatformsResponse>()
                    .withMarshaller(new ListSigningPlatformsRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listSigningPlatformsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all signing platforms available in AWS Signer that match the request parameters. If additional jobs remain
     * to be listed, AWS Signer returns a <code>nextToken</code> value. Use this value in subsequent calls to
     * <code>ListSigningJobs</code> to fetch the remaining values. You can continue calling <code>ListSigningJobs</code>
     * with your <code>maxResults</code> parameter and with new values that AWS Signer returns in the
     * <code>nextToken</code> parameter until all of your signing jobs have been returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSigningPlatforms(software.amazon.awssdk.services.signer.model.ListSigningPlatformsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.signer.paginators.ListSigningPlatformsPublisher publisher = client.listSigningPlatformsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.signer.paginators.ListSigningPlatformsPublisher publisher = client.listSigningPlatformsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.signer.model.ListSigningPlatformsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.signer.model.ListSigningPlatformsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSigningPlatforms(software.amazon.awssdk.services.signer.model.ListSigningPlatformsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSigningPlatformsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException You signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.ListSigningPlatforms
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningPlatforms" target="_top">AWS
     *      API Documentation</a>
     */
    public ListSigningPlatformsPublisher listSigningPlatformsPaginator(ListSigningPlatformsRequest listSigningPlatformsRequest) {
        return new ListSigningPlatformsPublisher(this, applyPaginatorUserAgent(listSigningPlatformsRequest));
    }

    /**
     * <p>
     * Lists all available signing profiles in your AWS account. Returns only profiles with an <code>ACTIVE</code>
     * status unless the <code>includeCanceled</code> request field is set to <code>true</code>. If additional jobs
     * remain to be listed, AWS Signer returns a <code>nextToken</code> value. Use this value in subsequent calls to
     * <code>ListSigningJobs</code> to fetch the remaining values. You can continue calling <code>ListSigningJobs</code>
     * with your <code>maxResults</code> parameter and with new values that AWS Signer returns in the
     * <code>nextToken</code> parameter until all of your signing jobs have been returned.
     * </p>
     *
     * @param listSigningProfilesRequest
     * @return A Java Future containing the result of the ListSigningProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.ListSigningProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSigningProfilesResponse> listSigningProfiles(
            ListSigningProfilesRequest listSigningProfilesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSigningProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSigningProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListSigningProfilesRequest, ListSigningProfilesResponse>()
                    .withMarshaller(new ListSigningProfilesRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listSigningProfilesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all available signing profiles in your AWS account. Returns only profiles with an <code>ACTIVE</code>
     * status unless the <code>includeCanceled</code> request field is set to <code>true</code>. If additional jobs
     * remain to be listed, AWS Signer returns a <code>nextToken</code> value. Use this value in subsequent calls to
     * <code>ListSigningJobs</code> to fetch the remaining values. You can continue calling <code>ListSigningJobs</code>
     * with your <code>maxResults</code> parameter and with new values that AWS Signer returns in the
     * <code>nextToken</code> parameter until all of your signing jobs have been returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSigningProfiles(software.amazon.awssdk.services.signer.model.ListSigningProfilesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.signer.paginators.ListSigningProfilesPublisher publisher = client.listSigningProfilesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.signer.paginators.ListSigningProfilesPublisher publisher = client.listSigningProfilesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.signer.model.ListSigningProfilesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.signer.model.ListSigningProfilesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSigningProfiles(software.amazon.awssdk.services.signer.model.ListSigningProfilesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSigningProfilesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.ListSigningProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/ListSigningProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSigningProfilesPublisher listSigningProfilesPaginator(ListSigningProfilesRequest listSigningProfilesRequest) {
        return new ListSigningProfilesPublisher(this, applyPaginatorUserAgent(listSigningProfilesRequest));
    }

    /**
     * <p>
     * Creates a signing profile. A signing profile is an AWS Signer template that can be used to carry out a
     * pre-defined signing job. For more information, see <a
     * href="http://docs.aws.amazon.com/signer/latest/developerguide/gs-profile.html"
     * >http://docs.aws.amazon.com/signer/latest/developerguide/gs-profile.html</a>
     * </p>
     *
     * @param putSigningProfileRequest
     * @return A Java Future containing the result of the PutSigningProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A specified resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException You signing certificate could not be validated.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.PutSigningProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/PutSigningProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutSigningProfileResponse> putSigningProfile(PutSigningProfileRequest putSigningProfileRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutSigningProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutSigningProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<PutSigningProfileRequest, PutSigningProfileResponse>()
                    .withMarshaller(new PutSigningProfileRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putSigningProfileRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Initiates a signing job to be performed on the code provided. Signing jobs are viewable by the
     * <code>ListSigningJobs</code> operation for two years after they are performed. Note the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must create an Amazon S3 source bucket. For more information, see <a
     * href="http://docs.aws.amazon.com/AmazonS3/latest/gsg/CreatingABucket.html">Create a Bucket</a> in the <i>Amazon
     * S3 Getting Started Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your S3 source bucket must be version enabled.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must create an S3 destination bucket. AWS Signer uses your S3 destination bucket to write your signed code.
     * </p>
     * </li>
     * <li>
     * <p>
     * You specify the name of the source and destination buckets when calling the <code>StartSigningJob</code>
     * operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must also specify a request token that identifies your request to AWS Signer.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can call the <a>DescribeSigningJob</a> and the <a>ListSigningJobs</a> actions after you call
     * <code>StartSigningJob</code>.
     * </p>
     * <p>
     * For a Java example that shows how to use this action, see <a
     * href="http://docs.aws.amazon.com/acm/latest/userguide/">http://docs.aws.amazon.com/acm/latest/userguide/</a>
     * </p>
     *
     * @param startSigningJobRequest
     * @return A Java Future containing the result of the StartSigningJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException You signing certificate could not be validated.</li>
     *         <li>ResourceNotFoundException A specified resource could not be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The signing job has been throttled.</li>
     *         <li>InternalServiceErrorException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SignerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SignerAsyncClient.StartSigningJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/signer-2017-08-25/StartSigningJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartSigningJobResponse> startSigningJob(StartSigningJobRequest startSigningJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSigningJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSigningJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<StartSigningJobRequest, StartSigningJobResponse>()
                    .withMarshaller(new StartSigningJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startSigningJobRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SignerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private <T extends SignerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
