/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.signer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A signing configuration that overrides the default encryption or hash algorithm of a signing job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SigningConfigurationOverrides implements SdkPojo, Serializable,
        ToCopyableBuilder<SigningConfigurationOverrides.Builder, SigningConfigurationOverrides> {
    private static final SdkField<String> ENCRYPTION_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SigningConfigurationOverrides::encryptionAlgorithmAsString))
            .setter(setter(Builder::encryptionAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionAlgorithm").build())
            .build();

    private static final SdkField<String> HASH_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SigningConfigurationOverrides::hashAlgorithmAsString)).setter(setter(Builder::hashAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hashAlgorithm").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENCRYPTION_ALGORITHM_FIELD,
            HASH_ALGORITHM_FIELD));

    private static final long serialVersionUID = 1L;

    private final String encryptionAlgorithm;

    private final String hashAlgorithm;

    private SigningConfigurationOverrides(BuilderImpl builder) {
        this.encryptionAlgorithm = builder.encryptionAlgorithm;
        this.hashAlgorithm = builder.hashAlgorithm;
    }

    /**
     * <p>
     * A specified override of the default encryption algorithm that is used in an AWS Signer job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #encryptionAlgorithm} will return {@link EncryptionAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #encryptionAlgorithmAsString}.
     * </p>
     * 
     * @return A specified override of the default encryption algorithm that is used in an AWS Signer job.
     * @see EncryptionAlgorithm
     */
    public EncryptionAlgorithm encryptionAlgorithm() {
        return EncryptionAlgorithm.fromValue(encryptionAlgorithm);
    }

    /**
     * <p>
     * A specified override of the default encryption algorithm that is used in an AWS Signer job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #encryptionAlgorithm} will return {@link EncryptionAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #encryptionAlgorithmAsString}.
     * </p>
     * 
     * @return A specified override of the default encryption algorithm that is used in an AWS Signer job.
     * @see EncryptionAlgorithm
     */
    public String encryptionAlgorithmAsString() {
        return encryptionAlgorithm;
    }

    /**
     * <p>
     * A specified override of the default hash algorithm that is used in an AWS Signer job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hashAlgorithm}
     * will return {@link HashAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hashAlgorithmAsString}.
     * </p>
     * 
     * @return A specified override of the default hash algorithm that is used in an AWS Signer job.
     * @see HashAlgorithm
     */
    public HashAlgorithm hashAlgorithm() {
        return HashAlgorithm.fromValue(hashAlgorithm);
    }

    /**
     * <p>
     * A specified override of the default hash algorithm that is used in an AWS Signer job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hashAlgorithm}
     * will return {@link HashAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hashAlgorithmAsString}.
     * </p>
     * 
     * @return A specified override of the default hash algorithm that is used in an AWS Signer job.
     * @see HashAlgorithm
     */
    public String hashAlgorithmAsString() {
        return hashAlgorithm;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(encryptionAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hashAlgorithmAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SigningConfigurationOverrides)) {
            return false;
        }
        SigningConfigurationOverrides other = (SigningConfigurationOverrides) obj;
        return Objects.equals(encryptionAlgorithmAsString(), other.encryptionAlgorithmAsString())
                && Objects.equals(hashAlgorithmAsString(), other.hashAlgorithmAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("SigningConfigurationOverrides").add("EncryptionAlgorithm", encryptionAlgorithmAsString())
                .add("HashAlgorithm", hashAlgorithmAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "encryptionAlgorithm":
            return Optional.ofNullable(clazz.cast(encryptionAlgorithmAsString()));
        case "hashAlgorithm":
            return Optional.ofNullable(clazz.cast(hashAlgorithmAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SigningConfigurationOverrides, T> g) {
        return obj -> g.apply((SigningConfigurationOverrides) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SigningConfigurationOverrides> {
        /**
         * <p>
         * A specified override of the default encryption algorithm that is used in an AWS Signer job.
         * </p>
         * 
         * @param encryptionAlgorithm
         *        A specified override of the default encryption algorithm that is used in an AWS Signer job.
         * @see EncryptionAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionAlgorithm
         */
        Builder encryptionAlgorithm(String encryptionAlgorithm);

        /**
         * <p>
         * A specified override of the default encryption algorithm that is used in an AWS Signer job.
         * </p>
         * 
         * @param encryptionAlgorithm
         *        A specified override of the default encryption algorithm that is used in an AWS Signer job.
         * @see EncryptionAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionAlgorithm
         */
        Builder encryptionAlgorithm(EncryptionAlgorithm encryptionAlgorithm);

        /**
         * <p>
         * A specified override of the default hash algorithm that is used in an AWS Signer job.
         * </p>
         * 
         * @param hashAlgorithm
         *        A specified override of the default hash algorithm that is used in an AWS Signer job.
         * @see HashAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HashAlgorithm
         */
        Builder hashAlgorithm(String hashAlgorithm);

        /**
         * <p>
         * A specified override of the default hash algorithm that is used in an AWS Signer job.
         * </p>
         * 
         * @param hashAlgorithm
         *        A specified override of the default hash algorithm that is used in an AWS Signer job.
         * @see HashAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HashAlgorithm
         */
        Builder hashAlgorithm(HashAlgorithm hashAlgorithm);
    }

    static final class BuilderImpl implements Builder {
        private String encryptionAlgorithm;

        private String hashAlgorithm;

        private BuilderImpl() {
        }

        private BuilderImpl(SigningConfigurationOverrides model) {
            encryptionAlgorithm(model.encryptionAlgorithm);
            hashAlgorithm(model.hashAlgorithm);
        }

        public final String getEncryptionAlgorithm() {
            return encryptionAlgorithm;
        }

        @Override
        public final Builder encryptionAlgorithm(String encryptionAlgorithm) {
            this.encryptionAlgorithm = encryptionAlgorithm;
            return this;
        }

        @Override
        public final Builder encryptionAlgorithm(EncryptionAlgorithm encryptionAlgorithm) {
            this.encryptionAlgorithm(encryptionAlgorithm.toString());
            return this;
        }

        public final void setEncryptionAlgorithm(String encryptionAlgorithm) {
            this.encryptionAlgorithm = encryptionAlgorithm;
        }

        public final String getHashAlgorithm() {
            return hashAlgorithm;
        }

        @Override
        public final Builder hashAlgorithm(String hashAlgorithm) {
            this.hashAlgorithm = hashAlgorithm;
            return this;
        }

        @Override
        public final Builder hashAlgorithm(HashAlgorithm hashAlgorithm) {
            this.hashAlgorithm(hashAlgorithm.toString());
            return this;
        }

        public final void setHashAlgorithm(String hashAlgorithm) {
            this.hashAlgorithm = hashAlgorithm;
        }

        @Override
        public SigningConfigurationOverrides build() {
            return new SigningConfigurationOverrides(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
