/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.signer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the ACM certificates and code signing configuration parameters that can be used by a given
 * code signing user.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SigningProfile implements SdkPojo, Serializable, ToCopyableBuilder<SigningProfile.Builder, SigningProfile> {
    private static final SdkField<String> PROFILE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SigningProfile::profileName)).setter(setter(Builder::profileName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("profileName").build()).build();

    private static final SdkField<SigningMaterial> SIGNING_MATERIAL_FIELD = SdkField
            .<SigningMaterial> builder(MarshallingType.SDK_POJO).getter(getter(SigningProfile::signingMaterial))
            .setter(setter(Builder::signingMaterial)).constructor(SigningMaterial::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signingMaterial").build()).build();

    private static final SdkField<String> PLATFORM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SigningProfile::platformId)).setter(setter(Builder::platformId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformId").build()).build();

    private static final SdkField<Map<String, String>> SIGNING_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(SigningProfile::signingParameters))
            .setter(setter(Builder::signingParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signingParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SigningProfile::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SigningProfile::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(SigningProfile::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROFILE_NAME_FIELD,
            SIGNING_MATERIAL_FIELD, PLATFORM_ID_FIELD, SIGNING_PARAMETERS_FIELD, STATUS_FIELD, ARN_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String profileName;

    private final SigningMaterial signingMaterial;

    private final String platformId;

    private final Map<String, String> signingParameters;

    private final String status;

    private final String arn;

    private final Map<String, String> tags;

    private SigningProfile(BuilderImpl builder) {
        this.profileName = builder.profileName;
        this.signingMaterial = builder.signingMaterial;
        this.platformId = builder.platformId;
        this.signingParameters = builder.signingParameters;
        this.status = builder.status;
        this.arn = builder.arn;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the signing profile.
     * </p>
     * 
     * @return The name of the signing profile.
     */
    public String profileName() {
        return profileName;
    }

    /**
     * <p>
     * The ACM certificate that is available for use by a signing profile.
     * </p>
     * 
     * @return The ACM certificate that is available for use by a signing profile.
     */
    public SigningMaterial signingMaterial() {
        return signingMaterial;
    }

    /**
     * <p>
     * The ID of a platform that is available for use by a signing profile.
     * </p>
     * 
     * @return The ID of a platform that is available for use by a signing profile.
     */
    public String platformId() {
        return platformId;
    }

    /**
     * Returns true if the SigningParameters property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSigningParameters() {
        return signingParameters != null && !(signingParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The parameters that are available for use by a code signing user.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSigningParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The parameters that are available for use by a code signing user.
     */
    public Map<String, String> signingParameters() {
        return signingParameters;
    }

    /**
     * <p>
     * The status of a code signing profile.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link SigningProfileStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a code signing profile.
     * @see SigningProfileStatus
     */
    public SigningProfileStatus status() {
        return SigningProfileStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of a code signing profile.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link SigningProfileStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a code signing profile.
     * @see SigningProfileStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) for the signing profile.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) for the signing profile.
     */
    public String arn() {
        return arn;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of tags associated with the signing profile.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of tags associated with the signing profile.
     */
    public Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(profileName());
        hashCode = 31 * hashCode + Objects.hashCode(signingMaterial());
        hashCode = 31 * hashCode + Objects.hashCode(platformId());
        hashCode = 31 * hashCode + Objects.hashCode(signingParameters());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SigningProfile)) {
            return false;
        }
        SigningProfile other = (SigningProfile) obj;
        return Objects.equals(profileName(), other.profileName()) && Objects.equals(signingMaterial(), other.signingMaterial())
                && Objects.equals(platformId(), other.platformId())
                && Objects.equals(signingParameters(), other.signingParameters())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(arn(), other.arn())
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SigningProfile").add("ProfileName", profileName()).add("SigningMaterial", signingMaterial())
                .add("PlatformId", platformId()).add("SigningParameters", signingParameters()).add("Status", statusAsString())
                .add("Arn", arn()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "profileName":
            return Optional.ofNullable(clazz.cast(profileName()));
        case "signingMaterial":
            return Optional.ofNullable(clazz.cast(signingMaterial()));
        case "platformId":
            return Optional.ofNullable(clazz.cast(platformId()));
        case "signingParameters":
            return Optional.ofNullable(clazz.cast(signingParameters()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SigningProfile, T> g) {
        return obj -> g.apply((SigningProfile) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SigningProfile> {
        /**
         * <p>
         * The name of the signing profile.
         * </p>
         * 
         * @param profileName
         *        The name of the signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder profileName(String profileName);

        /**
         * <p>
         * The ACM certificate that is available for use by a signing profile.
         * </p>
         * 
         * @param signingMaterial
         *        The ACM certificate that is available for use by a signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingMaterial(SigningMaterial signingMaterial);

        /**
         * <p>
         * The ACM certificate that is available for use by a signing profile.
         * </p>
         * This is a convenience that creates an instance of the {@link SigningMaterial.Builder} avoiding the need to
         * create one manually via {@link SigningMaterial#builder()}.
         *
         * When the {@link Consumer} completes, {@link SigningMaterial.Builder#build()} is called immediately and its
         * result is passed to {@link #signingMaterial(SigningMaterial)}.
         * 
         * @param signingMaterial
         *        a consumer that will call methods on {@link SigningMaterial.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signingMaterial(SigningMaterial)
         */
        default Builder signingMaterial(Consumer<SigningMaterial.Builder> signingMaterial) {
            return signingMaterial(SigningMaterial.builder().applyMutation(signingMaterial).build());
        }

        /**
         * <p>
         * The ID of a platform that is available for use by a signing profile.
         * </p>
         * 
         * @param platformId
         *        The ID of a platform that is available for use by a signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformId(String platformId);

        /**
         * <p>
         * The parameters that are available for use by a code signing user.
         * </p>
         * 
         * @param signingParameters
         *        The parameters that are available for use by a code signing user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingParameters(Map<String, String> signingParameters);

        /**
         * <p>
         * The status of a code signing profile.
         * </p>
         * 
         * @param status
         *        The status of a code signing profile.
         * @see SigningProfileStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningProfileStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a code signing profile.
         * </p>
         * 
         * @param status
         *        The status of a code signing profile.
         * @see SigningProfileStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningProfileStatus
         */
        Builder status(SigningProfileStatus status);

        /**
         * <p>
         * Amazon Resource Name (ARN) for the signing profile.
         * </p>
         * 
         * @param arn
         *        Amazon Resource Name (ARN) for the signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * A list of tags associated with the signing profile.
         * </p>
         * 
         * @param tags
         *        A list of tags associated with the signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String profileName;

        private SigningMaterial signingMaterial;

        private String platformId;

        private Map<String, String> signingParameters = DefaultSdkAutoConstructMap.getInstance();

        private String status;

        private String arn;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SigningProfile model) {
            profileName(model.profileName);
            signingMaterial(model.signingMaterial);
            platformId(model.platformId);
            signingParameters(model.signingParameters);
            status(model.status);
            arn(model.arn);
            tags(model.tags);
        }

        public final String getProfileName() {
            return profileName;
        }

        @Override
        public final Builder profileName(String profileName) {
            this.profileName = profileName;
            return this;
        }

        public final void setProfileName(String profileName) {
            this.profileName = profileName;
        }

        public final SigningMaterial.Builder getSigningMaterial() {
            return signingMaterial != null ? signingMaterial.toBuilder() : null;
        }

        @Override
        public final Builder signingMaterial(SigningMaterial signingMaterial) {
            this.signingMaterial = signingMaterial;
            return this;
        }

        public final void setSigningMaterial(SigningMaterial.BuilderImpl signingMaterial) {
            this.signingMaterial = signingMaterial != null ? signingMaterial.build() : null;
        }

        public final String getPlatformId() {
            return platformId;
        }

        @Override
        public final Builder platformId(String platformId) {
            this.platformId = platformId;
            return this;
        }

        public final void setPlatformId(String platformId) {
            this.platformId = platformId;
        }

        public final Map<String, String> getSigningParameters() {
            return signingParameters;
        }

        @Override
        public final Builder signingParameters(Map<String, String> signingParameters) {
            this.signingParameters = SigningParametersCopier.copy(signingParameters);
            return this;
        }

        public final void setSigningParameters(Map<String, String> signingParameters) {
            this.signingParameters = SigningParametersCopier.copy(signingParameters);
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(SigningProfileStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Map<String, String> getTags() {
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public SigningProfile build() {
            return new SigningProfile(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
