/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.signer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The hash algorithms that are available to a code signing job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HashAlgorithmOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<HashAlgorithmOptions.Builder, HashAlgorithmOptions> {
    private static final SdkField<List<String>> ALLOWED_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("allowedValues")
            .getter(getter(HashAlgorithmOptions::allowedValuesAsStrings))
            .setter(setter(Builder::allowedValuesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowedValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DEFAULT_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("defaultValue").getter(getter(HashAlgorithmOptions::defaultValueAsString))
            .setter(setter(Builder::defaultValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOWED_VALUES_FIELD,
            DEFAULT_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> allowedValues;

    private final String defaultValue;

    private HashAlgorithmOptions(BuilderImpl builder) {
        this.allowedValues = builder.allowedValues;
        this.defaultValue = builder.defaultValue;
    }

    /**
     * <p>
     * The set of accepted hash algorithms allowed in a code signing job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAllowedValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The set of accepted hash algorithms allowed in a code signing job.
     */
    public List<HashAlgorithm> allowedValues() {
        return HashAlgorithmsCopier.copyStringToEnum(allowedValues);
    }

    /**
     * Returns true if the AllowedValues property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAllowedValues() {
        return allowedValues != null && !(allowedValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of accepted hash algorithms allowed in a code signing job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAllowedValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The set of accepted hash algorithms allowed in a code signing job.
     */
    public List<String> allowedValuesAsStrings() {
        return allowedValues;
    }

    /**
     * <p>
     * The default hash algorithm that is used in a code signing job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #defaultValue} will
     * return {@link HashAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #defaultValueAsString}.
     * </p>
     * 
     * @return The default hash algorithm that is used in a code signing job.
     * @see HashAlgorithm
     */
    public HashAlgorithm defaultValue() {
        return HashAlgorithm.fromValue(defaultValue);
    }

    /**
     * <p>
     * The default hash algorithm that is used in a code signing job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #defaultValue} will
     * return {@link HashAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #defaultValueAsString}.
     * </p>
     * 
     * @return The default hash algorithm that is used in a code signing job.
     * @see HashAlgorithm
     */
    public String defaultValueAsString() {
        return defaultValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allowedValuesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(defaultValueAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HashAlgorithmOptions)) {
            return false;
        }
        HashAlgorithmOptions other = (HashAlgorithmOptions) obj;
        return Objects.equals(allowedValuesAsStrings(), other.allowedValuesAsStrings())
                && Objects.equals(defaultValueAsString(), other.defaultValueAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("HashAlgorithmOptions").add("AllowedValues", allowedValuesAsStrings())
                .add("DefaultValue", defaultValueAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "allowedValues":
            return Optional.ofNullable(clazz.cast(allowedValuesAsStrings()));
        case "defaultValue":
            return Optional.ofNullable(clazz.cast(defaultValueAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HashAlgorithmOptions, T> g) {
        return obj -> g.apply((HashAlgorithmOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HashAlgorithmOptions> {
        /**
         * <p>
         * The set of accepted hash algorithms allowed in a code signing job.
         * </p>
         * 
         * @param allowedValues
         *        The set of accepted hash algorithms allowed in a code signing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValuesWithStrings(Collection<String> allowedValues);

        /**
         * <p>
         * The set of accepted hash algorithms allowed in a code signing job.
         * </p>
         * 
         * @param allowedValues
         *        The set of accepted hash algorithms allowed in a code signing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValuesWithStrings(String... allowedValues);

        /**
         * <p>
         * The set of accepted hash algorithms allowed in a code signing job.
         * </p>
         * 
         * @param allowedValues
         *        The set of accepted hash algorithms allowed in a code signing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValues(Collection<HashAlgorithm> allowedValues);

        /**
         * <p>
         * The set of accepted hash algorithms allowed in a code signing job.
         * </p>
         * 
         * @param allowedValues
         *        The set of accepted hash algorithms allowed in a code signing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValues(HashAlgorithm... allowedValues);

        /**
         * <p>
         * The default hash algorithm that is used in a code signing job.
         * </p>
         * 
         * @param defaultValue
         *        The default hash algorithm that is used in a code signing job.
         * @see HashAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HashAlgorithm
         */
        Builder defaultValue(String defaultValue);

        /**
         * <p>
         * The default hash algorithm that is used in a code signing job.
         * </p>
         * 
         * @param defaultValue
         *        The default hash algorithm that is used in a code signing job.
         * @see HashAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HashAlgorithm
         */
        Builder defaultValue(HashAlgorithm defaultValue);
    }

    static final class BuilderImpl implements Builder {
        private List<String> allowedValues = DefaultSdkAutoConstructList.getInstance();

        private String defaultValue;

        private BuilderImpl() {
        }

        private BuilderImpl(HashAlgorithmOptions model) {
            allowedValuesWithStrings(model.allowedValues);
            defaultValue(model.defaultValue);
        }

        public final Collection<String> getAllowedValues() {
            return allowedValues;
        }

        @Override
        public final Builder allowedValuesWithStrings(Collection<String> allowedValues) {
            this.allowedValues = HashAlgorithmsCopier.copy(allowedValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedValuesWithStrings(String... allowedValues) {
            allowedValuesWithStrings(Arrays.asList(allowedValues));
            return this;
        }

        @Override
        public final Builder allowedValues(Collection<HashAlgorithm> allowedValues) {
            this.allowedValues = HashAlgorithmsCopier.copyEnumToString(allowedValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedValues(HashAlgorithm... allowedValues) {
            allowedValues(Arrays.asList(allowedValues));
            return this;
        }

        public final void setAllowedValues(Collection<String> allowedValues) {
            this.allowedValues = HashAlgorithmsCopier.copy(allowedValues);
        }

        public final String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public final Builder defaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        @Override
        public final Builder defaultValue(HashAlgorithm defaultValue) {
            this.defaultValue(defaultValue == null ? null : defaultValue.toString());
            return this;
        }

        public final void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public HashAlgorithmOptions build() {
            return new HashAlgorithmOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
