/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.signer.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the ACM certificates and code signing configuration parameters that can be used by a given
 * code signing user.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SigningProfile implements SdkPojo, Serializable, ToCopyableBuilder<SigningProfile.Builder, SigningProfile> {
    private static final SdkField<String> PROFILE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("profileName").getter(getter(SigningProfile::profileName)).setter(setter(Builder::profileName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("profileName").build()).build();

    private static final SdkField<String> PROFILE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("profileVersion").getter(getter(SigningProfile::profileVersion)).setter(setter(Builder::profileVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("profileVersion").build()).build();

    private static final SdkField<String> PROFILE_VERSION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("profileVersionArn").getter(getter(SigningProfile::profileVersionArn))
            .setter(setter(Builder::profileVersionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("profileVersionArn").build()).build();

    private static final SdkField<SigningMaterial> SIGNING_MATERIAL_FIELD = SdkField
            .<SigningMaterial> builder(MarshallingType.SDK_POJO).memberName("signingMaterial")
            .getter(getter(SigningProfile::signingMaterial)).setter(setter(Builder::signingMaterial))
            .constructor(SigningMaterial::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signingMaterial").build()).build();

    private static final SdkField<SignatureValidityPeriod> SIGNATURE_VALIDITY_PERIOD_FIELD = SdkField
            .<SignatureValidityPeriod> builder(MarshallingType.SDK_POJO).memberName("signatureValidityPeriod")
            .getter(getter(SigningProfile::signatureValidityPeriod)).setter(setter(Builder::signatureValidityPeriod))
            .constructor(SignatureValidityPeriod::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signatureValidityPeriod").build())
            .build();

    private static final SdkField<String> PLATFORM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platformId").getter(getter(SigningProfile::platformId)).setter(setter(Builder::platformId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformId").build()).build();

    private static final SdkField<String> PLATFORM_DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platformDisplayName").getter(getter(SigningProfile::platformDisplayName))
            .setter(setter(Builder::platformDisplayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformDisplayName").build())
            .build();

    private static final SdkField<Map<String, String>> SIGNING_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("signingParameters")
            .getter(getter(SigningProfile::signingParameters))
            .setter(setter(Builder::signingParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signingParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(SigningProfile::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(SigningProfile::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(SigningProfile::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROFILE_NAME_FIELD,
            PROFILE_VERSION_FIELD, PROFILE_VERSION_ARN_FIELD, SIGNING_MATERIAL_FIELD, SIGNATURE_VALIDITY_PERIOD_FIELD,
            PLATFORM_ID_FIELD, PLATFORM_DISPLAY_NAME_FIELD, SIGNING_PARAMETERS_FIELD, STATUS_FIELD, ARN_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String profileName;

    private final String profileVersion;

    private final String profileVersionArn;

    private final SigningMaterial signingMaterial;

    private final SignatureValidityPeriod signatureValidityPeriod;

    private final String platformId;

    private final String platformDisplayName;

    private final Map<String, String> signingParameters;

    private final String status;

    private final String arn;

    private final Map<String, String> tags;

    private SigningProfile(BuilderImpl builder) {
        this.profileName = builder.profileName;
        this.profileVersion = builder.profileVersion;
        this.profileVersionArn = builder.profileVersionArn;
        this.signingMaterial = builder.signingMaterial;
        this.signatureValidityPeriod = builder.signatureValidityPeriod;
        this.platformId = builder.platformId;
        this.platformDisplayName = builder.platformDisplayName;
        this.signingParameters = builder.signingParameters;
        this.status = builder.status;
        this.arn = builder.arn;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the signing profile.
     * </p>
     * 
     * @return The name of the signing profile.
     */
    public final String profileName() {
        return profileName;
    }

    /**
     * <p>
     * The version of a signing profile.
     * </p>
     * 
     * @return The version of a signing profile.
     */
    public final String profileVersion() {
        return profileVersion;
    }

    /**
     * <p>
     * The ARN of a signing profile, including the profile version.
     * </p>
     * 
     * @return The ARN of a signing profile, including the profile version.
     */
    public final String profileVersionArn() {
        return profileVersionArn;
    }

    /**
     * <p>
     * The ACM certificate that is available for use by a signing profile.
     * </p>
     * 
     * @return The ACM certificate that is available for use by a signing profile.
     */
    public final SigningMaterial signingMaterial() {
        return signingMaterial;
    }

    /**
     * <p>
     * The validity period for a signing job created using this signing profile.
     * </p>
     * 
     * @return The validity period for a signing job created using this signing profile.
     */
    public final SignatureValidityPeriod signatureValidityPeriod() {
        return signatureValidityPeriod;
    }

    /**
     * <p>
     * The ID of a platform that is available for use by a signing profile.
     * </p>
     * 
     * @return The ID of a platform that is available for use by a signing profile.
     */
    public final String platformId() {
        return platformId;
    }

    /**
     * <p>
     * The name of the signing platform.
     * </p>
     * 
     * @return The name of the signing platform.
     */
    public final String platformDisplayName() {
        return platformDisplayName;
    }

    /**
     * For responses, this returns true if the service returned a value for the SigningParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSigningParameters() {
        return signingParameters != null && !(signingParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The parameters that are available for use by a code signing user.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSigningParameters} method.
     * </p>
     * 
     * @return The parameters that are available for use by a code signing user.
     */
    public final Map<String, String> signingParameters() {
        return signingParameters;
    }

    /**
     * <p>
     * The status of a code signing profile.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link SigningProfileStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a code signing profile.
     * @see SigningProfileStatus
     */
    public final SigningProfileStatus status() {
        return SigningProfileStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of a code signing profile.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link SigningProfileStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a code signing profile.
     * @see SigningProfileStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the signing profile.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the signing profile.
     */
    public final String arn() {
        return arn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of tags associated with the signing profile.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags associated with the signing profile.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(profileName());
        hashCode = 31 * hashCode + Objects.hashCode(profileVersion());
        hashCode = 31 * hashCode + Objects.hashCode(profileVersionArn());
        hashCode = 31 * hashCode + Objects.hashCode(signingMaterial());
        hashCode = 31 * hashCode + Objects.hashCode(signatureValidityPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(platformId());
        hashCode = 31 * hashCode + Objects.hashCode(platformDisplayName());
        hashCode = 31 * hashCode + Objects.hashCode(hasSigningParameters() ? signingParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SigningProfile)) {
            return false;
        }
        SigningProfile other = (SigningProfile) obj;
        return Objects.equals(profileName(), other.profileName()) && Objects.equals(profileVersion(), other.profileVersion())
                && Objects.equals(profileVersionArn(), other.profileVersionArn())
                && Objects.equals(signingMaterial(), other.signingMaterial())
                && Objects.equals(signatureValidityPeriod(), other.signatureValidityPeriod())
                && Objects.equals(platformId(), other.platformId())
                && Objects.equals(platformDisplayName(), other.platformDisplayName())
                && hasSigningParameters() == other.hasSigningParameters()
                && Objects.equals(signingParameters(), other.signingParameters())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(arn(), other.arn())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SigningProfile").add("ProfileName", profileName()).add("ProfileVersion", profileVersion())
                .add("ProfileVersionArn", profileVersionArn()).add("SigningMaterial", signingMaterial())
                .add("SignatureValidityPeriod", signatureValidityPeriod()).add("PlatformId", platformId())
                .add("PlatformDisplayName", platformDisplayName())
                .add("SigningParameters", hasSigningParameters() ? signingParameters() : null).add("Status", statusAsString())
                .add("Arn", arn()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "profileName":
            return Optional.ofNullable(clazz.cast(profileName()));
        case "profileVersion":
            return Optional.ofNullable(clazz.cast(profileVersion()));
        case "profileVersionArn":
            return Optional.ofNullable(clazz.cast(profileVersionArn()));
        case "signingMaterial":
            return Optional.ofNullable(clazz.cast(signingMaterial()));
        case "signatureValidityPeriod":
            return Optional.ofNullable(clazz.cast(signatureValidityPeriod()));
        case "platformId":
            return Optional.ofNullable(clazz.cast(platformId()));
        case "platformDisplayName":
            return Optional.ofNullable(clazz.cast(platformDisplayName()));
        case "signingParameters":
            return Optional.ofNullable(clazz.cast(signingParameters()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SigningProfile, T> g) {
        return obj -> g.apply((SigningProfile) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SigningProfile> {
        /**
         * <p>
         * The name of the signing profile.
         * </p>
         * 
         * @param profileName
         *        The name of the signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder profileName(String profileName);

        /**
         * <p>
         * The version of a signing profile.
         * </p>
         * 
         * @param profileVersion
         *        The version of a signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder profileVersion(String profileVersion);

        /**
         * <p>
         * The ARN of a signing profile, including the profile version.
         * </p>
         * 
         * @param profileVersionArn
         *        The ARN of a signing profile, including the profile version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder profileVersionArn(String profileVersionArn);

        /**
         * <p>
         * The ACM certificate that is available for use by a signing profile.
         * </p>
         * 
         * @param signingMaterial
         *        The ACM certificate that is available for use by a signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingMaterial(SigningMaterial signingMaterial);

        /**
         * <p>
         * The ACM certificate that is available for use by a signing profile.
         * </p>
         * This is a convenience that creates an instance of the {@link SigningMaterial.Builder} avoiding the need to
         * create one manually via {@link SigningMaterial#builder()}.
         *
         * When the {@link Consumer} completes, {@link SigningMaterial.Builder#build()} is called immediately and its
         * result is passed to {@link #signingMaterial(SigningMaterial)}.
         * 
         * @param signingMaterial
         *        a consumer that will call methods on {@link SigningMaterial.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signingMaterial(SigningMaterial)
         */
        default Builder signingMaterial(Consumer<SigningMaterial.Builder> signingMaterial) {
            return signingMaterial(SigningMaterial.builder().applyMutation(signingMaterial).build());
        }

        /**
         * <p>
         * The validity period for a signing job created using this signing profile.
         * </p>
         * 
         * @param signatureValidityPeriod
         *        The validity period for a signing job created using this signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signatureValidityPeriod(SignatureValidityPeriod signatureValidityPeriod);

        /**
         * <p>
         * The validity period for a signing job created using this signing profile.
         * </p>
         * This is a convenience that creates an instance of the {@link SignatureValidityPeriod.Builder} avoiding the
         * need to create one manually via {@link SignatureValidityPeriod#builder()}.
         *
         * When the {@link Consumer} completes, {@link SignatureValidityPeriod.Builder#build()} is called immediately
         * and its result is passed to {@link #signatureValidityPeriod(SignatureValidityPeriod)}.
         * 
         * @param signatureValidityPeriod
         *        a consumer that will call methods on {@link SignatureValidityPeriod.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signatureValidityPeriod(SignatureValidityPeriod)
         */
        default Builder signatureValidityPeriod(Consumer<SignatureValidityPeriod.Builder> signatureValidityPeriod) {
            return signatureValidityPeriod(SignatureValidityPeriod.builder().applyMutation(signatureValidityPeriod).build());
        }

        /**
         * <p>
         * The ID of a platform that is available for use by a signing profile.
         * </p>
         * 
         * @param platformId
         *        The ID of a platform that is available for use by a signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformId(String platformId);

        /**
         * <p>
         * The name of the signing platform.
         * </p>
         * 
         * @param platformDisplayName
         *        The name of the signing platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformDisplayName(String platformDisplayName);

        /**
         * <p>
         * The parameters that are available for use by a code signing user.
         * </p>
         * 
         * @param signingParameters
         *        The parameters that are available for use by a code signing user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingParameters(Map<String, String> signingParameters);

        /**
         * <p>
         * The status of a code signing profile.
         * </p>
         * 
         * @param status
         *        The status of a code signing profile.
         * @see SigningProfileStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningProfileStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a code signing profile.
         * </p>
         * 
         * @param status
         *        The status of a code signing profile.
         * @see SigningProfileStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningProfileStatus
         */
        Builder status(SigningProfileStatus status);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the signing profile.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) for the signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * A list of tags associated with the signing profile.
         * </p>
         * 
         * @param tags
         *        A list of tags associated with the signing profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String profileName;

        private String profileVersion;

        private String profileVersionArn;

        private SigningMaterial signingMaterial;

        private SignatureValidityPeriod signatureValidityPeriod;

        private String platformId;

        private String platformDisplayName;

        private Map<String, String> signingParameters = DefaultSdkAutoConstructMap.getInstance();

        private String status;

        private String arn;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SigningProfile model) {
            profileName(model.profileName);
            profileVersion(model.profileVersion);
            profileVersionArn(model.profileVersionArn);
            signingMaterial(model.signingMaterial);
            signatureValidityPeriod(model.signatureValidityPeriod);
            platformId(model.platformId);
            platformDisplayName(model.platformDisplayName);
            signingParameters(model.signingParameters);
            status(model.status);
            arn(model.arn);
            tags(model.tags);
        }

        public final String getProfileName() {
            return profileName;
        }

        public final void setProfileName(String profileName) {
            this.profileName = profileName;
        }

        @Override
        @Transient
        public final Builder profileName(String profileName) {
            this.profileName = profileName;
            return this;
        }

        public final String getProfileVersion() {
            return profileVersion;
        }

        public final void setProfileVersion(String profileVersion) {
            this.profileVersion = profileVersion;
        }

        @Override
        @Transient
        public final Builder profileVersion(String profileVersion) {
            this.profileVersion = profileVersion;
            return this;
        }

        public final String getProfileVersionArn() {
            return profileVersionArn;
        }

        public final void setProfileVersionArn(String profileVersionArn) {
            this.profileVersionArn = profileVersionArn;
        }

        @Override
        @Transient
        public final Builder profileVersionArn(String profileVersionArn) {
            this.profileVersionArn = profileVersionArn;
            return this;
        }

        public final SigningMaterial.Builder getSigningMaterial() {
            return signingMaterial != null ? signingMaterial.toBuilder() : null;
        }

        public final void setSigningMaterial(SigningMaterial.BuilderImpl signingMaterial) {
            this.signingMaterial = signingMaterial != null ? signingMaterial.build() : null;
        }

        @Override
        @Transient
        public final Builder signingMaterial(SigningMaterial signingMaterial) {
            this.signingMaterial = signingMaterial;
            return this;
        }

        public final SignatureValidityPeriod.Builder getSignatureValidityPeriod() {
            return signatureValidityPeriod != null ? signatureValidityPeriod.toBuilder() : null;
        }

        public final void setSignatureValidityPeriod(SignatureValidityPeriod.BuilderImpl signatureValidityPeriod) {
            this.signatureValidityPeriod = signatureValidityPeriod != null ? signatureValidityPeriod.build() : null;
        }

        @Override
        @Transient
        public final Builder signatureValidityPeriod(SignatureValidityPeriod signatureValidityPeriod) {
            this.signatureValidityPeriod = signatureValidityPeriod;
            return this;
        }

        public final String getPlatformId() {
            return platformId;
        }

        public final void setPlatformId(String platformId) {
            this.platformId = platformId;
        }

        @Override
        @Transient
        public final Builder platformId(String platformId) {
            this.platformId = platformId;
            return this;
        }

        public final String getPlatformDisplayName() {
            return platformDisplayName;
        }

        public final void setPlatformDisplayName(String platformDisplayName) {
            this.platformDisplayName = platformDisplayName;
        }

        @Override
        @Transient
        public final Builder platformDisplayName(String platformDisplayName) {
            this.platformDisplayName = platformDisplayName;
            return this;
        }

        public final Map<String, String> getSigningParameters() {
            if (signingParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return signingParameters;
        }

        public final void setSigningParameters(Map<String, String> signingParameters) {
            this.signingParameters = SigningParametersCopier.copy(signingParameters);
        }

        @Override
        @Transient
        public final Builder signingParameters(Map<String, String> signingParameters) {
            this.signingParameters = SigningParametersCopier.copy(signingParameters);
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(SigningProfileStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        @Transient
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        @Override
        public SigningProfile build() {
            return new SigningProfile(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
