/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.signer.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.signer.SignerClient;
import software.amazon.awssdk.services.signer.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.signer.model.DescribeSigningJobRequest;
import software.amazon.awssdk.services.signer.model.DescribeSigningJobResponse;
import software.amazon.awssdk.services.signer.model.SignerRequest;
import software.amazon.awssdk.services.signer.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultSignerWaiter implements SignerWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final SignerClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeSigningJobResponse> successfulSigningJobWaiter;

    private DefaultSignerWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = SignerClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.successfulSigningJobWaiter = Waiter.builder(DescribeSigningJobResponse.class)
                .acceptors(successfulSigningJobWaiterAcceptors())
                .overrideConfiguration(successfulSigningJobWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeSigningJobResponse> waitUntilSuccessfulSigningJob(
            DescribeSigningJobRequest describeSigningJobRequest) {
        return successfulSigningJobWaiter.run(() -> client.describeSigningJob(applyWaitersUserAgent(describeSigningJobRequest)));
    }

    @Override
    public WaiterResponse<DescribeSigningJobResponse> waitUntilSuccessfulSigningJob(
            DescribeSigningJobRequest describeSigningJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return successfulSigningJobWaiter.run(() -> client.describeSigningJob(applyWaitersUserAgent(describeSigningJobRequest)),
                successfulSigningJobWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeSigningJobResponse>> successfulSigningJobWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeSigningJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "Succeeded");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "Failed");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(
                error -> Objects.equals(errorCode(error), "ResourceNotFoundException"),
                "A waiter acceptor was matched on error condition (ResourceNotFoundException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration successfulSigningJobWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(25);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static SignerWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends SignerRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements SignerWaiter.Builder {
        private SignerClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public SignerWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public SignerWaiter.Builder client(SignerClient client) {
            this.client = client;
            return this;
        }

        public SignerWaiter build() {
            return new DefaultSignerWaiter(this);
        }
    }
}
