/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.simspaceweaver;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.simspaceweaver.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.simspaceweaver.internal.SimSpaceWeaverServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.simspaceweaver.model.AccessDeniedException;
import software.amazon.awssdk.services.simspaceweaver.model.ConflictException;
import software.amazon.awssdk.services.simspaceweaver.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.simspaceweaver.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.simspaceweaver.model.DeleteAppRequest;
import software.amazon.awssdk.services.simspaceweaver.model.DeleteAppResponse;
import software.amazon.awssdk.services.simspaceweaver.model.DeleteSimulationRequest;
import software.amazon.awssdk.services.simspaceweaver.model.DeleteSimulationResponse;
import software.amazon.awssdk.services.simspaceweaver.model.DescribeAppRequest;
import software.amazon.awssdk.services.simspaceweaver.model.DescribeAppResponse;
import software.amazon.awssdk.services.simspaceweaver.model.DescribeSimulationRequest;
import software.amazon.awssdk.services.simspaceweaver.model.DescribeSimulationResponse;
import software.amazon.awssdk.services.simspaceweaver.model.InternalServerException;
import software.amazon.awssdk.services.simspaceweaver.model.ListAppsRequest;
import software.amazon.awssdk.services.simspaceweaver.model.ListAppsResponse;
import software.amazon.awssdk.services.simspaceweaver.model.ListSimulationsRequest;
import software.amazon.awssdk.services.simspaceweaver.model.ListSimulationsResponse;
import software.amazon.awssdk.services.simspaceweaver.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.simspaceweaver.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.simspaceweaver.model.ResourceNotFoundException;
import software.amazon.awssdk.services.simspaceweaver.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.simspaceweaver.model.SimSpaceWeaverException;
import software.amazon.awssdk.services.simspaceweaver.model.StartAppRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StartAppResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StartClockRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StartClockResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StartSimulationRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StartSimulationResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StopAppRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StopAppResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StopClockRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StopClockResponse;
import software.amazon.awssdk.services.simspaceweaver.model.StopSimulationRequest;
import software.amazon.awssdk.services.simspaceweaver.model.StopSimulationResponse;
import software.amazon.awssdk.services.simspaceweaver.model.TagResourceRequest;
import software.amazon.awssdk.services.simspaceweaver.model.TagResourceResponse;
import software.amazon.awssdk.services.simspaceweaver.model.TooManyTagsException;
import software.amazon.awssdk.services.simspaceweaver.model.UntagResourceRequest;
import software.amazon.awssdk.services.simspaceweaver.model.UntagResourceResponse;
import software.amazon.awssdk.services.simspaceweaver.model.ValidationException;
import software.amazon.awssdk.services.simspaceweaver.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.DeleteAppRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.DeleteSimulationRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.DescribeAppRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.DescribeSimulationRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.ListAppsRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.ListSimulationsRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StartAppRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StartClockRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StartSimulationRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StopAppRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StopClockRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.StopSimulationRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.simspaceweaver.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SimSpaceWeaverClient}.
 *
 * @see SimSpaceWeaverClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSimSpaceWeaverClient implements SimSpaceWeaverClient {
    private static final Logger log = Logger.loggerFor(DefaultSimSpaceWeaverClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSimSpaceWeaverClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "SimSpaceWeaver" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a snapshot of the specified simulation. A snapshot is a file that contains simulation state data at a
     * specific time. The state data saved in a snapshot includes entity data from the State Fabric, the simulation
     * configuration specified in the schema, and the clock tick number. You can use the snapshot to initialize a new
     * simulation. For more information about snapshots, see <a
     * href="https://docs.aws.amazon.com/simspaceweaver/latest/userguide/working-with_snapshots.html">Snapshots</a> in
     * the <i>SimSpace Weaver User Guide</i>.
     * </p>
     * <p>
     * You specify a <code>Destination</code> when you create a snapshot. The <code>Destination</code> is the name of an
     * Amazon S3 bucket and an optional <code>ObjectKeyPrefix</code>. The <code>ObjectKeyPrefix</code> is usually the
     * name of a folder in the bucket. SimSpace Weaver creates a <code>snapshot</code> folder inside the
     * <code>Destination</code> and places the snapshot file there.
     * </p>
     * <p>
     * The snapshot file is an Amazon S3 object. It has an object key with the form:
     * <code> <i>object-key-prefix</i>/snapshot/<i>simulation-name</i>-<i>YYMMdd</i>-<i>HHmm</i>-<i>ss</i>.zip</code>,
     * where:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code> <i>YY</i> </code> is the 2-digit year
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>MM</i> </code> is the 2-digit month
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>dd</i> </code> is the 2-digit day of the month
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>HH</i> </code> is the 2-digit hour (24-hour clock)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>mm</i> </code> is the 2-digit minutes
     * </p>
     * </li>
     * <li>
     * <p>
     * <code> <i>ss</i> </code> is the 2-digit seconds
     * </p>
     * </li>
     * </ul>
     *
     * @param createSnapshotRequest
     * @return Result of the CreateSnapshot operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.CreateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/CreateSnapshot" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateSnapshotResponse createSnapshot(CreateSnapshotRequest createSnapshotRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, ConflictException, AwsServiceException,
            SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSnapshotResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");

            return clientHandler.execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                    .withOperationName("CreateSnapshot").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the instance of the given custom app.
     * </p>
     *
     * @param deleteAppRequest
     * @return Result of the DeleteApp operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.DeleteApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/DeleteApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAppResponse deleteApp(DeleteAppRequest deleteAppRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, ConflictException, AwsServiceException,
            SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAppResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApp");

            return clientHandler.execute(new ClientExecutionParams<DeleteAppRequest, DeleteAppResponse>()
                    .withOperationName("DeleteApp").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteAppRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAppRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes all SimSpace Weaver resources assigned to the given simulation.
     * </p>
     * <note>
     * <p>
     * Your simulation uses resources in other Amazon Web Services. This API operation doesn't delete resources in other
     * Amazon Web Services.
     * </p>
     * </note>
     *
     * @param deleteSimulationRequest
     * @return Result of the DeleteSimulation operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.DeleteSimulation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/DeleteSimulation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSimulationResponse deleteSimulation(DeleteSimulationRequest deleteSimulationRequest)
            throws ResourceNotFoundException, InternalServerException, AccessDeniedException, ValidationException,
            ConflictException, AwsServiceException, SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSimulationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSimulationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSimulationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSimulationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSimulation");

            return clientHandler.execute(new ClientExecutionParams<DeleteSimulationRequest, DeleteSimulationResponse>()
                    .withOperationName("DeleteSimulation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteSimulationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSimulationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the state of the given custom app.
     * </p>
     *
     * @param describeAppRequest
     * @return Result of the DescribeApp operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.DescribeApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/DescribeApp" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeAppResponse describeApp(DescribeAppRequest describeAppRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, AwsServiceException, SdkClientException,
            SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAppResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApp");

            return clientHandler.execute(new ClientExecutionParams<DescribeAppRequest, DescribeAppResponse>()
                    .withOperationName("DescribeApp").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeAppRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAppRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current state of the given simulation.
     * </p>
     *
     * @param describeSimulationRequest
     * @return Result of the DescribeSimulation operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.DescribeSimulation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/DescribeSimulation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSimulationResponse describeSimulation(DescribeSimulationRequest describeSimulationRequest)
            throws ResourceNotFoundException, InternalServerException, AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSimulationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSimulationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSimulationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSimulationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSimulation");

            return clientHandler.execute(new ClientExecutionParams<DescribeSimulationRequest, DescribeSimulationResponse>()
                    .withOperationName("DescribeSimulation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeSimulationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSimulationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all custom apps or service apps for the given simulation and domain.
     * </p>
     *
     * @param listAppsRequest
     * @return Result of the ListApps operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.ListApps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/ListApps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAppsResponse listApps(ListAppsRequest listAppsRequest) throws ResourceNotFoundException, InternalServerException,
            AccessDeniedException, ValidationException, AwsServiceException, SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAppsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApps");

            return clientHandler.execute(new ClientExecutionParams<ListAppsRequest, ListAppsResponse>()
                    .withOperationName("ListApps").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listAppsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAppsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the SimSpace Weaver simulations in the Amazon Web Services account used to make the API call.
     * </p>
     *
     * @param listSimulationsRequest
     * @return Result of the ListSimulations operation returned by the service.
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.ListSimulations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/ListSimulations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSimulationsResponse listSimulations(ListSimulationsRequest listSimulationsRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, AwsServiceException, SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSimulationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSimulationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSimulationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSimulationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSimulations");

            return clientHandler.execute(new ClientExecutionParams<ListSimulationsRequest, ListSimulationsResponse>()
                    .withOperationName("ListSimulations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listSimulationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSimulationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags on a SimSpace Weaver resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a custom app with the configuration specified in the simulation schema.
     * </p>
     *
     * @param startAppRequest
     * @return Result of the StartApp operation returned by the service.
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.StartApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StartApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartAppResponse startApp(StartAppRequest startAppRequest) throws InternalServerException, AccessDeniedException,
            ValidationException, ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException,
            SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartAppResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartApp");

            return clientHandler.execute(new ClientExecutionParams<StartAppRequest, StartAppResponse>()
                    .withOperationName("StartApp").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startAppRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartAppRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the simulation clock.
     * </p>
     *
     * @param startClockRequest
     * @return Result of the StartClock operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.StartClock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StartClock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartClockResponse startClock(StartClockRequest startClockRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, ConflictException, AwsServiceException,
            SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartClockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartClockResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startClockRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startClockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartClock");

            return clientHandler.execute(new ClientExecutionParams<StartClockRequest, StartClockResponse>()
                    .withOperationName("StartClock").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startClockRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartClockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a simulation with the given name. You must choose to start your simulation from a schema or from a
     * snapshot. For more information about the schema, see the <a
     * href="https://docs.aws.amazon.com/simspaceweaver/latest/userguide/schema-reference.html">schema reference</a> in
     * the <i>SimSpace Weaver User Guide</i>. For more information about snapshots, see <a
     * href="https://docs.aws.amazon.com/simspaceweaver/latest/userguide/working-with_snapshots.html">Snapshots</a> in
     * the <i>SimSpace Weaver User Guide</i>.
     * </p>
     *
     * @param startSimulationRequest
     * @return Result of the StartSimulation operation returned by the service.
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.StartSimulation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StartSimulation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartSimulationResponse startSimulation(StartSimulationRequest startSimulationRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ServiceQuotaExceededException, ConflictException, AwsServiceException,
            SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartSimulationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartSimulationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSimulationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSimulationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSimulation");

            return clientHandler.execute(new ClientExecutionParams<StartSimulationRequest, StartSimulationResponse>()
                    .withOperationName("StartSimulation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startSimulationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartSimulationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the given custom app and shuts down all of its allocated compute resources.
     * </p>
     *
     * @param stopAppRequest
     * @return Result of the StopApp operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.StopApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StopApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopAppResponse stopApp(StopAppRequest stopAppRequest) throws ResourceNotFoundException, InternalServerException,
            AccessDeniedException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopAppResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopApp");

            return clientHandler.execute(new ClientExecutionParams<StopAppRequest, StopAppResponse>()
                    .withOperationName("StopApp").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopAppRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopAppRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the simulation clock.
     * </p>
     *
     * @param stopClockRequest
     * @return Result of the StopClock operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.StopClock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StopClock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopClockResponse stopClock(StopClockRequest stopClockRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, ConflictException, AwsServiceException,
            SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopClockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopClockResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopClockRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopClockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopClock");

            return clientHandler.execute(new ClientExecutionParams<StopClockRequest, StopClockResponse>()
                    .withOperationName("StopClock").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopClockRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopClockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the given simulation.
     * </p>
     * <important>
     * <p>
     * You can't restart a simulation after you stop it. If you want to restart a simulation, then you must stop it,
     * delete it, and start a new instance of it.
     * </p>
     * </important>
     *
     * @param stopSimulationRequest
     * @return Result of the StopSimulation operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.StopSimulation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/StopSimulation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopSimulationResponse stopSimulation(StopSimulationRequest stopSimulationRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, ConflictException, AwsServiceException,
            SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopSimulationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopSimulationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopSimulationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopSimulationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopSimulation");

            return clientHandler.execute(new ClientExecutionParams<StopSimulationRequest, StopSimulationResponse>()
                    .withOperationName("StopSimulation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopSimulationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopSimulationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to a SimSpace Weaver resource. For more information about tags, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services resources</a>
     * in the <i>Amazon Web Services General Reference</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws TooManyTagsException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws TooManyTagsException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a SimSpace Weaver resource. For more information about tags, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services resources</a>
     * in the <i>Amazon Web Services General Reference</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SimSpaceWeaverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SimSpaceWeaverClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/simspaceweaver-2022-10-28/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, SimSpaceWeaverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SimSpaceWeaver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        SimSpaceWeaverServiceClientConfigurationBuilder serviceConfigBuilder = new SimSpaceWeaverServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(SimSpaceWeaverException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final SimSpaceWeaverServiceClientConfiguration serviceClientConfiguration() {
        return new SimSpaceWeaverServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
