/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.snowball.transform.JobResourceMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains an array of <code>S3Resource</code> objects. Each <code>S3Resource</code> object represents an Amazon S3
 * bucket that your transferred data will be exported from or imported into.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobResource implements StructuredPojo, ToCopyableBuilder<JobResource.Builder, JobResource> {
    private final List<S3Resource> s3Resources;

    private final List<LambdaResource> lambdaResources;

    private JobResource(BuilderImpl builder) {
        this.s3Resources = builder.s3Resources;
        this.lambdaResources = builder.lambdaResources;
    }

    /**
     * <p>
     * An array of <code>S3Resource</code> objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of <code>S3Resource</code> objects.
     */
    public List<S3Resource> s3Resources() {
        return s3Resources;
    }

    /**
     * <p>
     * The Python-language Lambda functions for this job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The Python-language Lambda functions for this job.
     */
    public List<LambdaResource> lambdaResources() {
        return lambdaResources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Resources());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaResources());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobResource)) {
            return false;
        }
        JobResource other = (JobResource) obj;
        return Objects.equals(s3Resources(), other.s3Resources()) && Objects.equals(lambdaResources(), other.lambdaResources());
    }

    @Override
    public String toString() {
        return ToString.builder("JobResource").add("S3Resources", s3Resources()).add("LambdaResources", lambdaResources())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Resources":
            return Optional.ofNullable(clazz.cast(s3Resources()));
        case "LambdaResources":
            return Optional.ofNullable(clazz.cast(lambdaResources()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        JobResourceMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, JobResource> {
        /**
         * <p>
         * An array of <code>S3Resource</code> objects.
         * </p>
         * 
         * @param s3Resources
         *        An array of <code>S3Resource</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Resources(Collection<S3Resource> s3Resources);

        /**
         * <p>
         * An array of <code>S3Resource</code> objects.
         * </p>
         * 
         * @param s3Resources
         *        An array of <code>S3Resource</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Resources(S3Resource... s3Resources);

        /**
         * <p>
         * An array of <code>S3Resource</code> objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<S3Resource>.Builder} avoiding the need to
         * create one manually via {@link List<S3Resource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3Resource>.Builder#build()} is called immediately and its
         * result is passed to {@link #s3Resources(List<S3Resource>)}.
         * 
         * @param s3Resources
         *        a consumer that will call methods on {@link List<S3Resource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Resources(List<S3Resource>)
         */
        Builder s3Resources(Consumer<S3Resource.Builder>... s3Resources);

        /**
         * <p>
         * The Python-language Lambda functions for this job.
         * </p>
         * 
         * @param lambdaResources
         *        The Python-language Lambda functions for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaResources(Collection<LambdaResource> lambdaResources);

        /**
         * <p>
         * The Python-language Lambda functions for this job.
         * </p>
         * 
         * @param lambdaResources
         *        The Python-language Lambda functions for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaResources(LambdaResource... lambdaResources);

        /**
         * <p>
         * The Python-language Lambda functions for this job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LambdaResource>.Builder} avoiding the need
         * to create one manually via {@link List<LambdaResource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LambdaResource>.Builder#build()} is called immediately and
         * its result is passed to {@link #lambdaResources(List<LambdaResource>)}.
         * 
         * @param lambdaResources
         *        a consumer that will call methods on {@link List<LambdaResource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lambdaResources(List<LambdaResource>)
         */
        Builder lambdaResources(Consumer<LambdaResource.Builder>... lambdaResources);
    }

    static final class BuilderImpl implements Builder {
        private List<S3Resource> s3Resources = DefaultSdkAutoConstructList.getInstance();

        private List<LambdaResource> lambdaResources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JobResource model) {
            s3Resources(model.s3Resources);
            lambdaResources(model.lambdaResources);
        }

        public final Collection<S3Resource.Builder> getS3Resources() {
            return s3Resources != null ? s3Resources.stream().map(S3Resource::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder s3Resources(Collection<S3Resource> s3Resources) {
            this.s3Resources = S3ResourceListCopier.copy(s3Resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder s3Resources(S3Resource... s3Resources) {
            s3Resources(Arrays.asList(s3Resources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder s3Resources(Consumer<S3Resource.Builder>... s3Resources) {
            s3Resources(Stream.of(s3Resources).map(c -> S3Resource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setS3Resources(Collection<S3Resource.BuilderImpl> s3Resources) {
            this.s3Resources = S3ResourceListCopier.copyFromBuilder(s3Resources);
        }

        public final Collection<LambdaResource.Builder> getLambdaResources() {
            return lambdaResources != null ? lambdaResources.stream().map(LambdaResource::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder lambdaResources(Collection<LambdaResource> lambdaResources) {
            this.lambdaResources = LambdaResourceListCopier.copy(lambdaResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lambdaResources(LambdaResource... lambdaResources) {
            lambdaResources(Arrays.asList(lambdaResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lambdaResources(Consumer<LambdaResource.Builder>... lambdaResources) {
            lambdaResources(Stream.of(lambdaResources).map(c -> LambdaResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setLambdaResources(Collection<LambdaResource.BuilderImpl> lambdaResources) {
            this.lambdaResources = LambdaResourceListCopier.copyFromBuilder(lambdaResources);
        }

        @Override
        public JobResource build() {
            return new JobResource(this);
        }
    }
}
