/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.snowball.transform.NotificationMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Amazon Simple Notification Service (Amazon SNS) notification settings associated with a specific job. The
 * <code>Notification</code> object is returned as a part of the response syntax of the <code>DescribeJob</code> action
 * in the <code>JobMetadata</code> data type.
 * </p>
 * <p>
 * When the notification settings are defined during job creation, you can choose to notify based on a specific set of
 * job states using the <code>JobStatesToNotify</code> array of strings, or you can specify that you want to have Amazon
 * SNS notifications sent out for all job states with <code>NotifyAll</code> set to true.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Notification implements StructuredPojo, ToCopyableBuilder<Notification.Builder, Notification> {
    private final String snsTopicARN;

    private final List<String> jobStatesToNotify;

    private final Boolean notifyAllJobStates;

    private Notification(BuilderImpl builder) {
        this.snsTopicARN = builder.snsTopicARN;
        this.jobStatesToNotify = builder.jobStatesToNotify;
        this.notifyAllJobStates = builder.notifyAllJobStates;
    }

    /**
     * <p>
     * The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon Resource Names
     * (ARNs) for topics by using the <a
     * href="http://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API action.
     * </p>
     * <p>
     * You can subscribe email addresses to an Amazon SNS topic through the AWS Management Console, or by using the <a
     * href="http://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> AWS Simple Notification Service
     * (SNS) API action.
     * </p>
     * 
     * @return The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon
     *         Resource Names (ARNs) for topics by using the <a
     *         href="http://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API
     *         action.</p>
     *         <p>
     *         You can subscribe email addresses to an Amazon SNS topic through the AWS Management Console, or by using
     *         the <a href="http://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> AWS Simple
     *         Notification Service (SNS) API action.
     */
    public String snsTopicARN() {
        return snsTopicARN;
    }

    /**
     * <p>
     * The list of job states that will trigger a notification for this job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The list of job states that will trigger a notification for this job.
     * @see JobState
     */
    public List<String> jobStatesToNotify() {
        return jobStatesToNotify;
    }

    /**
     * <p>
     * Any change in job state will trigger a notification for this job.
     * </p>
     * 
     * @return Any change in job state will trigger a notification for this job.
     */
    public Boolean notifyAllJobStates() {
        return notifyAllJobStates;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((snsTopicARN() == null) ? 0 : snsTopicARN().hashCode());
        hashCode = 31 * hashCode + ((jobStatesToNotify() == null) ? 0 : jobStatesToNotify().hashCode());
        hashCode = 31 * hashCode + ((notifyAllJobStates() == null) ? 0 : notifyAllJobStates().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Notification)) {
            return false;
        }
        Notification other = (Notification) obj;
        if (other.snsTopicARN() == null ^ this.snsTopicARN() == null) {
            return false;
        }
        if (other.snsTopicARN() != null && !other.snsTopicARN().equals(this.snsTopicARN())) {
            return false;
        }
        if (other.jobStatesToNotify() == null ^ this.jobStatesToNotify() == null) {
            return false;
        }
        if (other.jobStatesToNotify() != null && !other.jobStatesToNotify().equals(this.jobStatesToNotify())) {
            return false;
        }
        if (other.notifyAllJobStates() == null ^ this.notifyAllJobStates() == null) {
            return false;
        }
        if (other.notifyAllJobStates() != null && !other.notifyAllJobStates().equals(this.notifyAllJobStates())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (snsTopicARN() != null) {
            sb.append("SnsTopicARN: ").append(snsTopicARN()).append(",");
        }
        if (jobStatesToNotify() != null) {
            sb.append("JobStatesToNotify: ").append(jobStatesToNotify()).append(",");
        }
        if (notifyAllJobStates() != null) {
            sb.append("NotifyAllJobStates: ").append(notifyAllJobStates()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        NotificationMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Notification> {
        /**
         * <p>
         * The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon Resource
         * Names (ARNs) for topics by using the <a
         * href="http://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API action.
         * </p>
         * <p>
         * You can subscribe email addresses to an Amazon SNS topic through the AWS Management Console, or by using the
         * <a href="http://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> AWS Simple Notification
         * Service (SNS) API action.
         * </p>
         * 
         * @param snsTopicARN
         *        The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon
         *        Resource Names (ARNs) for topics by using the <a
         *        href="http://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API
         *        action.</p>
         *        <p>
         *        You can subscribe email addresses to an Amazon SNS topic through the AWS Management Console, or by
         *        using the <a href="http://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> AWS
         *        Simple Notification Service (SNS) API action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicARN(String snsTopicARN);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @see JobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobState
         */
        Builder jobStatesToNotify(Collection<String> jobStatesToNotify);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @see JobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobState
         */
        Builder jobStatesToNotify(String... jobStatesToNotify);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @see JobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobState
         */
        Builder jobStatesToNotify(JobState... jobStatesToNotify);

        /**
         * <p>
         * Any change in job state will trigger a notification for this job.
         * </p>
         * 
         * @param notifyAllJobStates
         *        Any change in job state will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifyAllJobStates(Boolean notifyAllJobStates);
    }

    private static final class BuilderImpl implements Builder {
        private String snsTopicARN;

        private List<String> jobStatesToNotify;

        private Boolean notifyAllJobStates;

        private BuilderImpl() {
        }

        private BuilderImpl(Notification model) {
            setSnsTopicARN(model.snsTopicARN);
            setJobStatesToNotify(model.jobStatesToNotify);
            setNotifyAllJobStates(model.notifyAllJobStates);
        }

        public final String getSnsTopicARN() {
            return snsTopicARN;
        }

        @Override
        public final Builder snsTopicARN(String snsTopicARN) {
            this.snsTopicARN = snsTopicARN;
            return this;
        }

        public final void setSnsTopicARN(String snsTopicARN) {
            this.snsTopicARN = snsTopicARN;
        }

        public final Collection<String> getJobStatesToNotify() {
            return jobStatesToNotify;
        }

        @Override
        public final Builder jobStatesToNotify(Collection<String> jobStatesToNotify) {
            this.jobStatesToNotify = JobStateListCopier.copy(jobStatesToNotify);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStatesToNotify(String... jobStatesToNotify) {
            jobStatesToNotify(Arrays.asList(jobStatesToNotify));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStatesToNotify(JobState... jobStatesToNotify) {
            jobStatesToNotify(Arrays.asList(jobStatesToNotify).stream().map(Object::toString).collect(Collectors.toList()));
            return this;
        }

        public final void setJobStatesToNotify(Collection<String> jobStatesToNotify) {
            this.jobStatesToNotify = JobStateListCopier.copy(jobStatesToNotify);
        }

        public final Boolean getNotifyAllJobStates() {
            return notifyAllJobStates;
        }

        @Override
        public final Builder notifyAllJobStates(Boolean notifyAllJobStates) {
            this.notifyAllJobStates = notifyAllJobStates;
            return this;
        }

        public final void setNotifyAllJobStates(Boolean notifyAllJobStates) {
            this.notifyAllJobStates = notifyAllJobStates;
        }

        @Override
        public Notification build() {
            return new Notification(this);
        }
    }
}
