/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.snowball.transform.ClusterMetadataMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains metadata about a specific cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ClusterMetadata implements StructuredPojo, ToCopyableBuilder<ClusterMetadata.Builder, ClusterMetadata> {
    private final String clusterId;

    private final String description;

    private final String kmsKeyARN;

    private final String roleARN;

    private final String clusterState;

    private final String jobType;

    private final String snowballType;

    private final Instant creationDate;

    private final JobResource resources;

    private final String addressId;

    private final String shippingOption;

    private final Notification notification;

    private final String forwardingAddressId;

    private ClusterMetadata(BuilderImpl builder) {
        this.clusterId = builder.clusterId;
        this.description = builder.description;
        this.kmsKeyARN = builder.kmsKeyARN;
        this.roleARN = builder.roleARN;
        this.clusterState = builder.clusterState;
        this.jobType = builder.jobType;
        this.snowballType = builder.snowballType;
        this.creationDate = builder.creationDate;
        this.resources = builder.resources;
        this.addressId = builder.addressId;
        this.shippingOption = builder.shippingOption;
        this.notification = builder.notification;
        this.forwardingAddressId = builder.forwardingAddressId;
    }

    /**
     * <p>
     * The automatically generated ID for a cluster.
     * </p>
     * 
     * @return The automatically generated ID for a cluster.
     */
    public String clusterId() {
        return clusterId;
    }

    /**
     * <p>
     * The optional description of the cluster.
     * </p>
     * 
     * @return The optional description of the cluster.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The <code>KmsKeyARN</code> Amazon Resource Name (ARN) associated with this cluster. This ARN was created using
     * the <a href="http://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html">CreateKey</a> API action in
     * AWS Key Management Service (AWS KMS).
     * </p>
     * 
     * @return The <code>KmsKeyARN</code> Amazon Resource Name (ARN) associated with this cluster. This ARN was created
     *         using the <a href="http://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html">CreateKey</a>
     *         API action in AWS Key Management Service (AWS KMS).
     */
    public String kmsKeyARN() {
        return kmsKeyARN;
    }

    /**
     * <p>
     * The role ARN associated with this cluster. This ARN was created using the <a
     * href="http://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a> API action in AWS
     * Identity and Access Management (IAM).
     * </p>
     * 
     * @return The role ARN associated with this cluster. This ARN was created using the <a
     *         href="http://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a> API action
     *         in AWS Identity and Access Management (IAM).
     */
    public String roleARN() {
        return roleARN;
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #clusterState} will
     * return {@link ClusterState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #clusterStateString}.
     * </p>
     * 
     * @return The current status of the cluster.
     * @see ClusterState
     */
    public ClusterState clusterState() {
        return ClusterState.fromValue(clusterState);
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #clusterState} will
     * return {@link ClusterState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #clusterStateString}.
     * </p>
     * 
     * @return The current status of the cluster.
     * @see ClusterState
     */
    public String clusterStateString() {
        return clusterState;
    }

    /**
     * <p>
     * The type of job for this cluster. Currently, the only job type supported for clusters is <code>LOCAL_USE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobType} will
     * return {@link JobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobTypeString}.
     * </p>
     * 
     * @return The type of job for this cluster. Currently, the only job type supported for clusters is
     *         <code>LOCAL_USE</code>.
     * @see JobType
     */
    public JobType jobType() {
        return JobType.fromValue(jobType);
    }

    /**
     * <p>
     * The type of job for this cluster. Currently, the only job type supported for clusters is <code>LOCAL_USE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobType} will
     * return {@link JobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobTypeString}.
     * </p>
     * 
     * @return The type of job for this cluster. Currently, the only job type supported for clusters is
     *         <code>LOCAL_USE</code>.
     * @see JobType
     */
    public String jobTypeString() {
        return jobType;
    }

    /**
     * <p>
     * The type of AWS Snowball appliance to use for this cluster. Currently, the only supported appliance type for
     * cluster jobs is <code>EDGE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #snowballType} will
     * return {@link SnowballType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #snowballTypeString}.
     * </p>
     * 
     * @return The type of AWS Snowball appliance to use for this cluster. Currently, the only supported appliance type
     *         for cluster jobs is <code>EDGE</code>.
     * @see SnowballType
     */
    public SnowballType snowballType() {
        return SnowballType.fromValue(snowballType);
    }

    /**
     * <p>
     * The type of AWS Snowball appliance to use for this cluster. Currently, the only supported appliance type for
     * cluster jobs is <code>EDGE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #snowballType} will
     * return {@link SnowballType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #snowballTypeString}.
     * </p>
     * 
     * @return The type of AWS Snowball appliance to use for this cluster. Currently, the only supported appliance type
     *         for cluster jobs is <code>EDGE</code>.
     * @see SnowballType
     */
    public String snowballTypeString() {
        return snowballType;
    }

    /**
     * <p>
     * The creation date for this cluster.
     * </p>
     * 
     * @return The creation date for this cluster.
     */
    public Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
     * <a>LambdaResource</a> objects.
     * </p>
     * 
     * @return The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
     *         <a>LambdaResource</a> objects.
     */
    public JobResource resources() {
        return resources;
    }

    /**
     * <p>
     * The automatically generated ID for a specific address.
     * </p>
     * 
     * @return The automatically generated ID for a specific address.
     */
    public String addressId() {
        return addressId;
    }

    /**
     * <p>
     * The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each Snowball
     * Edge appliance, rather it represents how quickly each appliance moves to its destination while in transit.
     * Regional shipping speeds are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * In Australia, you have access to express shipping. Typically, appliances shipped express are delivered in about a
     * day.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped express are
     * delivered in about a day. In addition, most countries in the EU have access to standard shipping, which typically
     * takes less than a week, one way.
     * </p>
     * </li>
     * <li>
     * <p>
     * In India, Snowball Edges are delivered in one to seven days.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the US, you have access to one-day shipping and two-day shipping.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shippingOption}
     * will return {@link ShippingOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #shippingOptionString}.
     * </p>
     * 
     * @return The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
     *         Snowball Edge appliance, rather it represents how quickly each appliance moves to its destination while
     *         in transit. Regional shipping speeds are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         In Australia, you have access to express shipping. Typically, appliances shipped express are delivered in
     *         about a day.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped
     *         express are delivered in about a day. In addition, most countries in the EU have access to standard
     *         shipping, which typically takes less than a week, one way.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In India, Snowball Edges are delivered in one to seven days.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the US, you have access to one-day shipping and two-day shipping.
     *         </p>
     *         </li>
     * @see ShippingOption
     */
    public ShippingOption shippingOption() {
        return ShippingOption.fromValue(shippingOption);
    }

    /**
     * <p>
     * The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each Snowball
     * Edge appliance, rather it represents how quickly each appliance moves to its destination while in transit.
     * Regional shipping speeds are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * In Australia, you have access to express shipping. Typically, appliances shipped express are delivered in about a
     * day.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped express are
     * delivered in about a day. In addition, most countries in the EU have access to standard shipping, which typically
     * takes less than a week, one way.
     * </p>
     * </li>
     * <li>
     * <p>
     * In India, Snowball Edges are delivered in one to seven days.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the US, you have access to one-day shipping and two-day shipping.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shippingOption}
     * will return {@link ShippingOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #shippingOptionString}.
     * </p>
     * 
     * @return The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
     *         Snowball Edge appliance, rather it represents how quickly each appliance moves to its destination while
     *         in transit. Regional shipping speeds are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         In Australia, you have access to express shipping. Typically, appliances shipped express are delivered in
     *         about a day.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped
     *         express are delivered in about a day. In addition, most countries in the EU have access to standard
     *         shipping, which typically takes less than a week, one way.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In India, Snowball Edges are delivered in one to seven days.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the US, you have access to one-day shipping and two-day shipping.
     *         </p>
     *         </li>
     * @see ShippingOption
     */
    public String shippingOptionString() {
        return shippingOption;
    }

    /**
     * <p>
     * The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
     * </p>
     * 
     * @return The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
     */
    public Notification notification() {
        return notification;
    }

    /**
     * <p>
     * The ID of the address that you want a cluster shipped to, after it will be shipped to its primary address. This
     * field is not supported in most regions.
     * </p>
     * 
     * @return The ID of the address that you want a cluster shipped to, after it will be shipped to its primary
     *         address. This field is not supported in most regions.
     */
    public String forwardingAddressId() {
        return forwardingAddressId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clusterId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyARN());
        hashCode = 31 * hashCode + Objects.hashCode(roleARN());
        hashCode = 31 * hashCode + Objects.hashCode(clusterStateString());
        hashCode = 31 * hashCode + Objects.hashCode(jobTypeString());
        hashCode = 31 * hashCode + Objects.hashCode(snowballTypeString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        hashCode = 31 * hashCode + Objects.hashCode(addressId());
        hashCode = 31 * hashCode + Objects.hashCode(shippingOptionString());
        hashCode = 31 * hashCode + Objects.hashCode(notification());
        hashCode = 31 * hashCode + Objects.hashCode(forwardingAddressId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClusterMetadata)) {
            return false;
        }
        ClusterMetadata other = (ClusterMetadata) obj;
        return Objects.equals(clusterId(), other.clusterId()) && Objects.equals(description(), other.description())
                && Objects.equals(kmsKeyARN(), other.kmsKeyARN()) && Objects.equals(roleARN(), other.roleARN())
                && Objects.equals(clusterStateString(), other.clusterStateString())
                && Objects.equals(jobTypeString(), other.jobTypeString())
                && Objects.equals(snowballTypeString(), other.snowballTypeString())
                && Objects.equals(creationDate(), other.creationDate()) && Objects.equals(resources(), other.resources())
                && Objects.equals(addressId(), other.addressId())
                && Objects.equals(shippingOptionString(), other.shippingOptionString())
                && Objects.equals(notification(), other.notification())
                && Objects.equals(forwardingAddressId(), other.forwardingAddressId());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (clusterId() != null) {
            sb.append("ClusterId: ").append(clusterId()).append(",");
        }
        if (description() != null) {
            sb.append("Description: ").append(description()).append(",");
        }
        if (kmsKeyARN() != null) {
            sb.append("KmsKeyARN: ").append(kmsKeyARN()).append(",");
        }
        if (roleARN() != null) {
            sb.append("RoleARN: ").append(roleARN()).append(",");
        }
        if (clusterStateString() != null) {
            sb.append("ClusterState: ").append(clusterStateString()).append(",");
        }
        if (jobTypeString() != null) {
            sb.append("JobType: ").append(jobTypeString()).append(",");
        }
        if (snowballTypeString() != null) {
            sb.append("SnowballType: ").append(snowballTypeString()).append(",");
        }
        if (creationDate() != null) {
            sb.append("CreationDate: ").append(creationDate()).append(",");
        }
        if (resources() != null) {
            sb.append("Resources: ").append(resources()).append(",");
        }
        if (addressId() != null) {
            sb.append("AddressId: ").append(addressId()).append(",");
        }
        if (shippingOptionString() != null) {
            sb.append("ShippingOption: ").append(shippingOptionString()).append(",");
        }
        if (notification() != null) {
            sb.append("Notification: ").append(notification()).append(",");
        }
        if (forwardingAddressId() != null) {
            sb.append("ForwardingAddressId: ").append(forwardingAddressId()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClusterId":
            return Optional.of(clazz.cast(clusterId()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "KmsKeyARN":
            return Optional.of(clazz.cast(kmsKeyARN()));
        case "RoleARN":
            return Optional.of(clazz.cast(roleARN()));
        case "ClusterState":
            return Optional.of(clazz.cast(clusterStateString()));
        case "JobType":
            return Optional.of(clazz.cast(jobTypeString()));
        case "SnowballType":
            return Optional.of(clazz.cast(snowballTypeString()));
        case "CreationDate":
            return Optional.of(clazz.cast(creationDate()));
        case "Resources":
            return Optional.of(clazz.cast(resources()));
        case "AddressId":
            return Optional.of(clazz.cast(addressId()));
        case "ShippingOption":
            return Optional.of(clazz.cast(shippingOptionString()));
        case "Notification":
            return Optional.of(clazz.cast(notification()));
        case "ForwardingAddressId":
            return Optional.of(clazz.cast(forwardingAddressId()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ClusterMetadataMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ClusterMetadata> {
        /**
         * <p>
         * The automatically generated ID for a cluster.
         * </p>
         * 
         * @param clusterId
         *        The automatically generated ID for a cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterId(String clusterId);

        /**
         * <p>
         * The optional description of the cluster.
         * </p>
         * 
         * @param description
         *        The optional description of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The <code>KmsKeyARN</code> Amazon Resource Name (ARN) associated with this cluster. This ARN was created
         * using the <a href="http://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html">CreateKey</a> API
         * action in AWS Key Management Service (AWS KMS).
         * </p>
         * 
         * @param kmsKeyARN
         *        The <code>KmsKeyARN</code> Amazon Resource Name (ARN) associated with this cluster. This ARN was
         *        created using the <a
         *        href="http://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html">CreateKey</a> API action
         *        in AWS Key Management Service (AWS KMS).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyARN(String kmsKeyARN);

        /**
         * <p>
         * The role ARN associated with this cluster. This ARN was created using the <a
         * href="http://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a> API action in
         * AWS Identity and Access Management (IAM).
         * </p>
         * 
         * @param roleARN
         *        The role ARN associated with this cluster. This ARN was created using the <a
         *        href="http://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a> API
         *        action in AWS Identity and Access Management (IAM).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleARN(String roleARN);

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param clusterState
         *        The current status of the cluster.
         * @see ClusterState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterState
         */
        Builder clusterState(String clusterState);

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param clusterState
         *        The current status of the cluster.
         * @see ClusterState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterState
         */
        Builder clusterState(ClusterState clusterState);

        /**
         * <p>
         * The type of job for this cluster. Currently, the only job type supported for clusters is
         * <code>LOCAL_USE</code>.
         * </p>
         * 
         * @param jobType
         *        The type of job for this cluster. Currently, the only job type supported for clusters is
         *        <code>LOCAL_USE</code>.
         * @see JobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobType
         */
        Builder jobType(String jobType);

        /**
         * <p>
         * The type of job for this cluster. Currently, the only job type supported for clusters is
         * <code>LOCAL_USE</code>.
         * </p>
         * 
         * @param jobType
         *        The type of job for this cluster. Currently, the only job type supported for clusters is
         *        <code>LOCAL_USE</code>.
         * @see JobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobType
         */
        Builder jobType(JobType jobType);

        /**
         * <p>
         * The type of AWS Snowball appliance to use for this cluster. Currently, the only supported appliance type for
         * cluster jobs is <code>EDGE</code>.
         * </p>
         * 
         * @param snowballType
         *        The type of AWS Snowball appliance to use for this cluster. Currently, the only supported appliance
         *        type for cluster jobs is <code>EDGE</code>.
         * @see SnowballType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnowballType
         */
        Builder snowballType(String snowballType);

        /**
         * <p>
         * The type of AWS Snowball appliance to use for this cluster. Currently, the only supported appliance type for
         * cluster jobs is <code>EDGE</code>.
         * </p>
         * 
         * @param snowballType
         *        The type of AWS Snowball appliance to use for this cluster. Currently, the only supported appliance
         *        type for cluster jobs is <code>EDGE</code>.
         * @see SnowballType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnowballType
         */
        Builder snowballType(SnowballType snowballType);

        /**
         * <p>
         * The creation date for this cluster.
         * </p>
         * 
         * @param creationDate
         *        The creation date for this cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
         * <a>LambdaResource</a> objects.
         * </p>
         * 
         * @param resources
         *        The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
         *        <a>LambdaResource</a> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(JobResource resources);

        /**
         * <p>
         * The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
         * <a>LambdaResource</a> objects.
         * </p>
         * This is a convenience that creates an instance of the {@link JobResource.Builder} avoiding the need to create
         * one manually via {@link JobResource#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobResource.Builder#build()} is called immediately and its result
         * is passed to {@link #resources(JobResource)}.
         * 
         * @param resources
         *        a consumer that will call methods on {@link JobResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(JobResource)
         */
        default Builder resources(Consumer<JobResource.Builder> resources) {
            return resources(JobResource.builder().apply(resources).build());
        }

        /**
         * <p>
         * The automatically generated ID for a specific address.
         * </p>
         * 
         * @param addressId
         *        The automatically generated ID for a specific address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addressId(String addressId);

        /**
         * <p>
         * The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
         * Snowball Edge appliance, rather it represents how quickly each appliance moves to its destination while in
         * transit. Regional shipping speeds are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * In Australia, you have access to express shipping. Typically, appliances shipped express are delivered in
         * about a day.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped express
         * are delivered in about a day. In addition, most countries in the EU have access to standard shipping, which
         * typically takes less than a week, one way.
         * </p>
         * </li>
         * <li>
         * <p>
         * In India, Snowball Edges are delivered in one to seven days.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the US, you have access to one-day shipping and two-day shipping.
         * </p>
         * </li>
         * </ul>
         * 
         * @param shippingOption
         *        The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
         *        Snowball Edge appliance, rather it represents how quickly each appliance moves to its destination
         *        while in transit. Regional shipping speeds are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        In Australia, you have access to express shipping. Typically, appliances shipped express are delivered
         *        in about a day.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped
         *        express are delivered in about a day. In addition, most countries in the EU have access to standard
         *        shipping, which typically takes less than a week, one way.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In India, Snowball Edges are delivered in one to seven days.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the US, you have access to one-day shipping and two-day shipping.
         *        </p>
         *        </li>
         * @see ShippingOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShippingOption
         */
        Builder shippingOption(String shippingOption);

        /**
         * <p>
         * The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
         * Snowball Edge appliance, rather it represents how quickly each appliance moves to its destination while in
         * transit. Regional shipping speeds are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * In Australia, you have access to express shipping. Typically, appliances shipped express are delivered in
         * about a day.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped express
         * are delivered in about a day. In addition, most countries in the EU have access to standard shipping, which
         * typically takes less than a week, one way.
         * </p>
         * </li>
         * <li>
         * <p>
         * In India, Snowball Edges are delivered in one to seven days.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the US, you have access to one-day shipping and two-day shipping.
         * </p>
         * </li>
         * </ul>
         * 
         * @param shippingOption
         *        The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
         *        Snowball Edge appliance, rather it represents how quickly each appliance moves to its destination
         *        while in transit. Regional shipping speeds are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        In Australia, you have access to express shipping. Typically, appliances shipped express are delivered
         *        in about a day.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped
         *        express are delivered in about a day. In addition, most countries in the EU have access to standard
         *        shipping, which typically takes less than a week, one way.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In India, Snowball Edges are delivered in one to seven days.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the US, you have access to one-day shipping and two-day shipping.
         *        </p>
         *        </li>
         * @see ShippingOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShippingOption
         */
        Builder shippingOption(ShippingOption shippingOption);

        /**
         * <p>
         * The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
         * </p>
         * 
         * @param notification
         *        The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notification(Notification notification);

        /**
         * <p>
         * The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link Notification.Builder} avoiding the need to
         * create one manually via {@link Notification#builder()}.
         *
         * When the {@link Consumer} completes, {@link Notification.Builder#build()} is called immediately and its
         * result is passed to {@link #notification(Notification)}.
         * 
         * @param notification
         *        a consumer that will call methods on {@link Notification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notification(Notification)
         */
        default Builder notification(Consumer<Notification.Builder> notification) {
            return notification(Notification.builder().apply(notification).build());
        }

        /**
         * <p>
         * The ID of the address that you want a cluster shipped to, after it will be shipped to its primary address.
         * This field is not supported in most regions.
         * </p>
         * 
         * @param forwardingAddressId
         *        The ID of the address that you want a cluster shipped to, after it will be shipped to its primary
         *        address. This field is not supported in most regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardingAddressId(String forwardingAddressId);
    }

    static final class BuilderImpl implements Builder {
        private String clusterId;

        private String description;

        private String kmsKeyARN;

        private String roleARN;

        private String clusterState;

        private String jobType;

        private String snowballType;

        private Instant creationDate;

        private JobResource resources;

        private String addressId;

        private String shippingOption;

        private Notification notification;

        private String forwardingAddressId;

        private BuilderImpl() {
        }

        private BuilderImpl(ClusterMetadata model) {
            clusterId(model.clusterId);
            description(model.description);
            kmsKeyARN(model.kmsKeyARN);
            roleARN(model.roleARN);
            clusterState(model.clusterState);
            jobType(model.jobType);
            snowballType(model.snowballType);
            creationDate(model.creationDate);
            resources(model.resources);
            addressId(model.addressId);
            shippingOption(model.shippingOption);
            notification(model.notification);
            forwardingAddressId(model.forwardingAddressId);
        }

        public final String getClusterId() {
            return clusterId;
        }

        @Override
        public final Builder clusterId(String clusterId) {
            this.clusterId = clusterId;
            return this;
        }

        public final void setClusterId(String clusterId) {
            this.clusterId = clusterId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getKmsKeyARN() {
            return kmsKeyARN;
        }

        @Override
        public final Builder kmsKeyARN(String kmsKeyARN) {
            this.kmsKeyARN = kmsKeyARN;
            return this;
        }

        public final void setKmsKeyARN(String kmsKeyARN) {
            this.kmsKeyARN = kmsKeyARN;
        }

        public final String getRoleARN() {
            return roleARN;
        }

        @Override
        public final Builder roleARN(String roleARN) {
            this.roleARN = roleARN;
            return this;
        }

        public final void setRoleARN(String roleARN) {
            this.roleARN = roleARN;
        }

        public final String getClusterState() {
            return clusterState;
        }

        @Override
        public final Builder clusterState(String clusterState) {
            this.clusterState = clusterState;
            return this;
        }

        @Override
        public final Builder clusterState(ClusterState clusterState) {
            this.clusterState(clusterState.toString());
            return this;
        }

        public final void setClusterState(String clusterState) {
            this.clusterState = clusterState;
        }

        public final String getJobType() {
            return jobType;
        }

        @Override
        public final Builder jobType(String jobType) {
            this.jobType = jobType;
            return this;
        }

        @Override
        public final Builder jobType(JobType jobType) {
            this.jobType(jobType.toString());
            return this;
        }

        public final void setJobType(String jobType) {
            this.jobType = jobType;
        }

        public final String getSnowballType() {
            return snowballType;
        }

        @Override
        public final Builder snowballType(String snowballType) {
            this.snowballType = snowballType;
            return this;
        }

        @Override
        public final Builder snowballType(SnowballType snowballType) {
            this.snowballType(snowballType.toString());
            return this;
        }

        public final void setSnowballType(String snowballType) {
            this.snowballType = snowballType;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        public final JobResource.Builder getResources() {
            return resources != null ? resources.toBuilder() : null;
        }

        @Override
        public final Builder resources(JobResource resources) {
            this.resources = resources;
            return this;
        }

        public final void setResources(JobResource.BuilderImpl resources) {
            this.resources = resources != null ? resources.build() : null;
        }

        public final String getAddressId() {
            return addressId;
        }

        @Override
        public final Builder addressId(String addressId) {
            this.addressId = addressId;
            return this;
        }

        public final void setAddressId(String addressId) {
            this.addressId = addressId;
        }

        public final String getShippingOption() {
            return shippingOption;
        }

        @Override
        public final Builder shippingOption(String shippingOption) {
            this.shippingOption = shippingOption;
            return this;
        }

        @Override
        public final Builder shippingOption(ShippingOption shippingOption) {
            this.shippingOption(shippingOption.toString());
            return this;
        }

        public final void setShippingOption(String shippingOption) {
            this.shippingOption = shippingOption;
        }

        public final Notification.Builder getNotification() {
            return notification != null ? notification.toBuilder() : null;
        }

        @Override
        public final Builder notification(Notification notification) {
            this.notification = notification;
            return this;
        }

        public final void setNotification(Notification.BuilderImpl notification) {
            this.notification = notification != null ? notification.build() : null;
        }

        public final String getForwardingAddressId() {
            return forwardingAddressId;
        }

        @Override
        public final Builder forwardingAddressId(String forwardingAddressId) {
            this.forwardingAddressId = forwardingAddressId;
            return this;
        }

        public final void setForwardingAddressId(String forwardingAddressId) {
            this.forwardingAddressId = forwardingAddressId;
        }

        @Override
        public ClusterMetadata build() {
            return new ClusterMetadata(this);
        }
    }
}
