/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball;

import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.client.ClientExecutionParams;
import software.amazon.awssdk.core.client.ClientHandler;
import software.amazon.awssdk.core.client.SdkClientHandler;
import software.amazon.awssdk.core.config.ClientConfiguration;
import software.amazon.awssdk.core.config.SyncClientConfiguration;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory;
import software.amazon.awssdk.services.snowball.model.CancelClusterRequest;
import software.amazon.awssdk.services.snowball.model.CancelClusterResponse;
import software.amazon.awssdk.services.snowball.model.CancelJobRequest;
import software.amazon.awssdk.services.snowball.model.CancelJobResponse;
import software.amazon.awssdk.services.snowball.model.ClusterLimitExceededException;
import software.amazon.awssdk.services.snowball.model.CreateAddressRequest;
import software.amazon.awssdk.services.snowball.model.CreateAddressResponse;
import software.amazon.awssdk.services.snowball.model.CreateClusterRequest;
import software.amazon.awssdk.services.snowball.model.CreateClusterResponse;
import software.amazon.awssdk.services.snowball.model.CreateJobRequest;
import software.amazon.awssdk.services.snowball.model.CreateJobResponse;
import software.amazon.awssdk.services.snowball.model.DescribeAddressRequest;
import software.amazon.awssdk.services.snowball.model.DescribeAddressResponse;
import software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest;
import software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse;
import software.amazon.awssdk.services.snowball.model.DescribeClusterRequest;
import software.amazon.awssdk.services.snowball.model.DescribeClusterResponse;
import software.amazon.awssdk.services.snowball.model.DescribeJobRequest;
import software.amazon.awssdk.services.snowball.model.DescribeJobResponse;
import software.amazon.awssdk.services.snowball.model.GetJobManifestRequest;
import software.amazon.awssdk.services.snowball.model.GetJobManifestResponse;
import software.amazon.awssdk.services.snowball.model.GetJobUnlockCodeRequest;
import software.amazon.awssdk.services.snowball.model.GetJobUnlockCodeResponse;
import software.amazon.awssdk.services.snowball.model.GetSnowballUsageRequest;
import software.amazon.awssdk.services.snowball.model.GetSnowballUsageResponse;
import software.amazon.awssdk.services.snowball.model.InvalidAddressException;
import software.amazon.awssdk.services.snowball.model.InvalidInputCombinationException;
import software.amazon.awssdk.services.snowball.model.InvalidJobStateException;
import software.amazon.awssdk.services.snowball.model.InvalidNextTokenException;
import software.amazon.awssdk.services.snowball.model.InvalidResourceException;
import software.amazon.awssdk.services.snowball.model.KMSRequestFailedException;
import software.amazon.awssdk.services.snowball.model.ListClusterJobsRequest;
import software.amazon.awssdk.services.snowball.model.ListClusterJobsResponse;
import software.amazon.awssdk.services.snowball.model.ListClustersRequest;
import software.amazon.awssdk.services.snowball.model.ListClustersResponse;
import software.amazon.awssdk.services.snowball.model.ListJobsRequest;
import software.amazon.awssdk.services.snowball.model.ListJobsResponse;
import software.amazon.awssdk.services.snowball.model.SnowballException;
import software.amazon.awssdk.services.snowball.model.UnsupportedAddressException;
import software.amazon.awssdk.services.snowball.model.UpdateClusterRequest;
import software.amazon.awssdk.services.snowball.model.UpdateClusterResponse;
import software.amazon.awssdk.services.snowball.model.UpdateJobRequest;
import software.amazon.awssdk.services.snowball.model.UpdateJobResponse;
import software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPaginator;
import software.amazon.awssdk.services.snowball.paginators.ListJobsPaginator;
import software.amazon.awssdk.services.snowball.transform.CancelClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CancelClusterResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CancelJobResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateAddressRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateAddressResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateClusterResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateJobResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressesRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressesResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeClusterResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeJobResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobManifestRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobManifestResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobUnlockCodeRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobUnlockCodeResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.GetSnowballUsageRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetSnowballUsageResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClusterJobsRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClusterJobsResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClustersResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListJobsResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateClusterResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateJobResponseUnmarshaller;

/**
 * Internal implementation of {@link SnowballClient}.
 *
 * @see SnowballClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnowballClient implements SnowballClient {
    private final ClientHandler clientHandler;

    private final SdkJsonProtocolFactory protocolFactory;

    private final ClientConfiguration clientConfiguration;

    protected DefaultSnowballClient(SyncClientConfiguration clientConfiguration) {
        this.clientHandler = new SdkClientHandler(clientConfiguration, null);
        this.protocolFactory = init();
        this.clientConfiguration = clientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a cluster job. You can only cancel a cluster job while it's in the <code>AwaitingQuorum</code> status.
     * You'll have at least an hour after creating a cluster job to cancel it.
     * </p>
     *
     * @param cancelClusterRequest
     * @return Result of the CancelCluster operation returned by the service.
     * @throws KMSRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CancelCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CancelCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelClusterResponse cancelCluster(CancelClusterRequest cancelClusterRequest) throws KMSRequestFailedException,
            InvalidJobStateException, InvalidResourceException, SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<CancelClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CancelClusterResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CancelClusterRequest, CancelClusterResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(cancelClusterRequest).withMarshaller(new CancelClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Cancels the specified job. You can only cancel a job before its <code>JobState</code> value changes to
     * <code>PreparingAppliance</code>. Requesting the <code>ListJobs</code> or <code>DescribeJob</code> action will
     * return a job's <code>JobState</code> as part of the response element data returned.
     * </p>
     *
     * @param cancelJobRequest
     * @return Result of the CancelJob operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws KMSRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CancelJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelJobResponse cancelJob(CancelJobRequest cancelJobRequest) throws InvalidResourceException,
            InvalidJobStateException, KMSRequestFailedException, SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CancelJobResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest)
                .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an address for a Snowball to be shipped to. In most regions, addresses are validated at the time of
     * creation. The address you provide must be located within the serviceable area of your region. If the address is
     * invalid or unsupported, then an exception is thrown.
     * </p>
     *
     * @param createAddressRequest
     * @return Result of the CreateAddress operation returned by the service.
     * @throws InvalidAddressException
     *         The address provided was invalid. Check the address with your region's carrier, and try again.
     * @throws UnsupportedAddressException
     *         The address is either outside the serviceable area for your region, or an error occurred. Check the
     *         address with your region's carrier and try again. If the issue persists, contact AWS Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CreateAddress
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAddressResponse createAddress(CreateAddressRequest createAddressRequest) throws InvalidAddressException,
            UnsupportedAddressException, SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<CreateAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateAddressResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateAddressRequest, CreateAddressResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createAddressRequest).withMarshaller(new CreateAddressRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an empty cluster. Each cluster supports five nodes. You use the <a>CreateJob</a> action separately to
     * create the jobs for each of these nodes. The cluster does not ship until these five node jobs have been created.
     * </p>
     *
     * @param createClusterRequest
     * @return Result of the CreateCluster operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws KMSRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidInputCombinationException
     *         Job or cluster creation failed. One ore more inputs were invalid. Confirm that the
     *         <a>CreateClusterRequest&#36SnowballType</a> value supports your <a>CreateJobRequest&#36JobType</a>, and
     *         try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CreateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws InvalidResourceException,
            KMSRequestFailedException, InvalidInputCombinationException, SdkServiceException, SdkClientException,
            SnowballException {

        HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateClusterResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createClusterRequest).withMarshaller(new CreateClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a job to the other job attributes are inherited from the cluster.
     * </p>
     *
     * @param createJobRequest
     * @return Result of the CreateJob operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws KMSRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidInputCombinationException
     *         Job or cluster creation failed. One ore more inputs were invalid. Confirm that the
     *         <a>CreateClusterRequest&#36SnowballType</a> value supports your <a>CreateJobRequest&#36JobType</a>, and
     *         try again.
     * @throws ClusterLimitExceededException
     *         Job creation failed. Currently, clusters support five nodes. If you have less than five nodes for your
     *         cluster and you have more nodes to create for this cluster, try again and create jobs until your cluster
     *         has exactly five notes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CreateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateJobResponse createJob(CreateJobRequest createJobRequest) throws InvalidResourceException,
            KMSRequestFailedException, InvalidInputCombinationException, ClusterLimitExceededException, SdkServiceException,
            SdkClientException, SnowballException {

        HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateJobResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(createJobRequest)
                .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Takes an <code>AddressId</code> and returns specific details about that address in the form of an
     * <code>Address</code> object.
     * </p>
     *
     * @param describeAddressRequest
     * @return Result of the DescribeAddress operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeAddress
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAddressResponse describeAddress(DescribeAddressRequest describeAddressRequest)
            throws InvalidResourceException, SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<DescribeAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeAddressResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeAddressRequest, DescribeAddressResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeAddressRequest).withMarshaller(new DescribeAddressRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a specified number of <code>ADDRESS</code> objects. Calling this API in one of the US regions will return
     * addresses from the list of all addresses associated with this account in all US regions.
     * </p>
     *
     * @param describeAddressesRequest
     * @return Result of the DescribeAddresses operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddresses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAddressesResponse describeAddresses(DescribeAddressesRequest describeAddressesRequest)
            throws InvalidResourceException, InvalidNextTokenException, SdkServiceException, SdkClientException,
            SnowballException {

        HttpResponseHandler<DescribeAddressesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeAddressesResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeAddressesRequest, DescribeAddressesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeAddressesRequest).withMarshaller(new DescribeAddressesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a specified number of <code>ADDRESS</code> objects. Calling this API in one of the US regions will return
     * addresses from the list of all addresses associated with this account in all US regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAddresses(software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPaginator responses = client.describeAddressesIterable(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPaginator responses = client
     *             .describeAddressesIterable(request);
     *     for (software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPaginator responses = client.describeAddressesIterable(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAddresses(software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAddressesRequest
     * @return Result of the DescribeAddresses operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddresses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAddressesPaginator describeAddressesIterable(DescribeAddressesRequest describeAddressesRequest)
            throws InvalidResourceException, InvalidNextTokenException, SdkServiceException, SdkClientException,
            SnowballException {
        return new DescribeAddressesPaginator(this, describeAddressesRequest);
    }

    /**
     * <p>
     * Returns information about a specific cluster including shipping information, cluster status, and other important
     * metadata.
     * </p>
     *
     * @param describeClusterRequest
     * @return Result of the DescribeCluster operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeClusterResponse describeCluster(DescribeClusterRequest describeClusterRequest)
            throws InvalidResourceException, SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeClusterResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeClusterRequest).withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about a specific job including shipping information, job status, and other important
     * metadata.
     * </p>
     *
     * @param describeJobRequest
     * @return Result of the DescribeJob operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobResponse describeJob(DescribeJobRequest describeJobRequest) throws InvalidResourceException,
            SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<DescribeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeJobResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeJobRequest).withMarshaller(new DescribeJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a link to an Amazon S3 presigned URL for the manifest file associated with the specified
     * <code>JobId</code> value. You can access the manifest file for up to 60 minutes after this request has been made.
     * To access the manifest file after 60 minutes have passed, you'll have to make another call to the
     * <code>GetJobManifest</code> action.
     * </p>
     * <p>
     * The manifest is an encrypted file that you can download after your job enters the <code>WithCustomer</code>
     * status. The manifest is decrypted by using the <code>UnlockCode</code> code value, when you pass both values to
     * the Snowball through the Snowball client when the client is started for the first time.
     * </p>
     * <p>
     * As a best practice, we recommend that you don't save a copy of an <code>UnlockCode</code> value in the same
     * location as the manifest file for that job. Saving these separately helps prevent unauthorized parties from
     * gaining access to the Snowball associated with that job.
     * </p>
     * <p>
     * The credentials of a given job, including its manifest file and unlock code, expire 90 days after the job is
     * created.
     * </p>
     *
     * @param getJobManifestRequest
     * @return Result of the GetJobManifest operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.GetJobManifest
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetJobManifest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobManifestResponse getJobManifest(GetJobManifestRequest getJobManifestRequest) throws InvalidResourceException,
            InvalidJobStateException, SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<GetJobManifestResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetJobManifestResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetJobManifestRequest, GetJobManifestResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getJobManifestRequest).withMarshaller(new GetJobManifestRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the <code>UnlockCode</code> code value for the specified job. A particular <code>UnlockCode</code> value
     * can be accessed for up to 90 days after the associated job has been created.
     * </p>
     * <p>
     * The <code>UnlockCode</code> value is a 29-character code with 25 alphanumeric characters and 4 hyphens. This code
     * is used to decrypt the manifest file when it is passed along with the manifest to the Snowball through the
     * Snowball client when the client is started for the first time.
     * </p>
     * <p>
     * As a best practice, we recommend that you don't save a copy of the <code>UnlockCode</code> in the same location
     * as the manifest file for that job. Saving these separately helps prevent unauthorized parties from gaining access
     * to the Snowball associated with that job.
     * </p>
     *
     * @param getJobUnlockCodeRequest
     * @return Result of the GetJobUnlockCode operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.GetJobUnlockCode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetJobUnlockCode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobUnlockCodeResponse getJobUnlockCode(GetJobUnlockCodeRequest getJobUnlockCodeRequest)
            throws InvalidResourceException, InvalidJobStateException, SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<GetJobUnlockCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetJobUnlockCodeResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetJobUnlockCodeRequest, GetJobUnlockCodeResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getJobUnlockCodeRequest).withMarshaller(new GetJobUnlockCodeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about the Snowball service limit for your account, and also the number of Snowballs your
     * account has in use.
     * </p>
     * <p>
     * The default service limit for the number of Snowballs that you can have at one time is 1. If you want to increase
     * your service limit, contact AWS Support.
     * </p>
     *
     * @param getSnowballUsageRequest
     * @return Result of the GetSnowballUsage operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.GetSnowballUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetSnowballUsage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSnowballUsageResponse getSnowballUsage(GetSnowballUsageRequest getSnowballUsageRequest) throws SdkServiceException,
            SdkClientException, SnowballException {

        HttpResponseHandler<GetSnowballUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetSnowballUsageResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetSnowballUsageRequest, GetSnowballUsageResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getSnowballUsageRequest).withMarshaller(new GetSnowballUsageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object is for a job in the specified cluster and contains a job's state, a job's ID, and other information.
     * </p>
     *
     * @param listClusterJobsRequest
     * @return Result of the ListClusterJobs operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListClusterJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListClusterJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListClusterJobsResponse listClusterJobs(ListClusterJobsRequest listClusterJobsRequest)
            throws InvalidResourceException, InvalidNextTokenException, SdkServiceException, SdkClientException,
            SnowballException {

        HttpResponseHandler<ListClusterJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListClusterJobsResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListClusterJobsRequest, ListClusterJobsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listClusterJobsRequest).withMarshaller(new ListClusterJobsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of <code>ClusterListEntry</code> objects of the specified length. Each
     * <code>ClusterListEntry</code> object contains a cluster's state, a cluster's ID, and other important status
     * information.
     * </p>
     *
     * @param listClustersRequest
     * @return Result of the ListClusters operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws InvalidNextTokenException,
            SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListClustersResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listClustersRequest).withMarshaller(new ListClustersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object contains a job's state, a job's ID, and a value that indicates whether the job is a job part, in the case
     * of export jobs. Calling this API action in one of the US regions will return jobs from the list of all jobs
     * associated with this account in all US regions.
     * </p>
     *
     * @param listJobsRequest
     * @return Result of the ListJobs operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws InvalidNextTokenException, SdkServiceException,
            SdkClientException, SnowballException {

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(new JsonOperationMetadata()
                .withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListJobsResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest)
                .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object contains a job's state, a job's ID, and a value that indicates whether the job is a job part, in the case
     * of export jobs. Calling this API action in one of the US regions will return jobs from the list of all jobs
     * associated with this account in all US regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.snowball.model.ListJobsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.ListJobsPaginator responses = client.listJobsIterable(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.snowball.paginators.ListJobsPaginator responses = client.listJobsIterable(request);
     *     for (software.amazon.awssdk.services.snowball.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.ListJobsPaginator responses = client.listJobsIterable(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.snowball.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return Result of the ListJobs operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsPaginator listJobsIterable(ListJobsRequest listJobsRequest) throws InvalidNextTokenException,
            SdkServiceException, SdkClientException, SnowballException {
        return new ListJobsPaginator(this, listJobsRequest);
    }

    /**
     * <p>
     * While a cluster's <code>ClusterState</code> value is in the <code>AwaitingQuorum</code> state, you can update
     * some of the information associated with a cluster. Once the cluster changes to a different job state, usually 60
     * minutes after the cluster being created, this action is no longer available.
     * </p>
     *
     * @param updateClusterRequest
     * @return Result of the UpdateCluster operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws KMSRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidInputCombinationException
     *         Job or cluster creation failed. One ore more inputs were invalid. Confirm that the
     *         <a>CreateClusterRequest&#36SnowballType</a> value supports your <a>CreateJobRequest&#36JobType</a>, and
     *         try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.UpdateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateClusterResponse updateCluster(UpdateClusterRequest updateClusterRequest) throws InvalidResourceException,
            InvalidJobStateException, KMSRequestFailedException, InvalidInputCombinationException, SdkServiceException,
            SdkClientException, SnowballException {

        HttpResponseHandler<UpdateClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UpdateClusterResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UpdateClusterRequest, UpdateClusterResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateClusterRequest).withMarshaller(new UpdateClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * While a job's <code>JobState</code> value is <code>New</code>, you can update some of the information associated
     * with a job. Once the job changes to a different job state, usually within 60 minutes of the job being created,
     * this action is no longer available.
     * </p>
     *
     * @param updateJobRequest
     * @return Result of the UpdateJob operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws KMSRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidInputCombinationException
     *         Job or cluster creation failed. One ore more inputs were invalid. Confirm that the
     *         <a>CreateClusterRequest&#36SnowballType</a> value supports your <a>CreateJobRequest&#36JobType</a>, and
     *         try again.
     * @throws ClusterLimitExceededException
     *         Job creation failed. Currently, clusters support five nodes. If you have less than five nodes for your
     *         cluster and you have more nodes to create for this cluster, try again and create jobs until your cluster
     *         has exactly five notes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.UpdateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateJobResponse updateJob(UpdateJobRequest updateJobRequest) throws InvalidResourceException,
            InvalidJobStateException, KMSRequestFailedException, InvalidInputCombinationException, ClusterLimitExceededException,
            SdkServiceException, SdkClientException, SnowballException {

        HttpResponseHandler<UpdateJobResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UpdateJobResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UpdateJobRequest, UpdateJobResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(updateJobRequest)
                .withMarshaller(new UpdateJobRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<SdkServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory init() {
        return new SdkJsonProtocolFactory(new JsonClientMetadata()
                .withProtocolVersion("1.1")
                .withSupportsCbor(false)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.snowball.model.SnowballException.class)
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidJobStateException").withModeledClass(
                                InvalidJobStateException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidResourceException").withModeledClass(
                                InvalidResourceException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("KMSRequestFailedException").withModeledClass(
                                KMSRequestFailedException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("UnsupportedAddressException").withModeledClass(
                                UnsupportedAddressException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ClusterLimitExceededException").withModeledClass(
                                ClusterLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidInputCombinationException").withModeledClass(
                                InvalidInputCombinationException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidNextTokenException").withModeledClass(
                                InvalidNextTokenException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidAddressException").withModeledClass(
                                InvalidAddressException.class)));
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
