/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball.model;

import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.snowball.transform.JobLogsMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains job logs. Whenever Snowball is used to import data into or export data out of Amazon S3, you'll have the
 * option of downloading a PDF job report. Job logs are returned as a part of the response syntax of the
 * <code>DescribeJob</code> action in the <code>JobMetadata</code> data type. The job logs can be accessed for up to 60
 * minutes after this request has been made. To access any of the job logs after 60 minutes have passed, you'll have to
 * make another call to the <code>DescribeJob</code> action.
 * </p>
 * <p>
 * For import jobs, the PDF job report becomes available at the end of the import process. For export jobs, your job
 * report typically becomes available while the Snowball for your job part is being delivered to you.
 * </p>
 * <p>
 * The job report provides you insight into the state of your Amazon S3 data transfer. The report includes details about
 * your job or job part for your records.
 * </p>
 * <p>
 * For deeper visibility into the status of your transferred objects, you can look at the two associated logs: a success
 * log and a failure log. The logs are saved in comma-separated value (CSV) format, and the name of each log includes
 * the ID of the job or job part that the log describes.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class JobLogs implements StructuredPojo, ToCopyableBuilder<JobLogs.Builder, JobLogs> {
    private final String jobCompletionReportURI;

    private final String jobSuccessLogURI;

    private final String jobFailureLogURI;

    private JobLogs(BuilderImpl builder) {
        this.jobCompletionReportURI = builder.jobCompletionReportURI;
        this.jobSuccessLogURI = builder.jobSuccessLogURI;
        this.jobFailureLogURI = builder.jobFailureLogURI;
    }

    /**
     * <p>
     * A link to an Amazon S3 presigned URL where the job completion report is located.
     * </p>
     * 
     * @return A link to an Amazon S3 presigned URL where the job completion report is located.
     */
    public String jobCompletionReportURI() {
        return jobCompletionReportURI;
    }

    /**
     * <p>
     * A link to an Amazon S3 presigned URL where the job success log is located.
     * </p>
     * 
     * @return A link to an Amazon S3 presigned URL where the job success log is located.
     */
    public String jobSuccessLogURI() {
        return jobSuccessLogURI;
    }

    /**
     * <p>
     * A link to an Amazon S3 presigned URL where the job failure log is located.
     * </p>
     * 
     * @return A link to an Amazon S3 presigned URL where the job failure log is located.
     */
    public String jobFailureLogURI() {
        return jobFailureLogURI;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobCompletionReportURI());
        hashCode = 31 * hashCode + Objects.hashCode(jobSuccessLogURI());
        hashCode = 31 * hashCode + Objects.hashCode(jobFailureLogURI());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobLogs)) {
            return false;
        }
        JobLogs other = (JobLogs) obj;
        return Objects.equals(jobCompletionReportURI(), other.jobCompletionReportURI())
                && Objects.equals(jobSuccessLogURI(), other.jobSuccessLogURI())
                && Objects.equals(jobFailureLogURI(), other.jobFailureLogURI());
    }

    @Override
    public String toString() {
        return ToString.builder("JobLogs").add("JobCompletionReportURI", jobCompletionReportURI())
                .add("JobSuccessLogURI", jobSuccessLogURI()).add("JobFailureLogURI", jobFailureLogURI()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "JobCompletionReportURI":
            return Optional.of(clazz.cast(jobCompletionReportURI()));
        case "JobSuccessLogURI":
            return Optional.of(clazz.cast(jobSuccessLogURI()));
        case "JobFailureLogURI":
            return Optional.of(clazz.cast(jobFailureLogURI()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        JobLogsMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, JobLogs> {
        /**
         * <p>
         * A link to an Amazon S3 presigned URL where the job completion report is located.
         * </p>
         * 
         * @param jobCompletionReportURI
         *        A link to an Amazon S3 presigned URL where the job completion report is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobCompletionReportURI(String jobCompletionReportURI);

        /**
         * <p>
         * A link to an Amazon S3 presigned URL where the job success log is located.
         * </p>
         * 
         * @param jobSuccessLogURI
         *        A link to an Amazon S3 presigned URL where the job success log is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobSuccessLogURI(String jobSuccessLogURI);

        /**
         * <p>
         * A link to an Amazon S3 presigned URL where the job failure log is located.
         * </p>
         * 
         * @param jobFailureLogURI
         *        A link to an Amazon S3 presigned URL where the job failure log is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobFailureLogURI(String jobFailureLogURI);
    }

    static final class BuilderImpl implements Builder {
        private String jobCompletionReportURI;

        private String jobSuccessLogURI;

        private String jobFailureLogURI;

        private BuilderImpl() {
        }

        private BuilderImpl(JobLogs model) {
            jobCompletionReportURI(model.jobCompletionReportURI);
            jobSuccessLogURI(model.jobSuccessLogURI);
            jobFailureLogURI(model.jobFailureLogURI);
        }

        public final String getJobCompletionReportURI() {
            return jobCompletionReportURI;
        }

        @Override
        public final Builder jobCompletionReportURI(String jobCompletionReportURI) {
            this.jobCompletionReportURI = jobCompletionReportURI;
            return this;
        }

        public final void setJobCompletionReportURI(String jobCompletionReportURI) {
            this.jobCompletionReportURI = jobCompletionReportURI;
        }

        public final String getJobSuccessLogURI() {
            return jobSuccessLogURI;
        }

        @Override
        public final Builder jobSuccessLogURI(String jobSuccessLogURI) {
            this.jobSuccessLogURI = jobSuccessLogURI;
            return this;
        }

        public final void setJobSuccessLogURI(String jobSuccessLogURI) {
            this.jobSuccessLogURI = jobSuccessLogURI;
        }

        public final String getJobFailureLogURI() {
            return jobFailureLogURI;
        }

        @Override
        public final Builder jobFailureLogURI(String jobFailureLogURI) {
            this.jobFailureLogURI = jobFailureLogURI;
            return this;
        }

        public final void setJobFailureLogURI(String jobFailureLogURI) {
            this.jobFailureLogURI = jobFailureLogURI;
        }

        @Override
        public JobLogs build() {
            return new JobLogs(this);
        }
    }
}
