/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball;

import java.util.concurrent.CompletableFuture;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.client.AsyncClientHandler;
import software.amazon.awssdk.core.client.ClientExecutionParams;
import software.amazon.awssdk.core.client.SdkAsyncClientHandler;
import software.amazon.awssdk.core.config.AsyncClientConfiguration;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory;
import software.amazon.awssdk.services.snowball.model.CancelClusterRequest;
import software.amazon.awssdk.services.snowball.model.CancelClusterResponse;
import software.amazon.awssdk.services.snowball.model.CancelJobRequest;
import software.amazon.awssdk.services.snowball.model.CancelJobResponse;
import software.amazon.awssdk.services.snowball.model.ClusterLimitExceededException;
import software.amazon.awssdk.services.snowball.model.CreateAddressRequest;
import software.amazon.awssdk.services.snowball.model.CreateAddressResponse;
import software.amazon.awssdk.services.snowball.model.CreateClusterRequest;
import software.amazon.awssdk.services.snowball.model.CreateClusterResponse;
import software.amazon.awssdk.services.snowball.model.CreateJobRequest;
import software.amazon.awssdk.services.snowball.model.CreateJobResponse;
import software.amazon.awssdk.services.snowball.model.DescribeAddressRequest;
import software.amazon.awssdk.services.snowball.model.DescribeAddressResponse;
import software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest;
import software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse;
import software.amazon.awssdk.services.snowball.model.DescribeClusterRequest;
import software.amazon.awssdk.services.snowball.model.DescribeClusterResponse;
import software.amazon.awssdk.services.snowball.model.DescribeJobRequest;
import software.amazon.awssdk.services.snowball.model.DescribeJobResponse;
import software.amazon.awssdk.services.snowball.model.GetJobManifestRequest;
import software.amazon.awssdk.services.snowball.model.GetJobManifestResponse;
import software.amazon.awssdk.services.snowball.model.GetJobUnlockCodeRequest;
import software.amazon.awssdk.services.snowball.model.GetJobUnlockCodeResponse;
import software.amazon.awssdk.services.snowball.model.GetSnowballUsageRequest;
import software.amazon.awssdk.services.snowball.model.GetSnowballUsageResponse;
import software.amazon.awssdk.services.snowball.model.InvalidAddressException;
import software.amazon.awssdk.services.snowball.model.InvalidInputCombinationException;
import software.amazon.awssdk.services.snowball.model.InvalidJobStateException;
import software.amazon.awssdk.services.snowball.model.InvalidNextTokenException;
import software.amazon.awssdk.services.snowball.model.InvalidResourceException;
import software.amazon.awssdk.services.snowball.model.KMSRequestFailedException;
import software.amazon.awssdk.services.snowball.model.ListClusterJobsRequest;
import software.amazon.awssdk.services.snowball.model.ListClusterJobsResponse;
import software.amazon.awssdk.services.snowball.model.ListClustersRequest;
import software.amazon.awssdk.services.snowball.model.ListClustersResponse;
import software.amazon.awssdk.services.snowball.model.ListJobsRequest;
import software.amazon.awssdk.services.snowball.model.ListJobsResponse;
import software.amazon.awssdk.services.snowball.model.UnsupportedAddressException;
import software.amazon.awssdk.services.snowball.model.UpdateClusterRequest;
import software.amazon.awssdk.services.snowball.model.UpdateClusterResponse;
import software.amazon.awssdk.services.snowball.model.UpdateJobRequest;
import software.amazon.awssdk.services.snowball.model.UpdateJobResponse;
import software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPublisher;
import software.amazon.awssdk.services.snowball.paginators.ListJobsPublisher;
import software.amazon.awssdk.services.snowball.transform.CancelClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CancelClusterResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CancelJobResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateAddressRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateAddressResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateClusterResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateJobResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressesRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressesResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeClusterResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeJobResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobManifestRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobManifestResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobUnlockCodeRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobUnlockCodeResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.GetSnowballUsageRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetSnowballUsageResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClusterJobsRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClusterJobsResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClustersResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListJobsResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateClusterResponseUnmarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateJobResponseUnmarshaller;

/**
 * Internal implementation of {@link SnowballAsyncClient}.
 *
 * @see SnowballAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnowballAsyncClient implements SnowballAsyncClient {
    private final AsyncClientHandler clientHandler;

    private final SdkJsonProtocolFactory protocolFactory;

    protected DefaultSnowballAsyncClient(AsyncClientConfiguration clientConfiguration) {
        this.clientHandler = new SdkAsyncClientHandler(clientConfiguration, null);
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a cluster job. You can only cancel a cluster job while it's in the <code>AwaitingQuorum</code> status.
     * You'll have at least an hour after creating a cluster job to cancel it.
     * </p>
     *
     * @param cancelClusterRequest
     * @return A Java Future containing the result of the CancelCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>KMSRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CancelCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CancelCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelClusterResponse> cancelCluster(CancelClusterRequest cancelClusterRequest) {

        HttpResponseHandler<CancelClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CancelClusterResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CancelClusterRequest, CancelClusterResponse>()
                .withMarshaller(new CancelClusterRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(cancelClusterRequest));
    }

    /**
     * <p>
     * Cancels the specified job. You can only cancel a job before its <code>JobState</code> value changes to
     * <code>PreparingAppliance</code>. Requesting the <code>ListJobs</code> or <code>DescribeJob</code> action will
     * return a job's <code>JobState</code> as part of the response element data returned.
     * </p>
     *
     * @param cancelJobRequest
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>KMSRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CancelJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {

        HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CancelJobResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>()
                .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest));
    }

    /**
     * <p>
     * Creates an address for a Snowball to be shipped to. In most regions, addresses are validated at the time of
     * creation. The address you provide must be located within the serviceable area of your region. If the address is
     * invalid or unsupported, then an exception is thrown.
     * </p>
     *
     * @param createAddressRequest
     * @return A Java Future containing the result of the CreateAddress operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidAddressException The address provided was invalid. Check the address with your region's
     *         carrier, and try again.</li>
     *         <li>UnsupportedAddressException The address is either outside the serviceable area for your region, or an
     *         error occurred. Check the address with your region's carrier and try again. If the issue persists,
     *         contact AWS Support.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CreateAddress
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAddressResponse> createAddress(CreateAddressRequest createAddressRequest) {

        HttpResponseHandler<CreateAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateAddressResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateAddressRequest, CreateAddressResponse>()
                .withMarshaller(new CreateAddressRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createAddressRequest));
    }

    /**
     * <p>
     * Creates an empty cluster. Each cluster supports five nodes. You use the <a>CreateJob</a> action separately to
     * create the jobs for each of these nodes. The cluster does not ship until these five node jobs have been created.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>KMSRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One ore more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CreateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {

        HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateClusterResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createClusterRequest));
    }

    /**
     * <p>
     * Creates a job to the other job attributes are inherited from the cluster.
     * </p>
     *
     * @param createJobRequest
     * @return A Java Future containing the result of the CreateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>KMSRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One ore more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>ClusterLimitExceededException Job creation failed. Currently, clusters support five nodes. If you
     *         have less than five nodes for your cluster and you have more nodes to create for this cluster, try again
     *         and create jobs until your cluster has exactly five notes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CreateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobResponse> createJob(CreateJobRequest createJobRequest) {

        HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateJobResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>()
                .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createJobRequest));
    }

    /**
     * <p>
     * Takes an <code>AddressId</code> and returns specific details about that address in the form of an
     * <code>Address</code> object.
     * </p>
     *
     * @param describeAddressRequest
     * @return A Java Future containing the result of the DescribeAddress operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeAddress
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAddressResponse> describeAddress(DescribeAddressRequest describeAddressRequest) {

        HttpResponseHandler<DescribeAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeAddressResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeAddressRequest, DescribeAddressResponse>()
                .withMarshaller(new DescribeAddressRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAddressRequest));
    }

    /**
     * <p>
     * Returns a specified number of <code>ADDRESS</code> objects. Calling this API in one of the US regions will return
     * addresses from the list of all addresses associated with this account in all US regions.
     * </p>
     *
     * @param describeAddressesRequest
     * @return A Java Future containing the result of the DescribeAddresses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddresses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAddressesResponse> describeAddresses(DescribeAddressesRequest describeAddressesRequest) {

        HttpResponseHandler<DescribeAddressesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeAddressesResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeAddressesRequest, DescribeAddressesResponse>()
                .withMarshaller(new DescribeAddressesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAddressesRequest));
    }

    /**
     * <p>
     * Returns information about a specific cluster including shipping information, cluster status, and other important
     * metadata.
     * </p>
     *
     * @param describeClusterRequest
     * @return A Java Future containing the result of the DescribeCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeClusterResponse> describeCluster(DescribeClusterRequest describeClusterRequest) {

        HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeClusterResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeClusterRequest));
    }

    /**
     * <p>
     * Returns information about a specific job including shipping information, job status, and other important
     * metadata.
     * </p>
     *
     * @param describeJobRequest
     * @return A Java Future containing the result of the DescribeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobResponse> describeJob(DescribeJobRequest describeJobRequest) {

        HttpResponseHandler<DescribeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeJobResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                .withMarshaller(new DescribeJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeJobRequest));
    }

    /**
     * <p>
     * Returns a link to an Amazon S3 presigned URL for the manifest file associated with the specified
     * <code>JobId</code> value. You can access the manifest file for up to 60 minutes after this request has been made.
     * To access the manifest file after 60 minutes have passed, you'll have to make another call to the
     * <code>GetJobManifest</code> action.
     * </p>
     * <p>
     * The manifest is an encrypted file that you can download after your job enters the <code>WithCustomer</code>
     * status. The manifest is decrypted by using the <code>UnlockCode</code> code value, when you pass both values to
     * the Snowball through the Snowball client when the client is started for the first time.
     * </p>
     * <p>
     * As a best practice, we recommend that you don't save a copy of an <code>UnlockCode</code> value in the same
     * location as the manifest file for that job. Saving these separately helps prevent unauthorized parties from
     * gaining access to the Snowball associated with that job.
     * </p>
     * <p>
     * The credentials of a given job, including its manifest file and unlock code, expire 90 days after the job is
     * created.
     * </p>
     *
     * @param getJobManifestRequest
     * @return A Java Future containing the result of the GetJobManifest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.GetJobManifest
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetJobManifest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobManifestResponse> getJobManifest(GetJobManifestRequest getJobManifestRequest) {

        HttpResponseHandler<GetJobManifestResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetJobManifestResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetJobManifestRequest, GetJobManifestResponse>()
                .withMarshaller(new GetJobManifestRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getJobManifestRequest));
    }

    /**
     * <p>
     * Returns the <code>UnlockCode</code> code value for the specified job. A particular <code>UnlockCode</code> value
     * can be accessed for up to 90 days after the associated job has been created.
     * </p>
     * <p>
     * The <code>UnlockCode</code> value is a 29-character code with 25 alphanumeric characters and 4 hyphens. This code
     * is used to decrypt the manifest file when it is passed along with the manifest to the Snowball through the
     * Snowball client when the client is started for the first time.
     * </p>
     * <p>
     * As a best practice, we recommend that you don't save a copy of the <code>UnlockCode</code> in the same location
     * as the manifest file for that job. Saving these separately helps prevent unauthorized parties from gaining access
     * to the Snowball associated with that job.
     * </p>
     *
     * @param getJobUnlockCodeRequest
     * @return A Java Future containing the result of the GetJobUnlockCode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.GetJobUnlockCode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetJobUnlockCode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobUnlockCodeResponse> getJobUnlockCode(GetJobUnlockCodeRequest getJobUnlockCodeRequest) {

        HttpResponseHandler<GetJobUnlockCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetJobUnlockCodeResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetJobUnlockCodeRequest, GetJobUnlockCodeResponse>()
                .withMarshaller(new GetJobUnlockCodeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getJobUnlockCodeRequest));
    }

    /**
     * <p>
     * Returns information about the Snowball service limit for your account, and also the number of Snowballs your
     * account has in use.
     * </p>
     * <p>
     * The default service limit for the number of Snowballs that you can have at one time is 1. If you want to increase
     * your service limit, contact AWS Support.
     * </p>
     *
     * @param getSnowballUsageRequest
     * @return A Java Future containing the result of the GetSnowballUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.GetSnowballUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetSnowballUsage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSnowballUsageResponse> getSnowballUsage(GetSnowballUsageRequest getSnowballUsageRequest) {

        HttpResponseHandler<GetSnowballUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetSnowballUsageResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetSnowballUsageRequest, GetSnowballUsageResponse>()
                .withMarshaller(new GetSnowballUsageRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getSnowballUsageRequest));
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object is for a job in the specified cluster and contains a job's state, a job's ID, and other information.
     * </p>
     *
     * @param listClusterJobsRequest
     * @return A Java Future containing the result of the ListClusterJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListClusterJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListClusterJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListClusterJobsResponse> listClusterJobs(ListClusterJobsRequest listClusterJobsRequest) {

        HttpResponseHandler<ListClusterJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListClusterJobsResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListClusterJobsRequest, ListClusterJobsResponse>()
                .withMarshaller(new ListClusterJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listClusterJobsRequest));
    }

    /**
     * <p>
     * Returns an array of <code>ClusterListEntry</code> objects of the specified length. Each
     * <code>ClusterListEntry</code> object contains a cluster's state, a cluster's ID, and other important status
     * information.
     * </p>
     *
     * @param listClustersRequest
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListClustersResponse> listClusters(ListClustersRequest listClustersRequest) {

        HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListClustersResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                .withMarshaller(new ListClustersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listClustersRequest));
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object contains a job's state, a job's ID, and a value that indicates whether the job is a job part, in the case
     * of export jobs. Calling this API action in one of the US regions will return jobs from the list of all jobs
     * associated with this account in all US regions.
     * </p>
     *
     * @param listJobsRequest
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(new JsonOperationMetadata()
                .withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListJobsResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>()
                .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest));
    }

    /**
     * <p>
     * While a cluster's <code>ClusterState</code> value is in the <code>AwaitingQuorum</code> state, you can update
     * some of the information associated with a cluster. Once the cluster changes to a different job state, usually 60
     * minutes after the cluster being created, this action is no longer available.
     * </p>
     *
     * @param updateClusterRequest
     * @return A Java Future containing the result of the UpdateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>KMSRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One ore more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.UpdateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateClusterResponse> updateCluster(UpdateClusterRequest updateClusterRequest) {

        HttpResponseHandler<UpdateClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UpdateClusterResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UpdateClusterRequest, UpdateClusterResponse>()
                .withMarshaller(new UpdateClusterRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateClusterRequest));
    }

    /**
     * <p>
     * While a job's <code>JobState</code> value is <code>New</code>, you can update some of the information associated
     * with a job. Once the job changes to a different job state, usually within 60 minutes of the job being created,
     * this action is no longer available.
     * </p>
     *
     * @param updateJobRequest
     * @return A Java Future containing the result of the UpdateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>KMSRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One ore more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>ClusterLimitExceededException Job creation failed. Currently, clusters support five nodes. If you
     *         have less than five nodes for your cluster and you have more nodes to create for this cluster, try again
     *         and create jobs until your cluster has exactly five notes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.UpdateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobResponse> updateJob(UpdateJobRequest updateJobRequest) {

        HttpResponseHandler<UpdateJobResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UpdateJobResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UpdateJobRequest, UpdateJobResponse>()
                .withMarshaller(new UpdateJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateJobRequest));
    }

    /**
     * <p>
     * Returns a specified number of <code>ADDRESS</code> objects. Calling this API in one of the US regions will return
     * addresses from the list of all addresses associated with this account in all US regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAddresses(software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the forEach helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPublisher publisher = client.describeAddressesPaginator(request);
     * CompletableFuture<Void> future = publisher.forEach(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPublisher publisher = client.describeAddressesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAddresses(software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAddressesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddresses" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeAddressesPublisher describeAddressesPaginator(DescribeAddressesRequest describeAddressesRequest) {
        return new DescribeAddressesPublisher(this, describeAddressesRequest);
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object contains a job's state, a job's ID, and a value that indicates whether the job is a job part, in the case
     * of export jobs. Calling this API action in one of the US regions will return jobs from the list of all jobs
     * associated with this account in all US regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.snowball.model.ListJobsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the forEach helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.forEach(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.snowball.model.ListJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.snowball.model.ListJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.snowball.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListJobsPublisher listJobsPaginator(ListJobsRequest listJobsRequest) {
        return new ListJobsPublisher(this, listJobsRequest);
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory init() {
        return new SdkJsonProtocolFactory(new JsonClientMetadata()
                .withProtocolVersion("1.1")
                .withSupportsCbor(false)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.snowball.model.SnowballException.class)
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidJobStateException").withModeledClass(
                                InvalidJobStateException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidResourceException").withModeledClass(
                                InvalidResourceException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("KMSRequestFailedException").withModeledClass(
                                KMSRequestFailedException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("UnsupportedAddressException").withModeledClass(
                                UnsupportedAddressException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ClusterLimitExceededException").withModeledClass(
                                ClusterLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidInputCombinationException").withModeledClass(
                                InvalidInputCombinationException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidNextTokenException").withModeledClass(
                                InvalidNextTokenException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidAddressException").withModeledClass(
                                InvalidAddressException.class)));
    }

    private HttpResponseHandler<SdkServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }
}
