/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Amazon Simple Notification Service (Amazon SNS) notification settings associated with a specific job. The
 * <code>Notification</code> object is returned as a part of the response syntax of the <code>DescribeJob</code> action
 * in the <code>JobMetadata</code> data type.
 * </p>
 * <p>
 * When the notification settings are defined during job creation, you can choose to notify based on a specific set of
 * job states using the <code>JobStatesToNotify</code> array of strings, or you can specify that you want to have Amazon
 * SNS notifications sent out for all job states with <code>NotifyAll</code> set to true.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Notification implements SdkPojo, Serializable, ToCopyableBuilder<Notification.Builder, Notification> {
    private static final SdkField<String> SNS_TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Notification::snsTopicARN)).setter(setter(Builder::snsTopicARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsTopicARN").build()).build();

    private static final SdkField<List<String>> JOB_STATES_TO_NOTIFY_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Notification::jobStatesToNotifyAsStrings))
            .setter(setter(Builder::jobStatesToNotifyWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobStatesToNotify").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> NOTIFY_ALL_JOB_STATES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(Notification::notifyAllJobStates)).setter(setter(Builder::notifyAllJobStates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotifyAll").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SNS_TOPIC_ARN_FIELD,
            JOB_STATES_TO_NOTIFY_FIELD, NOTIFY_ALL_JOB_STATES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String snsTopicARN;

    private final List<String> jobStatesToNotify;

    private final Boolean notifyAllJobStates;

    private Notification(BuilderImpl builder) {
        this.snsTopicARN = builder.snsTopicARN;
        this.jobStatesToNotify = builder.jobStatesToNotify;
        this.notifyAllJobStates = builder.notifyAllJobStates;
    }

    /**
     * <p>
     * The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon Resource Names
     * (ARNs) for topics by using the <a
     * href="https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API action.
     * </p>
     * <p>
     * You can subscribe email addresses to an Amazon SNS topic through the AWS Management Console, or by using the <a
     * href="https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> AWS Simple Notification
     * Service (SNS) API action.
     * </p>
     * 
     * @return The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon
     *         Resource Names (ARNs) for topics by using the <a
     *         href="https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API
     *         action.</p>
     *         <p>
     *         You can subscribe email addresses to an Amazon SNS topic through the AWS Management Console, or by using
     *         the <a href="https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> AWS Simple
     *         Notification Service (SNS) API action.
     */
    public String snsTopicARN() {
        return snsTopicARN;
    }

    /**
     * <p>
     * The list of job states that will trigger a notification for this job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasJobStatesToNotify()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of job states that will trigger a notification for this job.
     */
    public List<JobState> jobStatesToNotify() {
        return JobStateListCopier.copyStringToEnum(jobStatesToNotify);
    }

    /**
     * Returns true if the JobStatesToNotify property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasJobStatesToNotify() {
        return jobStatesToNotify != null && !(jobStatesToNotify instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of job states that will trigger a notification for this job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasJobStatesToNotify()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of job states that will trigger a notification for this job.
     */
    public List<String> jobStatesToNotifyAsStrings() {
        return jobStatesToNotify;
    }

    /**
     * <p>
     * Any change in job state will trigger a notification for this job.
     * </p>
     * 
     * @return Any change in job state will trigger a notification for this job.
     */
    public Boolean notifyAllJobStates() {
        return notifyAllJobStates;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(snsTopicARN());
        hashCode = 31 * hashCode + Objects.hashCode(jobStatesToNotifyAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(notifyAllJobStates());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Notification)) {
            return false;
        }
        Notification other = (Notification) obj;
        return Objects.equals(snsTopicARN(), other.snsTopicARN())
                && Objects.equals(jobStatesToNotifyAsStrings(), other.jobStatesToNotifyAsStrings())
                && Objects.equals(notifyAllJobStates(), other.notifyAllJobStates());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Notification").add("SnsTopicARN", snsTopicARN())
                .add("JobStatesToNotify", jobStatesToNotifyAsStrings()).add("NotifyAllJobStates", notifyAllJobStates()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SnsTopicARN":
            return Optional.ofNullable(clazz.cast(snsTopicARN()));
        case "JobStatesToNotify":
            return Optional.ofNullable(clazz.cast(jobStatesToNotifyAsStrings()));
        case "NotifyAllJobStates":
            return Optional.ofNullable(clazz.cast(notifyAllJobStates()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Notification, T> g) {
        return obj -> g.apply((Notification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Notification> {
        /**
         * <p>
         * The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon Resource
         * Names (ARNs) for topics by using the <a
         * href="https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API action.
         * </p>
         * <p>
         * You can subscribe email addresses to an Amazon SNS topic through the AWS Management Console, or by using the
         * <a href="https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> AWS Simple Notification
         * Service (SNS) API action.
         * </p>
         * 
         * @param snsTopicARN
         *        The new SNS <code>TopicArn</code> that you want to associate with this job. You can create Amazon
         *        Resource Names (ARNs) for topics by using the <a
         *        href="https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html">CreateTopic</a> Amazon SNS API
         *        action.</p>
         *        <p>
         *        You can subscribe email addresses to an Amazon SNS topic through the AWS Management Console, or by
         *        using the <a href="https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html">Subscribe</a> AWS
         *        Simple Notification Service (SNS) API action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicARN(String snsTopicARN);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatesToNotifyWithStrings(Collection<String> jobStatesToNotify);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatesToNotifyWithStrings(String... jobStatesToNotify);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatesToNotify(Collection<JobState> jobStatesToNotify);

        /**
         * <p>
         * The list of job states that will trigger a notification for this job.
         * </p>
         * 
         * @param jobStatesToNotify
         *        The list of job states that will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatesToNotify(JobState... jobStatesToNotify);

        /**
         * <p>
         * Any change in job state will trigger a notification for this job.
         * </p>
         * 
         * @param notifyAllJobStates
         *        Any change in job state will trigger a notification for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifyAllJobStates(Boolean notifyAllJobStates);
    }

    static final class BuilderImpl implements Builder {
        private String snsTopicARN;

        private List<String> jobStatesToNotify = DefaultSdkAutoConstructList.getInstance();

        private Boolean notifyAllJobStates;

        private BuilderImpl() {
        }

        private BuilderImpl(Notification model) {
            snsTopicARN(model.snsTopicARN);
            jobStatesToNotifyWithStrings(model.jobStatesToNotify);
            notifyAllJobStates(model.notifyAllJobStates);
        }

        public final String getSnsTopicARN() {
            return snsTopicARN;
        }

        @Override
        public final Builder snsTopicARN(String snsTopicARN) {
            this.snsTopicARN = snsTopicARN;
            return this;
        }

        public final void setSnsTopicARN(String snsTopicARN) {
            this.snsTopicARN = snsTopicARN;
        }

        public final Collection<String> getJobStatesToNotifyAsStrings() {
            return jobStatesToNotify;
        }

        @Override
        public final Builder jobStatesToNotifyWithStrings(Collection<String> jobStatesToNotify) {
            this.jobStatesToNotify = JobStateListCopier.copy(jobStatesToNotify);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStatesToNotifyWithStrings(String... jobStatesToNotify) {
            jobStatesToNotifyWithStrings(Arrays.asList(jobStatesToNotify));
            return this;
        }

        @Override
        public final Builder jobStatesToNotify(Collection<JobState> jobStatesToNotify) {
            this.jobStatesToNotify = JobStateListCopier.copyEnumToString(jobStatesToNotify);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStatesToNotify(JobState... jobStatesToNotify) {
            jobStatesToNotify(Arrays.asList(jobStatesToNotify));
            return this;
        }

        public final void setJobStatesToNotifyWithStrings(Collection<String> jobStatesToNotify) {
            this.jobStatesToNotify = JobStateListCopier.copy(jobStatesToNotify);
        }

        public final Boolean getNotifyAllJobStates() {
            return notifyAllJobStates;
        }

        @Override
        public final Builder notifyAllJobStates(Boolean notifyAllJobStates) {
            this.notifyAllJobStates = notifyAllJobStates;
            return this;
        }

        public final void setNotifyAllJobStates(Boolean notifyAllJobStates) {
            this.notifyAllJobStates = notifyAllJobStates;
        }

        @Override
        public Notification build() {
            return new Notification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
