/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains metadata about a specific cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClusterMetadata implements SdkPojo, Serializable, ToCopyableBuilder<ClusterMetadata.Builder, ClusterMetadata> {
    private static final SdkField<String> CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::clusterId)).setter(setter(Builder::clusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::kmsKeyARN)).setter(setter(Builder::kmsKeyARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyARN").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::roleARN)).setter(setter(Builder::roleARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleARN").build()).build();

    private static final SdkField<String> CLUSTER_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::clusterStateAsString)).setter(setter(Builder::clusterState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterState").build()).build();

    private static final SdkField<String> JOB_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::jobTypeAsString)).setter(setter(Builder::jobType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobType").build()).build();

    private static final SdkField<String> SNOWBALL_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::snowballTypeAsString)).setter(setter(Builder::snowballType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnowballType").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ClusterMetadata::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final SdkField<JobResource> RESOURCES_FIELD = SdkField.<JobResource> builder(MarshallingType.SDK_POJO)
            .getter(getter(ClusterMetadata::resources)).setter(setter(Builder::resources)).constructor(JobResource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build()).build();

    private static final SdkField<String> ADDRESS_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::addressId)).setter(setter(Builder::addressId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressId").build()).build();

    private static final SdkField<String> SHIPPING_OPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::shippingOptionAsString)).setter(setter(Builder::shippingOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShippingOption").build()).build();

    private static final SdkField<Notification> NOTIFICATION_FIELD = SdkField.<Notification> builder(MarshallingType.SDK_POJO)
            .getter(getter(ClusterMetadata::notification)).setter(setter(Builder::notification))
            .constructor(Notification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Notification").build()).build();

    private static final SdkField<String> FORWARDING_ADDRESS_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClusterMetadata::forwardingAddressId)).setter(setter(Builder::forwardingAddressId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForwardingAddressId").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_ID_FIELD,
            DESCRIPTION_FIELD, KMS_KEY_ARN_FIELD, ROLE_ARN_FIELD, CLUSTER_STATE_FIELD, JOB_TYPE_FIELD, SNOWBALL_TYPE_FIELD,
            CREATION_DATE_FIELD, RESOURCES_FIELD, ADDRESS_ID_FIELD, SHIPPING_OPTION_FIELD, NOTIFICATION_FIELD,
            FORWARDING_ADDRESS_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String clusterId;

    private final String description;

    private final String kmsKeyARN;

    private final String roleARN;

    private final String clusterState;

    private final String jobType;

    private final String snowballType;

    private final Instant creationDate;

    private final JobResource resources;

    private final String addressId;

    private final String shippingOption;

    private final Notification notification;

    private final String forwardingAddressId;

    private ClusterMetadata(BuilderImpl builder) {
        this.clusterId = builder.clusterId;
        this.description = builder.description;
        this.kmsKeyARN = builder.kmsKeyARN;
        this.roleARN = builder.roleARN;
        this.clusterState = builder.clusterState;
        this.jobType = builder.jobType;
        this.snowballType = builder.snowballType;
        this.creationDate = builder.creationDate;
        this.resources = builder.resources;
        this.addressId = builder.addressId;
        this.shippingOption = builder.shippingOption;
        this.notification = builder.notification;
        this.forwardingAddressId = builder.forwardingAddressId;
    }

    /**
     * <p>
     * The automatically generated ID for a cluster.
     * </p>
     * 
     * @return The automatically generated ID for a cluster.
     */
    public String clusterId() {
        return clusterId;
    }

    /**
     * <p>
     * The optional description of the cluster.
     * </p>
     * 
     * @return The optional description of the cluster.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The <code>KmsKeyARN</code> Amazon Resource Name (ARN) associated with this cluster. This ARN was created using
     * the <a href="https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html">CreateKey</a> API action in
     * AWS Key Management Service (AWS KMS).
     * </p>
     * 
     * @return The <code>KmsKeyARN</code> Amazon Resource Name (ARN) associated with this cluster. This ARN was created
     *         using the <a href="https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html">CreateKey</a>
     *         API action in AWS Key Management Service (AWS KMS).
     */
    public String kmsKeyARN() {
        return kmsKeyARN;
    }

    /**
     * <p>
     * The role ARN associated with this cluster. This ARN was created using the <a
     * href="https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a> API action in AWS
     * Identity and Access Management (IAM).
     * </p>
     * 
     * @return The role ARN associated with this cluster. This ARN was created using the <a
     *         href="https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a> API action
     *         in AWS Identity and Access Management (IAM).
     */
    public String roleARN() {
        return roleARN;
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #clusterState} will
     * return {@link ClusterState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #clusterStateAsString}.
     * </p>
     * 
     * @return The current status of the cluster.
     * @see ClusterState
     */
    public ClusterState clusterState() {
        return ClusterState.fromValue(clusterState);
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #clusterState} will
     * return {@link ClusterState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #clusterStateAsString}.
     * </p>
     * 
     * @return The current status of the cluster.
     * @see ClusterState
     */
    public String clusterStateAsString() {
        return clusterState;
    }

    /**
     * <p>
     * The type of job for this cluster. Currently, the only job type supported for clusters is <code>LOCAL_USE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobType} will
     * return {@link JobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobTypeAsString}.
     * </p>
     * 
     * @return The type of job for this cluster. Currently, the only job type supported for clusters is
     *         <code>LOCAL_USE</code>.
     * @see JobType
     */
    public JobType jobType() {
        return JobType.fromValue(jobType);
    }

    /**
     * <p>
     * The type of job for this cluster. Currently, the only job type supported for clusters is <code>LOCAL_USE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobType} will
     * return {@link JobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobTypeAsString}.
     * </p>
     * 
     * @return The type of job for this cluster. Currently, the only job type supported for clusters is
     *         <code>LOCAL_USE</code>.
     * @see JobType
     */
    public String jobTypeAsString() {
        return jobType;
    }

    /**
     * <p>
     * The type of AWS Snowball device to use for this cluster. Currently, the only supported device type for cluster
     * jobs is <code>EDGE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #snowballType} will
     * return {@link SnowballType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #snowballTypeAsString}.
     * </p>
     * 
     * @return The type of AWS Snowball device to use for this cluster. Currently, the only supported device type for
     *         cluster jobs is <code>EDGE</code>.
     * @see SnowballType
     */
    public SnowballType snowballType() {
        return SnowballType.fromValue(snowballType);
    }

    /**
     * <p>
     * The type of AWS Snowball device to use for this cluster. Currently, the only supported device type for cluster
     * jobs is <code>EDGE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #snowballType} will
     * return {@link SnowballType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #snowballTypeAsString}.
     * </p>
     * 
     * @return The type of AWS Snowball device to use for this cluster. Currently, the only supported device type for
     *         cluster jobs is <code>EDGE</code>.
     * @see SnowballType
     */
    public String snowballTypeAsString() {
        return snowballType;
    }

    /**
     * <p>
     * The creation date for this cluster.
     * </p>
     * 
     * @return The creation date for this cluster.
     */
    public Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
     * <a>LambdaResource</a> objects.
     * </p>
     * 
     * @return The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
     *         <a>LambdaResource</a> objects.
     */
    public JobResource resources() {
        return resources;
    }

    /**
     * <p>
     * The automatically generated ID for a specific address.
     * </p>
     * 
     * @return The automatically generated ID for a specific address.
     */
    public String addressId() {
        return addressId;
    }

    /**
     * <p>
     * The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each Snowball
     * Edge device, rather it represents how quickly each device moves to its destination while in transit. Regional
     * shipping speeds are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * In Australia, you have access to express shipping. Typically, devices shipped express are delivered in about a
     * day.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped express are
     * delivered in about a day. In addition, most countries in the EU have access to standard shipping, which typically
     * takes less than a week, one way.
     * </p>
     * </li>
     * <li>
     * <p>
     * In India, Snowball Edges are delivered in one to seven days.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the US, you have access to one-day shipping and two-day shipping.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shippingOption}
     * will return {@link ShippingOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #shippingOptionAsString}.
     * </p>
     * 
     * @return The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
     *         Snowball Edge device, rather it represents how quickly each device moves to its destination while in
     *         transit. Regional shipping speeds are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         In Australia, you have access to express shipping. Typically, devices shipped express are delivered in
     *         about a day.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped
     *         express are delivered in about a day. In addition, most countries in the EU have access to standard
     *         shipping, which typically takes less than a week, one way.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In India, Snowball Edges are delivered in one to seven days.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the US, you have access to one-day shipping and two-day shipping.
     *         </p>
     *         </li>
     * @see ShippingOption
     */
    public ShippingOption shippingOption() {
        return ShippingOption.fromValue(shippingOption);
    }

    /**
     * <p>
     * The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each Snowball
     * Edge device, rather it represents how quickly each device moves to its destination while in transit. Regional
     * shipping speeds are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * In Australia, you have access to express shipping. Typically, devices shipped express are delivered in about a
     * day.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped express are
     * delivered in about a day. In addition, most countries in the EU have access to standard shipping, which typically
     * takes less than a week, one way.
     * </p>
     * </li>
     * <li>
     * <p>
     * In India, Snowball Edges are delivered in one to seven days.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the US, you have access to one-day shipping and two-day shipping.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shippingOption}
     * will return {@link ShippingOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #shippingOptionAsString}.
     * </p>
     * 
     * @return The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
     *         Snowball Edge device, rather it represents how quickly each device moves to its destination while in
     *         transit. Regional shipping speeds are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         In Australia, you have access to express shipping. Typically, devices shipped express are delivered in
     *         about a day.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped
     *         express are delivered in about a day. In addition, most countries in the EU have access to standard
     *         shipping, which typically takes less than a week, one way.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In India, Snowball Edges are delivered in one to seven days.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the US, you have access to one-day shipping and two-day shipping.
     *         </p>
     *         </li>
     * @see ShippingOption
     */
    public String shippingOptionAsString() {
        return shippingOption;
    }

    /**
     * <p>
     * The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
     * </p>
     * 
     * @return The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
     */
    public Notification notification() {
        return notification;
    }

    /**
     * <p>
     * The ID of the address that you want a cluster shipped to, after it will be shipped to its primary address. This
     * field is not supported in most regions.
     * </p>
     * 
     * @return The ID of the address that you want a cluster shipped to, after it will be shipped to its primary
     *         address. This field is not supported in most regions.
     */
    public String forwardingAddressId() {
        return forwardingAddressId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clusterId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyARN());
        hashCode = 31 * hashCode + Objects.hashCode(roleARN());
        hashCode = 31 * hashCode + Objects.hashCode(clusterStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(jobTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(snowballTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        hashCode = 31 * hashCode + Objects.hashCode(addressId());
        hashCode = 31 * hashCode + Objects.hashCode(shippingOptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(notification());
        hashCode = 31 * hashCode + Objects.hashCode(forwardingAddressId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClusterMetadata)) {
            return false;
        }
        ClusterMetadata other = (ClusterMetadata) obj;
        return Objects.equals(clusterId(), other.clusterId()) && Objects.equals(description(), other.description())
                && Objects.equals(kmsKeyARN(), other.kmsKeyARN()) && Objects.equals(roleARN(), other.roleARN())
                && Objects.equals(clusterStateAsString(), other.clusterStateAsString())
                && Objects.equals(jobTypeAsString(), other.jobTypeAsString())
                && Objects.equals(snowballTypeAsString(), other.snowballTypeAsString())
                && Objects.equals(creationDate(), other.creationDate()) && Objects.equals(resources(), other.resources())
                && Objects.equals(addressId(), other.addressId())
                && Objects.equals(shippingOptionAsString(), other.shippingOptionAsString())
                && Objects.equals(notification(), other.notification())
                && Objects.equals(forwardingAddressId(), other.forwardingAddressId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ClusterMetadata").add("ClusterId", clusterId()).add("Description", description())
                .add("KmsKeyARN", kmsKeyARN()).add("RoleARN", roleARN()).add("ClusterState", clusterStateAsString())
                .add("JobType", jobTypeAsString()).add("SnowballType", snowballTypeAsString())
                .add("CreationDate", creationDate()).add("Resources", resources()).add("AddressId", addressId())
                .add("ShippingOption", shippingOptionAsString()).add("Notification", notification())
                .add("ForwardingAddressId", forwardingAddressId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClusterId":
            return Optional.ofNullable(clazz.cast(clusterId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "KmsKeyARN":
            return Optional.ofNullable(clazz.cast(kmsKeyARN()));
        case "RoleARN":
            return Optional.ofNullable(clazz.cast(roleARN()));
        case "ClusterState":
            return Optional.ofNullable(clazz.cast(clusterStateAsString()));
        case "JobType":
            return Optional.ofNullable(clazz.cast(jobTypeAsString()));
        case "SnowballType":
            return Optional.ofNullable(clazz.cast(snowballTypeAsString()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "AddressId":
            return Optional.ofNullable(clazz.cast(addressId()));
        case "ShippingOption":
            return Optional.ofNullable(clazz.cast(shippingOptionAsString()));
        case "Notification":
            return Optional.ofNullable(clazz.cast(notification()));
        case "ForwardingAddressId":
            return Optional.ofNullable(clazz.cast(forwardingAddressId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ClusterMetadata, T> g) {
        return obj -> g.apply((ClusterMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClusterMetadata> {
        /**
         * <p>
         * The automatically generated ID for a cluster.
         * </p>
         * 
         * @param clusterId
         *        The automatically generated ID for a cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterId(String clusterId);

        /**
         * <p>
         * The optional description of the cluster.
         * </p>
         * 
         * @param description
         *        The optional description of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The <code>KmsKeyARN</code> Amazon Resource Name (ARN) associated with this cluster. This ARN was created
         * using the <a href="https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html">CreateKey</a> API
         * action in AWS Key Management Service (AWS KMS).
         * </p>
         * 
         * @param kmsKeyARN
         *        The <code>KmsKeyARN</code> Amazon Resource Name (ARN) associated with this cluster. This ARN was
         *        created using the <a
         *        href="https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html">CreateKey</a> API action
         *        in AWS Key Management Service (AWS KMS).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyARN(String kmsKeyARN);

        /**
         * <p>
         * The role ARN associated with this cluster. This ARN was created using the <a
         * href="https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a> API action in
         * AWS Identity and Access Management (IAM).
         * </p>
         * 
         * @param roleARN
         *        The role ARN associated with this cluster. This ARN was created using the <a
         *        href="https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a> API
         *        action in AWS Identity and Access Management (IAM).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleARN(String roleARN);

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param clusterState
         *        The current status of the cluster.
         * @see ClusterState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterState
         */
        Builder clusterState(String clusterState);

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param clusterState
         *        The current status of the cluster.
         * @see ClusterState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterState
         */
        Builder clusterState(ClusterState clusterState);

        /**
         * <p>
         * The type of job for this cluster. Currently, the only job type supported for clusters is
         * <code>LOCAL_USE</code>.
         * </p>
         * 
         * @param jobType
         *        The type of job for this cluster. Currently, the only job type supported for clusters is
         *        <code>LOCAL_USE</code>.
         * @see JobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobType
         */
        Builder jobType(String jobType);

        /**
         * <p>
         * The type of job for this cluster. Currently, the only job type supported for clusters is
         * <code>LOCAL_USE</code>.
         * </p>
         * 
         * @param jobType
         *        The type of job for this cluster. Currently, the only job type supported for clusters is
         *        <code>LOCAL_USE</code>.
         * @see JobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobType
         */
        Builder jobType(JobType jobType);

        /**
         * <p>
         * The type of AWS Snowball device to use for this cluster. Currently, the only supported device type for
         * cluster jobs is <code>EDGE</code>.
         * </p>
         * 
         * @param snowballType
         *        The type of AWS Snowball device to use for this cluster. Currently, the only supported device type for
         *        cluster jobs is <code>EDGE</code>.
         * @see SnowballType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnowballType
         */
        Builder snowballType(String snowballType);

        /**
         * <p>
         * The type of AWS Snowball device to use for this cluster. Currently, the only supported device type for
         * cluster jobs is <code>EDGE</code>.
         * </p>
         * 
         * @param snowballType
         *        The type of AWS Snowball device to use for this cluster. Currently, the only supported device type for
         *        cluster jobs is <code>EDGE</code>.
         * @see SnowballType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnowballType
         */
        Builder snowballType(SnowballType snowballType);

        /**
         * <p>
         * The creation date for this cluster.
         * </p>
         * 
         * @param creationDate
         *        The creation date for this cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
         * <a>LambdaResource</a> objects.
         * </p>
         * 
         * @param resources
         *        The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
         *        <a>LambdaResource</a> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(JobResource resources);

        /**
         * <p>
         * The arrays of <a>JobResource</a> objects that can include updated <a>S3Resource</a> objects or
         * <a>LambdaResource</a> objects.
         * </p>
         * This is a convenience that creates an instance of the {@link JobResource.Builder} avoiding the need to create
         * one manually via {@link JobResource#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobResource.Builder#build()} is called immediately and its result
         * is passed to {@link #resources(JobResource)}.
         * 
         * @param resources
         *        a consumer that will call methods on {@link JobResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(JobResource)
         */
        default Builder resources(Consumer<JobResource.Builder> resources) {
            return resources(JobResource.builder().applyMutation(resources).build());
        }

        /**
         * <p>
         * The automatically generated ID for a specific address.
         * </p>
         * 
         * @param addressId
         *        The automatically generated ID for a specific address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addressId(String addressId);

        /**
         * <p>
         * The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
         * Snowball Edge device, rather it represents how quickly each device moves to its destination while in transit.
         * Regional shipping speeds are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * In Australia, you have access to express shipping. Typically, devices shipped express are delivered in about
         * a day.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped express
         * are delivered in about a day. In addition, most countries in the EU have access to standard shipping, which
         * typically takes less than a week, one way.
         * </p>
         * </li>
         * <li>
         * <p>
         * In India, Snowball Edges are delivered in one to seven days.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the US, you have access to one-day shipping and two-day shipping.
         * </p>
         * </li>
         * </ul>
         * 
         * @param shippingOption
         *        The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
         *        Snowball Edge device, rather it represents how quickly each device moves to its destination while in
         *        transit. Regional shipping speeds are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        In Australia, you have access to express shipping. Typically, devices shipped express are delivered in
         *        about a day.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped
         *        express are delivered in about a day. In addition, most countries in the EU have access to standard
         *        shipping, which typically takes less than a week, one way.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In India, Snowball Edges are delivered in one to seven days.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the US, you have access to one-day shipping and two-day shipping.
         *        </p>
         *        </li>
         * @see ShippingOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShippingOption
         */
        Builder shippingOption(String shippingOption);

        /**
         * <p>
         * The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
         * Snowball Edge device, rather it represents how quickly each device moves to its destination while in transit.
         * Regional shipping speeds are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * In Australia, you have access to express shipping. Typically, devices shipped express are delivered in about
         * a day.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped express
         * are delivered in about a day. In addition, most countries in the EU have access to standard shipping, which
         * typically takes less than a week, one way.
         * </p>
         * </li>
         * <li>
         * <p>
         * In India, Snowball Edges are delivered in one to seven days.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the US, you have access to one-day shipping and two-day shipping.
         * </p>
         * </li>
         * </ul>
         * 
         * @param shippingOption
         *        The shipping speed for each node in this cluster. This speed doesn't dictate how soon you'll get each
         *        Snowball Edge device, rather it represents how quickly each device moves to its destination while in
         *        transit. Regional shipping speeds are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        In Australia, you have access to express shipping. Typically, devices shipped express are delivered in
         *        about a day.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the European Union (EU), you have access to express shipping. Typically, Snowball Edges shipped
         *        express are delivered in about a day. In addition, most countries in the EU have access to standard
         *        shipping, which typically takes less than a week, one way.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In India, Snowball Edges are delivered in one to seven days.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the US, you have access to one-day shipping and two-day shipping.
         *        </p>
         *        </li>
         * @see ShippingOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShippingOption
         */
        Builder shippingOption(ShippingOption shippingOption);

        /**
         * <p>
         * The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
         * </p>
         * 
         * @param notification
         *        The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notification(Notification notification);

        /**
         * <p>
         * The Amazon Simple Notification Service (Amazon SNS) notification settings for this cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link Notification.Builder} avoiding the need to
         * create one manually via {@link Notification#builder()}.
         *
         * When the {@link Consumer} completes, {@link Notification.Builder#build()} is called immediately and its
         * result is passed to {@link #notification(Notification)}.
         * 
         * @param notification
         *        a consumer that will call methods on {@link Notification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notification(Notification)
         */
        default Builder notification(Consumer<Notification.Builder> notification) {
            return notification(Notification.builder().applyMutation(notification).build());
        }

        /**
         * <p>
         * The ID of the address that you want a cluster shipped to, after it will be shipped to its primary address.
         * This field is not supported in most regions.
         * </p>
         * 
         * @param forwardingAddressId
         *        The ID of the address that you want a cluster shipped to, after it will be shipped to its primary
         *        address. This field is not supported in most regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardingAddressId(String forwardingAddressId);
    }

    static final class BuilderImpl implements Builder {
        private String clusterId;

        private String description;

        private String kmsKeyARN;

        private String roleARN;

        private String clusterState;

        private String jobType;

        private String snowballType;

        private Instant creationDate;

        private JobResource resources;

        private String addressId;

        private String shippingOption;

        private Notification notification;

        private String forwardingAddressId;

        private BuilderImpl() {
        }

        private BuilderImpl(ClusterMetadata model) {
            clusterId(model.clusterId);
            description(model.description);
            kmsKeyARN(model.kmsKeyARN);
            roleARN(model.roleARN);
            clusterState(model.clusterState);
            jobType(model.jobType);
            snowballType(model.snowballType);
            creationDate(model.creationDate);
            resources(model.resources);
            addressId(model.addressId);
            shippingOption(model.shippingOption);
            notification(model.notification);
            forwardingAddressId(model.forwardingAddressId);
        }

        public final String getClusterId() {
            return clusterId;
        }

        @Override
        public final Builder clusterId(String clusterId) {
            this.clusterId = clusterId;
            return this;
        }

        public final void setClusterId(String clusterId) {
            this.clusterId = clusterId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getKmsKeyARN() {
            return kmsKeyARN;
        }

        @Override
        public final Builder kmsKeyARN(String kmsKeyARN) {
            this.kmsKeyARN = kmsKeyARN;
            return this;
        }

        public final void setKmsKeyARN(String kmsKeyARN) {
            this.kmsKeyARN = kmsKeyARN;
        }

        public final String getRoleARN() {
            return roleARN;
        }

        @Override
        public final Builder roleARN(String roleARN) {
            this.roleARN = roleARN;
            return this;
        }

        public final void setRoleARN(String roleARN) {
            this.roleARN = roleARN;
        }

        public final String getClusterStateAsString() {
            return clusterState;
        }

        @Override
        public final Builder clusterState(String clusterState) {
            this.clusterState = clusterState;
            return this;
        }

        @Override
        public final Builder clusterState(ClusterState clusterState) {
            this.clusterState(clusterState == null ? null : clusterState.toString());
            return this;
        }

        public final void setClusterState(String clusterState) {
            this.clusterState = clusterState;
        }

        public final String getJobTypeAsString() {
            return jobType;
        }

        @Override
        public final Builder jobType(String jobType) {
            this.jobType = jobType;
            return this;
        }

        @Override
        public final Builder jobType(JobType jobType) {
            this.jobType(jobType == null ? null : jobType.toString());
            return this;
        }

        public final void setJobType(String jobType) {
            this.jobType = jobType;
        }

        public final String getSnowballTypeAsString() {
            return snowballType;
        }

        @Override
        public final Builder snowballType(String snowballType) {
            this.snowballType = snowballType;
            return this;
        }

        @Override
        public final Builder snowballType(SnowballType snowballType) {
            this.snowballType(snowballType == null ? null : snowballType.toString());
            return this;
        }

        public final void setSnowballType(String snowballType) {
            this.snowballType = snowballType;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        public final JobResource.Builder getResources() {
            return resources != null ? resources.toBuilder() : null;
        }

        @Override
        public final Builder resources(JobResource resources) {
            this.resources = resources;
            return this;
        }

        public final void setResources(JobResource.BuilderImpl resources) {
            this.resources = resources != null ? resources.build() : null;
        }

        public final String getAddressId() {
            return addressId;
        }

        @Override
        public final Builder addressId(String addressId) {
            this.addressId = addressId;
            return this;
        }

        public final void setAddressId(String addressId) {
            this.addressId = addressId;
        }

        public final String getShippingOptionAsString() {
            return shippingOption;
        }

        @Override
        public final Builder shippingOption(String shippingOption) {
            this.shippingOption = shippingOption;
            return this;
        }

        @Override
        public final Builder shippingOption(ShippingOption shippingOption) {
            this.shippingOption(shippingOption == null ? null : shippingOption.toString());
            return this;
        }

        public final void setShippingOption(String shippingOption) {
            this.shippingOption = shippingOption;
        }

        public final Notification.Builder getNotification() {
            return notification != null ? notification.toBuilder() : null;
        }

        @Override
        public final Builder notification(Notification notification) {
            this.notification = notification;
            return this;
        }

        public final void setNotification(Notification.BuilderImpl notification) {
            this.notification = notification != null ? notification.build() : null;
        }

        public final String getForwardingAddressId() {
            return forwardingAddressId;
        }

        @Override
        public final Builder forwardingAddressId(String forwardingAddressId) {
            this.forwardingAddressId = forwardingAddressId;
            return this;
        }

        public final void setForwardingAddressId(String forwardingAddressId) {
            this.forwardingAddressId = forwardingAddressId;
        }

        @Override
        public ClusterMetadata build() {
            return new ClusterMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
