/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.snowball.model.CancelClusterRequest;
import software.amazon.awssdk.services.snowball.model.CancelClusterResponse;
import software.amazon.awssdk.services.snowball.model.CancelJobRequest;
import software.amazon.awssdk.services.snowball.model.CancelJobResponse;
import software.amazon.awssdk.services.snowball.model.ClusterLimitExceededException;
import software.amazon.awssdk.services.snowball.model.CreateAddressRequest;
import software.amazon.awssdk.services.snowball.model.CreateAddressResponse;
import software.amazon.awssdk.services.snowball.model.CreateClusterRequest;
import software.amazon.awssdk.services.snowball.model.CreateClusterResponse;
import software.amazon.awssdk.services.snowball.model.CreateJobRequest;
import software.amazon.awssdk.services.snowball.model.CreateJobResponse;
import software.amazon.awssdk.services.snowball.model.DescribeAddressRequest;
import software.amazon.awssdk.services.snowball.model.DescribeAddressResponse;
import software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest;
import software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse;
import software.amazon.awssdk.services.snowball.model.DescribeClusterRequest;
import software.amazon.awssdk.services.snowball.model.DescribeClusterResponse;
import software.amazon.awssdk.services.snowball.model.DescribeJobRequest;
import software.amazon.awssdk.services.snowball.model.DescribeJobResponse;
import software.amazon.awssdk.services.snowball.model.Ec2RequestFailedException;
import software.amazon.awssdk.services.snowball.model.GetJobManifestRequest;
import software.amazon.awssdk.services.snowball.model.GetJobManifestResponse;
import software.amazon.awssdk.services.snowball.model.GetJobUnlockCodeRequest;
import software.amazon.awssdk.services.snowball.model.GetJobUnlockCodeResponse;
import software.amazon.awssdk.services.snowball.model.GetSnowballUsageRequest;
import software.amazon.awssdk.services.snowball.model.GetSnowballUsageResponse;
import software.amazon.awssdk.services.snowball.model.GetSoftwareUpdatesRequest;
import software.amazon.awssdk.services.snowball.model.GetSoftwareUpdatesResponse;
import software.amazon.awssdk.services.snowball.model.InvalidAddressException;
import software.amazon.awssdk.services.snowball.model.InvalidInputCombinationException;
import software.amazon.awssdk.services.snowball.model.InvalidJobStateException;
import software.amazon.awssdk.services.snowball.model.InvalidNextTokenException;
import software.amazon.awssdk.services.snowball.model.InvalidResourceException;
import software.amazon.awssdk.services.snowball.model.KmsRequestFailedException;
import software.amazon.awssdk.services.snowball.model.ListClusterJobsRequest;
import software.amazon.awssdk.services.snowball.model.ListClusterJobsResponse;
import software.amazon.awssdk.services.snowball.model.ListClustersRequest;
import software.amazon.awssdk.services.snowball.model.ListClustersResponse;
import software.amazon.awssdk.services.snowball.model.ListCompatibleImagesRequest;
import software.amazon.awssdk.services.snowball.model.ListCompatibleImagesResponse;
import software.amazon.awssdk.services.snowball.model.ListJobsRequest;
import software.amazon.awssdk.services.snowball.model.ListJobsResponse;
import software.amazon.awssdk.services.snowball.model.SnowballException;
import software.amazon.awssdk.services.snowball.model.SnowballRequest;
import software.amazon.awssdk.services.snowball.model.UnsupportedAddressException;
import software.amazon.awssdk.services.snowball.model.UpdateClusterRequest;
import software.amazon.awssdk.services.snowball.model.UpdateClusterResponse;
import software.amazon.awssdk.services.snowball.model.UpdateJobRequest;
import software.amazon.awssdk.services.snowball.model.UpdateJobResponse;
import software.amazon.awssdk.services.snowball.paginators.DescribeAddressesIterable;
import software.amazon.awssdk.services.snowball.paginators.ListJobsIterable;
import software.amazon.awssdk.services.snowball.transform.CancelClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateAddressRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressesRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobManifestRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobUnlockCodeRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetSnowballUsageRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetSoftwareUpdatesRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClusterJobsRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListCompatibleImagesRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateJobRequestMarshaller;

/**
 * Internal implementation of {@link SnowballClient}.
 *
 * @see SnowballClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnowballClient implements SnowballClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSnowballClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a cluster job. You can only cancel a cluster job while it's in the <code>AwaitingQuorum</code> status.
     * You'll have at least an hour after creating a cluster job to cancel it.
     * </p>
     *
     * @param cancelClusterRequest
     * @return Result of the CancelCluster operation returned by the service.
     * @throws KmsRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CancelCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CancelCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelClusterResponse cancelCluster(CancelClusterRequest cancelClusterRequest) throws KmsRequestFailedException,
            InvalidJobStateException, InvalidResourceException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelCluster");

            return clientHandler.execute(new ClientExecutionParams<CancelClusterRequest, CancelClusterResponse>()
                    .withOperationName("CancelCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelClusterRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelClusterRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels the specified job. You can only cancel a job before its <code>JobState</code> value changes to
     * <code>PreparingAppliance</code>. Requesting the <code>ListJobs</code> or <code>DescribeJob</code> action returns
     * a job's <code>JobState</code> as part of the response element data returned.
     * </p>
     *
     * @param cancelJobRequest
     * @return Result of the CancelJob operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws KmsRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CancelJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelJobResponse cancelJob(CancelJobRequest cancelJobRequest) throws InvalidResourceException,
            InvalidJobStateException, KmsRequestFailedException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");

            return clientHandler.execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>()
                    .withOperationName("CancelJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CancelJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an address for a Snowball to be shipped to. In most regions, addresses are validated at the time of
     * creation. The address you provide must be located within the serviceable area of your region. If the address is
     * invalid or unsupported, then an exception is thrown.
     * </p>
     *
     * @param createAddressRequest
     * @return Result of the CreateAddress operation returned by the service.
     * @throws InvalidAddressException
     *         The address provided was invalid. Check the address with your region's carrier, and try again.
     * @throws UnsupportedAddressException
     *         The address is either outside the serviceable area for your region, or an error occurred. Check the
     *         address with your region's carrier and try again. If the issue persists, contact AWS Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CreateAddress
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAddressResponse createAddress(CreateAddressRequest createAddressRequest) throws InvalidAddressException,
            UnsupportedAddressException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAddressResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAddressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAddress");

            return clientHandler.execute(new ClientExecutionParams<CreateAddressRequest, CreateAddressResponse>()
                    .withOperationName("CreateAddress").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAddressRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAddressRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAddressRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an empty cluster. Each cluster supports five nodes. You use the <a>CreateJob</a> action separately to
     * create the jobs for each of these nodes. The cluster does not ship until these five node jobs have been created.
     * </p>
     *
     * @param createClusterRequest
     * @return Result of the CreateCluster operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws KmsRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidInputCombinationException
     *         Job or cluster creation failed. One or more inputs were invalid. Confirm that the
     *         <a>CreateClusterRequest&#36SnowballType</a> value supports your <a>CreateJobRequest&#36JobType</a>, and
     *         try again.
     * @throws Ec2RequestFailedException
     *         Your IAM user lacks the necessary Amazon EC2 permissions to perform the attempted action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CreateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws InvalidResourceException,
            KmsRequestFailedException, InvalidInputCombinationException, Ec2RequestFailedException, AwsServiceException,
            SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");

            return clientHandler.execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                    .withOperationName("CreateCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a job to the other job attributes are inherited from the cluster.
     * </p>
     *
     * @param createJobRequest
     * @return Result of the CreateJob operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws KmsRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidInputCombinationException
     *         Job or cluster creation failed. One or more inputs were invalid. Confirm that the
     *         <a>CreateClusterRequest&#36SnowballType</a> value supports your <a>CreateJobRequest&#36JobType</a>, and
     *         try again.
     * @throws ClusterLimitExceededException
     *         Job creation failed. Currently, clusters support five nodes. If you have less than five nodes for your
     *         cluster and you have more nodes to create for this cluster, try again and create jobs until your cluster
     *         has exactly five notes.
     * @throws Ec2RequestFailedException
     *         Your IAM user lacks the necessary Amazon EC2 permissions to perform the attempted action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.CreateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateJobResponse createJob(CreateJobRequest createJobRequest) throws InvalidResourceException,
            KmsRequestFailedException, InvalidInputCombinationException, ClusterLimitExceededException,
            Ec2RequestFailedException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");

            return clientHandler.execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>()
                    .withOperationName("CreateJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Takes an <code>AddressId</code> and returns specific details about that address in the form of an
     * <code>Address</code> object.
     * </p>
     *
     * @param describeAddressRequest
     * @return Result of the DescribeAddress operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeAddress
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAddressResponse describeAddress(DescribeAddressRequest describeAddressRequest)
            throws InvalidResourceException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAddressResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAddressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAddress");

            return clientHandler.execute(new ClientExecutionParams<DescribeAddressRequest, DescribeAddressResponse>()
                    .withOperationName("DescribeAddress").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAddressRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAddressRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAddressRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a specified number of <code>ADDRESS</code> objects. Calling this API in one of the US regions will return
     * addresses from the list of all addresses associated with this account in all US regions.
     * </p>
     *
     * @param describeAddressesRequest
     * @return Result of the DescribeAddresses operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddresses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAddressesResponse describeAddresses(DescribeAddressesRequest describeAddressesRequest)
            throws InvalidResourceException, InvalidNextTokenException, AwsServiceException, SdkClientException,
            SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAddressesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAddressesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAddresses");

            return clientHandler.execute(new ClientExecutionParams<DescribeAddressesRequest, DescribeAddressesResponse>()
                    .withOperationName("DescribeAddresses").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAddressesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAddressesRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAddressesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a specified number of <code>ADDRESS</code> objects. Calling this API in one of the US regions will return
     * addresses from the list of all addresses associated with this account in all US regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAddresses(software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.DescribeAddressesIterable responses = client.describeAddressesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.snowball.paginators.DescribeAddressesIterable responses = client
     *             .describeAddressesPaginator(request);
     *     for (software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.DescribeAddressesIterable responses = client.describeAddressesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAddresses(software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAddressesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddresses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAddressesIterable describeAddressesPaginator(DescribeAddressesRequest describeAddressesRequest)
            throws InvalidResourceException, InvalidNextTokenException, AwsServiceException, SdkClientException,
            SnowballException {
        return new DescribeAddressesIterable(this, applyPaginatorUserAgent(describeAddressesRequest));
    }

    /**
     * <p>
     * Returns information about a specific cluster including shipping information, cluster status, and other important
     * metadata.
     * </p>
     *
     * @param describeClusterRequest
     * @return Result of the DescribeCluster operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeClusterResponse describeCluster(DescribeClusterRequest describeClusterRequest)
            throws InvalidResourceException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCluster");

            return clientHandler.execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                    .withOperationName("DescribeCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClusterRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specific job including shipping information, job status, and other important
     * metadata.
     * </p>
     *
     * @param describeJobRequest
     * @return Result of the DescribeJob operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.DescribeJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobResponse describeJob(DescribeJobRequest describeJobRequest) throws InvalidResourceException,
            AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                    .withOperationName("DescribeJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a link to an Amazon S3 presigned URL for the manifest file associated with the specified
     * <code>JobId</code> value. You can access the manifest file for up to 60 minutes after this request has been made.
     * To access the manifest file after 60 minutes have passed, you'll have to make another call to the
     * <code>GetJobManifest</code> action.
     * </p>
     * <p>
     * The manifest is an encrypted file that you can download after your job enters the <code>WithCustomer</code>
     * status. The manifest is decrypted by using the <code>UnlockCode</code> code value, when you pass both values to
     * the Snowball through the Snowball client when the client is started for the first time.
     * </p>
     * <p>
     * As a best practice, we recommend that you don't save a copy of an <code>UnlockCode</code> value in the same
     * location as the manifest file for that job. Saving these separately helps prevent unauthorized parties from
     * gaining access to the Snowball associated with that job.
     * </p>
     * <p>
     * The credentials of a given job, including its manifest file and unlock code, expire 90 days after the job is
     * created.
     * </p>
     *
     * @param getJobManifestRequest
     * @return Result of the GetJobManifest operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.GetJobManifest
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetJobManifest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobManifestResponse getJobManifest(GetJobManifestRequest getJobManifestRequest) throws InvalidResourceException,
            InvalidJobStateException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetJobManifestResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobManifestResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobManifest");

            return clientHandler.execute(new ClientExecutionParams<GetJobManifestRequest, GetJobManifestResponse>()
                    .withOperationName("GetJobManifest").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getJobManifestRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetJobManifestRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobManifestRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the <code>UnlockCode</code> code value for the specified job. A particular <code>UnlockCode</code> value
     * can be accessed for up to 90 days after the associated job has been created.
     * </p>
     * <p>
     * The <code>UnlockCode</code> value is a 29-character code with 25 alphanumeric characters and 4 hyphens. This code
     * is used to decrypt the manifest file when it is passed along with the manifest to the Snowball through the
     * Snowball client when the client is started for the first time.
     * </p>
     * <p>
     * As a best practice, we recommend that you don't save a copy of the <code>UnlockCode</code> in the same location
     * as the manifest file for that job. Saving these separately helps prevent unauthorized parties from gaining access
     * to the Snowball associated with that job.
     * </p>
     *
     * @param getJobUnlockCodeRequest
     * @return Result of the GetJobUnlockCode operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.GetJobUnlockCode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetJobUnlockCode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobUnlockCodeResponse getJobUnlockCode(GetJobUnlockCodeRequest getJobUnlockCodeRequest)
            throws InvalidResourceException, InvalidJobStateException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetJobUnlockCodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobUnlockCodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobUnlockCode");

            return clientHandler.execute(new ClientExecutionParams<GetJobUnlockCodeRequest, GetJobUnlockCodeResponse>()
                    .withOperationName("GetJobUnlockCode").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getJobUnlockCodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetJobUnlockCodeRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobUnlockCodeRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the Snowball service limit for your account, and also the number of Snowballs your
     * account has in use.
     * </p>
     * <p>
     * The default service limit for the number of Snowballs that you can have at one time is 1. If you want to increase
     * your service limit, contact AWS Support.
     * </p>
     *
     * @param getSnowballUsageRequest
     * @return Result of the GetSnowballUsage operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.GetSnowballUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetSnowballUsage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSnowballUsageResponse getSnowballUsage(GetSnowballUsageRequest getSnowballUsageRequest) throws AwsServiceException,
            SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSnowballUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSnowballUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSnowballUsage");

            return clientHandler.execute(new ClientExecutionParams<GetSnowballUsageRequest, GetSnowballUsageResponse>()
                    .withOperationName("GetSnowballUsage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSnowballUsageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSnowballUsageRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSnowballUsageRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an Amazon S3 presigned URL for an update file associated with a specified <code>JobId</code>.
     * </p>
     *
     * @param getSoftwareUpdatesRequest
     * @return Result of the GetSoftwareUpdates operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.GetSoftwareUpdates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetSoftwareUpdates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSoftwareUpdatesResponse getSoftwareUpdates(GetSoftwareUpdatesRequest getSoftwareUpdatesRequest)
            throws InvalidResourceException, InvalidJobStateException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSoftwareUpdatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSoftwareUpdatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSoftwareUpdates");

            return clientHandler.execute(new ClientExecutionParams<GetSoftwareUpdatesRequest, GetSoftwareUpdatesResponse>()
                    .withOperationName("GetSoftwareUpdates").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSoftwareUpdatesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSoftwareUpdatesRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSoftwareUpdatesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object is for a job in the specified cluster and contains a job's state, a job's ID, and other information.
     * </p>
     *
     * @param listClusterJobsRequest
     * @return Result of the ListClusterJobs operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListClusterJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListClusterJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListClusterJobsResponse listClusterJobs(ListClusterJobsRequest listClusterJobsRequest)
            throws InvalidResourceException, InvalidNextTokenException, AwsServiceException, SdkClientException,
            SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListClusterJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListClusterJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusterJobs");

            return clientHandler.execute(new ClientExecutionParams<ListClusterJobsRequest, ListClusterJobsResponse>()
                    .withOperationName("ListClusterJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listClusterJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListClusterJobsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClusterJobsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>ClusterListEntry</code> objects of the specified length. Each
     * <code>ClusterListEntry</code> object contains a cluster's state, a cluster's ID, and other important status
     * information.
     * </p>
     *
     * @param listClustersRequest
     * @return Result of the ListClusters operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusters");

            return clientHandler.execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                    .withOperationName("ListClusters").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListClustersRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClustersRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action returns a list of the different Amazon EC2 Amazon Machine Images (AMIs) that are owned by your AWS
     * account that would be supported for use on a Snowball Edge device. Currently, supported AMIs are based on the
     * CentOS 7 (x86_64) - with Updates HVM, Ubuntu Server 14.04 LTS (HVM), and Ubuntu 16.04 LTS - Xenial (HVM) images,
     * available on the AWS Marketplace.
     * </p>
     *
     * @param listCompatibleImagesRequest
     * @return Result of the ListCompatibleImages operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws Ec2RequestFailedException
     *         Your IAM user lacks the necessary Amazon EC2 permissions to perform the attempted action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListCompatibleImages
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListCompatibleImages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListCompatibleImagesResponse listCompatibleImages(ListCompatibleImagesRequest listCompatibleImagesRequest)
            throws InvalidNextTokenException, Ec2RequestFailedException, AwsServiceException, SdkClientException,
            SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCompatibleImagesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCompatibleImagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCompatibleImages");

            return clientHandler.execute(new ClientExecutionParams<ListCompatibleImagesRequest, ListCompatibleImagesResponse>()
                    .withOperationName("ListCompatibleImages").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listCompatibleImagesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCompatibleImagesRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCompatibleImagesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object contains a job's state, a job's ID, and a value that indicates whether the job is a job part, in the case
     * of export jobs. Calling this API action in one of the US regions will return jobs from the list of all jobs
     * associated with this account in all US regions.
     * </p>
     *
     * @param listJobsRequest
     * @return Result of the ListJobs operation returned by the service.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws InvalidNextTokenException, AwsServiceException,
            SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");

            return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>()
                    .withOperationName("ListJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object contains a job's state, a job's ID, and a value that indicates whether the job is a job part, in the case
     * of export jobs. Calling this API action in one of the US regions will return jobs from the list of all jobs
     * associated with this account in all US regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.snowball.model.ListJobsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.snowball.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.snowball.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.snowball.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The <code>NextToken</code> string was altered unexpectedly, and the operation has stopped. Run the
     *         operation without changing the <code>NextToken</code> string, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsIterable listJobsPaginator(ListJobsRequest listJobsRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, SnowballException {
        return new ListJobsIterable(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * While a cluster's <code>ClusterState</code> value is in the <code>AwaitingQuorum</code> state, you can update
     * some of the information associated with a cluster. Once the cluster changes to a different job state, usually 60
     * minutes after the cluster being created, this action is no longer available.
     * </p>
     *
     * @param updateClusterRequest
     * @return Result of the UpdateCluster operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws KmsRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidInputCombinationException
     *         Job or cluster creation failed. One or more inputs were invalid. Confirm that the
     *         <a>CreateClusterRequest&#36SnowballType</a> value supports your <a>CreateJobRequest&#36JobType</a>, and
     *         try again.
     * @throws Ec2RequestFailedException
     *         Your IAM user lacks the necessary Amazon EC2 permissions to perform the attempted action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.UpdateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateClusterResponse updateCluster(UpdateClusterRequest updateClusterRequest) throws InvalidResourceException,
            InvalidJobStateException, KmsRequestFailedException, InvalidInputCombinationException, Ec2RequestFailedException,
            AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCluster");

            return clientHandler.execute(new ClientExecutionParams<UpdateClusterRequest, UpdateClusterResponse>()
                    .withOperationName("UpdateCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateClusterRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateClusterRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * While a job's <code>JobState</code> value is <code>New</code>, you can update some of the information associated
     * with a job. Once the job changes to a different job state, usually within 60 minutes of the job being created,
     * this action is no longer available.
     * </p>
     *
     * @param updateJobRequest
     * @return Result of the UpdateJob operation returned by the service.
     * @throws InvalidResourceException
     *         The specified resource can't be found. Check the information you provided in your last request, and try
     *         again.
     * @throws InvalidJobStateException
     *         The action can't be performed because the job's current state doesn't allow that action to be performed.
     * @throws KmsRequestFailedException
     *         The provided AWS Key Management Service key lacks the permissions to perform the specified
     *         <a>CreateJob</a> or <a>UpdateJob</a> action.
     * @throws InvalidInputCombinationException
     *         Job or cluster creation failed. One or more inputs were invalid. Confirm that the
     *         <a>CreateClusterRequest&#36SnowballType</a> value supports your <a>CreateJobRequest&#36JobType</a>, and
     *         try again.
     * @throws ClusterLimitExceededException
     *         Job creation failed. Currently, clusters support five nodes. If you have less than five nodes for your
     *         cluster and you have more nodes to create for this cluster, try again and create jobs until your cluster
     *         has exactly five notes.
     * @throws Ec2RequestFailedException
     *         Your IAM user lacks the necessary Amazon EC2 permissions to perform the attempted action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnowballException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnowballClient.UpdateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateJobResponse updateJob(UpdateJobRequest updateJobRequest) throws InvalidResourceException,
            InvalidJobStateException, KmsRequestFailedException, InvalidInputCombinationException, ClusterLimitExceededException,
            Ec2RequestFailedException, AwsServiceException, SdkClientException, SnowballException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJob");

            return clientHandler.execute(new ClientExecutionParams<UpdateJobRequest, UpdateJobResponse>()
                    .withOperationName("UpdateJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new UpdateJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SnowballException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidJobStateException")
                                .exceptionBuilderSupplier(InvalidJobStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceException")
                                .exceptionBuilderSupplier(InvalidResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedAddressException")
                                .exceptionBuilderSupplier(UnsupportedAddressException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClusterLimitExceededException")
                                .exceptionBuilderSupplier(ClusterLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputCombinationException")
                                .exceptionBuilderSupplier(InvalidInputCombinationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("Ec2RequestFailedException")
                                .exceptionBuilderSupplier(Ec2RequestFailedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSRequestFailedException")
                                .exceptionBuilderSupplier(KmsRequestFailedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidAddressException")
                                .exceptionBuilderSupplier(InvalidAddressException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SnowballRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
