/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowball;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.snowball.model.CancelClusterRequest;
import software.amazon.awssdk.services.snowball.model.CancelClusterResponse;
import software.amazon.awssdk.services.snowball.model.CancelJobRequest;
import software.amazon.awssdk.services.snowball.model.CancelJobResponse;
import software.amazon.awssdk.services.snowball.model.ClusterLimitExceededException;
import software.amazon.awssdk.services.snowball.model.ConflictException;
import software.amazon.awssdk.services.snowball.model.CreateAddressRequest;
import software.amazon.awssdk.services.snowball.model.CreateAddressResponse;
import software.amazon.awssdk.services.snowball.model.CreateClusterRequest;
import software.amazon.awssdk.services.snowball.model.CreateClusterResponse;
import software.amazon.awssdk.services.snowball.model.CreateJobRequest;
import software.amazon.awssdk.services.snowball.model.CreateJobResponse;
import software.amazon.awssdk.services.snowball.model.CreateLongTermPricingRequest;
import software.amazon.awssdk.services.snowball.model.CreateLongTermPricingResponse;
import software.amazon.awssdk.services.snowball.model.CreateReturnShippingLabelRequest;
import software.amazon.awssdk.services.snowball.model.CreateReturnShippingLabelResponse;
import software.amazon.awssdk.services.snowball.model.DescribeAddressRequest;
import software.amazon.awssdk.services.snowball.model.DescribeAddressResponse;
import software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest;
import software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse;
import software.amazon.awssdk.services.snowball.model.DescribeClusterRequest;
import software.amazon.awssdk.services.snowball.model.DescribeClusterResponse;
import software.amazon.awssdk.services.snowball.model.DescribeJobRequest;
import software.amazon.awssdk.services.snowball.model.DescribeJobResponse;
import software.amazon.awssdk.services.snowball.model.DescribeReturnShippingLabelRequest;
import software.amazon.awssdk.services.snowball.model.DescribeReturnShippingLabelResponse;
import software.amazon.awssdk.services.snowball.model.Ec2RequestFailedException;
import software.amazon.awssdk.services.snowball.model.GetJobManifestRequest;
import software.amazon.awssdk.services.snowball.model.GetJobManifestResponse;
import software.amazon.awssdk.services.snowball.model.GetJobUnlockCodeRequest;
import software.amazon.awssdk.services.snowball.model.GetJobUnlockCodeResponse;
import software.amazon.awssdk.services.snowball.model.GetSnowballUsageRequest;
import software.amazon.awssdk.services.snowball.model.GetSnowballUsageResponse;
import software.amazon.awssdk.services.snowball.model.GetSoftwareUpdatesRequest;
import software.amazon.awssdk.services.snowball.model.GetSoftwareUpdatesResponse;
import software.amazon.awssdk.services.snowball.model.InvalidAddressException;
import software.amazon.awssdk.services.snowball.model.InvalidInputCombinationException;
import software.amazon.awssdk.services.snowball.model.InvalidJobStateException;
import software.amazon.awssdk.services.snowball.model.InvalidNextTokenException;
import software.amazon.awssdk.services.snowball.model.InvalidResourceException;
import software.amazon.awssdk.services.snowball.model.KmsRequestFailedException;
import software.amazon.awssdk.services.snowball.model.ListClusterJobsRequest;
import software.amazon.awssdk.services.snowball.model.ListClusterJobsResponse;
import software.amazon.awssdk.services.snowball.model.ListClustersRequest;
import software.amazon.awssdk.services.snowball.model.ListClustersResponse;
import software.amazon.awssdk.services.snowball.model.ListCompatibleImagesRequest;
import software.amazon.awssdk.services.snowball.model.ListCompatibleImagesResponse;
import software.amazon.awssdk.services.snowball.model.ListJobsRequest;
import software.amazon.awssdk.services.snowball.model.ListJobsResponse;
import software.amazon.awssdk.services.snowball.model.ListLongTermPricingRequest;
import software.amazon.awssdk.services.snowball.model.ListLongTermPricingResponse;
import software.amazon.awssdk.services.snowball.model.ReturnShippingLabelAlreadyExistsException;
import software.amazon.awssdk.services.snowball.model.SnowballException;
import software.amazon.awssdk.services.snowball.model.SnowballRequest;
import software.amazon.awssdk.services.snowball.model.UnsupportedAddressException;
import software.amazon.awssdk.services.snowball.model.UpdateClusterRequest;
import software.amazon.awssdk.services.snowball.model.UpdateClusterResponse;
import software.amazon.awssdk.services.snowball.model.UpdateJobRequest;
import software.amazon.awssdk.services.snowball.model.UpdateJobResponse;
import software.amazon.awssdk.services.snowball.model.UpdateJobShipmentStateRequest;
import software.amazon.awssdk.services.snowball.model.UpdateJobShipmentStateResponse;
import software.amazon.awssdk.services.snowball.model.UpdateLongTermPricingRequest;
import software.amazon.awssdk.services.snowball.model.UpdateLongTermPricingResponse;
import software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPublisher;
import software.amazon.awssdk.services.snowball.paginators.ListJobsPublisher;
import software.amazon.awssdk.services.snowball.transform.CancelClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateAddressRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateLongTermPricingRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.CreateReturnShippingLabelRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeAddressesRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.DescribeReturnShippingLabelRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobManifestRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetJobUnlockCodeRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetSnowballUsageRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.GetSoftwareUpdatesRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClusterJobsRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListCompatibleImagesRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.ListLongTermPricingRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateClusterRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateJobRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateJobShipmentStateRequestMarshaller;
import software.amazon.awssdk.services.snowball.transform.UpdateLongTermPricingRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SnowballAsyncClient}.
 *
 * @see SnowballAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnowballAsyncClient implements SnowballAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSnowballAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSnowballAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a cluster job. You can only cancel a cluster job while it's in the <code>AwaitingQuorum</code> status.
     * You'll have at least an hour after creating a cluster job to cancel it.
     * </p>
     *
     * @param cancelClusterRequest
     * @return A Java Future containing the result of the CancelCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>KmsRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CancelCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CancelCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelClusterResponse> cancelCluster(CancelClusterRequest cancelClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelClusterRequest, CancelClusterResponse>()
                            .withOperationName("CancelCluster")
                            .withMarshaller(new CancelClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelClusterRequest));
            CompletableFuture<CancelClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels the specified job. You can only cancel a job before its <code>JobState</code> value changes to
     * <code>PreparingAppliance</code>. Requesting the <code>ListJobs</code> or <code>DescribeJob</code> action returns
     * a job's <code>JobState</code> as part of the response element data returned.
     * </p>
     *
     * @param cancelJobRequest
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>KmsRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>().withOperationName("CancelJob")
                            .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelJobRequest));
            CompletableFuture<CancelJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an address for a Snow device to be shipped to. In most regions, addresses are validated at the time of
     * creation. The address you provide must be located within the serviceable area of your region. If the address is
     * invalid or unsupported, then an exception is thrown.
     * </p>
     *
     * @param createAddressRequest
     * @return A Java Future containing the result of the CreateAddress operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidAddressException The address provided was invalid. Check the address with your region's
     *         carrier, and try again.</li>
     *         <li>UnsupportedAddressException The address is either outside the serviceable area for your region, or an
     *         error occurred. Check the address with your region's carrier and try again. If the issue persists,
     *         contact AWS Support.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CreateAddress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAddressResponse> createAddress(CreateAddressRequest createAddressRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAddressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAddress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAddressResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAddressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAddressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAddressRequest, CreateAddressResponse>()
                            .withOperationName("CreateAddress")
                            .withMarshaller(new CreateAddressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAddressRequest));
            CompletableFuture<CreateAddressResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an empty cluster. Each cluster supports five nodes. You use the <a>CreateJob</a> action separately to
     * create the jobs for each of these nodes. The cluster does not ship until these five node jobs have been created.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>KmsRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One or more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>Ec2RequestFailedException Your IAM user lacks the necessary Amazon EC2 permissions to perform the
     *         attempted action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                            .withOperationName("CreateCluster")
                            .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createClusterRequest));
            CompletableFuture<CreateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a job to import or export data between Amazon S3 and your on-premises data center. Your AWS account must
     * have the right trust policies and permissions in place to create a job for a Snow device. If you're creating a
     * job for a node in a cluster, you only need to provide the <code>clusterId</code> value; the other job attributes
     * are inherited from the cluster.
     * </p>
     * <note>
     * <p>
     * Only the Snowball; Edge device type is supported when ordering clustered jobs.
     * </p>
     * <p>
     * The device capacity is optional.
     * </p>
     * <p>
     * Availability of device types differ by AWS Region. For more information about Region availability, see <a
     * href="https://aws.amazon.com/about-aws/global-infrastructure/regional-product-services/?p=ngi&amp;loc=4">AWS
     * Regional Services</a>.
     * </p>
     * </note>
     * <p/>
     * <p class="title">
     * <b>AWS Snow Family device types and their capacities.</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Snow Family device type: <b>SNC1_SSD</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Capacity: T14
     * </p>
     * </li>
     * <li>
     * <p>
     * Description: Snowcone
     * </p>
     * </li>
     * </ul>
     * <p/></li>
     * <li>
     * <p>
     * Snow Family device type: <b>SNC1_HDD</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Capacity: T8
     * </p>
     * </li>
     * <li>
     * <p>
     * Description: Snowcone
     * </p>
     * </li>
     * </ul>
     * <p/></li>
     * <li>
     * <p>
     * Device type: <b>EDGE_S</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Capacity: T98
     * </p>
     * </li>
     * <li>
     * <p>
     * Description: Snowball Edge Storage Optimized for data transfer only
     * </p>
     * </li>
     * </ul>
     * <p/></li>
     * <li>
     * <p>
     * Device type: <b>EDGE_CG</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Capacity: T42
     * </p>
     * </li>
     * <li>
     * <p>
     * Description: Snowball Edge Compute Optimized with GPU
     * </p>
     * </li>
     * </ul>
     * <p/></li>
     * <li>
     * <p>
     * Device type: <b>EDGE_C</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Capacity: T42
     * </p>
     * </li>
     * <li>
     * <p>
     * Description: Snowball Edge Compute Optimized without GPU
     * </p>
     * </li>
     * </ul>
     * <p/></li>
     * <li>
     * <p>
     * Device type: <b>EDGE</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Capacity: T100
     * </p>
     * </li>
     * <li>
     * <p>
     * Description: Snowball Edge Storage Optimized with EC2 Compute
     * </p>
     * </li>
     * </ul>
     * <p/></li>
     * <li>
     * <p>
     * Device type: <b>STANDARD</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Capacity: T50
     * </p>
     * </li>
     * <li>
     * <p>
     * Description: Original Snowball device
     * </p>
     * <note>
     * <p>
     * This device is only available in the Ningxia, Beijing, and Singapore AWS Regions.
     * </p>
     * </note></li>
     * </ul>
     * <p/></li>
     * <li>
     * <p>
     * Device type: <b>STANDARD</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Capacity: T80
     * </p>
     * </li>
     * <li>
     * <p>
     * Description: Original Snowball device
     * </p>
     * <note>
     * <p>
     * This device is only available in the Ningxia, Beijing, and Singapore AWS Regions.
     * </p>
     * </note></li>
     * </ul>
     * <p/></li>
     * </ul>
     *
     * @param createJobRequest
     * @return A Java Future containing the result of the CreateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>KmsRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One or more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>ClusterLimitExceededException Job creation failed. Currently, clusters support five nodes. If you
     *         have fewer than five nodes for your cluster and you have more nodes to create for this cluster, try again
     *         and create jobs until your cluster has exactly five nodes.</li>
     *         <li>Ec2RequestFailedException Your IAM user lacks the necessary Amazon EC2 permissions to perform the
     *         attempted action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CreateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobResponse> createJob(CreateJobRequest createJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>().withOperationName("CreateJob")
                            .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(createJobRequest));
            CompletableFuture<CreateJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a job with the long-term usage option for a device. The long-term usage is a 1-year or 3-year long-term
     * pricing type for the device. You are billed upfront, and AWS provides discounts for long-term pricing.
     * </p>
     *
     * @param createLongTermPricingRequest
     * @return A Java Future containing the result of the CreateLongTermPricing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CreateLongTermPricing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateLongTermPricing"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLongTermPricingResponse> createLongTermPricing(
            CreateLongTermPricingRequest createLongTermPricingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLongTermPricingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLongTermPricing");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLongTermPricingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLongTermPricingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLongTermPricingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLongTermPricingRequest, CreateLongTermPricingResponse>()
                            .withOperationName("CreateLongTermPricing")
                            .withMarshaller(new CreateLongTermPricingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLongTermPricingRequest));
            CompletableFuture<CreateLongTermPricingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a shipping label that will be used to return the Snow device to AWS.
     * </p>
     *
     * @param createReturnShippingLabelRequest
     * @return A Java Future containing the result of the CreateReturnShippingLabel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One or more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>ConflictException You get this exception when you call <code>CreateReturnShippingLabel</code> more
     *         than once when other requests are not completed.</li>
     *         <li>ReturnShippingLabelAlreadyExistsException You get this exception if you call
     *         <code>CreateReturnShippingLabel</code> and a valid return shipping label already exists. In this case,
     *         use <code>DescribeReturnShippingLabel</code> to get the url.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.CreateReturnShippingLabel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/CreateReturnShippingLabel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateReturnShippingLabelResponse> createReturnShippingLabel(
            CreateReturnShippingLabelRequest createReturnShippingLabelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createReturnShippingLabelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateReturnShippingLabel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateReturnShippingLabelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateReturnShippingLabelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateReturnShippingLabelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateReturnShippingLabelRequest, CreateReturnShippingLabelResponse>()
                            .withOperationName("CreateReturnShippingLabel")
                            .withMarshaller(new CreateReturnShippingLabelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createReturnShippingLabelRequest));
            CompletableFuture<CreateReturnShippingLabelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Takes an <code>AddressId</code> and returns specific details about that address in the form of an
     * <code>Address</code> object.
     * </p>
     *
     * @param describeAddressRequest
     * @return A Java Future containing the result of the DescribeAddress operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeAddress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAddressResponse> describeAddress(DescribeAddressRequest describeAddressRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAddressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAddress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAddressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAddressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAddressRequest, DescribeAddressResponse>()
                            .withOperationName("DescribeAddress")
                            .withMarshaller(new DescribeAddressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAddressRequest));
            CompletableFuture<DescribeAddressResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a specified number of <code>ADDRESS</code> objects. Calling this API in one of the US regions will return
     * addresses from the list of all addresses associated with this account in all US regions.
     * </p>
     *
     * @param describeAddressesRequest
     * @return A Java Future containing the result of the DescribeAddresses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeAddresses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddresses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAddressesResponse> describeAddresses(DescribeAddressesRequest describeAddressesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAddressesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAddresses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAddressesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAddressesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAddressesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAddressesRequest, DescribeAddressesResponse>()
                            .withOperationName("DescribeAddresses")
                            .withMarshaller(new DescribeAddressesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAddressesRequest));
            CompletableFuture<DescribeAddressesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a specified number of <code>ADDRESS</code> objects. Calling this API in one of the US regions will return
     * addresses from the list of all addresses associated with this account in all US regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAddresses(software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPublisher publisher = client.describeAddressesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.DescribeAddressesPublisher publisher = client.describeAddressesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.snowball.model.DescribeAddressesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAddresses(software.amazon.awssdk.services.snowball.model.DescribeAddressesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAddressesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeAddresses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeAddresses" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeAddressesPublisher describeAddressesPaginator(DescribeAddressesRequest describeAddressesRequest) {
        return new DescribeAddressesPublisher(this, applyPaginatorUserAgent(describeAddressesRequest));
    }

    /**
     * <p>
     * Returns information about a specific cluster including shipping information, cluster status, and other important
     * metadata.
     * </p>
     *
     * @param describeClusterRequest
     * @return A Java Future containing the result of the DescribeCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeClusterResponse> describeCluster(DescribeClusterRequest describeClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                            .withOperationName("DescribeCluster")
                            .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeClusterRequest));
            CompletableFuture<DescribeClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specific job including shipping information, job status, and other important
     * metadata.
     * </p>
     *
     * @param describeJobRequest
     * @return A Java Future containing the result of the DescribeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobResponse> describeJob(DescribeJobRequest describeJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                            .withOperationName("DescribeJob").withMarshaller(new DescribeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobRequest));
            CompletableFuture<DescribeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Information on the shipping label of a Snow device that is being returned to AWS.
     * </p>
     *
     * @param describeReturnShippingLabelRequest
     * @return A Java Future containing the result of the DescribeReturnShippingLabel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>ConflictException You get this exception when you call <code>CreateReturnShippingLabel</code> more
     *         than once when other requests are not completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.DescribeReturnShippingLabel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/DescribeReturnShippingLabel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeReturnShippingLabelResponse> describeReturnShippingLabel(
            DescribeReturnShippingLabelRequest describeReturnShippingLabelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReturnShippingLabelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReturnShippingLabel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeReturnShippingLabelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeReturnShippingLabelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeReturnShippingLabelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeReturnShippingLabelRequest, DescribeReturnShippingLabelResponse>()
                            .withOperationName("DescribeReturnShippingLabel")
                            .withMarshaller(new DescribeReturnShippingLabelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeReturnShippingLabelRequest));
            CompletableFuture<DescribeReturnShippingLabelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a link to an Amazon S3 presigned URL for the manifest file associated with the specified
     * <code>JobId</code> value. You can access the manifest file for up to 60 minutes after this request has been made.
     * To access the manifest file after 60 minutes have passed, you'll have to make another call to the
     * <code>GetJobManifest</code> action.
     * </p>
     * <p>
     * The manifest is an encrypted file that you can download after your job enters the <code>WithCustomer</code>
     * status. The manifest is decrypted by using the <code>UnlockCode</code> code value, when you pass both values to
     * the Snow device through the Snowball client when the client is started for the first time.
     * </p>
     * <p>
     * As a best practice, we recommend that you don't save a copy of an <code>UnlockCode</code> value in the same
     * location as the manifest file for that job. Saving these separately helps prevent unauthorized parties from
     * gaining access to the Snow device associated with that job.
     * </p>
     * <p>
     * The credentials of a given job, including its manifest file and unlock code, expire 360 days after the job is
     * created.
     * </p>
     *
     * @param getJobManifestRequest
     * @return A Java Future containing the result of the GetJobManifest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.GetJobManifest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetJobManifest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobManifestResponse> getJobManifest(GetJobManifestRequest getJobManifestRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobManifestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobManifest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobManifestResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetJobManifestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetJobManifestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobManifestRequest, GetJobManifestResponse>()
                            .withOperationName("GetJobManifest")
                            .withMarshaller(new GetJobManifestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getJobManifestRequest));
            CompletableFuture<GetJobManifestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the <code>UnlockCode</code> code value for the specified job. A particular <code>UnlockCode</code> value
     * can be accessed for up to 360 days after the associated job has been created.
     * </p>
     * <p>
     * The <code>UnlockCode</code> value is a 29-character code with 25 alphanumeric characters and 4 hyphens. This code
     * is used to decrypt the manifest file when it is passed along with the manifest to the Snow device through the
     * Snowball client when the client is started for the first time.
     * </p>
     * <p>
     * As a best practice, we recommend that you don't save a copy of the <code>UnlockCode</code> in the same location
     * as the manifest file for that job. Saving these separately helps prevent unauthorized parties from gaining access
     * to the Snow device associated with that job.
     * </p>
     *
     * @param getJobUnlockCodeRequest
     * @return A Java Future containing the result of the GetJobUnlockCode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.GetJobUnlockCode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetJobUnlockCode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobUnlockCodeResponse> getJobUnlockCode(GetJobUnlockCodeRequest getJobUnlockCodeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobUnlockCodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobUnlockCode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobUnlockCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetJobUnlockCodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetJobUnlockCodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobUnlockCodeRequest, GetJobUnlockCodeResponse>()
                            .withOperationName("GetJobUnlockCode")
                            .withMarshaller(new GetJobUnlockCodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getJobUnlockCodeRequest));
            CompletableFuture<GetJobUnlockCodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the Snow Family service limit for your account, and also the number of Snow devices
     * your account has in use.
     * </p>
     * <p>
     * The default service limit for the number of Snow devices that you can have at one time is 1. If you want to
     * increase your service limit, contact AWS Support.
     * </p>
     *
     * @param getSnowballUsageRequest
     * @return A Java Future containing the result of the GetSnowballUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.GetSnowballUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetSnowballUsage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSnowballUsageResponse> getSnowballUsage(GetSnowballUsageRequest getSnowballUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSnowballUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSnowballUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSnowballUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSnowballUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSnowballUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSnowballUsageRequest, GetSnowballUsageResponse>()
                            .withOperationName("GetSnowballUsage")
                            .withMarshaller(new GetSnowballUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSnowballUsageRequest));
            CompletableFuture<GetSnowballUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an Amazon S3 presigned URL for an update file associated with a specified <code>JobId</code>.
     * </p>
     *
     * @param getSoftwareUpdatesRequest
     * @return A Java Future containing the result of the GetSoftwareUpdates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.GetSoftwareUpdates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/GetSoftwareUpdates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSoftwareUpdatesResponse> getSoftwareUpdates(GetSoftwareUpdatesRequest getSoftwareUpdatesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSoftwareUpdatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSoftwareUpdates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSoftwareUpdatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSoftwareUpdatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSoftwareUpdatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSoftwareUpdatesRequest, GetSoftwareUpdatesResponse>()
                            .withOperationName("GetSoftwareUpdates")
                            .withMarshaller(new GetSoftwareUpdatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSoftwareUpdatesRequest));
            CompletableFuture<GetSoftwareUpdatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object is for a job in the specified cluster and contains a job's state, a job's ID, and other information.
     * </p>
     *
     * @param listClusterJobsRequest
     * @return A Java Future containing the result of the ListClusterJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListClusterJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListClusterJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListClusterJobsResponse> listClusterJobs(ListClusterJobsRequest listClusterJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClusterJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusterJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListClusterJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListClusterJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListClusterJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListClusterJobsRequest, ListClusterJobsResponse>()
                            .withOperationName("ListClusterJobs")
                            .withMarshaller(new ListClusterJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listClusterJobsRequest));
            CompletableFuture<ListClusterJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>ClusterListEntry</code> objects of the specified length. Each
     * <code>ClusterListEntry</code> object contains a cluster's state, a cluster's ID, and other important status
     * information.
     * </p>
     *
     * @param listClustersRequest
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListClustersResponse> listClusters(ListClustersRequest listClustersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                            .withOperationName("ListClusters").withMarshaller(new ListClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listClustersRequest));
            CompletableFuture<ListClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action returns a list of the different Amazon EC2 Amazon Machine Images (AMIs) that are owned by your AWS
     * account that would be supported for use on a Snow device. Currently, supported AMIs are based on the CentOS 7
     * (x86_64) - with Updates HVM, Ubuntu Server 14.04 LTS (HVM), and Ubuntu 16.04 LTS - Xenial (HVM) images, available
     * on the AWS Marketplace.
     * </p>
     *
     * @param listCompatibleImagesRequest
     * @return A Java Future containing the result of the ListCompatibleImages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>Ec2RequestFailedException Your IAM user lacks the necessary Amazon EC2 permissions to perform the
     *         attempted action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListCompatibleImages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListCompatibleImages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCompatibleImagesResponse> listCompatibleImages(
            ListCompatibleImagesRequest listCompatibleImagesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCompatibleImagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCompatibleImages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCompatibleImagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCompatibleImagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCompatibleImagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCompatibleImagesRequest, ListCompatibleImagesResponse>()
                            .withOperationName("ListCompatibleImages")
                            .withMarshaller(new ListCompatibleImagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCompatibleImagesRequest));
            CompletableFuture<ListCompatibleImagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object contains a job's state, a job's ID, and a value that indicates whether the job is a job part, in the case
     * of export jobs. Calling this API action in one of the US regions will return jobs from the list of all jobs
     * associated with this account in all US regions.
     * </p>
     *
     * @param listJobsRequest
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listJobsRequest));
            CompletableFuture<ListJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>JobListEntry</code> objects of the specified length. Each <code>JobListEntry</code>
     * object contains a job's state, a job's ID, and a value that indicates whether the job is a job part, in the case
     * of export jobs. Calling this API action in one of the US regions will return jobs from the list of all jobs
     * associated with this account in all US regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.snowball.model.ListJobsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowball.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.snowball.model.ListJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.snowball.model.ListJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.snowball.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListJobsPublisher listJobsPaginator(ListJobsRequest listJobsRequest) {
        return new ListJobsPublisher(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * Lists all long-term pricing types.
     * </p>
     *
     * @param listLongTermPricingRequest
     * @return A Java Future containing the result of the ListLongTermPricing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> string was altered unexpectedly, and the
     *         operation has stopped. Run the operation without changing the <code>NextToken</code> string, and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.ListLongTermPricing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/ListLongTermPricing" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLongTermPricingResponse> listLongTermPricing(
            ListLongTermPricingRequest listLongTermPricingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLongTermPricingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLongTermPricing");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLongTermPricingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLongTermPricingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLongTermPricingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLongTermPricingRequest, ListLongTermPricingResponse>()
                            .withOperationName("ListLongTermPricing")
                            .withMarshaller(new ListLongTermPricingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLongTermPricingRequest));
            CompletableFuture<ListLongTermPricingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * While a cluster's <code>ClusterState</code> value is in the <code>AwaitingQuorum</code> state, you can update
     * some of the information associated with a cluster. Once the cluster changes to a different job state, usually 60
     * minutes after the cluster being created, this action is no longer available.
     * </p>
     *
     * @param updateClusterRequest
     * @return A Java Future containing the result of the UpdateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>KmsRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One or more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>Ec2RequestFailedException Your IAM user lacks the necessary Amazon EC2 permissions to perform the
     *         attempted action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateClusterResponse> updateCluster(UpdateClusterRequest updateClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateClusterRequest, UpdateClusterResponse>()
                            .withOperationName("UpdateCluster")
                            .withMarshaller(new UpdateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateClusterRequest));
            CompletableFuture<UpdateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * While a job's <code>JobState</code> value is <code>New</code>, you can update some of the information associated
     * with a job. Once the job changes to a different job state, usually within 60 minutes of the job being created,
     * this action is no longer available.
     * </p>
     *
     * @param updateJobRequest
     * @return A Java Future containing the result of the UpdateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>KmsRequestFailedException The provided AWS Key Management Service key lacks the permissions to
     *         perform the specified <a>CreateJob</a> or <a>UpdateJob</a> action.</li>
     *         <li>InvalidInputCombinationException Job or cluster creation failed. One or more inputs were invalid.
     *         Confirm that the <a>CreateClusterRequest&#36SnowballType</a> value supports your
     *         <a>CreateJobRequest&#36JobType</a>, and try again.</li>
     *         <li>ClusterLimitExceededException Job creation failed. Currently, clusters support five nodes. If you
     *         have fewer than five nodes for your cluster and you have more nodes to create for this cluster, try again
     *         and create jobs until your cluster has exactly five nodes.</li>
     *         <li>Ec2RequestFailedException Your IAM user lacks the necessary Amazon EC2 permissions to perform the
     *         attempted action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.UpdateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobResponse> updateJob(UpdateJobRequest updateJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobRequest, UpdateJobResponse>().withOperationName("UpdateJob")
                            .withMarshaller(new UpdateJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateJobRequest));
            CompletableFuture<UpdateJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the state when a shipment state changes to a different state.
     * </p>
     *
     * @param updateJobShipmentStateRequest
     * @return A Java Future containing the result of the UpdateJobShipmentState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>InvalidJobStateException The action can't be performed because the job's current state doesn't allow
     *         that action to be performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.UpdateJobShipmentState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateJobShipmentState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobShipmentStateResponse> updateJobShipmentState(
            UpdateJobShipmentStateRequest updateJobShipmentStateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobShipmentStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobShipmentState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateJobShipmentStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateJobShipmentStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateJobShipmentStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobShipmentStateRequest, UpdateJobShipmentStateResponse>()
                            .withOperationName("UpdateJobShipmentState")
                            .withMarshaller(new UpdateJobShipmentStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateJobShipmentStateRequest));
            CompletableFuture<UpdateJobShipmentStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the long-term pricing type.
     * </p>
     *
     * @param updateLongTermPricingRequest
     * @return A Java Future containing the result of the UpdateLongTermPricing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidResourceException The specified resource can't be found. Check the information you provided in
     *         your last request, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowballException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SnowballAsyncClient.UpdateLongTermPricing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snowball-2016-06-30/UpdateLongTermPricing"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLongTermPricingResponse> updateLongTermPricing(
            UpdateLongTermPricingRequest updateLongTermPricingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLongTermPricingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snowball");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLongTermPricing");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLongTermPricingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLongTermPricingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLongTermPricingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLongTermPricingRequest, UpdateLongTermPricingResponse>()
                            .withOperationName("UpdateLongTermPricing")
                            .withMarshaller(new UpdateLongTermPricingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLongTermPricingRequest));
            CompletableFuture<UpdateLongTermPricingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SnowballException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidJobStateException")
                                .exceptionBuilderSupplier(InvalidJobStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReturnShippingLabelAlreadyExistsException")
                                .exceptionBuilderSupplier(ReturnShippingLabelAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedAddressException")
                                .exceptionBuilderSupplier(UnsupportedAddressException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("Ec2RequestFailedException")
                                .exceptionBuilderSupplier(Ec2RequestFailedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSRequestFailedException")
                                .exceptionBuilderSupplier(KmsRequestFailedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidAddressException")
                                .exceptionBuilderSupplier(InvalidAddressException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceException")
                                .exceptionBuilderSupplier(InvalidResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClusterLimitExceededException")
                                .exceptionBuilderSupplier(ClusterLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputCombinationException")
                                .exceptionBuilderSupplier(InvalidInputCombinationException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SnowballRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
