/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.snowdevicemanagement;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.snowdevicemanagement.model.AccessDeniedException;
import software.amazon.awssdk.services.snowdevicemanagement.model.CancelTaskRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.CancelTaskResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.CreateTaskRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.CreateTaskResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeDeviceEc2InstancesRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeDeviceEc2InstancesResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeDeviceRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeDeviceResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeExecutionRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeExecutionResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeTaskRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.DescribeTaskResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.InternalServerException;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListDeviceResourcesRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListDeviceResourcesResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListDevicesRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListDevicesResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListExecutionsRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListExecutionsResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListTasksRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.ListTasksResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ResourceNotFoundException;
import software.amazon.awssdk.services.snowdevicemanagement.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.snowdevicemanagement.model.SnowDeviceManagementException;
import software.amazon.awssdk.services.snowdevicemanagement.model.SnowDeviceManagementRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.TagResourceRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.TagResourceResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ThrottlingException;
import software.amazon.awssdk.services.snowdevicemanagement.model.UntagResourceRequest;
import software.amazon.awssdk.services.snowdevicemanagement.model.UntagResourceResponse;
import software.amazon.awssdk.services.snowdevicemanagement.model.ValidationException;
import software.amazon.awssdk.services.snowdevicemanagement.paginators.ListDeviceResourcesPublisher;
import software.amazon.awssdk.services.snowdevicemanagement.paginators.ListDevicesPublisher;
import software.amazon.awssdk.services.snowdevicemanagement.paginators.ListExecutionsPublisher;
import software.amazon.awssdk.services.snowdevicemanagement.paginators.ListTasksPublisher;
import software.amazon.awssdk.services.snowdevicemanagement.transform.CancelTaskRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.CreateTaskRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.DescribeDeviceEc2InstancesRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.DescribeDeviceRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.DescribeExecutionRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.DescribeTaskRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListDeviceResourcesRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListDevicesRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListExecutionsRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.ListTasksRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.snowdevicemanagement.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SnowDeviceManagementAsyncClient}.
 *
 * @see SnowDeviceManagementAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnowDeviceManagementAsyncClient implements SnowDeviceManagementAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSnowDeviceManagementAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SnowDeviceManagementServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSnowDeviceManagementAsyncClient(SnowDeviceManagementServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Sends a cancel request for a specified task. You can cancel a task only if it's still in a <code>QUEUED</code>
     * state. Tasks that are already running can't be cancelled.
     * </p>
     * <note>
     * <p>
     * A task might still run if it's processed from the queue before the <code>CancelTask</code> operation changes the
     * task's state.
     * </p>
     * </note>
     *
     * @param cancelTaskRequest
     * @return A Java Future containing the result of the CancelTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.CancelTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/CancelTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelTaskResponse> cancelTask(CancelTaskRequest cancelTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelTaskRequest, CancelTaskResponse>().withOperationName("CancelTask")
                            .withMarshaller(new CancelTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelTaskRequest));
            CompletableFuture<CancelTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Instructs one or more devices to start a task, such as unlocking or rebooting.
     * </p>
     *
     * @param createTaskRequest
     * @return A Java Future containing the result of the CreateTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.CreateTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/CreateTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTaskResponse> createTask(CreateTaskRequest createTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTaskRequest, CreateTaskResponse>().withOperationName("CreateTask")
                            .withMarshaller(new CreateTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTaskRequest));
            CompletableFuture<CreateTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks device-specific information, such as the device type, software version, IP addresses, and lock status.
     * </p>
     *
     * @param describeDeviceRequest
     * @return A Java Future containing the result of the DescribeDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.DescribeDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/DescribeDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDeviceResponse> describeDevice(DescribeDeviceRequest describeDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDeviceRequest, DescribeDeviceResponse>()
                            .withOperationName("DescribeDevice")
                            .withMarshaller(new DescribeDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDeviceRequest));
            CompletableFuture<DescribeDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks the current state of the Amazon EC2 instances. The output is similar to <code>describeDevice</code>, but
     * the results are sourced from the device cache in the Amazon Web Services Cloud and include a subset of the
     * available fields.
     * </p>
     *
     * @param describeDeviceEc2InstancesRequest
     * @return A Java Future containing the result of the DescribeDeviceEc2Instances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.DescribeDeviceEc2Instances
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/DescribeDeviceEc2Instances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDeviceEc2InstancesResponse> describeDeviceEc2Instances(
            DescribeDeviceEc2InstancesRequest describeDeviceEc2InstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceEc2InstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDeviceEc2Instances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDeviceEc2InstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDeviceEc2InstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDeviceEc2InstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDeviceEc2InstancesRequest, DescribeDeviceEc2InstancesResponse>()
                            .withOperationName("DescribeDeviceEc2Instances")
                            .withMarshaller(new DescribeDeviceEc2InstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDeviceEc2InstancesRequest));
            CompletableFuture<DescribeDeviceEc2InstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks the status of a remote task running on one or more target devices.
     * </p>
     *
     * @param describeExecutionRequest
     * @return A Java Future containing the result of the DescribeExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.DescribeExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/DescribeExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeExecutionResponse> describeExecution(DescribeExecutionRequest describeExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeExecutionRequest, DescribeExecutionResponse>()
                            .withOperationName("DescribeExecution")
                            .withMarshaller(new DescribeExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeExecutionRequest));
            CompletableFuture<DescribeExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks the metadata for a given task on a device.
     * </p>
     *
     * @param describeTaskRequest
     * @return A Java Future containing the result of the DescribeTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.DescribeTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/DescribeTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTaskResponse> describeTask(DescribeTaskRequest describeTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTaskRequest, DescribeTaskResponse>()
                            .withOperationName("DescribeTask").withMarshaller(new DescribeTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTaskRequest));
            CompletableFuture<DescribeTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the Amazon Web Services resources available for a device. Currently, Amazon EC2 instances are
     * the only supported resource type.
     * </p>
     *
     * @param listDeviceResourcesRequest
     * @return A Java Future containing the result of the ListDeviceResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListDeviceResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListDeviceResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeviceResourcesResponse> listDeviceResources(
            ListDeviceResourcesRequest listDeviceResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeviceResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeviceResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeviceResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeviceResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeviceResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeviceResourcesRequest, ListDeviceResourcesResponse>()
                            .withOperationName("ListDeviceResources")
                            .withMarshaller(new ListDeviceResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDeviceResourcesRequest));
            CompletableFuture<ListDeviceResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the Amazon Web Services resources available for a device. Currently, Amazon EC2 instances are
     * the only supported resource type.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDeviceResources(software.amazon.awssdk.services.snowdevicemanagement.model.ListDeviceResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowdevicemanagement.paginators.ListDeviceResourcesPublisher publisher = client.listDeviceResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowdevicemanagement.paginators.ListDeviceResourcesPublisher publisher = client.listDeviceResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.snowdevicemanagement.model.ListDeviceResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.snowdevicemanagement.model.ListDeviceResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeviceResources(software.amazon.awssdk.services.snowdevicemanagement.model.ListDeviceResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDeviceResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListDeviceResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListDeviceResources"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDeviceResourcesPublisher listDeviceResourcesPaginator(ListDeviceResourcesRequest listDeviceResourcesRequest) {
        return new ListDeviceResourcesPublisher(this, applyPaginatorUserAgent(listDeviceResourcesRequest));
    }

    /**
     * <p>
     * Returns a list of all devices on your Amazon Web Services account that have Amazon Web Services Snow Device
     * Management enabled in the Amazon Web Services Region where the command is run.
     * </p>
     *
     * @param listDevicesRequest
     * @return A Java Future containing the result of the ListDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListDevices"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDevicesResponse> listDevices(ListDevicesRequest listDevicesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDevicesRequest, ListDevicesResponse>()
                            .withOperationName("ListDevices").withMarshaller(new ListDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDevicesRequest));
            CompletableFuture<ListDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all devices on your Amazon Web Services account that have Amazon Web Services Snow Device
     * Management enabled in the Amazon Web Services Region where the command is run.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDevices(software.amazon.awssdk.services.snowdevicemanagement.model.ListDevicesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowdevicemanagement.paginators.ListDevicesPublisher publisher = client.listDevicesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowdevicemanagement.paginators.ListDevicesPublisher publisher = client.listDevicesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.snowdevicemanagement.model.ListDevicesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.snowdevicemanagement.model.ListDevicesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDevices(software.amazon.awssdk.services.snowdevicemanagement.model.ListDevicesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDevicesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListDevices"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDevicesPublisher listDevicesPaginator(ListDevicesRequest listDevicesRequest) {
        return new ListDevicesPublisher(this, applyPaginatorUserAgent(listDevicesRequest));
    }

    /**
     * <p>
     * Returns the status of tasks for one or more target devices.
     * </p>
     *
     * @param listExecutionsRequest
     * @return A Java Future containing the result of the ListExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListExecutionsResponse> listExecutions(ListExecutionsRequest listExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExecutionsRequest, ListExecutionsResponse>()
                            .withOperationName("ListExecutions")
                            .withMarshaller(new ListExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listExecutionsRequest));
            CompletableFuture<ListExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the status of tasks for one or more target devices.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listExecutions(software.amazon.awssdk.services.snowdevicemanagement.model.ListExecutionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowdevicemanagement.paginators.ListExecutionsPublisher publisher = client.listExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowdevicemanagement.paginators.ListExecutionsPublisher publisher = client.listExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.snowdevicemanagement.model.ListExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.snowdevicemanagement.model.ListExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExecutions(software.amazon.awssdk.services.snowdevicemanagement.model.ListExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listExecutionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListExecutionsPublisher listExecutionsPaginator(ListExecutionsRequest listExecutionsRequest) {
        return new ListExecutionsPublisher(this, applyPaginatorUserAgent(listExecutionsRequest));
    }

    /**
     * <p>
     * Returns a list of tags for a managed device or task.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tasks that can be filtered by state.
     * </p>
     *
     * @param listTasksRequest
     * @return A Java Future containing the result of the ListTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTasksResponse> listTasks(ListTasksRequest listTasksRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTasksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTasksRequest, ListTasksResponse>().withOperationName("ListTasks")
                            .withMarshaller(new ListTasksRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTasksRequest));
            CompletableFuture<ListTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tasks that can be filtered by state.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTasks(software.amazon.awssdk.services.snowdevicemanagement.model.ListTasksRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowdevicemanagement.paginators.ListTasksPublisher publisher = client.listTasksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.snowdevicemanagement.paginators.ListTasksPublisher publisher = client.listTasksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.snowdevicemanagement.model.ListTasksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.snowdevicemanagement.model.ListTasksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTasks(software.amazon.awssdk.services.snowdevicemanagement.model.ListTasksRequest)} operation.</b>
     * </p>
     *
     * @param listTasksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.ListTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/ListTasks"
     *      target="_top">AWS API Documentation</a>
     */
    public ListTasksPublisher listTasksPaginator(ListTasksRequest listTasksRequest) {
        return new ListTasksPublisher(this, applyPaginatorUserAgent(listTasksRequest));
    }

    /**
     * <p>
     * Adds or replaces tags on a device or task.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from a device or task.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnowDeviceManagementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample SnowDeviceManagementAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/snow-device-management-2021-08-04/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Snow Device Management");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SnowDeviceManagementServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SnowDeviceManagementException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SnowDeviceManagementRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
