/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns.model;

import java.nio.ByteBuffer;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.core.runtime.StandardMemberCopier;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The user-specified message attribute value. For string data types, the value attribute has the same restrictions on
 * the content as the message body. For more information, see <a
 * href="http://docs.aws.amazon.com/sns/latest/api/API_Publish.html">Publish</a>.
 * </p>
 * <p>
 * Name, type, and value must not be empty or null. In addition, the message body should not be empty or null. All parts
 * of the message attribute, including name, type, and value, are included in the message size restriction, which is
 * currently 256 KB (262,144 bytes). For more information, see <a
 * href="http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html">Using Amazon SNS Message Attributes</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class MessageAttributeValue implements ToCopyableBuilder<MessageAttributeValue.Builder, MessageAttributeValue> {
    private final String dataType;

    private final String stringValue;

    private final ByteBuffer binaryValue;

    private MessageAttributeValue(BuilderImpl builder) {
        this.dataType = builder.dataType;
        this.stringValue = builder.stringValue;
        this.binaryValue = builder.binaryValue;
    }

    /**
     * <p>
     * Amazon SNS supports the following logical data types: String, Number, and Binary. For more information, see <a
     * href="http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes">Message
     * Attribute Data Types</a>.
     * </p>
     * 
     * @return Amazon SNS supports the following logical data types: String, Number, and Binary. For more information,
     *         see <a href=
     *         "http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes"
     *         >Message Attribute Data Types</a>.
     */
    public String dataType() {
        return dataType;
    }

    /**
     * <p>
     * Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
     * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
     * >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
     * </p>
     * 
     * @return Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
     *         href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
     *         >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
     */
    public String stringValue() {
        return stringValue;
    }

    /**
     * <p>
     * Binary type attributes can store any binary data, for example, compressed data, encrypted data, or images.
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return Binary type attributes can store any binary data, for example, compressed data, encrypted data, or
     *         images.
     */
    public ByteBuffer binaryValue() {
        return binaryValue == null ? null : binaryValue.asReadOnlyBuffer();
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((dataType() == null) ? 0 : dataType().hashCode());
        hashCode = 31 * hashCode + ((stringValue() == null) ? 0 : stringValue().hashCode());
        hashCode = 31 * hashCode + ((binaryValue() == null) ? 0 : binaryValue().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageAttributeValue)) {
            return false;
        }
        MessageAttributeValue other = (MessageAttributeValue) obj;
        if (other.dataType() == null ^ this.dataType() == null) {
            return false;
        }
        if (other.dataType() != null && !other.dataType().equals(this.dataType())) {
            return false;
        }
        if (other.stringValue() == null ^ this.stringValue() == null) {
            return false;
        }
        if (other.stringValue() != null && !other.stringValue().equals(this.stringValue())) {
            return false;
        }
        if (other.binaryValue() == null ^ this.binaryValue() == null) {
            return false;
        }
        if (other.binaryValue() != null && !other.binaryValue().equals(this.binaryValue())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (dataType() != null) {
            sb.append("DataType: ").append(dataType()).append(",");
        }
        if (stringValue() != null) {
            sb.append("StringValue: ").append(stringValue()).append(",");
        }
        if (binaryValue() != null) {
            sb.append("BinaryValue: ").append(binaryValue()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DataType":
            return Optional.of(clazz.cast(dataType()));
        case "StringValue":
            return Optional.of(clazz.cast(stringValue()));
        case "BinaryValue":
            return Optional.of(clazz.cast(binaryValue()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, MessageAttributeValue> {
        /**
         * <p>
         * Amazon SNS supports the following logical data types: String, Number, and Binary. For more information, see
         * <a href="http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes">
         * Message Attribute Data Types</a>.
         * </p>
         * 
         * @param dataType
         *        Amazon SNS supports the following logical data types: String, Number, and Binary. For more
         *        information, see <a href=
         *        "http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes"
         *        >Message Attribute Data Types</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);

        /**
         * <p>
         * Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         * >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * 
         * @param stringValue
         *        Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         *        >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringValue(String stringValue);

        /**
         * <p>
         * Binary type attributes can store any binary data, for example, compressed data, encrypted data, or images.
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param binaryValue
         *        Binary type attributes can store any binary data, for example, compressed data, encrypted data, or
         *        images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryValue(ByteBuffer binaryValue);
    }

    static final class BuilderImpl implements Builder {
        private String dataType;

        private String stringValue;

        private ByteBuffer binaryValue;

        private BuilderImpl() {
        }

        private BuilderImpl(MessageAttributeValue model) {
            dataType(model.dataType);
            stringValue(model.stringValue);
            binaryValue(model.binaryValue);
        }

        public final String getDataType() {
            return dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        public final String getStringValue() {
            return stringValue;
        }

        @Override
        public final Builder stringValue(String stringValue) {
            this.stringValue = stringValue;
            return this;
        }

        public final void setStringValue(String stringValue) {
            this.stringValue = stringValue;
        }

        public final ByteBuffer getBinaryValue() {
            return binaryValue;
        }

        @Override
        public final Builder binaryValue(ByteBuffer binaryValue) {
            this.binaryValue = StandardMemberCopier.copy(binaryValue);
            return this;
        }

        public final void setBinaryValue(ByteBuffer binaryValue) {
            this.binaryValue = StandardMemberCopier.copy(binaryValue);
        }

        @Override
        public MessageAttributeValue build() {
            return new MessageAttributeValue(this);
        }
    }
}
