/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The user-specified message attribute value. For string data types, the value attribute has the same restrictions on
 * the content as the message body. For more information, see <a
 * href="http://docs.aws.amazon.com/sns/latest/api/API_Publish.html">Publish</a>.
 * </p>
 * <p>
 * Name, type, and value must not be empty or null. In addition, the message body should not be empty or null. All parts
 * of the message attribute, including name, type, and value, are included in the message size restriction, which is
 * currently 256 KB (262,144 bytes). For more information, see <a
 * href="http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html">Using Amazon SNS Message Attributes</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MessageAttributeValue implements SdkPojo, Serializable,
        ToCopyableBuilder<MessageAttributeValue.Builder, MessageAttributeValue> {
    private static final SdkField<String> DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MessageAttributeValue::dataType)).setter(setter(Builder::dataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataType").build()).build();

    private static final SdkField<String> STRING_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MessageAttributeValue::stringValue)).setter(setter(Builder::stringValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StringValue").build()).build();

    private static final SdkField<SdkBytes> BINARY_VALUE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(MessageAttributeValue::binaryValue)).setter(setter(Builder::binaryValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BinaryValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_TYPE_FIELD,
            STRING_VALUE_FIELD, BINARY_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dataType;

    private final String stringValue;

    private final SdkBytes binaryValue;

    private MessageAttributeValue(BuilderImpl builder) {
        this.dataType = builder.dataType;
        this.stringValue = builder.stringValue;
        this.binaryValue = builder.binaryValue;
    }

    /**
     * <p>
     * Amazon SNS supports the following logical data types: String, String.Array, Number, and Binary. For more
     * information, see <a
     * href="http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes">Message
     * Attribute Data Types</a>.
     * </p>
     * 
     * @return Amazon SNS supports the following logical data types: String, String.Array, Number, and Binary. For more
     *         information, see <a href=
     *         "http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes"
     *         >Message Attribute Data Types</a>.
     */
    public String dataType() {
        return dataType;
    }

    /**
     * <p>
     * Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
     * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
     * >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
     * </p>
     * 
     * @return Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
     *         href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
     *         >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
     */
    public String stringValue() {
        return stringValue;
    }

    /**
     * <p>
     * Binary type attributes can store any binary data, for example, compressed data, encrypted data, or images.
     * </p>
     * 
     * @return Binary type attributes can store any binary data, for example, compressed data, encrypted data, or
     *         images.
     */
    public SdkBytes binaryValue() {
        return binaryValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        hashCode = 31 * hashCode + Objects.hashCode(stringValue());
        hashCode = 31 * hashCode + Objects.hashCode(binaryValue());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageAttributeValue)) {
            return false;
        }
        MessageAttributeValue other = (MessageAttributeValue) obj;
        return Objects.equals(dataType(), other.dataType()) && Objects.equals(stringValue(), other.stringValue())
                && Objects.equals(binaryValue(), other.binaryValue());
    }

    @Override
    public String toString() {
        return ToString.builder("MessageAttributeValue").add("DataType", dataType()).add("StringValue", stringValue())
                .add("BinaryValue", binaryValue()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DataType":
            return Optional.ofNullable(clazz.cast(dataType()));
        case "StringValue":
            return Optional.ofNullable(clazz.cast(stringValue()));
        case "BinaryValue":
            return Optional.ofNullable(clazz.cast(binaryValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MessageAttributeValue, T> g) {
        return obj -> g.apply((MessageAttributeValue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MessageAttributeValue> {
        /**
         * <p>
         * Amazon SNS supports the following logical data types: String, String.Array, Number, and Binary. For more
         * information, see <a
         * href="http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes"
         * >Message Attribute Data Types</a>.
         * </p>
         * 
         * @param dataType
         *        Amazon SNS supports the following logical data types: String, String.Array, Number, and Binary. For
         *        more information, see <a href=
         *        "http://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes"
         *        >Message Attribute Data Types</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);

        /**
         * <p>
         * Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         * >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * 
         * @param stringValue
         *        Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         *        >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringValue(String stringValue);

        /**
         * <p>
         * Binary type attributes can store any binary data, for example, compressed data, encrypted data, or images.
         * </p>
         * 
         * @param binaryValue
         *        Binary type attributes can store any binary data, for example, compressed data, encrypted data, or
         *        images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryValue(SdkBytes binaryValue);
    }

    static final class BuilderImpl implements Builder {
        private String dataType;

        private String stringValue;

        private SdkBytes binaryValue;

        private BuilderImpl() {
        }

        private BuilderImpl(MessageAttributeValue model) {
            dataType(model.dataType);
            stringValue(model.stringValue);
            binaryValue(model.binaryValue);
        }

        public final String getDataType() {
            return dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        public final String getStringValue() {
            return stringValue;
        }

        @Override
        public final Builder stringValue(String stringValue) {
            this.stringValue = stringValue;
            return this;
        }

        public final void setStringValue(String stringValue) {
            this.stringValue = stringValue;
        }

        public final ByteBuffer getBinaryValue() {
            return binaryValue == null ? null : binaryValue.asByteBuffer();
        }

        @Override
        public final Builder binaryValue(SdkBytes binaryValue) {
            this.binaryValue = StandardMemberCopier.copy(binaryValue);
            return this;
        }

        public final void setBinaryValue(ByteBuffer binaryValue) {
            binaryValue(binaryValue == null ? null : SdkBytes.fromByteBuffer(binaryValue));
        }

        @Override
        public MessageAttributeValue build() {
            return new MessageAttributeValue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
