/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.sns.model.AddPermissionRequest;
import software.amazon.awssdk.services.sns.model.AddPermissionResponse;
import software.amazon.awssdk.services.sns.model.AuthorizationErrorException;
import software.amazon.awssdk.services.sns.model.CheckIfPhoneNumberIsOptedOutRequest;
import software.amazon.awssdk.services.sns.model.CheckIfPhoneNumberIsOptedOutResponse;
import software.amazon.awssdk.services.sns.model.ConcurrentAccessException;
import software.amazon.awssdk.services.sns.model.ConfirmSubscriptionRequest;
import software.amazon.awssdk.services.sns.model.ConfirmSubscriptionResponse;
import software.amazon.awssdk.services.sns.model.CreatePlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.CreatePlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.CreatePlatformEndpointRequest;
import software.amazon.awssdk.services.sns.model.CreatePlatformEndpointResponse;
import software.amazon.awssdk.services.sns.model.CreateTopicRequest;
import software.amazon.awssdk.services.sns.model.CreateTopicResponse;
import software.amazon.awssdk.services.sns.model.DeleteEndpointRequest;
import software.amazon.awssdk.services.sns.model.DeleteEndpointResponse;
import software.amazon.awssdk.services.sns.model.DeletePlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.DeletePlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.DeleteTopicRequest;
import software.amazon.awssdk.services.sns.model.DeleteTopicResponse;
import software.amazon.awssdk.services.sns.model.EndpointDisabledException;
import software.amazon.awssdk.services.sns.model.FilterPolicyLimitExceededException;
import software.amazon.awssdk.services.sns.model.GetEndpointAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetEndpointAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetPlatformApplicationAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetPlatformApplicationAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetSmsAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetSmsAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetSubscriptionAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetSubscriptionAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetTopicAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetTopicAttributesResponse;
import software.amazon.awssdk.services.sns.model.InternalErrorException;
import software.amazon.awssdk.services.sns.model.InvalidParameterException;
import software.amazon.awssdk.services.sns.model.InvalidParameterValueException;
import software.amazon.awssdk.services.sns.model.InvalidSecurityException;
import software.amazon.awssdk.services.sns.model.KmsAccessDeniedException;
import software.amazon.awssdk.services.sns.model.KmsDisabledException;
import software.amazon.awssdk.services.sns.model.KmsInvalidStateException;
import software.amazon.awssdk.services.sns.model.KmsNotFoundException;
import software.amazon.awssdk.services.sns.model.KmsOptInRequiredException;
import software.amazon.awssdk.services.sns.model.KmsThrottlingException;
import software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.ListPhoneNumbersOptedOutRequest;
import software.amazon.awssdk.services.sns.model.ListPhoneNumbersOptedOutResponse;
import software.amazon.awssdk.services.sns.model.ListPlatformApplicationsRequest;
import software.amazon.awssdk.services.sns.model.ListPlatformApplicationsResponse;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicRequest;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicResponse;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsRequest;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsResponse;
import software.amazon.awssdk.services.sns.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.sns.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.sns.model.ListTopicsRequest;
import software.amazon.awssdk.services.sns.model.ListTopicsResponse;
import software.amazon.awssdk.services.sns.model.NotFoundException;
import software.amazon.awssdk.services.sns.model.OptInPhoneNumberRequest;
import software.amazon.awssdk.services.sns.model.OptInPhoneNumberResponse;
import software.amazon.awssdk.services.sns.model.PlatformApplicationDisabledException;
import software.amazon.awssdk.services.sns.model.PublishRequest;
import software.amazon.awssdk.services.sns.model.PublishResponse;
import software.amazon.awssdk.services.sns.model.RemovePermissionRequest;
import software.amazon.awssdk.services.sns.model.RemovePermissionResponse;
import software.amazon.awssdk.services.sns.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sns.model.SetEndpointAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetEndpointAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetPlatformApplicationAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetPlatformApplicationAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetSmsAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetSmsAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetSubscriptionAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetSubscriptionAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetTopicAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetTopicAttributesResponse;
import software.amazon.awssdk.services.sns.model.SnsException;
import software.amazon.awssdk.services.sns.model.SnsRequest;
import software.amazon.awssdk.services.sns.model.StaleTagException;
import software.amazon.awssdk.services.sns.model.SubscribeRequest;
import software.amazon.awssdk.services.sns.model.SubscribeResponse;
import software.amazon.awssdk.services.sns.model.SubscriptionLimitExceededException;
import software.amazon.awssdk.services.sns.model.TagLimitExceededException;
import software.amazon.awssdk.services.sns.model.TagPolicyException;
import software.amazon.awssdk.services.sns.model.TagResourceRequest;
import software.amazon.awssdk.services.sns.model.TagResourceResponse;
import software.amazon.awssdk.services.sns.model.ThrottledException;
import software.amazon.awssdk.services.sns.model.TopicLimitExceededException;
import software.amazon.awssdk.services.sns.model.UnsubscribeRequest;
import software.amazon.awssdk.services.sns.model.UnsubscribeResponse;
import software.amazon.awssdk.services.sns.model.UntagResourceRequest;
import software.amazon.awssdk.services.sns.model.UntagResourceResponse;
import software.amazon.awssdk.services.sns.paginators.ListEndpointsByPlatformApplicationPublisher;
import software.amazon.awssdk.services.sns.paginators.ListPlatformApplicationsPublisher;
import software.amazon.awssdk.services.sns.paginators.ListSubscriptionsByTopicPublisher;
import software.amazon.awssdk.services.sns.paginators.ListSubscriptionsPublisher;
import software.amazon.awssdk.services.sns.paginators.ListTopicsPublisher;
import software.amazon.awssdk.services.sns.transform.AddPermissionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CheckIfPhoneNumberIsOptedOutRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ConfirmSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreatePlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreatePlatformEndpointRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreateTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeleteEndpointRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeletePlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeleteTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetEndpointAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetPlatformApplicationAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetSmsAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetSubscriptionAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetTopicAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListEndpointsByPlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListPhoneNumbersOptedOutRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListPlatformApplicationsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListSubscriptionsByTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListTopicsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.OptInPhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.PublishRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetEndpointAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetPlatformApplicationAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetSmsAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetSubscriptionAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetTopicAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SubscribeRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.UnsubscribeRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SnsAsyncClient}.
 *
 * @see SnsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnsAsyncClient implements SnsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSnsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSnsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the
     * specified actions.
     * </p>
     *
     * @param addPermissionRequest
     * @return A Java Future containing the result of the AddPermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.AddPermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/AddPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddPermissionResponse> addPermission(AddPermissionRequest addPermissionRequest) {
        try {

            HttpResponseHandler<AddPermissionResponse> responseHandler = protocolFactory
                    .createResponseHandler(AddPermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<AddPermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddPermissionRequest, AddPermissionResponse>()
                            .withOperationName("AddPermission")
                            .withMarshaller(new AddPermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(addPermissionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your
     * account. You cannot send SMS messages to a number that is opted out.
     * </p>
     * <p>
     * To resume sending messages, you can opt in the number by using the <code>OptInPhoneNumber</code> action.
     * </p>
     *
     * @param checkIfPhoneNumberIsOptedOutRequest
     *        The input for the <code>CheckIfPhoneNumberIsOptedOut</code> action.
     * @return A Java Future containing the result of the CheckIfPhoneNumberIsOptedOut operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CheckIfPhoneNumberIsOptedOut
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CheckIfPhoneNumberIsOptedOut"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckIfPhoneNumberIsOptedOutResponse> checkIfPhoneNumberIsOptedOut(
            CheckIfPhoneNumberIsOptedOutRequest checkIfPhoneNumberIsOptedOutRequest) {
        try {

            HttpResponseHandler<CheckIfPhoneNumberIsOptedOutResponse> responseHandler = protocolFactory
                    .createResponseHandler(CheckIfPhoneNumberIsOptedOutResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CheckIfPhoneNumberIsOptedOutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckIfPhoneNumberIsOptedOutRequest, CheckIfPhoneNumberIsOptedOutResponse>()
                            .withOperationName("CheckIfPhoneNumberIsOptedOut")
                            .withMarshaller(new CheckIfPhoneNumberIsOptedOutRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(checkIfPhoneNumberIsOptedOutRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an
     * earlier <code>Subscribe</code> action. If the token is valid, the action creates a new subscription and returns
     * its Amazon Resource Name (ARN). This call requires an AWS signature only when the
     * <code>AuthenticateOnUnsubscribe</code> flag is set to "true".
     * </p>
     *
     * @param confirmSubscriptionRequest
     *        Input for ConfirmSubscription action.
     * @return A Java Future containing the result of the ConfirmSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubscriptionLimitExceededException Indicates that the customer already owns the maximum allowed
     *         number of subscriptions.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>FilterPolicyLimitExceededException Indicates that the number of filter polices in your AWS account
     *         exceeds the limit. To add more filter polices, submit an SNS Limit Increase case in the AWS Support
     *         Center.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ConfirmSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ConfirmSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ConfirmSubscriptionResponse> confirmSubscription(
            ConfirmSubscriptionRequest confirmSubscriptionRequest) {
        try {

            HttpResponseHandler<ConfirmSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(ConfirmSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ConfirmSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfirmSubscriptionRequest, ConfirmSubscriptionResponse>()
                            .withOperationName("ConfirmSubscription")
                            .withMarshaller(new ConfirmSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(confirmSubscriptionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a platform application object for one of the supported push notification services, such as APNS and FCM,
     * to which devices and mobile apps may register. You must specify PlatformPrincipal and PlatformCredential
     * attributes when using the <code>CreatePlatformApplication</code> action. The PlatformPrincipal is received from
     * the notification service. For APNS/APNS_SANDBOX, PlatformPrincipal is "SSL certificate". For FCM,
     * PlatformPrincipal is not applicable. For ADM, PlatformPrincipal is "client id". The PlatformCredential is also
     * received from the notification service. For WNS, PlatformPrincipal is "Package Security Identifier". For MPNS,
     * PlatformPrincipal is "TLS certificate". For Baidu, PlatformPrincipal is "API key".
     * </p>
     * <p>
     * For APNS/APNS_SANDBOX, PlatformCredential is "private key". For FCM, PlatformCredential is "API key". For ADM,
     * PlatformCredential is "client secret". For WNS, PlatformCredential is "secret key". For MPNS, PlatformCredential
     * is "private key". For Baidu, PlatformCredential is "secret key". The PlatformApplicationArn that is returned when
     * using <code>CreatePlatformApplication</code> is then used as an attribute for the
     * <code>CreatePlatformEndpoint</code> action.
     * </p>
     *
     * @param createPlatformApplicationRequest
     *        Input for CreatePlatformApplication action.
     * @return A Java Future containing the result of the CreatePlatformApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CreatePlatformApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreatePlatformApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlatformApplicationResponse> createPlatformApplication(
            CreatePlatformApplicationRequest createPlatformApplicationRequest) {
        try {

            HttpResponseHandler<CreatePlatformApplicationResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreatePlatformApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreatePlatformApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlatformApplicationRequest, CreatePlatformApplicationResponse>()
                            .withOperationName("CreatePlatformApplication")
                            .withMarshaller(new CreatePlatformApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPlatformApplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for a device and mobile app on one of the supported push notification services, such as FCM
     * and APNS. <code>CreatePlatformEndpoint</code> requires the PlatformApplicationArn that is returned from
     * <code>CreatePlatformApplication</code>. The EndpointArn that is returned when using
     * <code>CreatePlatformEndpoint</code> can then be used by the <code>Publish</code> action to send a message to a
     * mobile app or by the <code>Subscribe</code> action for subscription to a topic. The
     * <code>CreatePlatformEndpoint</code> action is idempotent, so if the requester already owns an endpoint with the
     * same device token and attributes, that endpoint's ARN is returned without creating a new endpoint. For more
     * information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile
     * Push Notifications</a>.
     * </p>
     * <p>
     * When using <code>CreatePlatformEndpoint</code> with Baidu, two attributes must be provided: ChannelId and UserId.
     * The token field must also contain the ChannelId. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePushBaiduEndpoint.html">Creating an Amazon SNS Endpoint
     * for Baidu</a>.
     * </p>
     *
     * @param createPlatformEndpointRequest
     *        Input for CreatePlatformEndpoint action.
     * @return A Java Future containing the result of the CreatePlatformEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CreatePlatformEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreatePlatformEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlatformEndpointResponse> createPlatformEndpoint(
            CreatePlatformEndpointRequest createPlatformEndpointRequest) {
        try {

            HttpResponseHandler<CreatePlatformEndpointResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreatePlatformEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreatePlatformEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlatformEndpointRequest, CreatePlatformEndpointResponse>()
                            .withOperationName("CreatePlatformEndpoint")
                            .withMarshaller(new CreatePlatformEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPlatformEndpointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a topic to which notifications can be published. Users can create at most 100,000 topics. For more
     * information, see <a href="http://aws.amazon.com/sns/">https://aws.amazon.com/sns</a>. This action is idempotent,
     * so if the requester already owns a topic with the specified name, that topic's ARN is returned without creating a
     * new topic.
     * </p>
     *
     * @param createTopicRequest
     *        Input for CreateTopic action.
     * @return A Java Future containing the result of the CreateTopic operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>TopicLimitExceededException Indicates that the customer already owns the maximum allowed number of
     *         topics.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>TagLimitExceededException Can't add more than 50 tags to a topic.</li>
     *         <li>StaleTagException A tag has been added to a resource with the same ARN as a deleted resource. Wait a
     *         short while and then retry the operation.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.CreateTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreateTopic" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTopicResponse> createTopic(CreateTopicRequest createTopicRequest) {
        try {

            HttpResponseHandler<CreateTopicResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateTopicResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateTopicResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTopicRequest, CreateTopicResponse>()
                            .withOperationName("CreateTopic").withMarshaller(new CreateTopicRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createTopicRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the endpoint for a device and mobile app from Amazon SNS. This action is idempotent. For more
     * information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile
     * Push Notifications</a>.
     * </p>
     * <p>
     * When you delete an endpoint that is also subscribed to a topic, then you must also unsubscribe the endpoint from
     * the topic.
     * </p>
     *
     * @param deleteEndpointRequest
     *        Input for DeleteEndpoint action.
     * @return A Java Future containing the result of the DeleteEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.DeleteEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeleteEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEndpointResponse> deleteEndpoint(DeleteEndpointRequest deleteEndpointRequest) {
        try {

            HttpResponseHandler<DeleteEndpointResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEndpointRequest, DeleteEndpointResponse>()
                            .withOperationName("DeleteEndpoint")
                            .withMarshaller(new DeleteEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteEndpointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a platform application object for one of the supported push notification services, such as APNS and FCM.
     * For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS
     * Mobile Push Notifications</a>.
     * </p>
     *
     * @param deletePlatformApplicationRequest
     *        Input for DeletePlatformApplication action.
     * @return A Java Future containing the result of the DeletePlatformApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.DeletePlatformApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeletePlatformApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePlatformApplicationResponse> deletePlatformApplication(
            DeletePlatformApplicationRequest deletePlatformApplicationRequest) {
        try {

            HttpResponseHandler<DeletePlatformApplicationResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeletePlatformApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeletePlatformApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePlatformApplicationRequest, DeletePlatformApplicationResponse>()
                            .withOperationName("DeletePlatformApplication")
                            .withMarshaller(new DeletePlatformApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deletePlatformApplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a topic and all its subscriptions. Deleting a topic might prevent some messages previously sent to the
     * topic from being delivered to subscribers. This action is idempotent, so deleting a topic that does not exist
     * does not result in an error.
     * </p>
     *
     * @param deleteTopicRequest
     * @return A Java Future containing the result of the DeleteTopic operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>StaleTagException A tag has been added to a resource with the same ARN as a deleted resource. Wait a
     *         short while and then retry the operation.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.DeleteTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeleteTopic" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTopicResponse> deleteTopic(DeleteTopicRequest deleteTopicRequest) {
        try {

            HttpResponseHandler<DeleteTopicResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteTopicResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteTopicResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTopicRequest, DeleteTopicResponse>()
                            .withOperationName("DeleteTopic").withMarshaller(new DeleteTopicRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTopicRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the endpoint attributes for a device on one of the supported push notification services, such as FCM
     * and APNS. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>.
     * </p>
     *
     * @param getEndpointAttributesRequest
     *        Input for GetEndpointAttributes action.
     * @return A Java Future containing the result of the GetEndpointAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetEndpointAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetEndpointAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetEndpointAttributesResponse> getEndpointAttributes(
            GetEndpointAttributesRequest getEndpointAttributesRequest) {
        try {

            HttpResponseHandler<GetEndpointAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetEndpointAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetEndpointAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEndpointAttributesRequest, GetEndpointAttributesResponse>()
                            .withOperationName("GetEndpointAttributes")
                            .withMarshaller(new GetEndpointAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getEndpointAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the attributes of the platform application object for the supported push notification services, such as
     * APNS and FCM. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     *
     * @param getPlatformApplicationAttributesRequest
     *        Input for GetPlatformApplicationAttributes action.
     * @return A Java Future containing the result of the GetPlatformApplicationAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetPlatformApplicationAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetPlatformApplicationAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlatformApplicationAttributesResponse> getPlatformApplicationAttributes(
            GetPlatformApplicationAttributesRequest getPlatformApplicationAttributesRequest) {
        try {

            HttpResponseHandler<GetPlatformApplicationAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetPlatformApplicationAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetPlatformApplicationAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlatformApplicationAttributesRequest, GetPlatformApplicationAttributesResponse>()
                            .withOperationName("GetPlatformApplicationAttributes")
                            .withMarshaller(new GetPlatformApplicationAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getPlatformApplicationAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the settings for sending SMS messages from your account.
     * </p>
     * <p>
     * These settings are set with the <code>SetSMSAttributes</code> action.
     * </p>
     *
     * @param getSmsAttributesRequest
     *        The input for the <code>GetSMSAttributes</code> request.
     * @return A Java Future containing the result of the GetSMSAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetSMSAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetSMSAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSmsAttributesResponse> getSMSAttributes(GetSmsAttributesRequest getSmsAttributesRequest) {
        try {

            HttpResponseHandler<GetSmsAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetSmsAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetSmsAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSmsAttributesRequest, GetSmsAttributesResponse>()
                            .withOperationName("GetSMSAttributes")
                            .withMarshaller(new GetSmsAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSmsAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all of the properties of a subscription.
     * </p>
     *
     * @param getSubscriptionAttributesRequest
     *        Input for GetSubscriptionAttributes.
     * @return A Java Future containing the result of the GetSubscriptionAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetSubscriptionAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetSubscriptionAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSubscriptionAttributesResponse> getSubscriptionAttributes(
            GetSubscriptionAttributesRequest getSubscriptionAttributesRequest) {
        try {

            HttpResponseHandler<GetSubscriptionAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetSubscriptionAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetSubscriptionAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSubscriptionAttributesRequest, GetSubscriptionAttributesResponse>()
                            .withOperationName("GetSubscriptionAttributes")
                            .withMarshaller(new GetSubscriptionAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSubscriptionAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of
     * the user.
     * </p>
     *
     * @param getTopicAttributesRequest
     *        Input for GetTopicAttributes action.
     * @return A Java Future containing the result of the GetTopicAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.GetTopicAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetTopicAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTopicAttributesResponse> getTopicAttributes(GetTopicAttributesRequest getTopicAttributesRequest) {
        try {

            HttpResponseHandler<GetTopicAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(GetTopicAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<GetTopicAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTopicAttributesRequest, GetTopicAttributesResponse>()
                            .withOperationName("GetTopicAttributes")
                            .withMarshaller(new GetTopicAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getTopicAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as FCM and
     * APNS. The results for <code>ListEndpointsByPlatformApplication</code> are paginated and return a limited list of
     * endpoints, up to 100. If additional records are available after the first page results, then a NextToken string
     * will be returned. To receive the next page, you call <code>ListEndpointsByPlatformApplication</code> again using
     * the NextToken string received from the previous call. When there are no more records to return, NextToken will be
     * null. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listEndpointsByPlatformApplicationRequest
     *        Input for ListEndpointsByPlatformApplication action.
     * @return A Java Future containing the result of the ListEndpointsByPlatformApplication operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListEndpointsByPlatformApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListEndpointsByPlatformApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEndpointsByPlatformApplicationResponse> listEndpointsByPlatformApplication(
            ListEndpointsByPlatformApplicationRequest listEndpointsByPlatformApplicationRequest) {
        try {

            HttpResponseHandler<ListEndpointsByPlatformApplicationResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListEndpointsByPlatformApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListEndpointsByPlatformApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEndpointsByPlatformApplicationRequest, ListEndpointsByPlatformApplicationResponse>()
                            .withOperationName("ListEndpointsByPlatformApplication")
                            .withMarshaller(new ListEndpointsByPlatformApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEndpointsByPlatformApplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as FCM and
     * APNS. The results for <code>ListEndpointsByPlatformApplication</code> are paginated and return a limited list of
     * endpoints, up to 100. If additional records are available after the first page results, then a NextToken string
     * will be returned. To receive the next page, you call <code>ListEndpointsByPlatformApplication</code> again using
     * the NextToken string received from the previous call. When there are no more records to return, NextToken will be
     * null. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEndpointsByPlatformApplication(software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListEndpointsByPlatformApplicationPublisher publisher = client.listEndpointsByPlatformApplicationPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListEndpointsByPlatformApplicationPublisher publisher = client.listEndpointsByPlatformApplicationPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEndpointsByPlatformApplication(software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEndpointsByPlatformApplicationRequest
     *        Input for ListEndpointsByPlatformApplication action.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListEndpointsByPlatformApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListEndpointsByPlatformApplication"
     *      target="_top">AWS API Documentation</a>
     */
    public ListEndpointsByPlatformApplicationPublisher listEndpointsByPlatformApplicationPaginator(
            ListEndpointsByPlatformApplicationRequest listEndpointsByPlatformApplicationRequest) {
        return new ListEndpointsByPlatformApplicationPublisher(this,
                applyPaginatorUserAgent(listEndpointsByPlatformApplicationRequest));
    }

    /**
     * <p>
     * Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.
     * </p>
     * <p>
     * The results for <code>ListPhoneNumbersOptedOut</code> are paginated, and each page returns up to 100 phone
     * numbers. If additional phone numbers are available after the first page of results, then a <code>NextToken</code>
     * string will be returned. To receive the next page, you call <code>ListPhoneNumbersOptedOut</code> again using the
     * <code>NextToken</code> string received from the previous call. When there are no more records to return,
     * <code>NextToken</code> will be null.
     * </p>
     *
     * @param listPhoneNumbersOptedOutRequest
     *        The input for the <code>ListPhoneNumbersOptedOut</code> action.
     * @return A Java Future containing the result of the ListPhoneNumbersOptedOut operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListPhoneNumbersOptedOut
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListPhoneNumbersOptedOut" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPhoneNumbersOptedOutResponse> listPhoneNumbersOptedOut(
            ListPhoneNumbersOptedOutRequest listPhoneNumbersOptedOutRequest) {
        try {

            HttpResponseHandler<ListPhoneNumbersOptedOutResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListPhoneNumbersOptedOutResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListPhoneNumbersOptedOutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPhoneNumbersOptedOutRequest, ListPhoneNumbersOptedOutResponse>()
                            .withOperationName("ListPhoneNumbersOptedOut")
                            .withMarshaller(new ListPhoneNumbersOptedOutRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPhoneNumbersOptedOutRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the platform application objects for the supported push notification services, such as APNS and FCM. The
     * results for <code>ListPlatformApplications</code> are paginated and return a limited list of applications, up to
     * 100. If additional records are available after the first page results, then a NextToken string will be returned.
     * To receive the next page, you call <code>ListPlatformApplications</code> using the NextToken string received from
     * the previous call. When there are no more records to return, NextToken will be null. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 15 transactions per second (TPS).
     * </p>
     *
     * @param listPlatformApplicationsRequest
     *        Input for ListPlatformApplications action.
     * @return A Java Future containing the result of the ListPlatformApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListPlatformApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListPlatformApplications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlatformApplicationsResponse> listPlatformApplications(
            ListPlatformApplicationsRequest listPlatformApplicationsRequest) {
        try {

            HttpResponseHandler<ListPlatformApplicationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListPlatformApplicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListPlatformApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlatformApplicationsRequest, ListPlatformApplicationsResponse>()
                            .withOperationName("ListPlatformApplications")
                            .withMarshaller(new ListPlatformApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPlatformApplicationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the platform application objects for the supported push notification services, such as APNS and FCM. The
     * results for <code>ListPlatformApplications</code> are paginated and return a limited list of applications, up to
     * 100. If additional records are available after the first page results, then a NextToken string will be returned.
     * To receive the next page, you call <code>ListPlatformApplications</code> using the NextToken string received from
     * the previous call. When there are no more records to return, NextToken will be null. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 15 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPlatformApplications(software.amazon.awssdk.services.sns.model.ListPlatformApplicationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListPlatformApplicationsPublisher publisher = client.listPlatformApplicationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListPlatformApplicationsPublisher publisher = client.listPlatformApplicationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sns.model.ListPlatformApplicationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sns.model.ListPlatformApplicationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlatformApplications(software.amazon.awssdk.services.sns.model.ListPlatformApplicationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPlatformApplicationsRequest
     *        Input for ListPlatformApplications action.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListPlatformApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListPlatformApplications" target="_top">AWS
     *      API Documentation</a>
     */
    public ListPlatformApplicationsPublisher listPlatformApplicationsPaginator(
            ListPlatformApplicationsRequest listPlatformApplicationsRequest) {
        return new ListPlatformApplicationsPublisher(this, applyPaginatorUserAgent(listPlatformApplicationsRequest));
    }

    /**
     * <p>
     * Returns a list of the requester's subscriptions. Each call returns a limited list of subscriptions, up to 100. If
     * there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter
     * in a new <code>ListSubscriptions</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listSubscriptionsRequest
     *        Input for ListSubscriptions action.
     * @return A Java Future containing the result of the ListSubscriptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListSubscriptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSubscriptionsResponse> listSubscriptions(ListSubscriptionsRequest listSubscriptionsRequest) {
        try {

            HttpResponseHandler<ListSubscriptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListSubscriptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListSubscriptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSubscriptionsRequest, ListSubscriptionsResponse>()
                            .withOperationName("ListSubscriptions")
                            .withMarshaller(new ListSubscriptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listSubscriptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the subscriptions to a specific topic. Each call returns a limited list of subscriptions, up to
     * 100. If there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code>
     * parameter in a new <code>ListSubscriptionsByTopic</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listSubscriptionsByTopicRequest
     *        Input for ListSubscriptionsByTopic action.
     * @return A Java Future containing the result of the ListSubscriptionsByTopic operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListSubscriptionsByTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptionsByTopic" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSubscriptionsByTopicResponse> listSubscriptionsByTopic(
            ListSubscriptionsByTopicRequest listSubscriptionsByTopicRequest) {
        try {

            HttpResponseHandler<ListSubscriptionsByTopicResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListSubscriptionsByTopicResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListSubscriptionsByTopicResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSubscriptionsByTopicRequest, ListSubscriptionsByTopicResponse>()
                            .withOperationName("ListSubscriptionsByTopic")
                            .withMarshaller(new ListSubscriptionsByTopicRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listSubscriptionsByTopicRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the subscriptions to a specific topic. Each call returns a limited list of subscriptions, up to
     * 100. If there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code>
     * parameter in a new <code>ListSubscriptionsByTopic</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSubscriptionsByTopic(software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListSubscriptionsByTopicPublisher publisher = client.listSubscriptionsByTopicPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListSubscriptionsByTopicPublisher publisher = client.listSubscriptionsByTopicPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSubscriptionsByTopic(software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSubscriptionsByTopicRequest
     *        Input for ListSubscriptionsByTopic action.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListSubscriptionsByTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptionsByTopic" target="_top">AWS
     *      API Documentation</a>
     */
    public ListSubscriptionsByTopicPublisher listSubscriptionsByTopicPaginator(
            ListSubscriptionsByTopicRequest listSubscriptionsByTopicRequest) {
        return new ListSubscriptionsByTopicPublisher(this, applyPaginatorUserAgent(listSubscriptionsByTopicRequest));
    }

    /**
     * <p>
     * Returns a list of the requester's subscriptions. Each call returns a limited list of subscriptions, up to 100. If
     * there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter
     * in a new <code>ListSubscriptions</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSubscriptions(software.amazon.awssdk.services.sns.model.ListSubscriptionsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListSubscriptionsPublisher publisher = client.listSubscriptionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListSubscriptionsPublisher publisher = client.listSubscriptionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sns.model.ListSubscriptionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sns.model.ListSubscriptionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSubscriptions(software.amazon.awssdk.services.sns.model.ListSubscriptionsRequest)} operation.</b>
     * </p>
     *
     * @param listSubscriptionsRequest
     *        Input for ListSubscriptions action.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListSubscriptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptions" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSubscriptionsPublisher listSubscriptionsPaginator(ListSubscriptionsRequest listSubscriptionsRequest) {
        return new ListSubscriptionsPublisher(this, applyPaginatorUserAgent(listSubscriptionsRequest));
    }

    /**
     * <p>
     * List all tags added to the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon Simple
     * Notification Service Developer Guide</i>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Can't tag resource. Verify that the topic exists.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100. If there are
     * more topics, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter in a new
     * <code>ListTopics</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listTopicsRequest
     * @return A Java Future containing the result of the ListTopics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListTopics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListTopics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTopicsResponse> listTopics(ListTopicsRequest listTopicsRequest) {
        try {

            HttpResponseHandler<ListTopicsResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListTopicsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListTopicsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTopicsRequest, ListTopicsResponse>().withOperationName("ListTopics")
                            .withMarshaller(new ListTopicsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTopicsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100. If there are
     * more topics, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter in a new
     * <code>ListTopics</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTopics(software.amazon.awssdk.services.sns.model.ListTopicsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListTopicsPublisher publisher = client.listTopicsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListTopicsPublisher publisher = client.listTopicsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sns.model.ListTopicsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sns.model.ListTopicsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTopics(software.amazon.awssdk.services.sns.model.ListTopicsRequest)} operation.</b>
     * </p>
     *
     * @param listTopicsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.ListTopics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListTopics" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTopicsPublisher listTopicsPaginator(ListTopicsRequest listTopicsRequest) {
        return new ListTopicsPublisher(this, applyPaginatorUserAgent(listTopicsRequest));
    }

    /**
     * <p>
     * Use this request to opt in a phone number that is opted out, which enables you to resume sending SMS messages to
     * the number.
     * </p>
     * <p>
     * You can opt in a phone number only once every 30 days.
     * </p>
     *
     * @param optInPhoneNumberRequest
     *        Input for the OptInPhoneNumber action.
     * @return A Java Future containing the result of the OptInPhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.OptInPhoneNumber
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/OptInPhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<OptInPhoneNumberResponse> optInPhoneNumber(OptInPhoneNumberRequest optInPhoneNumberRequest) {
        try {

            HttpResponseHandler<OptInPhoneNumberResponse> responseHandler = protocolFactory
                    .createResponseHandler(OptInPhoneNumberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<OptInPhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<OptInPhoneNumberRequest, OptInPhoneNumberResponse>()
                            .withOperationName("OptInPhoneNumber")
                            .withMarshaller(new OptInPhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(optInPhoneNumberRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a message to an Amazon SNS topic or sends a text message (SMS message) directly to a phone number.
     * </p>
     * <p>
     * If you send a message to a topic, Amazon SNS delivers the message to each endpoint that is subscribed to the
     * topic. The format of the message depends on the notification protocol for each subscribed endpoint.
     * </p>
     * <p>
     * When a <code>messageId</code> is returned, the message has been saved and Amazon SNS will attempt to deliver it
     * shortly.
     * </p>
     * <p>
     * To use the <code>Publish</code> action for sending a message to a mobile endpoint, such as an app on a Kindle
     * device or mobile phone, you must specify the EndpointArn for the TargetArn parameter. The EndpointArn is returned
     * when making a call with the <code>CreatePlatformEndpoint</code> action.
     * </p>
     * <p>
     * For more information about formatting messages, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/mobile-push-send-custommessage.html">Send Custom
     * Platform-Specific Payloads in Messages to Mobile Devices</a>.
     * </p>
     *
     * @param publishRequest
     *        Input for Publish action.
     * @return A Java Future containing the result of the Publish operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InvalidParameterValueException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>EndpointDisabledException Exception error indicating endpoint disabled.</li>
     *         <li>PlatformApplicationDisabledException Exception error indicating platform application disabled.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>KmsDisabledException The request was rejected because the specified customer master key (CMK) isn't
     *         enabled.</li>
     *         <li>KmsInvalidStateException The request was rejected because the state of the specified resource isn't
     *         valid for this request. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.</li>
     *         <li>KmsNotFoundException The request was rejected because the specified entity or resource can't be
     *         found.</li>
     *         <li>KmsOptInRequiredException The AWS access key ID needs a subscription for the service.</li>
     *         <li>KmsThrottlingException The request was denied due to request throttling. For more information about
     *         throttling, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#requests-per-second">Limits</a>
     *         in the <i>AWS Key Management Service Developer Guide.</i></li>
     *         <li>KmsAccessDeniedException The ciphertext references a key that doesn't exist or that you don't have
     *         access to.</li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.Publish
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Publish" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PublishResponse> publish(PublishRequest publishRequest) {
        try {

            HttpResponseHandler<PublishResponse> responseHandler = protocolFactory
                    .createResponseHandler(PublishResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<PublishResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PublishRequest, PublishResponse>().withOperationName("Publish")
                            .withMarshaller(new PublishRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(publishRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a statement from a topic's access control policy.
     * </p>
     *
     * @param removePermissionRequest
     *        Input for RemovePermission action.
     * @return A Java Future containing the result of the RemovePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.RemovePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemovePermissionResponse> removePermission(RemovePermissionRequest removePermissionRequest) {
        try {

            HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory
                    .createResponseHandler(RemovePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<RemovePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                            .withOperationName("RemovePermission")
                            .withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(removePermissionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the attributes for an endpoint for a device on one of the supported push notification services, such as FCM
     * and APNS. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>.
     * </p>
     *
     * @param setEndpointAttributesRequest
     *        Input for SetEndpointAttributes action.
     * @return A Java Future containing the result of the SetEndpointAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetEndpointAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetEndpointAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SetEndpointAttributesResponse> setEndpointAttributes(
            SetEndpointAttributesRequest setEndpointAttributesRequest) {
        try {

            HttpResponseHandler<SetEndpointAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetEndpointAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetEndpointAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetEndpointAttributesRequest, SetEndpointAttributesResponse>()
                            .withOperationName("SetEndpointAttributes")
                            .withMarshaller(new SetEndpointAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(setEndpointAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the attributes of the platform application object for the supported push notification services, such as APNS
     * and FCM. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>. For information on configuring attributes for message delivery status,
     * see <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-msg-status.html">Using Amazon SNS Application
     * Attributes for Message Delivery Status</a>.
     * </p>
     *
     * @param setPlatformApplicationAttributesRequest
     *        Input for SetPlatformApplicationAttributes action.
     * @return A Java Future containing the result of the SetPlatformApplicationAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetPlatformApplicationAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetPlatformApplicationAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SetPlatformApplicationAttributesResponse> setPlatformApplicationAttributes(
            SetPlatformApplicationAttributesRequest setPlatformApplicationAttributesRequest) {
        try {

            HttpResponseHandler<SetPlatformApplicationAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetPlatformApplicationAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetPlatformApplicationAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetPlatformApplicationAttributesRequest, SetPlatformApplicationAttributesResponse>()
                            .withOperationName("SetPlatformApplicationAttributes")
                            .withMarshaller(new SetPlatformApplicationAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(setPlatformApplicationAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this request to set the default settings for sending SMS messages and receiving daily SMS usage reports.
     * </p>
     * <p>
     * You can override some of these settings for a single message when you use the <code>Publish</code> action with
     * the <code>MessageAttributes.entry.N</code> parameter. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sms_publish-to-phone.html">Sending an SMS Message</a> in the
     * <i>Amazon SNS Developer Guide</i>.
     * </p>
     *
     * @param setSmsAttributesRequest
     *        The input for the SetSMSAttributes action.
     * @return A Java Future containing the result of the SetSMSAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ThrottledException Indicates that the rate at which requests have been submitted for this action
     *         exceeds the limit for your account.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetSMSAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetSMSAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SetSmsAttributesResponse> setSMSAttributes(SetSmsAttributesRequest setSmsAttributesRequest) {
        try {

            HttpResponseHandler<SetSmsAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetSmsAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetSmsAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetSmsAttributesRequest, SetSmsAttributesResponse>()
                            .withOperationName("SetSMSAttributes")
                            .withMarshaller(new SetSmsAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(setSmsAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows a subscription owner to set an attribute of the subscription to a new value.
     * </p>
     *
     * @param setSubscriptionAttributesRequest
     *        Input for SetSubscriptionAttributes action.
     * @return A Java Future containing the result of the SetSubscriptionAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>FilterPolicyLimitExceededException Indicates that the number of filter polices in your AWS account
     *         exceeds the limit. To add more filter polices, submit an SNS Limit Increase case in the AWS Support
     *         Center.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetSubscriptionAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetSubscriptionAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SetSubscriptionAttributesResponse> setSubscriptionAttributes(
            SetSubscriptionAttributesRequest setSubscriptionAttributesRequest) {
        try {

            HttpResponseHandler<SetSubscriptionAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetSubscriptionAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetSubscriptionAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetSubscriptionAttributesRequest, SetSubscriptionAttributesResponse>()
                            .withOperationName("SetSubscriptionAttributes")
                            .withMarshaller(new SetSubscriptionAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(setSubscriptionAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows a topic owner to set an attribute of the topic to a new value.
     * </p>
     *
     * @param setTopicAttributesRequest
     *        Input for SetTopicAttributes action.
     * @return A Java Future containing the result of the SetTopicAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.SetTopicAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetTopicAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SetTopicAttributesResponse> setTopicAttributes(SetTopicAttributesRequest setTopicAttributesRequest) {
        try {

            HttpResponseHandler<SetTopicAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(SetTopicAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SetTopicAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetTopicAttributesRequest, SetTopicAttributesResponse>()
                            .withOperationName("SetTopicAttributes")
                            .withMarshaller(new SetTopicAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(setTopicAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Prepares to subscribe an endpoint by sending the endpoint a confirmation message. To actually create a
     * subscription, the endpoint owner must call the <code>ConfirmSubscription</code> action with the token from the
     * confirmation message. Confirmation tokens are valid for three days.
     * </p>
     * <p>
     * This action is throttled at 100 transactions per second (TPS).
     * </p>
     *
     * @param subscribeRequest
     *        Input for Subscribe action.
     * @return A Java Future containing the result of the Subscribe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubscriptionLimitExceededException Indicates that the customer already owns the maximum allowed
     *         number of subscriptions.</li>
     *         <li>FilterPolicyLimitExceededException Indicates that the number of filter polices in your AWS account
     *         exceeds the limit. To add more filter polices, submit an SNS Limit Increase case in the AWS Support
     *         Center.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.Subscribe
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Subscribe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SubscribeResponse> subscribe(SubscribeRequest subscribeRequest) {
        try {

            HttpResponseHandler<SubscribeResponse> responseHandler = protocolFactory
                    .createResponseHandler(SubscribeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<SubscribeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SubscribeRequest, SubscribeResponse>().withOperationName("Subscribe")
                            .withMarshaller(new SubscribeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(subscribeRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add tags to the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon SNS Developer
     * Guide</i>.
     * </p>
     * <p>
     * When you use topic tags, keep the following guidelines in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Adding more than 50 tags to a topic isn't recommended.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags don't have any semantic meaning. Amazon SNS interprets tags as character strings.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * A new tag with a key identical to that of an existing tag overwrites the existing tag.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tagging actions are limited to 10 TPS per AWS account, per AWS region. If your application requires a higher
     * throughput, file a <a
     * href="https://console.aws.amazon.com/support/home#/case/create?issueType=technical">technical support
     * request</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Can't tag resource. Verify that the topic exists.</li>
     *         <li>TagLimitExceededException Can't add more than 50 tags to a topic.</li>
     *         <li>StaleTagException A tag has been added to a resource with the same ARN as a deleted resource. Wait a
     *         short while and then retry the operation.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the
     * subscription or the topic's owner can unsubscribe, and an AWS signature is required. If the
     * <code>Unsubscribe</code> call does not require authentication and the requester is not the subscription owner, a
     * final cancellation message is delivered to the endpoint, so that the endpoint owner can easily resubscribe to the
     * topic if the <code>Unsubscribe</code> request was unintended.
     * </p>
     * <p>
     * This action is throttled at 100 transactions per second (TPS).
     * </p>
     *
     * @param unsubscribeRequest
     *        Input for Unsubscribe action.
     * @return A Java Future containing the result of the Unsubscribe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>NotFoundException Indicates that the requested resource does not exist.</li>
     *         <li>InvalidSecurityException The credential signature isn't valid. You must use an HTTPS endpoint and
     *         sign your request using Signature Version 4.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.Unsubscribe
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Unsubscribe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UnsubscribeResponse> unsubscribe(UnsubscribeRequest unsubscribeRequest) {
        try {

            HttpResponseHandler<UnsubscribeResponse> responseHandler = protocolFactory
                    .createResponseHandler(UnsubscribeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UnsubscribeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnsubscribeRequest, UnsubscribeResponse>()
                            .withOperationName("Unsubscribe").withMarshaller(new UnsubscribeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(unsubscribeRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove tags from the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon SNS Developer
     * Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Can't tag resource. Verify that the topic exists.</li>
     *         <li>TagLimitExceededException Can't add more than 50 tags to a topic.</li>
     *         <li>StaleTagException A tag has been added to a resource with the same ARN as a deleted resource. Wait a
     *         short while and then retry the operation.</li>
     *         <li>TagPolicyException The request doesn't comply with the IAM tag policy. Correct your request and then
     *         retry it.</li>
     *         <li>InvalidParameterException Indicates that a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>AuthorizationErrorException Indicates that the user has been denied access to the requested resource.
     *         </li>
     *         <li>ConcurrentAccessException Can't perform multiple operations on a tag simultaneously. Perform the
     *         operations sequentially.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SnsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SnsAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubscriptionLimitExceeded")
                                .exceptionBuilderSupplier(SubscriptionLimitExceededException::builder).httpStatusCode(403)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSThrottling")
                                .exceptionBuilderSupplier(KmsThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameter")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterValueInvalid")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EndpointDisabled")
                                .exceptionBuilderSupplier(EndpointDisabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FilterPolicyLimitExceeded")
                                .exceptionBuilderSupplier(FilterPolicyLimitExceededException::builder).httpStatusCode(403)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInvalidState")
                                .exceptionBuilderSupplier(KmsInvalidStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSNotFound")
                                .exceptionBuilderSupplier(KmsNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSOptInRequired")
                                .exceptionBuilderSupplier(KmsOptInRequiredException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StaleTag").exceptionBuilderSupplier(StaleTagException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSAccessDenied")
                                .exceptionBuilderSupplier(KmsAccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFound").exceptionBuilderSupplier(NotFoundException::builder)
                                .httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSDisabled")
                                .exceptionBuilderSupplier(KmsDisabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TopicLimitExceeded")
                                .exceptionBuilderSupplier(TopicLimitExceededException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicy").exceptionBuilderSupplier(TagPolicyException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceeded")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("Throttled").exceptionBuilderSupplier(ThrottledException::builder)
                                .httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PlatformApplicationDisabled")
                                .exceptionBuilderSupplier(PlatformApplicationDisabledException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalError")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationError")
                                .exceptionBuilderSupplier(AuthorizationErrorException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentAccess")
                                .exceptionBuilderSupplier(ConcurrentAccessException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSecurity")
                                .exceptionBuilderSupplier(InvalidSecurityException::builder).httpStatusCode(403).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(SnsException::builder).build();
    }

    private <T extends SnsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
