/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.sns.model.AddPermissionRequest;
import software.amazon.awssdk.services.sns.model.AddPermissionResponse;
import software.amazon.awssdk.services.sns.model.AuthorizationErrorException;
import software.amazon.awssdk.services.sns.model.CheckIfPhoneNumberIsOptedOutRequest;
import software.amazon.awssdk.services.sns.model.CheckIfPhoneNumberIsOptedOutResponse;
import software.amazon.awssdk.services.sns.model.ConcurrentAccessException;
import software.amazon.awssdk.services.sns.model.ConfirmSubscriptionRequest;
import software.amazon.awssdk.services.sns.model.ConfirmSubscriptionResponse;
import software.amazon.awssdk.services.sns.model.CreatePlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.CreatePlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.CreatePlatformEndpointRequest;
import software.amazon.awssdk.services.sns.model.CreatePlatformEndpointResponse;
import software.amazon.awssdk.services.sns.model.CreateTopicRequest;
import software.amazon.awssdk.services.sns.model.CreateTopicResponse;
import software.amazon.awssdk.services.sns.model.DeleteEndpointRequest;
import software.amazon.awssdk.services.sns.model.DeleteEndpointResponse;
import software.amazon.awssdk.services.sns.model.DeletePlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.DeletePlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.DeleteTopicRequest;
import software.amazon.awssdk.services.sns.model.DeleteTopicResponse;
import software.amazon.awssdk.services.sns.model.EndpointDisabledException;
import software.amazon.awssdk.services.sns.model.FilterPolicyLimitExceededException;
import software.amazon.awssdk.services.sns.model.GetEndpointAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetEndpointAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetPlatformApplicationAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetPlatformApplicationAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetSmsAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetSmsAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetSubscriptionAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetSubscriptionAttributesResponse;
import software.amazon.awssdk.services.sns.model.GetTopicAttributesRequest;
import software.amazon.awssdk.services.sns.model.GetTopicAttributesResponse;
import software.amazon.awssdk.services.sns.model.InternalErrorException;
import software.amazon.awssdk.services.sns.model.InvalidParameterException;
import software.amazon.awssdk.services.sns.model.InvalidParameterValueException;
import software.amazon.awssdk.services.sns.model.InvalidSecurityException;
import software.amazon.awssdk.services.sns.model.KmsAccessDeniedException;
import software.amazon.awssdk.services.sns.model.KmsDisabledException;
import software.amazon.awssdk.services.sns.model.KmsInvalidStateException;
import software.amazon.awssdk.services.sns.model.KmsNotFoundException;
import software.amazon.awssdk.services.sns.model.KmsOptInRequiredException;
import software.amazon.awssdk.services.sns.model.KmsThrottlingException;
import software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationRequest;
import software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationResponse;
import software.amazon.awssdk.services.sns.model.ListPhoneNumbersOptedOutRequest;
import software.amazon.awssdk.services.sns.model.ListPhoneNumbersOptedOutResponse;
import software.amazon.awssdk.services.sns.model.ListPlatformApplicationsRequest;
import software.amazon.awssdk.services.sns.model.ListPlatformApplicationsResponse;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicRequest;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicResponse;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsRequest;
import software.amazon.awssdk.services.sns.model.ListSubscriptionsResponse;
import software.amazon.awssdk.services.sns.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.sns.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.sns.model.ListTopicsRequest;
import software.amazon.awssdk.services.sns.model.ListTopicsResponse;
import software.amazon.awssdk.services.sns.model.NotFoundException;
import software.amazon.awssdk.services.sns.model.OptInPhoneNumberRequest;
import software.amazon.awssdk.services.sns.model.OptInPhoneNumberResponse;
import software.amazon.awssdk.services.sns.model.PlatformApplicationDisabledException;
import software.amazon.awssdk.services.sns.model.PublishRequest;
import software.amazon.awssdk.services.sns.model.PublishResponse;
import software.amazon.awssdk.services.sns.model.RemovePermissionRequest;
import software.amazon.awssdk.services.sns.model.RemovePermissionResponse;
import software.amazon.awssdk.services.sns.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sns.model.SetEndpointAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetEndpointAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetPlatformApplicationAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetPlatformApplicationAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetSmsAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetSmsAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetSubscriptionAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetSubscriptionAttributesResponse;
import software.amazon.awssdk.services.sns.model.SetTopicAttributesRequest;
import software.amazon.awssdk.services.sns.model.SetTopicAttributesResponse;
import software.amazon.awssdk.services.sns.model.SnsException;
import software.amazon.awssdk.services.sns.model.SnsRequest;
import software.amazon.awssdk.services.sns.model.StaleTagException;
import software.amazon.awssdk.services.sns.model.SubscribeRequest;
import software.amazon.awssdk.services.sns.model.SubscribeResponse;
import software.amazon.awssdk.services.sns.model.SubscriptionLimitExceededException;
import software.amazon.awssdk.services.sns.model.TagLimitExceededException;
import software.amazon.awssdk.services.sns.model.TagPolicyException;
import software.amazon.awssdk.services.sns.model.TagResourceRequest;
import software.amazon.awssdk.services.sns.model.TagResourceResponse;
import software.amazon.awssdk.services.sns.model.ThrottledException;
import software.amazon.awssdk.services.sns.model.TopicLimitExceededException;
import software.amazon.awssdk.services.sns.model.UnsubscribeRequest;
import software.amazon.awssdk.services.sns.model.UnsubscribeResponse;
import software.amazon.awssdk.services.sns.model.UntagResourceRequest;
import software.amazon.awssdk.services.sns.model.UntagResourceResponse;
import software.amazon.awssdk.services.sns.paginators.ListEndpointsByPlatformApplicationIterable;
import software.amazon.awssdk.services.sns.paginators.ListPlatformApplicationsIterable;
import software.amazon.awssdk.services.sns.paginators.ListSubscriptionsByTopicIterable;
import software.amazon.awssdk.services.sns.paginators.ListSubscriptionsIterable;
import software.amazon.awssdk.services.sns.paginators.ListTopicsIterable;
import software.amazon.awssdk.services.sns.transform.AddPermissionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CheckIfPhoneNumberIsOptedOutRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ConfirmSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreatePlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreatePlatformEndpointRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.CreateTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeleteEndpointRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeletePlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.DeleteTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetEndpointAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetPlatformApplicationAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetSmsAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetSubscriptionAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.GetTopicAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListEndpointsByPlatformApplicationRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListPhoneNumbersOptedOutRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListPlatformApplicationsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListSubscriptionsByTopicRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.ListTopicsRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.OptInPhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.PublishRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetEndpointAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetPlatformApplicationAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetSmsAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetSubscriptionAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SetTopicAttributesRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.SubscribeRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.UnsubscribeRequestMarshaller;
import software.amazon.awssdk.services.sns.transform.UntagResourceRequestMarshaller;

/**
 * Internal implementation of {@link SnsClient}.
 *
 * @see SnsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSnsClient implements SnsClient {
    private final SyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSnsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the
     * specified actions.
     * </p>
     *
     * @param addPermissionRequest
     * @return Result of the AddPermission operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.AddPermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/AddPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AddPermissionResponse addPermission(AddPermissionRequest addPermissionRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, NotFoundException, AwsServiceException, SdkClientException,
            SnsException {

        HttpResponseHandler<AddPermissionResponse> responseHandler = protocolFactory
                .createResponseHandler(AddPermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<AddPermissionRequest, AddPermissionResponse>()
                .withOperationName("AddPermission").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(addPermissionRequest)
                .withMarshaller(new AddPermissionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your
     * account. You cannot send SMS messages to a number that is opted out.
     * </p>
     * <p>
     * To resume sending messages, you can opt in the number by using the <code>OptInPhoneNumber</code> action.
     * </p>
     *
     * @param checkIfPhoneNumberIsOptedOutRequest
     *        The input for the <code>CheckIfPhoneNumberIsOptedOut</code> action.
     * @return Result of the CheckIfPhoneNumberIsOptedOut operation returned by the service.
     * @throws ThrottledException
     *         Indicates that the rate at which requests have been submitted for this action exceeds the limit for your
     *         account.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.CheckIfPhoneNumberIsOptedOut
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CheckIfPhoneNumberIsOptedOut"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CheckIfPhoneNumberIsOptedOutResponse checkIfPhoneNumberIsOptedOut(
            CheckIfPhoneNumberIsOptedOutRequest checkIfPhoneNumberIsOptedOutRequest) throws ThrottledException,
            InternalErrorException, AuthorizationErrorException, InvalidParameterException, AwsServiceException,
            SdkClientException, SnsException {

        HttpResponseHandler<CheckIfPhoneNumberIsOptedOutResponse> responseHandler = protocolFactory
                .createResponseHandler(CheckIfPhoneNumberIsOptedOutResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<CheckIfPhoneNumberIsOptedOutRequest, CheckIfPhoneNumberIsOptedOutResponse>()
                        .withOperationName("CheckIfPhoneNumberIsOptedOut").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(checkIfPhoneNumberIsOptedOutRequest)
                        .withMarshaller(new CheckIfPhoneNumberIsOptedOutRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an
     * earlier <code>Subscribe</code> action. If the token is valid, the action creates a new subscription and returns
     * its Amazon Resource Name (ARN). This call requires an AWS signature only when the
     * <code>AuthenticateOnUnsubscribe</code> flag is set to "true".
     * </p>
     *
     * @param confirmSubscriptionRequest
     *        Input for ConfirmSubscription action.
     * @return Result of the ConfirmSubscription operation returned by the service.
     * @throws SubscriptionLimitExceededException
     *         Indicates that the customer already owns the maximum allowed number of subscriptions.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws FilterPolicyLimitExceededException
     *         Indicates that the number of filter polices in your AWS account exceeds the limit. To add more filter
     *         polices, submit an SNS Limit Increase case in the AWS Support Center.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ConfirmSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ConfirmSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ConfirmSubscriptionResponse confirmSubscription(ConfirmSubscriptionRequest confirmSubscriptionRequest)
            throws SubscriptionLimitExceededException, InvalidParameterException, NotFoundException, InternalErrorException,
            AuthorizationErrorException, FilterPolicyLimitExceededException, AwsServiceException, SdkClientException,
            SnsException {

        HttpResponseHandler<ConfirmSubscriptionResponse> responseHandler = protocolFactory
                .createResponseHandler(ConfirmSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ConfirmSubscriptionRequest, ConfirmSubscriptionResponse>()
                .withOperationName("ConfirmSubscription").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(confirmSubscriptionRequest)
                .withMarshaller(new ConfirmSubscriptionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a platform application object for one of the supported push notification services, such as APNS and FCM,
     * to which devices and mobile apps may register. You must specify PlatformPrincipal and PlatformCredential
     * attributes when using the <code>CreatePlatformApplication</code> action. The PlatformPrincipal is received from
     * the notification service. For APNS/APNS_SANDBOX, PlatformPrincipal is "SSL certificate". For FCM,
     * PlatformPrincipal is not applicable. For ADM, PlatformPrincipal is "client id". The PlatformCredential is also
     * received from the notification service. For WNS, PlatformPrincipal is "Package Security Identifier". For MPNS,
     * PlatformPrincipal is "TLS certificate". For Baidu, PlatformPrincipal is "API key".
     * </p>
     * <p>
     * For APNS/APNS_SANDBOX, PlatformCredential is "private key". For FCM, PlatformCredential is "API key". For ADM,
     * PlatformCredential is "client secret". For WNS, PlatformCredential is "secret key". For MPNS, PlatformCredential
     * is "private key". For Baidu, PlatformCredential is "secret key". The PlatformApplicationArn that is returned when
     * using <code>CreatePlatformApplication</code> is then used as an attribute for the
     * <code>CreatePlatformEndpoint</code> action.
     * </p>
     *
     * @param createPlatformApplicationRequest
     *        Input for CreatePlatformApplication action.
     * @return Result of the CreatePlatformApplication operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.CreatePlatformApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreatePlatformApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreatePlatformApplicationResponse createPlatformApplication(
            CreatePlatformApplicationRequest createPlatformApplicationRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<CreatePlatformApplicationResponse> responseHandler = protocolFactory
                .createResponseHandler(CreatePlatformApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<CreatePlatformApplicationRequest, CreatePlatformApplicationResponse>()
                        .withOperationName("CreatePlatformApplication").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createPlatformApplicationRequest)
                        .withMarshaller(new CreatePlatformApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an endpoint for a device and mobile app on one of the supported push notification services, such as FCM
     * and APNS. <code>CreatePlatformEndpoint</code> requires the PlatformApplicationArn that is returned from
     * <code>CreatePlatformApplication</code>. The EndpointArn that is returned when using
     * <code>CreatePlatformEndpoint</code> can then be used by the <code>Publish</code> action to send a message to a
     * mobile app or by the <code>Subscribe</code> action for subscription to a topic. The
     * <code>CreatePlatformEndpoint</code> action is idempotent, so if the requester already owns an endpoint with the
     * same device token and attributes, that endpoint's ARN is returned without creating a new endpoint. For more
     * information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile
     * Push Notifications</a>.
     * </p>
     * <p>
     * When using <code>CreatePlatformEndpoint</code> with Baidu, two attributes must be provided: ChannelId and UserId.
     * The token field must also contain the ChannelId. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePushBaiduEndpoint.html">Creating an Amazon SNS Endpoint
     * for Baidu</a>.
     * </p>
     *
     * @param createPlatformEndpointRequest
     *        Input for CreatePlatformEndpoint action.
     * @return Result of the CreatePlatformEndpoint operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.CreatePlatformEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreatePlatformEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePlatformEndpointResponse createPlatformEndpoint(CreatePlatformEndpointRequest createPlatformEndpointRequest)
            throws InvalidParameterException, InternalErrorException, AuthorizationErrorException, NotFoundException,
            AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<CreatePlatformEndpointResponse> responseHandler = protocolFactory
                .createResponseHandler(CreatePlatformEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreatePlatformEndpointRequest, CreatePlatformEndpointResponse>()
                .withOperationName("CreatePlatformEndpoint").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createPlatformEndpointRequest)
                .withMarshaller(new CreatePlatformEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a topic to which notifications can be published. Users can create at most 100,000 topics. For more
     * information, see <a href="http://aws.amazon.com/sns/">https://aws.amazon.com/sns</a>. This action is idempotent,
     * so if the requester already owns a topic with the specified name, that topic's ARN is returned without creating a
     * new topic.
     * </p>
     *
     * @param createTopicRequest
     *        Input for CreateTopic action.
     * @return Result of the CreateTopic operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws TopicLimitExceededException
     *         Indicates that the customer already owns the maximum allowed number of topics.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws InvalidSecurityException
     *         The credential signature isn't valid. You must use an HTTPS endpoint and sign your request using
     *         Signature Version 4.
     * @throws TagLimitExceededException
     *         Can't add more than 50 tags to a topic.
     * @throws StaleTagException
     *         A tag has been added to a resource with the same ARN as a deleted resource. Wait a short while and then
     *         retry the operation.
     * @throws TagPolicyException
     *         The request doesn't comply with the IAM tag policy. Correct your request and then retry it.
     * @throws ConcurrentAccessException
     *         Can't perform multiple operations on a tag simultaneously. Perform the operations sequentially.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.CreateTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/CreateTopic" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTopicResponse createTopic(CreateTopicRequest createTopicRequest) throws InvalidParameterException,
            TopicLimitExceededException, InternalErrorException, AuthorizationErrorException, InvalidSecurityException,
            TagLimitExceededException, StaleTagException, TagPolicyException, ConcurrentAccessException, AwsServiceException,
            SdkClientException, SnsException {

        HttpResponseHandler<CreateTopicResponse> responseHandler = protocolFactory
                .createResponseHandler(CreateTopicResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateTopicRequest, CreateTopicResponse>()
                .withOperationName("CreateTopic").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createTopicRequest)
                .withMarshaller(new CreateTopicRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the endpoint for a device and mobile app from Amazon SNS. This action is idempotent. For more
     * information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile
     * Push Notifications</a>.
     * </p>
     * <p>
     * When you delete an endpoint that is also subscribed to a topic, then you must also unsubscribe the endpoint from
     * the topic.
     * </p>
     *
     * @param deleteEndpointRequest
     *        Input for DeleteEndpoint action.
     * @return Result of the DeleteEndpoint operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.DeleteEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeleteEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteEndpointResponse deleteEndpoint(DeleteEndpointRequest deleteEndpointRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<DeleteEndpointResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteEndpointRequest, DeleteEndpointResponse>()
                .withOperationName("DeleteEndpoint").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteEndpointRequest)
                .withMarshaller(new DeleteEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a platform application object for one of the supported push notification services, such as APNS and FCM.
     * For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS
     * Mobile Push Notifications</a>.
     * </p>
     *
     * @param deletePlatformApplicationRequest
     *        Input for DeletePlatformApplication action.
     * @return Result of the DeletePlatformApplication operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.DeletePlatformApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeletePlatformApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeletePlatformApplicationResponse deletePlatformApplication(
            DeletePlatformApplicationRequest deletePlatformApplicationRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<DeletePlatformApplicationResponse> responseHandler = protocolFactory
                .createResponseHandler(DeletePlatformApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<DeletePlatformApplicationRequest, DeletePlatformApplicationResponse>()
                        .withOperationName("DeletePlatformApplication").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deletePlatformApplicationRequest)
                        .withMarshaller(new DeletePlatformApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a topic and all its subscriptions. Deleting a topic might prevent some messages previously sent to the
     * topic from being delivered to subscribers. This action is idempotent, so deleting a topic that does not exist
     * does not result in an error.
     * </p>
     *
     * @param deleteTopicRequest
     * @return Result of the DeleteTopic operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws StaleTagException
     *         A tag has been added to a resource with the same ARN as a deleted resource. Wait a short while and then
     *         retry the operation.
     * @throws TagPolicyException
     *         The request doesn't comply with the IAM tag policy. Correct your request and then retry it.
     * @throws ConcurrentAccessException
     *         Can't perform multiple operations on a tag simultaneously. Perform the operations sequentially.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.DeleteTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/DeleteTopic" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTopicResponse deleteTopic(DeleteTopicRequest deleteTopicRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, NotFoundException, StaleTagException, TagPolicyException,
            ConcurrentAccessException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<DeleteTopicResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteTopicResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteTopicRequest, DeleteTopicResponse>()
                .withOperationName("DeleteTopic").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteTopicRequest)
                .withMarshaller(new DeleteTopicRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the endpoint attributes for a device on one of the supported push notification services, such as FCM
     * and APNS. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>.
     * </p>
     *
     * @param getEndpointAttributesRequest
     *        Input for GetEndpointAttributes action.
     * @return Result of the GetEndpointAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.GetEndpointAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetEndpointAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetEndpointAttributesResponse getEndpointAttributes(GetEndpointAttributesRequest getEndpointAttributesRequest)
            throws InvalidParameterException, InternalErrorException, AuthorizationErrorException, NotFoundException,
            AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<GetEndpointAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(GetEndpointAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetEndpointAttributesRequest, GetEndpointAttributesResponse>()
                .withOperationName("GetEndpointAttributes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getEndpointAttributesRequest)
                .withMarshaller(new GetEndpointAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the attributes of the platform application object for the supported push notification services, such as
     * APNS and FCM. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     *
     * @param getPlatformApplicationAttributesRequest
     *        Input for GetPlatformApplicationAttributes action.
     * @return Result of the GetPlatformApplicationAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.GetPlatformApplicationAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetPlatformApplicationAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPlatformApplicationAttributesResponse getPlatformApplicationAttributes(
            GetPlatformApplicationAttributesRequest getPlatformApplicationAttributesRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, NotFoundException, AwsServiceException, SdkClientException,
            SnsException {

        HttpResponseHandler<GetPlatformApplicationAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(GetPlatformApplicationAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<GetPlatformApplicationAttributesRequest, GetPlatformApplicationAttributesResponse>()
                        .withOperationName("GetPlatformApplicationAttributes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getPlatformApplicationAttributesRequest)
                        .withMarshaller(new GetPlatformApplicationAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the settings for sending SMS messages from your account.
     * </p>
     * <p>
     * These settings are set with the <code>SetSMSAttributes</code> action.
     * </p>
     *
     * @param getSmsAttributesRequest
     *        The input for the <code>GetSMSAttributes</code> request.
     * @return Result of the GetSMSAttributes operation returned by the service.
     * @throws ThrottledException
     *         Indicates that the rate at which requests have been submitted for this action exceeds the limit for your
     *         account.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.GetSMSAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetSMSAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSmsAttributesResponse getSMSAttributes(GetSmsAttributesRequest getSmsAttributesRequest) throws ThrottledException,
            InternalErrorException, AuthorizationErrorException, InvalidParameterException, AwsServiceException,
            SdkClientException, SnsException {

        HttpResponseHandler<GetSmsAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(GetSmsAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetSmsAttributesRequest, GetSmsAttributesResponse>()
                .withOperationName("GetSMSAttributes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getSmsAttributesRequest)
                .withMarshaller(new GetSmsAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns all of the properties of a subscription.
     * </p>
     *
     * @param getSubscriptionAttributesRequest
     *        Input for GetSubscriptionAttributes.
     * @return Result of the GetSubscriptionAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.GetSubscriptionAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetSubscriptionAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSubscriptionAttributesResponse getSubscriptionAttributes(
            GetSubscriptionAttributesRequest getSubscriptionAttributesRequest) throws InvalidParameterException,
            InternalErrorException, NotFoundException, AuthorizationErrorException, AwsServiceException, SdkClientException,
            SnsException {

        HttpResponseHandler<GetSubscriptionAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(GetSubscriptionAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<GetSubscriptionAttributesRequest, GetSubscriptionAttributesResponse>()
                        .withOperationName("GetSubscriptionAttributes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getSubscriptionAttributesRequest)
                        .withMarshaller(new GetSubscriptionAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of
     * the user.
     * </p>
     *
     * @param getTopicAttributesRequest
     *        Input for GetTopicAttributes action.
     * @return Result of the GetTopicAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws InvalidSecurityException
     *         The credential signature isn't valid. You must use an HTTPS endpoint and sign your request using
     *         Signature Version 4.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.GetTopicAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/GetTopicAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTopicAttributesResponse getTopicAttributes(GetTopicAttributesRequest getTopicAttributesRequest)
            throws InvalidParameterException, InternalErrorException, NotFoundException, AuthorizationErrorException,
            InvalidSecurityException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<GetTopicAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(GetTopicAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetTopicAttributesRequest, GetTopicAttributesResponse>()
                .withOperationName("GetTopicAttributes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getTopicAttributesRequest)
                .withMarshaller(new GetTopicAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as FCM and
     * APNS. The results for <code>ListEndpointsByPlatformApplication</code> are paginated and return a limited list of
     * endpoints, up to 100. If additional records are available after the first page results, then a NextToken string
     * will be returned. To receive the next page, you call <code>ListEndpointsByPlatformApplication</code> again using
     * the NextToken string received from the previous call. When there are no more records to return, NextToken will be
     * null. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listEndpointsByPlatformApplicationRequest
     *        Input for ListEndpointsByPlatformApplication action.
     * @return Result of the ListEndpointsByPlatformApplication operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListEndpointsByPlatformApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListEndpointsByPlatformApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEndpointsByPlatformApplicationResponse listEndpointsByPlatformApplication(
            ListEndpointsByPlatformApplicationRequest listEndpointsByPlatformApplicationRequest)
            throws InvalidParameterException, InternalErrorException, AuthorizationErrorException, NotFoundException,
            AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<ListEndpointsByPlatformApplicationResponse> responseHandler = protocolFactory
                .createResponseHandler(ListEndpointsByPlatformApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<ListEndpointsByPlatformApplicationRequest, ListEndpointsByPlatformApplicationResponse>()
                        .withOperationName("ListEndpointsByPlatformApplication").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listEndpointsByPlatformApplicationRequest)
                        .withMarshaller(new ListEndpointsByPlatformApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as FCM and
     * APNS. The results for <code>ListEndpointsByPlatformApplication</code> are paginated and return a limited list of
     * endpoints, up to 100. If additional records are available after the first page results, then a NextToken string
     * will be returned. To receive the next page, you call <code>ListEndpointsByPlatformApplication</code> again using
     * the NextToken string received from the previous call. When there are no more records to return, NextToken will be
     * null. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEndpointsByPlatformApplication(software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListEndpointsByPlatformApplicationIterable responses = client.listEndpointsByPlatformApplicationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sns.paginators.ListEndpointsByPlatformApplicationIterable responses = client
     *             .listEndpointsByPlatformApplicationPaginator(request);
     *     for (software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListEndpointsByPlatformApplicationIterable responses = client.listEndpointsByPlatformApplicationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEndpointsByPlatformApplication(software.amazon.awssdk.services.sns.model.ListEndpointsByPlatformApplicationRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEndpointsByPlatformApplicationRequest
     *        Input for ListEndpointsByPlatformApplication action.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListEndpointsByPlatformApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListEndpointsByPlatformApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEndpointsByPlatformApplicationIterable listEndpointsByPlatformApplicationPaginator(
            ListEndpointsByPlatformApplicationRequest listEndpointsByPlatformApplicationRequest)
            throws InvalidParameterException, InternalErrorException, AuthorizationErrorException, NotFoundException,
            AwsServiceException, SdkClientException, SnsException {
        return new ListEndpointsByPlatformApplicationIterable(this,
                applyPaginatorUserAgent(listEndpointsByPlatformApplicationRequest));
    }

    /**
     * <p>
     * Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.
     * </p>
     * <p>
     * The results for <code>ListPhoneNumbersOptedOut</code> are paginated, and each page returns up to 100 phone
     * numbers. If additional phone numbers are available after the first page of results, then a <code>NextToken</code>
     * string will be returned. To receive the next page, you call <code>ListPhoneNumbersOptedOut</code> again using the
     * <code>NextToken</code> string received from the previous call. When there are no more records to return,
     * <code>NextToken</code> will be null.
     * </p>
     *
     * @param listPhoneNumbersOptedOutRequest
     *        The input for the <code>ListPhoneNumbersOptedOut</code> action.
     * @return Result of the ListPhoneNumbersOptedOut operation returned by the service.
     * @throws ThrottledException
     *         Indicates that the rate at which requests have been submitted for this action exceeds the limit for your
     *         account.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListPhoneNumbersOptedOut
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListPhoneNumbersOptedOut" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPhoneNumbersOptedOutResponse listPhoneNumbersOptedOut(
            ListPhoneNumbersOptedOutRequest listPhoneNumbersOptedOutRequest) throws ThrottledException, InternalErrorException,
            AuthorizationErrorException, InvalidParameterException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<ListPhoneNumbersOptedOutResponse> responseHandler = protocolFactory
                .createResponseHandler(ListPhoneNumbersOptedOutResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<ListPhoneNumbersOptedOutRequest, ListPhoneNumbersOptedOutResponse>()
                        .withOperationName("ListPhoneNumbersOptedOut").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listPhoneNumbersOptedOutRequest)
                        .withMarshaller(new ListPhoneNumbersOptedOutRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the platform application objects for the supported push notification services, such as APNS and FCM. The
     * results for <code>ListPlatformApplications</code> are paginated and return a limited list of applications, up to
     * 100. If additional records are available after the first page results, then a NextToken string will be returned.
     * To receive the next page, you call <code>ListPlatformApplications</code> using the NextToken string received from
     * the previous call. When there are no more records to return, NextToken will be null. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 15 transactions per second (TPS).
     * </p>
     *
     * @param listPlatformApplicationsRequest
     *        Input for ListPlatformApplications action.
     * @return Result of the ListPlatformApplications operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListPlatformApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListPlatformApplications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPlatformApplicationsResponse listPlatformApplications(
            ListPlatformApplicationsRequest listPlatformApplicationsRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<ListPlatformApplicationsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListPlatformApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<ListPlatformApplicationsRequest, ListPlatformApplicationsResponse>()
                        .withOperationName("ListPlatformApplications").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listPlatformApplicationsRequest)
                        .withMarshaller(new ListPlatformApplicationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the platform application objects for the supported push notification services, such as APNS and FCM. The
     * results for <code>ListPlatformApplications</code> are paginated and return a limited list of applications, up to
     * 100. If additional records are available after the first page results, then a NextToken string will be returned.
     * To receive the next page, you call <code>ListPlatformApplications</code> using the NextToken string received from
     * the previous call. When there are no more records to return, NextToken will be null. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using Amazon SNS Mobile Push
     * Notifications</a>.
     * </p>
     * <p>
     * This action is throttled at 15 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPlatformApplications(software.amazon.awssdk.services.sns.model.ListPlatformApplicationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListPlatformApplicationsIterable responses = client.listPlatformApplicationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sns.paginators.ListPlatformApplicationsIterable responses = client
     *             .listPlatformApplicationsPaginator(request);
     *     for (software.amazon.awssdk.services.sns.model.ListPlatformApplicationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListPlatformApplicationsIterable responses = client.listPlatformApplicationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlatformApplications(software.amazon.awssdk.services.sns.model.ListPlatformApplicationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPlatformApplicationsRequest
     *        Input for ListPlatformApplications action.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListPlatformApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListPlatformApplications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPlatformApplicationsIterable listPlatformApplicationsPaginator(
            ListPlatformApplicationsRequest listPlatformApplicationsRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, AwsServiceException, SdkClientException, SnsException {
        return new ListPlatformApplicationsIterable(this, applyPaginatorUserAgent(listPlatformApplicationsRequest));
    }

    /**
     * <p>
     * Returns a list of the requester's subscriptions. Each call returns a limited list of subscriptions, up to 100. If
     * there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter
     * in a new <code>ListSubscriptions</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listSubscriptionsRequest
     *        Input for ListSubscriptions action.
     * @return Result of the ListSubscriptions operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListSubscriptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSubscriptionsResponse listSubscriptions(ListSubscriptionsRequest listSubscriptionsRequest)
            throws InvalidParameterException, InternalErrorException, AuthorizationErrorException, AwsServiceException,
            SdkClientException, SnsException {

        HttpResponseHandler<ListSubscriptionsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListSubscriptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListSubscriptionsRequest, ListSubscriptionsResponse>()
                .withOperationName("ListSubscriptions").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listSubscriptionsRequest)
                .withMarshaller(new ListSubscriptionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the requester's subscriptions. Each call returns a limited list of subscriptions, up to 100. If
     * there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter
     * in a new <code>ListSubscriptions</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSubscriptions(software.amazon.awssdk.services.sns.model.ListSubscriptionsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListSubscriptionsIterable responses = client.listSubscriptionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sns.paginators.ListSubscriptionsIterable responses = client
     *             .listSubscriptionsPaginator(request);
     *     for (software.amazon.awssdk.services.sns.model.ListSubscriptionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListSubscriptionsIterable responses = client.listSubscriptionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSubscriptions(software.amazon.awssdk.services.sns.model.ListSubscriptionsRequest)} operation.</b>
     * </p>
     *
     * @param listSubscriptionsRequest
     *        Input for ListSubscriptions action.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListSubscriptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSubscriptionsIterable listSubscriptionsPaginator(ListSubscriptionsRequest listSubscriptionsRequest)
            throws InvalidParameterException, InternalErrorException, AuthorizationErrorException, AwsServiceException,
            SdkClientException, SnsException {
        return new ListSubscriptionsIterable(this, applyPaginatorUserAgent(listSubscriptionsRequest));
    }

    /**
     * <p>
     * Returns a list of the subscriptions to a specific topic. Each call returns a limited list of subscriptions, up to
     * 100. If there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code>
     * parameter in a new <code>ListSubscriptionsByTopic</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listSubscriptionsByTopicRequest
     *        Input for ListSubscriptionsByTopic action.
     * @return Result of the ListSubscriptionsByTopic operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListSubscriptionsByTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptionsByTopic" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSubscriptionsByTopicResponse listSubscriptionsByTopic(
            ListSubscriptionsByTopicRequest listSubscriptionsByTopicRequest) throws InvalidParameterException,
            InternalErrorException, NotFoundException, AuthorizationErrorException, AwsServiceException, SdkClientException,
            SnsException {

        HttpResponseHandler<ListSubscriptionsByTopicResponse> responseHandler = protocolFactory
                .createResponseHandler(ListSubscriptionsByTopicResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<ListSubscriptionsByTopicRequest, ListSubscriptionsByTopicResponse>()
                        .withOperationName("ListSubscriptionsByTopic").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listSubscriptionsByTopicRequest)
                        .withMarshaller(new ListSubscriptionsByTopicRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the subscriptions to a specific topic. Each call returns a limited list of subscriptions, up to
     * 100. If there are more subscriptions, a <code>NextToken</code> is also returned. Use the <code>NextToken</code>
     * parameter in a new <code>ListSubscriptionsByTopic</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSubscriptionsByTopic(software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListSubscriptionsByTopicIterable responses = client.listSubscriptionsByTopicPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sns.paginators.ListSubscriptionsByTopicIterable responses = client
     *             .listSubscriptionsByTopicPaginator(request);
     *     for (software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListSubscriptionsByTopicIterable responses = client.listSubscriptionsByTopicPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSubscriptionsByTopic(software.amazon.awssdk.services.sns.model.ListSubscriptionsByTopicRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSubscriptionsByTopicRequest
     *        Input for ListSubscriptionsByTopic action.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListSubscriptionsByTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListSubscriptionsByTopic" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSubscriptionsByTopicIterable listSubscriptionsByTopicPaginator(
            ListSubscriptionsByTopicRequest listSubscriptionsByTopicRequest) throws InvalidParameterException,
            InternalErrorException, NotFoundException, AuthorizationErrorException, AwsServiceException, SdkClientException,
            SnsException {
        return new ListSubscriptionsByTopicIterable(this, applyPaginatorUserAgent(listSubscriptionsByTopicRequest));
    }

    /**
     * <p>
     * List all tags added to the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon Simple
     * Notification Service Developer Guide</i>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Can't tag resource. Verify that the topic exists.
     * @throws TagPolicyException
     *         The request doesn't comply with the IAM tag policy. Correct your request and then retry it.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws ConcurrentAccessException
     *         Can't perform multiple operations on a tag simultaneously. Perform the operations sequentially.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, TagPolicyException, InvalidParameterException, AuthorizationErrorException,
            ConcurrentAccessException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100. If there are
     * more topics, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter in a new
     * <code>ListTopics</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     *
     * @param listTopicsRequest
     * @return Result of the ListTopics operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListTopics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListTopics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTopicsResponse listTopics(ListTopicsRequest listTopicsRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<ListTopicsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListTopicsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListTopicsRequest, ListTopicsResponse>()
                .withOperationName("ListTopics").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTopicsRequest)
                .withMarshaller(new ListTopicsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100. If there are
     * more topics, a <code>NextToken</code> is also returned. Use the <code>NextToken</code> parameter in a new
     * <code>ListTopics</code> call to get further results.
     * </p>
     * <p>
     * This action is throttled at 30 transactions per second (TPS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTopics(software.amazon.awssdk.services.sns.model.ListTopicsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListTopicsIterable responses = client.listTopicsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sns.paginators.ListTopicsIterable responses = client.listTopicsPaginator(request);
     *     for (software.amazon.awssdk.services.sns.model.ListTopicsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sns.paginators.ListTopicsIterable responses = client.listTopicsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTopics(software.amazon.awssdk.services.sns.model.ListTopicsRequest)} operation.</b>
     * </p>
     *
     * @param listTopicsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.ListTopics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/ListTopics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTopicsIterable listTopicsPaginator(ListTopicsRequest listTopicsRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, AwsServiceException, SdkClientException, SnsException {
        return new ListTopicsIterable(this, applyPaginatorUserAgent(listTopicsRequest));
    }

    /**
     * <p>
     * Use this request to opt in a phone number that is opted out, which enables you to resume sending SMS messages to
     * the number.
     * </p>
     * <p>
     * You can opt in a phone number only once every 30 days.
     * </p>
     *
     * @param optInPhoneNumberRequest
     *        Input for the OptInPhoneNumber action.
     * @return Result of the OptInPhoneNumber operation returned by the service.
     * @throws ThrottledException
     *         Indicates that the rate at which requests have been submitted for this action exceeds the limit for your
     *         account.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.OptInPhoneNumber
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/OptInPhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public OptInPhoneNumberResponse optInPhoneNumber(OptInPhoneNumberRequest optInPhoneNumberRequest) throws ThrottledException,
            InternalErrorException, AuthorizationErrorException, InvalidParameterException, AwsServiceException,
            SdkClientException, SnsException {

        HttpResponseHandler<OptInPhoneNumberResponse> responseHandler = protocolFactory
                .createResponseHandler(OptInPhoneNumberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<OptInPhoneNumberRequest, OptInPhoneNumberResponse>()
                .withOperationName("OptInPhoneNumber").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(optInPhoneNumberRequest)
                .withMarshaller(new OptInPhoneNumberRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sends a message to an Amazon SNS topic or sends a text message (SMS message) directly to a phone number.
     * </p>
     * <p>
     * If you send a message to a topic, Amazon SNS delivers the message to each endpoint that is subscribed to the
     * topic. The format of the message depends on the notification protocol for each subscribed endpoint.
     * </p>
     * <p>
     * When a <code>messageId</code> is returned, the message has been saved and Amazon SNS will attempt to deliver it
     * shortly.
     * </p>
     * <p>
     * To use the <code>Publish</code> action for sending a message to a mobile endpoint, such as an app on a Kindle
     * device or mobile phone, you must specify the EndpointArn for the TargetArn parameter. The EndpointArn is returned
     * when making a call with the <code>CreatePlatformEndpoint</code> action.
     * </p>
     * <p>
     * For more information about formatting messages, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/mobile-push-send-custommessage.html">Send Custom
     * Platform-Specific Payloads in Messages to Mobile Devices</a>.
     * </p>
     *
     * @param publishRequest
     *        Input for Publish action.
     * @return Result of the Publish operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InvalidParameterValueException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws EndpointDisabledException
     *         Exception error indicating endpoint disabled.
     * @throws PlatformApplicationDisabledException
     *         Exception error indicating platform application disabled.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws KmsDisabledException
     *         The request was rejected because the specified customer master key (CMK) isn't enabled.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource isn't valid for this request. For
     *         more information, see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How
     *         Key State Affects Use of a Customer Master Key</a> in the <i>AWS Key Management Service Developer
     *         Guide</i>.
     * @throws KmsNotFoundException
     *         The request was rejected because the specified entity or resource can't be found.
     * @throws KmsOptInRequiredException
     *         The AWS access key ID needs a subscription for the service.
     * @throws KmsThrottlingException
     *         The request was denied due to request throttling. For more information about throttling, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#requests-per-second">Limits</a>
     *         in the <i>AWS Key Management Service Developer Guide.</i>
     * @throws KmsAccessDeniedException
     *         The ciphertext references a key that doesn't exist or that you don't have access to.
     * @throws InvalidSecurityException
     *         The credential signature isn't valid. You must use an HTTPS endpoint and sign your request using
     *         Signature Version 4.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.Publish
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Publish" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PublishResponse publish(PublishRequest publishRequest) throws InvalidParameterException,
            InvalidParameterValueException, InternalErrorException, NotFoundException, EndpointDisabledException,
            PlatformApplicationDisabledException, AuthorizationErrorException, KmsDisabledException, KmsInvalidStateException,
            KmsNotFoundException, KmsOptInRequiredException, KmsThrottlingException, KmsAccessDeniedException,
            InvalidSecurityException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<PublishResponse> responseHandler = protocolFactory.createResponseHandler(PublishResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PublishRequest, PublishResponse>().withOperationName("Publish")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(publishRequest)
                .withMarshaller(new PublishRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes a statement from a topic's access control policy.
     * </p>
     *
     * @param removePermissionRequest
     *        Input for RemovePermission action.
     * @return Result of the RemovePermission operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.RemovePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RemovePermissionResponse removePermission(RemovePermissionRequest removePermissionRequest)
            throws InvalidParameterException, InternalErrorException, AuthorizationErrorException, NotFoundException,
            AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory
                .createResponseHandler(RemovePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                .withOperationName("RemovePermission").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(removePermissionRequest)
                .withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the attributes for an endpoint for a device on one of the supported push notification services, such as FCM
     * and APNS. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>.
     * </p>
     *
     * @param setEndpointAttributesRequest
     *        Input for SetEndpointAttributes action.
     * @return Result of the SetEndpointAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.SetEndpointAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetEndpointAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetEndpointAttributesResponse setEndpointAttributes(SetEndpointAttributesRequest setEndpointAttributesRequest)
            throws InvalidParameterException, InternalErrorException, AuthorizationErrorException, NotFoundException,
            AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<SetEndpointAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(SetEndpointAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SetEndpointAttributesRequest, SetEndpointAttributesResponse>()
                .withOperationName("SetEndpointAttributes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(setEndpointAttributesRequest)
                .withMarshaller(new SetEndpointAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the attributes of the platform application object for the supported push notification services, such as APNS
     * and FCM. For more information, see <a href="https://docs.aws.amazon.com/sns/latest/dg/SNSMobilePush.html">Using
     * Amazon SNS Mobile Push Notifications</a>. For information on configuring attributes for message delivery status,
     * see <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-msg-status.html">Using Amazon SNS Application
     * Attributes for Message Delivery Status</a>.
     * </p>
     *
     * @param setPlatformApplicationAttributesRequest
     *        Input for SetPlatformApplicationAttributes action.
     * @return Result of the SetPlatformApplicationAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.SetPlatformApplicationAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetPlatformApplicationAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetPlatformApplicationAttributesResponse setPlatformApplicationAttributes(
            SetPlatformApplicationAttributesRequest setPlatformApplicationAttributesRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, NotFoundException, AwsServiceException, SdkClientException,
            SnsException {

        HttpResponseHandler<SetPlatformApplicationAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(SetPlatformApplicationAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<SetPlatformApplicationAttributesRequest, SetPlatformApplicationAttributesResponse>()
                        .withOperationName("SetPlatformApplicationAttributes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(setPlatformApplicationAttributesRequest)
                        .withMarshaller(new SetPlatformApplicationAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Use this request to set the default settings for sending SMS messages and receiving daily SMS usage reports.
     * </p>
     * <p>
     * You can override some of these settings for a single message when you use the <code>Publish</code> action with
     * the <code>MessageAttributes.entry.N</code> parameter. For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sms_publish-to-phone.html">Sending an SMS Message</a> in the
     * <i>Amazon SNS Developer Guide</i>.
     * </p>
     *
     * @param setSmsAttributesRequest
     *        The input for the SetSMSAttributes action.
     * @return Result of the SetSMSAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws ThrottledException
     *         Indicates that the rate at which requests have been submitted for this action exceeds the limit for your
     *         account.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.SetSMSAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetSMSAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetSmsAttributesResponse setSMSAttributes(SetSmsAttributesRequest setSmsAttributesRequest)
            throws InvalidParameterException, ThrottledException, InternalErrorException, AuthorizationErrorException,
            AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<SetSmsAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(SetSmsAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SetSmsAttributesRequest, SetSmsAttributesResponse>()
                .withOperationName("SetSMSAttributes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(setSmsAttributesRequest)
                .withMarshaller(new SetSmsAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Allows a subscription owner to set an attribute of the subscription to a new value.
     * </p>
     *
     * @param setSubscriptionAttributesRequest
     *        Input for SetSubscriptionAttributes action.
     * @return Result of the SetSubscriptionAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws FilterPolicyLimitExceededException
     *         Indicates that the number of filter polices in your AWS account exceeds the limit. To add more filter
     *         polices, submit an SNS Limit Increase case in the AWS Support Center.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.SetSubscriptionAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetSubscriptionAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public SetSubscriptionAttributesResponse setSubscriptionAttributes(
            SetSubscriptionAttributesRequest setSubscriptionAttributesRequest) throws InvalidParameterException,
            FilterPolicyLimitExceededException, InternalErrorException, NotFoundException, AuthorizationErrorException,
            AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<SetSubscriptionAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(SetSubscriptionAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<SetSubscriptionAttributesRequest, SetSubscriptionAttributesResponse>()
                        .withOperationName("SetSubscriptionAttributes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(setSubscriptionAttributesRequest)
                        .withMarshaller(new SetSubscriptionAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Allows a topic owner to set an attribute of the topic to a new value.
     * </p>
     *
     * @param setTopicAttributesRequest
     *        Input for SetTopicAttributes action.
     * @return Result of the SetTopicAttributes operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws InvalidSecurityException
     *         The credential signature isn't valid. You must use an HTTPS endpoint and sign your request using
     *         Signature Version 4.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.SetTopicAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/SetTopicAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetTopicAttributesResponse setTopicAttributes(SetTopicAttributesRequest setTopicAttributesRequest)
            throws InvalidParameterException, InternalErrorException, NotFoundException, AuthorizationErrorException,
            InvalidSecurityException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<SetTopicAttributesResponse> responseHandler = protocolFactory
                .createResponseHandler(SetTopicAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SetTopicAttributesRequest, SetTopicAttributesResponse>()
                .withOperationName("SetTopicAttributes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(setTopicAttributesRequest)
                .withMarshaller(new SetTopicAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Prepares to subscribe an endpoint by sending the endpoint a confirmation message. To actually create a
     * subscription, the endpoint owner must call the <code>ConfirmSubscription</code> action with the token from the
     * confirmation message. Confirmation tokens are valid for three days.
     * </p>
     * <p>
     * This action is throttled at 100 transactions per second (TPS).
     * </p>
     *
     * @param subscribeRequest
     *        Input for Subscribe action.
     * @return Result of the Subscribe operation returned by the service.
     * @throws SubscriptionLimitExceededException
     *         Indicates that the customer already owns the maximum allowed number of subscriptions.
     * @throws FilterPolicyLimitExceededException
     *         Indicates that the number of filter polices in your AWS account exceeds the limit. To add more filter
     *         polices, submit an SNS Limit Increase case in the AWS Support Center.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws InvalidSecurityException
     *         The credential signature isn't valid. You must use an HTTPS endpoint and sign your request using
     *         Signature Version 4.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.Subscribe
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Subscribe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SubscribeResponse subscribe(SubscribeRequest subscribeRequest) throws SubscriptionLimitExceededException,
            FilterPolicyLimitExceededException, InvalidParameterException, InternalErrorException, NotFoundException,
            AuthorizationErrorException, InvalidSecurityException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<SubscribeResponse> responseHandler = protocolFactory
                .createResponseHandler(SubscribeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SubscribeRequest, SubscribeResponse>()
                .withOperationName("Subscribe").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(subscribeRequest)
                .withMarshaller(new SubscribeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Add tags to the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon SNS Developer
     * Guide</i>.
     * </p>
     * <p>
     * When you use topic tags, keep the following guidelines in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Adding more than 50 tags to a topic isn't recommended.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags don't have any semantic meaning. Amazon SNS interprets tags as character strings.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * A new tag with a key identical to that of an existing tag overwrites the existing tag.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tagging actions are limited to 10 TPS per AWS account, per AWS region. If your application requires a higher
     * throughput, file a <a
     * href="https://console.aws.amazon.com/support/home#/case/create?issueType=technical">technical support
     * request</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Can't tag resource. Verify that the topic exists.
     * @throws TagLimitExceededException
     *         Can't add more than 50 tags to a topic.
     * @throws StaleTagException
     *         A tag has been added to a resource with the same ARN as a deleted resource. Wait a short while and then
     *         retry the operation.
     * @throws TagPolicyException
     *         The request doesn't comply with the IAM tag policy. Correct your request and then retry it.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws ConcurrentAccessException
     *         Can't perform multiple operations on a tag simultaneously. Perform the operations sequentially.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            TagLimitExceededException, StaleTagException, TagPolicyException, InvalidParameterException,
            AuthorizationErrorException, ConcurrentAccessException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the
     * subscription or the topic's owner can unsubscribe, and an AWS signature is required. If the
     * <code>Unsubscribe</code> call does not require authentication and the requester is not the subscription owner, a
     * final cancellation message is delivered to the endpoint, so that the endpoint owner can easily resubscribe to the
     * topic if the <code>Unsubscribe</code> request was unintended.
     * </p>
     * <p>
     * This action is throttled at 100 transactions per second (TPS).
     * </p>
     *
     * @param unsubscribeRequest
     *        Input for Unsubscribe action.
     * @return Result of the Unsubscribe operation returned by the service.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws NotFoundException
     *         Indicates that the requested resource does not exist.
     * @throws InvalidSecurityException
     *         The credential signature isn't valid. You must use an HTTPS endpoint and sign your request using
     *         Signature Version 4.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.Unsubscribe
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/Unsubscribe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UnsubscribeResponse unsubscribe(UnsubscribeRequest unsubscribeRequest) throws InvalidParameterException,
            InternalErrorException, AuthorizationErrorException, NotFoundException, InvalidSecurityException,
            AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<UnsubscribeResponse> responseHandler = protocolFactory
                .createResponseHandler(UnsubscribeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UnsubscribeRequest, UnsubscribeResponse>()
                .withOperationName("Unsubscribe").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(unsubscribeRequest)
                .withMarshaller(new UnsubscribeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Remove tags from the specified Amazon SNS topic. For an overview, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-tags.html">Amazon SNS Tags</a> in the <i>Amazon SNS Developer
     * Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Can't tag resource. Verify that the topic exists.
     * @throws TagLimitExceededException
     *         Can't add more than 50 tags to a topic.
     * @throws StaleTagException
     *         A tag has been added to a resource with the same ARN as a deleted resource. Wait a short while and then
     *         retry the operation.
     * @throws TagPolicyException
     *         The request doesn't comply with the IAM tag policy. Correct your request and then retry it.
     * @throws InvalidParameterException
     *         Indicates that a request parameter does not comply with the associated constraints.
     * @throws AuthorizationErrorException
     *         Indicates that the user has been denied access to the requested resource.
     * @throws ConcurrentAccessException
     *         Can't perform multiple operations on a tag simultaneously. Perform the operations sequentially.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SnsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SnsClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/sns-2010-03-31/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            TagLimitExceededException, StaleTagException, TagPolicyException, InvalidParameterException,
            AuthorizationErrorException, ConcurrentAccessException, AwsServiceException, SdkClientException, SnsException {

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubscriptionLimitExceeded")
                                .exceptionBuilderSupplier(SubscriptionLimitExceededException::builder).httpStatusCode(403)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSThrottling")
                                .exceptionBuilderSupplier(KmsThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameter")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterValueInvalid")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EndpointDisabled")
                                .exceptionBuilderSupplier(EndpointDisabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FilterPolicyLimitExceeded")
                                .exceptionBuilderSupplier(FilterPolicyLimitExceededException::builder).httpStatusCode(403)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInvalidState")
                                .exceptionBuilderSupplier(KmsInvalidStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSNotFound")
                                .exceptionBuilderSupplier(KmsNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSOptInRequired")
                                .exceptionBuilderSupplier(KmsOptInRequiredException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StaleTag").exceptionBuilderSupplier(StaleTagException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSAccessDenied")
                                .exceptionBuilderSupplier(KmsAccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFound").exceptionBuilderSupplier(NotFoundException::builder)
                                .httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSDisabled")
                                .exceptionBuilderSupplier(KmsDisabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TopicLimitExceeded")
                                .exceptionBuilderSupplier(TopicLimitExceededException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicy").exceptionBuilderSupplier(TagPolicyException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceeded")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("Throttled").exceptionBuilderSupplier(ThrottledException::builder)
                                .httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PlatformApplicationDisabled")
                                .exceptionBuilderSupplier(PlatformApplicationDisabledException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalError")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationError")
                                .exceptionBuilderSupplier(AuthorizationErrorException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentAccess")
                                .exceptionBuilderSupplier(ConcurrentAccessException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSecurity")
                                .exceptionBuilderSupplier(InvalidSecurityException::builder).httpStatusCode(403).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(SnsException::builder).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SnsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
