/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Input for CreateTopic action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateTopicRequest extends SnsRequest implements
        ToCopyableBuilder<CreateTopicRequest.Builder, CreateTopicRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTopicRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(CreateTopicRequest::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateTopicRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ATTRIBUTES_FIELD,
            TAGS_FIELD));

    private final String name;

    private final Map<String, String> attributes;

    private final List<Tag> tags;

    private CreateTopicRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.attributes = builder.attributes;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the topic you want to create.
     * </p>
     * <p>
     * Constraints: Topic names must be made up of only uppercase and lowercase ASCII letters, numbers, underscores, and
     * hyphens, and must be between 1 and 256 characters long.
     * </p>
     * 
     * @return The name of the topic you want to create.</p>
     *         <p>
     *         Constraints: Topic names must be made up of only uppercase and lowercase ASCII letters, numbers,
     *         underscores, and hyphens, and must be between 1 and 256 characters long.
     */
    public String name() {
        return name;
    }

    /**
     * Returns true if the Attributes property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of attributes with their corresponding values.
     * </p>
     * <p>
     * The following lists the names, descriptions, and values of the special request parameters that the
     * <code>CreateTopic</code> action uses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
     * endpoints.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DisplayName</code> – The display name to use for a topic with SMS subscriptions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Policy</code> – The policy that defines who can access your topic. By default, only the topic owner can
     * publish or subscribe to the topic.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following attribute applies only to <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html">server-side-encryption</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KmsMasterKeyId</code> - The ID of an AWS-managed customer master key (CMK) for Amazon SNS or a custom CMK.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html#sse-key-terms">Key Terms</a>. For
     * more examples, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
     * >KeyId</a> in the <i>AWS Key Management Service API Reference</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of attributes with their corresponding values.</p>
     *         <p>
     *         The following lists the names, descriptions, and values of the special request parameters that the
     *         <code>CreateTopic</code> action uses:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
     *         endpoints.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DisplayName</code> – The display name to use for a topic with SMS subscriptions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Policy</code> – The policy that defines who can access your topic. By default, only the topic owner
     *         can publish or subscribe to the topic.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following attribute applies only to <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html"
     *         >server-side-encryption</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KmsMasterKeyId</code> - The ID of an AWS-managed customer master key (CMK) for Amazon SNS or a
     *         custom CMK. For more information, see <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html#sse-key-terms">Key
     *         Terms</a>. For more examples, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
     *         >KeyId</a> in the <i>AWS Key Management Service API Reference</i>.
     *         </p>
     *         </li>
     */
    public Map<String, String> attributes() {
        return attributes;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of tags to add to a new topic.
     * </p>
     * <note>
     * <p>
     * To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
     * <code>sns:TagResource</code> permissions.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of tags to add to a new topic.</p> <note>
     *         <p>
     *         To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
     *         <code>sns:TagResource</code> permissions.
     *         </p>
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(attributes());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateTopicRequest)) {
            return false;
        }
        CreateTopicRequest other = (CreateTopicRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(attributes(), other.attributes())
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateTopicRequest").add("Name", name()).add("Attributes", attributes()).add("Tags", tags())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateTopicRequest, T> g) {
        return obj -> g.apply((CreateTopicRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SnsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateTopicRequest> {
        /**
         * <p>
         * The name of the topic you want to create.
         * </p>
         * <p>
         * Constraints: Topic names must be made up of only uppercase and lowercase ASCII letters, numbers, underscores,
         * and hyphens, and must be between 1 and 256 characters long.
         * </p>
         * 
         * @param name
         *        The name of the topic you want to create.</p>
         *        <p>
         *        Constraints: Topic names must be made up of only uppercase and lowercase ASCII letters, numbers,
         *        underscores, and hyphens, and must be between 1 and 256 characters long.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A map of attributes with their corresponding values.
         * </p>
         * <p>
         * The following lists the names, descriptions, and values of the special request parameters that the
         * <code>CreateTopic</code> action uses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to HTTP/S
         * endpoints.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DisplayName</code> – The display name to use for a topic with SMS subscriptions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Policy</code> – The policy that defines who can access your topic. By default, only the topic owner can
         * publish or subscribe to the topic.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following attribute applies only to <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html">server-side-encryption</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KmsMasterKeyId</code> - The ID of an AWS-managed customer master key (CMK) for Amazon SNS or a custom
         * CMK. For more information, see <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html#sse-key-terms">Key Terms</a>.
         * For more examples, see <a href=
         * "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
         * >KeyId</a> in the <i>AWS Key Management Service API Reference</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributes
         *        A map of attributes with their corresponding values.</p>
         *        <p>
         *        The following lists the names, descriptions, and values of the special request parameters that the
         *        <code>CreateTopic</code> action uses:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DeliveryPolicy</code> – The policy that defines how Amazon SNS retries failed deliveries to
         *        HTTP/S endpoints.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DisplayName</code> – The display name to use for a topic with SMS subscriptions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Policy</code> – The policy that defines who can access your topic. By default, only the topic
         *        owner can publish or subscribe to the topic.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following attribute applies only to <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html"
         *        >server-side-encryption</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KmsMasterKeyId</code> - The ID of an AWS-managed customer master key (CMK) for Amazon SNS or a
         *        custom CMK. For more information, see <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/sns-server-side-encryption.html#sse-key-terms">Key
         *        Terms</a>. For more examples, see <a href=
         *        "https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html#API_DescribeKey_RequestParameters"
         *        >KeyId</a> in the <i>AWS Key Management Service API Reference</i>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        /**
         * <p>
         * The list of tags to add to a new topic.
         * </p>
         * <note>
         * <p>
         * To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         * <code>sns:TagResource</code> permissions.
         * </p>
         * </note>
         * 
         * @param tags
         *        The list of tags to add to a new topic.</p> <note>
         *        <p>
         *        To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         *        <code>sns:TagResource</code> permissions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The list of tags to add to a new topic.
         * </p>
         * <note>
         * <p>
         * To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         * <code>sns:TagResource</code> permissions.
         * </p>
         * </note>
         * 
         * @param tags
         *        The list of tags to add to a new topic.</p> <note>
         *        <p>
         *        To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         *        <code>sns:TagResource</code> permissions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The list of tags to add to a new topic.
         * </p>
         * <note>
         * <p>
         * To be able to tag a topic on creation, you must have the <code>sns:CreateTopic</code> and
         * <code>sns:TagResource</code> permissions.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to
         * create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SnsRequest.BuilderImpl implements Builder {
        private String name;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateTopicRequest model) {
            super(model);
            name(model.name);
            attributes(model.attributes);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Map<String, String> getAttributes() {
            return attributes;
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = TopicAttributesMapCopier.copy(attributes);
            return this;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = TopicAttributesMapCopier.copy(attributes);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateTopicRequest build() {
            return new CreateTopicRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
